/* PonG-Edit: A gui editor for the pong files
 * Author: George Lebl
 * (c) 2000 Eazel, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include <gnome.h>
#include <glade/glade.h>
#include <pong/pong.h>
#include <pong/pong-glade.h>
#include <pong/pongelement.h>
#include <pong/pongparser.h>
#include <pong/pongutil.h>
#include <pong/pong-type.h>
#include <pong/pong-strings.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

#include "tree.h"
#include "glade-helper.h"
#include "pong-edit.h"
#include "util.h"

#include "pane-arguments.h"

static XMLTag *selected_argument = NULL;
static XMLTag *selected_widget = NULL;

void pane_arguments_clist_select_row	(GtkCList *clist,
					 int row, int column,
					 GdkEvent *event);
void pane_arguments_clist_unselect_row	(GtkCList *clist,
					 int row, int column,
					 GdkEvent *event);

void change_pane_argument_name		(GtkWidget *entry);
void change_pane_argument_value		(GtkWidget *entry);
void change_pane_argument_translatable	(GtkWidget *cb);

void pane_argument_add			(GtkWidget *w);
void pane_argument_remove		(GtkWidget *w);

static GList *per_list_connections = NULL;

static void
pane_argument_name_updated (XMLTag *tag,
			    const char *key,
			    const char *text,
			    gpointer data)
{
	int row;

	row = pong_edit_find_tag_row (GTK_CLIST (pane_arguments_clist), tag);

	if (row >= 0) {
		gtk_clist_set_text (GTK_CLIST (pane_arguments_clist),
				    row, 0, text);
	}
}

static void
pane_argument_value_updated (XMLTag *tag,
			     const char *key,
			     const char *text,
			     gpointer data)
{
	int row;

	row = pong_edit_find_tag_row (GTK_CLIST (pane_arguments_clist), tag);

	if (row >= 0) {
		gtk_clist_set_text (GTK_CLIST (pane_arguments_clist),
				    row, 1, text);
	}
}

static void
pane_argument_translatable_updated (XMLTag *tag,
				    const char *key,
				    const char *text,
				    gpointer data)
{
	int row;

	row = pong_edit_find_tag_row (GTK_CLIST (pane_arguments_clist), tag);

	if (row >= 0) {
		gboolean value;

		if (text != NULL)
			value = pong_bool_from_string (text);
		else
			value = FALSE;

		gtk_clist_set_text (GTK_CLIST (pane_arguments_clist),
				    row, 2, value ? "True" : "False");
	}
}

static void
pane_argument_killed (XMLTag *tag, gpointer data)
{
	int row;

	row = pong_edit_find_tag_row (GTK_CLIST (pane_arguments_clist), tag);

	if (row >= 0) {
		gtk_clist_remove (GTK_CLIST (pane_arguments_clist), row);
	}
}

static int
add_argument_to_clist (XMLTag *argument)
{
	int row;
	char *text[3];
	char *name, *value, *translatable;
	gboolean bool;
	int con_id;

	name = tag_get_text (argument, PONG_S_Name, NULL);
	value = tag_get_text (argument, PONG_S_Value, NULL);

	bool = tag_get_bool (argument, PONG_S_Translatable, FALSE);
	translatable = bool ? "True" : "False";

	text[0] = pong_sure_string (name);
	text[1] = pong_sure_string (value);
	text[2] = pong_sure_string (translatable);

	row = gtk_clist_append (GTK_CLIST (pane_arguments_clist), text);

	g_free (name);
	g_free (value);

	gtk_clist_set_row_data_full (GTK_CLIST (pane_arguments_clist), row,
				     tag_ref (argument),
				     (GtkDestroyNotify)tag_unref);

	con_id = tag_connect_text (argument, PONG_S_Name,
				   pane_argument_name_updated, NULL, NULL);
	connection_add (&per_list_connections, argument, con_id);
	con_id = tag_connect_text (argument, PONG_S_Value,
				   pane_argument_value_updated, NULL, NULL);
	connection_add (&per_list_connections, argument, con_id);
	con_id = tag_connect_text (argument, PONG_S_Translatable,
				   pane_argument_translatable_updated,
				   NULL, NULL);
	connection_add (&per_list_connections, argument, con_id);
	con_id = tag_connect_kill (argument, pane_argument_killed, NULL, NULL);
	connection_add (&per_list_connections, argument, con_id);

	return row;
}

static void
pane_argument_added (XMLTag *root, XMLTag *reference, gpointer data)
{
	const char *name = tag_peek_type (reference);

	if (name != NULL &&
	    pong_strcasecmp_no_locale (name, PONG_S_Argument) == 0) {
		add_argument_to_clist (reference);
	}
}

void
setup_pane_arguments_list (XMLTag *widget)
{
	GList *list, *li;
	int con_id;

	selected_widget = widget;

	gtk_clist_freeze (GTK_CLIST (pane_arguments_clist));

	list = tag_get_tags (widget, PONG_S_Argument);

	for (li = list; li != NULL; li = li->next) {
		XMLTag *tag = li->data;

		add_argument_to_clist (tag);
	}

	tree_free_tag_list (list);

	gtk_clist_thaw (GTK_CLIST (pane_arguments_clist));

	con_id = tag_connect_add (widget, pane_argument_added, NULL, NULL);
	connection_add (&per_list_connections, widget, con_id);
}

void
clear_pane_arguments_list (void)
{
	GtkWidget *w;

	selected_widget = NULL;

	connection_disconnect_all (&per_list_connections);

	gtk_clist_unselect_all (GTK_CLIST (pane_arguments_clist));
	gtk_clist_clear (GTK_CLIST (pane_arguments_clist));

	w = glade_helper_get (app_xml, "pane_argument_edit_table",
			      GTK_TYPE_TABLE);
	gtk_widget_set_sensitive (w, FALSE);
}

void
change_pane_argument_value (GtkWidget *entry)
{
	char *text;
	const char *key;

	if (selected_argument == NULL)
		return;

	text = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);

	if (text != NULL)
		g_strstrip (text);

	if (tag_get_bool (selected_argument, PONG_S_Translatable, FALSE))
		key = "_" PONG_S_Value;
	else
		key = PONG_S_Value;

	if (pong_string_empty (text))
		tag_set_text (selected_argument, key, NULL, NULL);
	else
		tag_set_text (selected_argument, key, text, NULL);

	g_free (text);
}

void
change_pane_argument_translatable (GtkWidget *cb)
{
	gboolean value;
	GList *list, *li;

	if (selected_argument == NULL)
		return;

	value = GTK_TOGGLE_BUTTON (cb)->active;

	tag_set_bool (selected_argument, PONG_S_Translatable, value);
	list = tag_get_tags (selected_argument, PONG_S_Value);
	for (li = list; li != NULL; li = li->next) {
		XMLTag *tag = li->data;
		tag_set_translatable (tag, value);
	}
	tree_free_tag_list (list);
}

void
change_pane_argument_name (GtkWidget *entry)
{
	gboolean success;

	if (selected_argument == NULL)
		return;

	/* if we got this far there HAS to be a selected widget */
	g_assert (selected_widget != NULL);

	success = tree_set_argument_name (selected_widget, selected_argument,
					  gtk_entry_get_text (GTK_ENTRY (entry)));

	pong_edit_set_red (entry, success ? FALSE : TRUE);
}

static void
setup_pane_argument_edit_area (XMLTag *argument)
{
	GtkWidget *w;
	char *text;
	gboolean value;

	/* The argument name */
	w = glade_helper_get (app_xml, "pane_argument_name_entry",
			      GTK_TYPE_ENTRY);
	text = tag_get_text (argument, PONG_S_Name, NULL);
	gtk_entry_set_text (GTK_ENTRY (w), pong_sure_string (text));
	g_free (text);

	/* The argument value */
	w = glade_helper_get (app_xml, "pane_argument_value_entry",
			      GTK_TYPE_ENTRY);
	text = tag_get_text (argument, PONG_S_Value, NULL);
	gtk_entry_set_text (GTK_ENTRY (w), pong_sure_string (text));
	g_free (text);

	/* value translatable */
	w = glade_helper_get (app_xml, "pane_argument_translatable_cb",
			      GTK_TYPE_CHECK_BUTTON);
	value = tag_get_bool (argument, PONG_S_Translatable, FALSE);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), value);
}

static void
clear_pane_argument_edit_area (void)
{
	GtkWidget *w;

	/* The argument name */
	w = glade_helper_get (app_xml, "pane_argument_name_entry",
			      GTK_TYPE_ENTRY);
	gtk_entry_set_text (GTK_ENTRY (w), "");

	/* The argument value */
	w = glade_helper_get (app_xml, "pane_argument_value_entry",
			      GTK_TYPE_ENTRY);
	gtk_entry_set_text (GTK_ENTRY (w), "");

	/* value translatable */
	w = glade_helper_get (app_xml, "pane_argument_translatable_cb",
			      GTK_TYPE_CHECK_BUTTON);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), FALSE);
}

void
pane_arguments_clist_select_row (GtkCList *clist, int row, int column,
				 GdkEvent *event)
{
	GtkWidget *w;
	XMLTag *tag = gtk_clist_get_row_data (clist, row);

	selected_argument = NULL;

	w = glade_helper_get (app_xml, "pane_argument_edit_table",
			      GTK_TYPE_TABLE);
	gtk_widget_set_sensitive (w, TRUE);

	setup_pane_argument_edit_area (tag);
	selected_argument = tag;
}

void
pane_arguments_clist_unselect_row (GtkCList *clist, int row, int column,
				   GdkEvent *event)
{
	GtkWidget *w;

	selected_argument = NULL;
	clear_pane_argument_edit_area ();

	w = glade_helper_get (app_xml, "pane_argument_edit_table",
			      GTK_TYPE_TABLE);
	gtk_widget_set_sensitive (w, FALSE);
}

void
pane_argument_add (GtkWidget *w)
{
	XMLTag *tag;

	if (selected_widget != NULL)
		tag = tree_widget_add_argument (selected_widget);
	else
		tag = NULL;

	clist_select_thing (GTK_CLIST (pane_arguments_clist), tag);

	tag_unref (tag);
}

void
pane_argument_remove (GtkWidget *w)
{
	if (selected_argument != NULL)
		tag_kill (selected_argument);
}
