/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include <libgnome/libgnome.h>

#include <gmime/gmime.h>

#include <pan/base/debug.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/util-mime.h>
#include <pan/base/util-file.h>

/***
****
***/

/**
 * parse_uu_begin_line
 * @param line the line to check for "begin filename mode"
 * @param filename if parse is successful, is set with the
 *        starting character of the filename.
 * @param mode if parse is successful, mode is set with the
 *        mode requested by the line.
 * @return 0 on success, -1 on failure
 */
int
uu_parse_begin_line (const gchar  * b,
                     const gchar ** file,
                            gint  * mode)
{
	gint m = 0;

	/* skip to the permissions */
	while (*b==' ') ++b;
	if (!isdigit((int)*b)) return -1;

	/* parse the permissions */
	while (isdigit((int)*b)) {
		m = m*8 + (*b-'0');
		++b;
	};

	/* skip to the filename */
	if (*b!=' ') return -1;
	while (*b==' ') ++b;
	if (*b < ' ') return -1;

	/* return the results */
	if (mode!=NULL) *mode = m;
	if (file!=NULL) *file = b;
	return 0;
}

/**
 * uu_is_beginning
 * @param line line to test & see if it's the beginning of a uu-encoded block
 * @return true if it is, false otherwise
 */
gboolean
uu_is_beginning_line (const gchar * line)
{
	return line!=NULL
	 	&& (line[0]=='b' || line[0]=='B')
	 	&& (line[1]=='e' || line[1]=='E')
	 	&& (line[2]=='g' || line[2]=='G')
	 	&& (line[3]=='i' || line[3]=='I')
	 	&& (line[4]=='n' || line[4]=='N')
		&& line[5]==' '
		&& !uu_parse_begin_line (line+6, NULL, NULL);
}


/**
 * uu_is_ending
 * @param line line to test & see if it's the end of a uu-encoded block
 * @return true if it is, false otherwise
 */
gboolean
uu_is_ending_line (const gchar * line)
{
	return line!=NULL
	 	&& (line[0]=='e' || line[0]=='E')
	 	&& (line[1]=='n' || line[1]=='N')
	 	&& (line[2]=='d' || line[2]=='D')
	 	&& (line[3]=='\0' || line[3]=='\n' || line[3]==' ');
}

void
uu_get_file_info (const gchar       * begin,
                  gchar            ** setme_filename,
                  gulong            * setme_mode)
{
	gchar * filename;
	const gchar * end;

	g_return_if_fail (uu_is_beginning_line(begin));
	g_return_if_fail (setme_filename != NULL);
	g_return_if_fail (setme_mode != NULL);

	*setme_mode = strtol (begin+6, NULL, 8);
	
	begin += 10;
	end = strchr (begin, '\n');
	filename = g_strndup (begin, end-begin);
	g_strstrip (filename);
	*setme_filename = filename;
}

int
uu_get_char_len (gint octet_len)
{
	gint char_len = (octet_len / 3) * 4;
	switch (octet_len % 3) {
		case 0: break;
		case 1: char_len += 2;
		case 2: char_len += 3;
	}
	return octet_len;
}

gboolean
is_uu_line (const gchar * line, gint len)
{
	gint i;
	gint octet_len;
	gint char_len;

	g_return_val_if_fail (is_nonempty_string(line), FALSE);

	if (*line=='\0' || len<1)
		return FALSE;

	if (len==1 && *line=='`')
		return TRUE;

	/* get octet length */
	octet_len = *line - 0x20;
	if (octet_len > 45)
		return FALSE;

	/* get character length */
	char_len = uu_get_char_len (octet_len);
	if (char_len+1 > len)
		return FALSE;

	/* make sure each character is in the uuencoded range */
	for (i=0; i<char_len; ++i)
		if (line[i+1]<0x20 || line[i+1]>0x60)
			return FALSE;

	/* looks okay */
	return TRUE;
}

static gboolean
stream_readln (GMimeStream *stream, GByteArray *line, off_t* startpos)
{
	char linebuf[1024];
	ssize_t len;

	g_return_val_if_fail (stream!=NULL, FALSE);
	g_return_val_if_fail (line!=NULL, FALSE);
	g_return_val_if_fail (startpos!=NULL, FALSE);

	/* where are we now? */
	*startpos = g_mime_stream_tell (stream);

	/* fill the line array */
	g_byte_array_set_size (line, 0);
	do {
		len = g_mime_stream_buffer_gets (stream, linebuf, sizeof (linebuf));
		if (len > 0)
			g_byte_array_append (line, linebuf, len);
	} while (len>0 && linebuf[len-1]!='\n');

	return len > 0;
}

typedef struct
{
	off_t start;
	off_t end;
	gchar * uu_decoded_filename;
	gboolean is_uu;
}
UUTempFilePart;

static void
separate_uu_from_non_uu (GMimeStream * istream,
                         GPtrArray * appendme_uu_file_parts)
{
	UUTempFilePart * cur = NULL;
	gboolean uu = FALSE;
	GByteArray * line;
	off_t linestart_pos = 0;

	/* sanity clause */
	g_return_if_fail (istream!=NULL);
	g_return_if_fail (appendme_uu_file_parts!=NULL);

	line = g_byte_array_new ();
	while (stream_readln (istream, line, &linestart_pos))
	{
		const gchar * line_str = line->data;
		/* beginning of a UU part */
		if (!uu && uu_is_beginning_line(line_str))
		{
			gulong mode = 0;
			gchar * decoded_filename = NULL;

			if (cur != NULL) {
				cur->end = linestart_pos;
				g_ptr_array_add (appendme_uu_file_parts, cur);
				cur = NULL;
			}

			/* new entry */
			uu = TRUE;
			uu_get_file_info (line_str, &decoded_filename, &mode);
			cur = g_new0 (UUTempFilePart, 1);
			cur->start = g_mime_stream_tell (istream);
			cur->is_uu = TRUE;
			cur->uu_decoded_filename = decoded_filename;
		}

		/* ending of a UU part */
		else if (uu && uu_is_ending_line(line_str))
		{
			if (cur != NULL) {
				cur->end = linestart_pos;
				g_ptr_array_add (appendme_uu_file_parts, cur);
				cur = NULL;
			}
			uu = FALSE;
		}

		/* all other lines */
		else
		{
			/* new non-uu part */
			if (cur == NULL) {
				cur = g_new0 (UUTempFilePart, 1);
				cur->start = linestart_pos;
				cur->is_uu = FALSE;
				cur->uu_decoded_filename = NULL;
			}
		}
	}

	/* close old entry */
	if (cur != NULL) {
		cur->end = g_mime_stream_tell (istream);
		g_ptr_array_add (appendme_uu_file_parts, cur);
		cur = NULL;
	}

	g_byte_array_free (line, TRUE);
}

static void
guess_part_type_from_filename (const gchar   * filename,
                               gchar        ** setme_type,
                               gchar        ** setme_subtype)
{
	const gchar * type = gnome_mime_type (filename);
	const gchar * delimiter = strchr (type, '/');
	*setme_type = g_strndup (type, delimiter-type);
	*setme_subtype = g_strdup (delimiter+1);
}

static void
look_for_uuencoded_data (GMimePart * parent, gpointer data)
{
	gint i;
	guint content_len;
	GPtrArray * parts;
	GHashTable * skip = (GHashTable*) data;

	if (g_hash_table_lookup (skip, parent) != NULL)
		return;
	if (!g_mime_content_type_is_type (parent->mime_type, "text", "plain"))
		return;

	content_len = 0;
	parts = g_ptr_array_new ();

	/* convert into N temp files -- uu and text parts */
	if (parent->content)
	{
		const GMimeDataWrapper * content;
		GMimeStream * in_stream;
		GMimeStream * in_buf;
		
		content = g_mime_part_get_content_object (parent);
		in_stream = g_mime_data_wrapper_get_stream ((GMimeDataWrapper*)content);
		in_buf = g_mime_stream_buffer_new (in_stream, GMIME_STREAM_BUFFER_BLOCK_READ);
		g_mime_stream_unref (in_stream);
		in_stream = NULL;
		
		separate_uu_from_non_uu (in_buf, parts);
		g_mime_stream_reset (in_buf);

		/* split? */
		if (parts->len>1 || (parts->len==1 && ((UUTempFilePart*)g_ptr_array_index(parts,0))->is_uu))
		{
			/* recast this part as a multipart/mixed instead of a text/plain */
			if (1) {
				GMimeContentType * mime_type;
				g_mime_part_set_content_object (parent, NULL);
				mime_type = g_mime_content_type_new ("multipart", "mixed");
				g_mime_part_set_content_type (parent, mime_type);
			}

			/* add the children */
			for (i=0; i<parts->len; ++i)
			{
				const UUTempFilePart * part = g_ptr_array_index(parts,i);
				GMimePart * child = NULL;

				if (part->is_uu)
				{
					gchar * pch = NULL;
					gchar * type = NULL;
					gchar * subtype = NULL;
					GMimeDataWrapper * child_wrapper;
					GMimeStream * child_stream;
					GMimeStream * child_filter_stream;
					GMimeFilter * child_filter;

					/* create the part */
					guess_part_type_from_filename (part->uu_decoded_filename, &type, &subtype);
					child = g_mime_part_new_with_type (type, subtype);

					/* set part's attributes */
					g_mime_part_set_filename (child, part->uu_decoded_filename);
					pch = g_strdup_printf ("<%s>", part->uu_decoded_filename);
					g_mime_part_set_content_id (child, pch);
					g_free (pch);
					g_mime_part_set_content_disposition (child, "inline");

					/* set the part's content */
					child_stream = g_mime_stream_substream (in_buf, part->start, part->end);
					child_filter_stream = g_mime_stream_filter_new_with_stream (child_stream);
					g_mime_stream_unref (child_stream);
					
					child_filter = g_mime_filter_basic_new_type (GMIME_FILTER_BASIC_UU_DEC);
					g_mime_stream_filter_add (GMIME_STREAM_FILTER(child_filter_stream), child_filter);
					child_wrapper = g_mime_data_wrapper_new_with_stream (child_filter_stream, GMIME_PART_ENCODING_8BIT);
					g_mime_stream_unref (child_filter_stream);
					
					g_mime_part_set_content_object (child, child_wrapper);
					
					g_free (type);
					g_free (subtype);
				}
				else
				{
					GMimeStream * child_stream;
					GMimeDataWrapper * child_wrapper;

					child = g_mime_part_new_with_type ("text", "plain");
					child_stream = g_mime_stream_substream (in_buf, part->start, part->end);
					child_wrapper = g_mime_data_wrapper_new_with_stream (child_stream, GMIME_PART_ENCODING_8BIT);
					g_mime_part_set_content_object (child, child_wrapper);
					g_mime_stream_unref (child_stream);
				}

				g_hash_table_insert (skip, child, child);
				g_mime_part_add_subpart (parent, child);
			}
		}

		g_mime_stream_unref (in_buf);
	}

	/* remove parts */
	for (i=0; i<parts->len; ++i) {
		UUTempFilePart * part = g_ptr_array_index(parts,i);
		g_free (part->uu_decoded_filename);
		g_free (part);
	}
	g_ptr_array_free (parts, TRUE);
}

/***
****
***/

static GMimeMessage*
construct_message_impl (GMimeStream * istream)
{
	GMimeMessage * message;
	GHashTable * skip;

	message = g_mime_parser_construct_message (istream, TRUE);

	/* walk through all the parts to break uu from non-uu */
	skip = g_hash_table_new (g_direct_hash, g_direct_equal);
	g_mime_message_foreach_part (message, look_for_uuencoded_data, skip);
	g_hash_table_destroy (skip);

	return message;
}

GMimeMessage*
pan_g_mime_parser_construct_message (const gchar * data)
{
	GMimeStream * istream;
	GMimeMessage * retval;

	istream = g_mime_stream_mem_new_with_buffer (data, strlen(data));
	retval = construct_message_impl (istream);
	g_mime_stream_unref (istream);

	return retval;
}

GMimeMessage*
pan_g_mime_parser_construct_message_from_file (FILE * fp)
{
	GMimeStream * istream;
	GMimeStream * buf_istream;
	GMimeMessage * message;

	istream = g_mime_stream_file_new (fp);
	buf_istream = g_mime_stream_buffer_new (istream, GMIME_STREAM_BUFFER_BLOCK_READ);
       	message = construct_message_impl (buf_istream);

	g_mime_stream_unref (buf_istream);
	g_mime_stream_unref (istream);

	return message;
}
