/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 8-*- */

/*This file is part of GNU MlView
 *
 *GNU MlView is free software; 
 *you can redistribute it and/or modify it under the terms of 
 *the GNU General Public License as 
 *published by the Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *GNU MlView is distributed in the hope 
 *that it will be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. 
 *If not, write to the Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *See COPYRIGHT file for copyright information.
 */

#include "mlview-validator.h"

#define MESSAGE_LINE_LENGTH 55

static void
validation_message (GArray *a_messages,
                    const gchar *a_msg_format, ...)
{
        xmlErrorPtr xml_err = NULL;
        gchar *msg = NULL;
        MlViewValidationMessage *out = NULL;

        g_return_if_fail (a_messages);
        
        xml_err = xmlGetLastError ();
        
        g_return_if_fail (xml_err);
        
        if (xml_err->domain == XML_FROM_VALID ||
            xml_err->domain == XML_FROM_RELAXNGV ||
            xml_err->domain == XML_FROM_SCHEMASV) {
                msg = mlview_utils_normalize_text 
                        (xml_err->message, "\n", 
                         "\n", " ", MESSAGE_LINE_LENGTH);

                g_return_if_fail (msg);     

                out = g_try_malloc (sizeof (MlViewValidationMessage));

                if (!out)
                        goto cleanup;

                out->message = msg;
                out->priority = xml_err->level;
                out->node = xml_err->node;
                
                if (out->node)
                        out->type = out->node->type;

                a_messages = g_array_append_val (a_messages, out);

                if (!a_messages) 
                        goto cleanup;
        }

        return;
        
 cleanup:
        if (msg) {
                g_free (msg);
                msg = NULL;
        }

        if (out) {
                g_free (out);
                out = NULL;
        }
}

MlViewValidationOutput *
mlview_validator_validate_with_dtd (MlViewXMLDocument *a_doc, 
				    xmlDtdPtr a_dtd,
                                    MlViewValidatorStatus *a_state)
{
        MlViewValidationOutput *msgs = NULL;
        xmlDocPtr xml_doc = NULL;
        xmlValidCtxt ctx = { 0 };
        int output = -1;

        g_return_val_if_fail (a_doc && MLVIEW_IS_XML_DOCUMENT (a_doc), NULL);
        g_return_val_if_fail (a_dtd, NULL);
        
        xml_doc = mlview_xml_document_get_native_document (a_doc);
        
        g_return_val_if_fail (xml_doc, NULL);
        
        msgs = mlview_validation_output_new (a_doc);
        
        if (!msgs || !msgs->messages)
                goto cleanup;
        
        ctx.userData = msgs->messages;
        ctx.error = (xmlValidityErrorFunc)validation_message;
        ctx.warning = (xmlValidityWarningFunc)validation_message;
        
        output = xmlValidateDtd (&ctx, xml_doc, a_dtd);
        if (a_state) {
                if (output == 1)
                        *a_state = VALIDATOR_VALID_DOC;
                else
                        *a_state = VALIDATOR_INVALID_DOC;
        }

        return msgs;

 cleanup:
        if (msgs) {
                mlview_validation_output_free (msgs);
                msgs = NULL;
        }
        
        if (a_state)
                *a_state = VALIDATOR_ERROR;

        return NULL;
}

MlViewValidationOutput *
mlview_validator_validate_with_rng (MlViewXMLDocument *a_doc, 
                                    xmlRelaxNGPtr a_rng,
                                    MlViewValidatorStatus *a_state)
{
        MlViewValidationOutput *msgs = NULL;
        xmlDocPtr xml_doc = NULL;
        xmlRelaxNGValidCtxtPtr ctx = NULL;
        int output = -1;

        g_return_val_if_fail (a_doc && MLVIEW_IS_XML_DOCUMENT (a_doc), NULL);
        g_return_val_if_fail (a_rng, NULL);

        xml_doc = mlview_xml_document_get_native_document (a_doc);
        
        g_return_val_if_fail (xml_doc, NULL);

        msgs = mlview_validation_output_new (a_doc);
        
        if (!msgs || !msgs->messages)
                goto cleanup;

        ctx = xmlRelaxNGNewValidCtxt (a_rng);

        if (!ctx)
                goto cleanup;

        xmlRelaxNGSetValidErrors (ctx, (xmlRelaxNGValidityErrorFunc)validation_message,
                                  (xmlRelaxNGValidityErrorFunc)validation_message, 
                                  msgs->messages);

        output = xmlRelaxNGValidateDoc (ctx, xml_doc);

        if (a_state) {
                if (output == 0)
                        *a_state = VALIDATOR_VALID_DOC;
                else if (output > 0)
                        *a_state = VALIDATOR_INVALID_DOC;
                else
                        *a_state = VALIDATOR_ERROR;
        }
        xmlRelaxNGFreeValidCtxt (ctx);

        return msgs;

 cleanup:     
        if (msgs) {
                mlview_validation_output_free (msgs);
                msgs = NULL;
        }
        
        if (ctx) {
                xmlRelaxNGFreeValidCtxt (ctx);
                ctx = NULL;
        }
        
        if (a_state)
                *a_state = VALIDATOR_ERROR;

        return NULL;
}

MlViewValidationOutput *
mlview_validator_validate_with_xsd (MlViewXMLDocument *a_doc, 
                                    xmlSchemaPtr a_xsd,
                                    MlViewValidatorStatus *a_state)
{
        MlViewValidationOutput *msgs = NULL;
        xmlDocPtr xml_doc = NULL;
        xmlSchemaValidCtxtPtr ctx = NULL;
        int output = -1;
        
        g_return_val_if_fail (a_doc && MLVIEW_IS_XML_DOCUMENT (a_doc), NULL);
        g_return_val_if_fail (a_xsd, NULL);
        
        xml_doc = mlview_xml_document_get_native_document (a_doc);
        
        g_return_val_if_fail (xml_doc, NULL);
        
        msgs = mlview_validation_output_new (a_doc);
        
        if (!msgs || !msgs->messages)
                goto cleanup;
        
        ctx = xmlSchemaNewValidCtxt (a_xsd);
        
        if (!ctx)
                goto cleanup;

        xmlSchemaSetValidErrors (ctx, (xmlSchemaValidityErrorFunc)validation_message,
                                 (xmlSchemaValidityErrorFunc)validation_message, 
                                 msgs->messages);
        
        output = xmlSchemaValidateDoc (ctx, xml_doc);
        
        if (a_state) {
                if (output == 0)
                        *a_state = VALIDATOR_VALID_DOC;
                else if (output > 0)
                        *a_state = VALIDATOR_INVALID_DOC;
                else
                        *a_state = VALIDATOR_ERROR;
        }

        xmlSchemaFreeValidCtxt (ctx);

        return msgs;

 cleanup:
        if (msgs) {
                mlview_validation_output_free (msgs);
                msgs = NULL;
        }
        
        if (ctx) {
                xmlSchemaFreeValidCtxt (ctx);
                ctx = NULL;
        }
        
        if (a_state)
                *a_state = VALIDATOR_ERROR;

        return NULL;
}

MlViewValidationOutput *
mlview_validator_validate_with_schema (MlViewXMLDocument *a_doc,
                                       MlViewSchema *a_schema,
                                       MlViewValidatorStatus *a_state)
{
        enum MlViewStatus status = MLVIEW_OK ;
        enum MlViewSchemaType schema_type = SCHEMA_TYPE_UNDEF ;
        gpointer native_schema = NULL ;

        g_return_val_if_fail (a_doc && MLVIEW_IS_XML_DOCUMENT (a_doc), NULL);
        g_return_val_if_fail (a_schema, NULL);

        status = mlview_schema_get_type (a_schema, &schema_type) ;
        g_return_val_if_fail (status == MLVIEW_OK 
                              && schema_type != SCHEMA_TYPE_UNDEF, NULL) ;

        status = mlview_schema_get_native_schema (a_schema, &native_schema) ;
        g_return_val_if_fail (status == MLVIEW_OK && native_schema, NULL) ;

        switch (schema_type) {
        case SCHEMA_TYPE_DTD:
                return mlview_validator_validate_with_dtd (a_doc, 
                                                           native_schema,
                                                           a_state);
        case SCHEMA_TYPE_RNG:
                return mlview_validator_validate_with_rng (a_doc, 
                                                           native_schema,
                                                           a_state);
        case SCHEMA_TYPE_XSD:
                return mlview_validator_validate_with_xsd (a_doc, 
                                                           native_schema,
                                                           a_state);
        default:
                g_assert_not_reached () ;
        };

        return NULL;
}

