/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 8-*- */

/*
 *This file is part of MlView
 *
 *MlView is free software; you can redistribute it and/or 
 *modify it under the terms of 
 *the GNU General Public License as published 
 *by the Free Software Foundation; either version 2
 *or (at your option) any later version.
 *
 *MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView.
 *see the file COPYING. 
 *If not, write to the 
 *Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *See COPYRIGHT file for copyright information.
 */

#define ENABLE_NLS 1

#include <stdio.h>
#include <gnome.h>
#include "config.h"
#include "mlview-editor.h"
#include "mlview-app.h"
#include "mlview-app-context.h"
#include "mlview-preferences.h"
#include "mlview-drop-manager.h"
#include "mlview-iview.h"

#define ENABLE_NLS 1

/**
 *@file 
 *the mlview application class definition.
 *It instanciates mlview's instance of  GnomeApp, 
 *fills it with an instance of MlViewEditor, builds the application menus
 *and toolbars and creates an instance of MlViewAppContext.
 */

#define PRIVATE(a_obj) a_obj->priv

struct _MlViewAppPriv {
        struct MlViewWidgetsHandle *widgets ;
        GtkUIManager *ui_manager ;
        GtkActionGroup *doc_required_action_group ;
        GtkActionGroup *doc_not_required_action_group ;
        guint main_menubar_merge_id ;
        guint main_toolbar_merge_id ;
} ;


/*
 *The current instance of the mlview application context.
 *Must be set by mlview_app. For the moment, 
 *there must be one application
 *context per process. That should be changed in the 
 *future to enforce
 *reentrency.
 */

/*common function declarations*/

/*callbacks ...*/
static void
display_about_dialog (void);

static  void document_name_changed_cb (MlViewAppContext *a_ctxt,
                                       MlViewXMLDocument *a_doc,
                                       MlViewApp *a_app) ;

static void view_swapped_cb (MlViewAppContext *a_ctxt,
                             MlViewIView *a_old_view,
                             MlViewIView *a_new_view,
                             MlViewApp *a_app) ;


static enum MlViewStatus build_widgets_handle (MlViewApp *a_app,
                                               GladeXML *a_glade_xml) ;

static enum MlViewStatus init_editor (MlViewApp *a_app,
                                      GladeXML *a_glade_xml) ;


static void mlview_app_new_menuitem_action_cb (GtkAction *a_action,
                                               MlViewApp *a_user_data) ;

static void mlview_app_open_menuitem_action_cb (GtkAction *a_action,
                                                MlViewApp *a_user_data) ;

static void mlview_app_openlocation_menuitem_action_cb (GtkAction *a_action,
                                                        MlViewApp *a_user_data) ;

static void mlview_app_save_menuitem_action_cb (GtkAction *a_action,
                                                MlViewApp *a_user_data) ;

static void mlview_app_save_as_menuitem_action_cb (GtkAction *a_action,
                                                   MlViewApp *a_user_data) ;

static void mlview_app_close_menuitem_action_cb (GtkAction *a_action,
                                                 MlViewApp *a_user_data) ;

static void mlview_app_quit_menuitem_action_cb (GtkAction *a_action,
                                                MlViewApp *a_user_data) ;

static void mlview_app_undo_menuitem_action_cb (GtkAction *a_action,
                                                MlViewApp *a_user_data) ;

static void mlview_app_redo_menuitem_action_cb (GtkAction *a_action,
                                                MlViewApp *a_user_data) ;

static void mlview_app_associated_schemas_menuitem_action_cb (GtkAction *a_action,
                                                              MlViewApp *a_app) ;

static void mlview_app_new_view_on_doc_menuitem_action_cb (GtkAction *a_action,
                                                           MlViewApp *a_user_data) ;

static void mlview_app_rename_view_menuitem_action_cb (GtkAction *a_action,
                                                       MlViewApp *a_user_data) ;


static void mlview_app_validate_dtd_menuitem_action_cb (GtkAction *a_action,
                                                        MlViewApp *a_user_data) ;

static void mlview_app_apply_xslt_dtd_menuitem_action_cb (GtkAction *a_action,
                                                          MlViewApp *a_user_data) ;

static void mlview_app_about_menuitem_action_cb (GtkAction *a_action,
                                                 MlViewApp *a_user_data) ;

static void mlview_app_edit_menu_action_cb (GtkAction *a_action,
                                            MlViewApp *a_user_data) ;

/*actions that require a document to be open*/
static GtkActionEntry gv_doc_required_actions [] = {

        /*File Menu*/
        {
                "SaveMenuitemAction", GTK_STOCK_SAVE, N_("_Save"), 
                "<control>S", N_("Save the XML Document"), 
                G_CALLBACK (mlview_app_save_menuitem_action_cb)},

        {
                "SaveAsMenuitemAction", GTK_STOCK_SAVE_AS, N_("Save _As"), 
                "<shift><control>S", N_("Save the XML Document in an other file"), 
                G_CALLBACK (mlview_app_save_as_menuitem_action_cb)},

        {
                "CloseMenuitemAction", GTK_STOCK_CLOSE, N_("_Close"), 
                "<control>W", N_("Close the XML Document"), 
                G_CALLBACK (mlview_app_close_menuitem_action_cb)},
        
        /*Edit Menu*/        
        {
                "EditMenuAction", NULL, N_("_Edit"), NULL, NULL, 
                G_CALLBACK (mlview_app_edit_menu_action_cb)
        },

        {
                "UndoMenuitemAction", GTK_STOCK_UNDO, N_("_Undo"), 
                "<control>Z", N_("Undo the last action"), 
                G_CALLBACK (mlview_app_undo_menuitem_action_cb)},
        
        {
                "RedoMenuitemAction", GTK_STOCK_REDO, N_("_Redo"), 
                "<shift><control>Z", N_("Undo the last action"),
                G_CALLBACK (mlview_app_redo_menuitem_action_cb)},
        /*
          {
                "FindMenuitemAction", GTK_STOCK_FIND, N_("Find node"),
                "<control>F", N_("Finds a node in the document"),
                G_CALLBACK (mlview_app_find_node_menuitem_action_cb) ;
                }*/

        /*Tools menu*/
        {"ToolsMenuAction", NULL, N_("_Tools"), NULL, NULL, NULL},

        {"NewViewOnDocMenuitemAction", NULL, N_("New _view"), 
         NULL, N_("Creates a new view on the current document"), 
         G_CALLBACK (mlview_app_new_view_on_doc_menuitem_action_cb)},
        
        {"RenameViewMenuitemAction", NULL, N_("Re_name view"), 
         NULL, N_("Rename the view"), 
         G_CALLBACK (mlview_app_rename_view_menuitem_action_cb)},

        {"AssociatedSchemasMenuitemAction", NULL, N_("Schemas"), 
         NULL, N_("Manage the schemas/DTDs associated to the document"), 
         G_CALLBACK (mlview_app_associated_schemas_menuitem_action_cb)},

        {"ValidateDocMenuitemAction", NULL, N_("Va_lidate document"),
         NULL, N_("Validate the document against the associated DTD"), 
         G_CALLBACK (mlview_app_validate_dtd_menuitem_action_cb)},

        {"ApplyXSLTMenuitemAction", NULL, N_("Apply an _XSL transform"),
         NULL, N_("Apply an XSL transform to the current document"), 
         G_CALLBACK (mlview_app_apply_xslt_dtd_menuitem_action_cb)},
} ;


static GtkActionEntry gv_doc_not_required_actions[] = {
        /*File Menu*/
        {"FileMenuAction", NULL, N_("_File"), NULL, NULL, NULL},

        {"NewMenuitemAction", GTK_STOCK_NEW, N_("_New"), 
         "<control>N", N_("Create a new XML Document"), 
         G_CALLBACK (mlview_app_new_menuitem_action_cb)},

        {"OpenMenuitemAction", GTK_STOCK_OPEN, N_("_Open"), 
         "<control>O", N_("Open an XML Document"), 
         G_CALLBACK (mlview_app_open_menuitem_action_cb)},

        {"OpenLocationMenuitemAction", 0, N_("Open _location"),
         "<control>L", N_("Open an XML Document from an URI"), 
         G_CALLBACK (mlview_app_openlocation_menuitem_action_cb)},
        
        {"QuitMenuitemAction", GTK_STOCK_QUIT, N_("_Quit"), 
         "<control>Q", N_("Quit the MlView application"), 
         G_CALLBACK (mlview_app_quit_menuitem_action_cb)},
                
        /*Help Menu*/
        {"HelpMenuAction", NULL, N_("_Help"), NULL, NULL, NULL},

        {"AboutMenuitemAction", GNOME_STOCK_ABOUT, N_("_About"),
         NULL, N_("About MlView ..."), 
         G_CALLBACK (mlview_app_about_menuitem_action_cb)}
} ;


/*
 *Displays the about dialog.
 */
static void
display_about_dialog ()
{
        static GtkWidget *about_dialog = NULL;
        guchar *path = NULL ;
        GdkPixbuf *pixbuf = NULL;

        const gchar *authors[] = {
                N_("Author and maintainer: Dodji Seketeli <dodji@gnome.org>"),
                "\n\n",
                N_("Substantial contributors:"),
                "\n",
                N_("Nicolas Centa <happypeng@free.fr>"),
                N_("Philippe Mechai <pmechai@free.fr>"),
                N_("Baptiste Mille-Mathias <bmm80@free.fr>"),
                "\n\n",
                N_("Former contributors:"),
                "\n",
                "Gael Chamoulaud<strider@gnome.org>", 
                "Stephane Bonhomme<s.bonhomme@wanadoo.fr>",
                NULL
        };

        const gchar *documenters[] =
                {"Dodji Seketeli<dodji@mlview.org>",
                 NULL
        };

        const gchar *translator_credits = _("translator_credits");

        if (about_dialog) {
                gtk_window_present (GTK_WINDOW (about_dialog));
                return;
        }
        path = gnome_program_locate_file (NULL, 
                                          GNOME_FILE_DOMAIN_APP_DATADIR,
                                          "mlview/mlview-app-icon.xpm",
                                          TRUE, NULL) ;
        if (path) {
                pixbuf = gdk_pixbuf_new_from_file (path, NULL);
                g_free (path) ;
                path = NULL ;
        }
        about_dialog = gnome_about_new 
                (PACKAGE, VERSION,
                 "Copyright \xc2\xa9 2001-2004 Dodji Seketeli\n",
                 _("A simple xml editor for GNOME"),
                 (const char **) authors,
                 (const char **) documenters,
                 strcmp
                 (translator_credits,
                  "translator_credits") !=
                 0 ? translator_credits :
                 "No translators, English by\n"
                 "Dodji Seketeli  <dodji@mlview.org>\n"
                 "Gael Chamoulaud <strider@mlview.org>",
                 pixbuf);

        if (pixbuf != NULL) {
                g_object_unref (pixbuf) ;
                pixbuf = NULL ;
        }
        g_object_add_weak_pointer (G_OBJECT (about_dialog),
                                   (void **) &about_dialog);
        gtk_widget_show (about_dialog);
}

/*=========================================================
 *General signal handlers for the differents menuitems
 *========================================================*/

static void
close_application (MlViewApp *a_this)
{
        MlViewEditor *editor = NULL;
        struct MlViewWidgetsHandle *handle = NULL ;
        gboolean is_ok = FALSE;

        g_return_if_fail (a_this && PRIVATE (a_this));
        
        editor = mlview_app_get_editor (a_this);
        g_return_if_fail (editor) ;
        
        handle = mlview_app_get_widgets_handle (a_this) ;
        g_return_if_fail (handle && handle->app_win) ;

        if (editor)
                is_ok = mlview_editor_close_all_xml_documents_interactive
                        (editor);

        if (is_ok == FALSE) {
                gtk_widget_show (handle->app_win) ;
                return;
        }

        gtk_widget_destroy (handle->app_win);
        gtk_main_quit ();
}

static gboolean
delete_event_cb (GtkWidget *a_widget,
                 GdkEvent *a_event,
                 MlViewApp *a_app)
{
        g_return_val_if_fail (GTK_IS_WIDGET (a_widget)
                              && a_app,
                              FALSE) ;

        close_application (a_app) ;
        return TRUE ;
}



static void
document_name_changed_cb (MlViewAppContext *a_ctxt,
                          MlViewXMLDocument *a_doc,
                          MlViewApp *a_app)
{
        gchar *doc_name = NULL ;
        xmlDoc *doc = NULL ;

        g_return_if_fail (a_ctxt
                          && MLVIEW_IS_APP_CONTEXT (a_ctxt)) ;
        g_return_if_fail (a_doc && MLVIEW_IS_XML_DOCUMENT (a_doc)) ;

        doc = mlview_xml_document_get_xml_document (a_doc) ;
        if (!doc) {
                mlview_utils_trace_info ("mlview_xml_document_get_xml_document() failed\n") ;
                return ;
        }
        if (doc->name) {
                doc_name = doc->name ;
        } else {
                doc_name = (gchar*)"untitled" ;
        }
        mlview_app_set_main_window_title (a_app, doc_name) ;
}

static void
view_swapped_cb (MlViewAppContext *a_ctxt,
                 MlViewIView *a_old_view,
                 MlViewIView *a_new_view,
                 MlViewApp *a_app)
{
        MlViewXMLDocument *mlview_doc = NULL ;
        xmlDoc *doc = NULL ;
        gchar *doc_name = NULL ;

        g_return_if_fail (a_ctxt && MLVIEW_IS_APP_CONTEXT (a_ctxt)
                          && a_new_view && MLVIEW_IS_IVIEW (a_new_view)
                          && a_app && PRIVATE (a_app)) ;

        mlview_iview_get_document (a_new_view, &mlview_doc) ;
        if (mlview_doc)
                doc = mlview_xml_document_get_xml_document (mlview_doc) ;
        if (!doc) {
                mlview_utils_trace_info 
                        ("mlview_xml_document_get_xml_document() failed\n") ;
                return ;
        }
        if (doc->name) {
                doc_name = doc->name ;
        } else {
                doc_name = (gchar*)"untitled" ;
        }
        mlview_app_set_main_window_title (a_app, doc_name) ;
}

static void
document_undo_state_changed_cb (MlViewAppContext *a_ctxt,
                                MlViewXMLDocument *a_doc,
                                MlViewApp *a_app)
{
        MlViewEditor *editor = NULL ;
        GtkUIManager *ui_manager = NULL ;
        GtkAction *undo_action = NULL,
                *redo_action = NULL ;                

        g_return_if_fail (a_ctxt
                          && MLVIEW_IS_APP_CONTEXT (a_ctxt)
                          && a_app) ;

        editor = mlview_app_context_get_element (a_ctxt, "MlViewEditor") ;
        if (!editor) {
                mlview_utils_trace_info ("Could not find the editor") ; 
                return ;
        }

        ui_manager = mlview_app_get_ui_manager (a_app) ;
        g_return_if_fail (ui_manager) ;

        undo_action = gtk_ui_manager_get_action (ui_manager, "/MainToolbar/UndoToolitem") ;
        g_return_if_fail (undo_action) ;
        redo_action = gtk_ui_manager_get_action (ui_manager, "/MainToolbar/RedoToolitem") ;

        if (!a_doc || !MLVIEW_IS_XML_DOCUMENT (a_doc)) {
                g_object_set (G_OBJECT (undo_action),
                              "sensitive", FALSE, NULL) ;
                g_object_set (G_OBJECT (redo_action),
                              "sensitive", FALSE, NULL) ;
                return ;
        }

        if (mlview_xml_document_can_undo_mutation (a_doc) == TRUE) {
                g_object_set (G_OBJECT (undo_action),
                              "sensitive", TRUE, NULL) ;
        } else {
                g_object_set (G_OBJECT (undo_action),
                              "sensitive", FALSE, NULL) ;
        }
        if (mlview_xml_document_can_redo_mutation (a_doc) == TRUE) {
                g_object_set (G_OBJECT (redo_action), "sensitive", TRUE, NULL) ;
                
        } else {
                g_object_set (G_OBJECT (redo_action), "sensitive", FALSE, NULL) ;
        }
}

static void
set_editing_enabled (MlViewApp *a_this,
                     gboolean a_enable)
{
        GtkUIManager *ui_manager = NULL ;
        MlViewEditor *editor = NULL ;
        MlViewXMLDocument *doc = NULL ;
        MlViewAppContext *context = NULL ;

        g_return_if_fail (a_this && PRIVATE (a_this)) ;
        g_return_if_fail (PRIVATE (a_this)->doc_required_action_group
                          && PRIVATE (a_this)->doc_not_required_action_group) ;

        ui_manager = mlview_app_get_ui_manager (a_this) ;
        g_return_if_fail (ui_manager) ;

        if (a_enable == TRUE) {
                gtk_action_group_set_sensitive (PRIVATE (a_this)->doc_required_action_group,
                                                TRUE) ;

        } else {
                gtk_action_group_set_sensitive (PRIVATE (a_this)->doc_required_action_group,
                                                FALSE) ;
        }
        /*now update the undo/redo action sensitivity*/
        editor = mlview_app_get_editor (a_this) ;
        if (!editor)
                return ;

        doc = mlview_editor_get_cur_doc (editor) ;
        context = mlview_editor_get_app_context (editor) ;
        g_return_if_fail (context) ;
        mlview_app_context_notify_document_undo_state_changed (context, doc) ;
}

static void
application_initialized_cb (MlViewAppContext *a_this,
                            gpointer a_user_data)
{
        MlViewApp *app = NULL ;
        g_return_if_fail (a_this && MLVIEW_IS_APP_CONTEXT (a_this)
                          && a_user_data) ;

        app = (MlViewApp*) a_user_data ;
        set_editing_enabled (app, FALSE) ;
}

static void
last_view_removed_cb (MlViewEditor *a_editor,
                      gpointer a_user_data)
{
        MlViewApp *app = NULL ;
        g_return_if_fail (a_editor 
                          && MLVIEW_IS_EDITOR (a_editor)
                          && a_user_data) ;
        app = (MlViewApp*) a_user_data ;
        set_editing_enabled (app, FALSE) ;
}

static void
first_view_added_cb (MlViewEditor *a_editor,
                     MlViewIView *a_view,
                     gpointer a_user_data)
{
        MlViewApp *app = NULL ;

        g_return_if_fail (a_editor && MLVIEW_IS_EDITOR (a_editor)
                          && MLVIEW_IS_IVIEW (a_view)
                          && a_user_data) ;        
        app = (MlViewApp*) a_user_data ;
        set_editing_enabled (app, TRUE) ;
}


static void 
mlview_app_new_menuitem_action_cb (GtkAction *a_action,
                                   MlViewApp *a_app)
{
        MlViewEditor *editor = NULL ;
        g_return_if_fail (a_action && a_app && PRIVATE (a_app)) ;
        
        editor = mlview_app_get_editor (a_app) ;
        g_return_if_fail (editor && MLVIEW_IS_EDITOR (editor)) ;

        mlview_editor_create_new_xml_document (editor);
}

static void 
mlview_app_open_menuitem_action_cb (GtkAction *a_action,
                                    MlViewApp *a_app)
{
        MlViewEditor *editor = NULL ;
        g_return_if_fail (a_action && a_app && PRIVATE (a_app)) ;
        
        editor = mlview_app_get_editor (a_app) ;
        g_return_if_fail (editor && MLVIEW_IS_EDITOR (editor)) ;
        
        mlview_editor_open_local_xml_document_interactive (editor) ;
}

static void
mlview_app_openlocation_menuitem_action_cb (GtkAction *a_action,
                                            MlViewApp *a_app)
{
        MlViewEditor *editor = NULL ;
        g_return_if_fail (a_action && a_app && PRIVATE (a_app)) ;
        
        editor = mlview_app_get_editor (a_app) ;
        g_return_if_fail (editor && MLVIEW_IS_EDITOR (editor)) ;
        
        mlview_editor_open_xml_document_interactive (editor) ;
}

static void 
mlview_app_save_menuitem_action_cb (GtkAction *a_action,
                                    MlViewApp *a_app)
{
        MlViewEditor *editor = NULL ;
        g_return_if_fail (a_action && a_app && PRIVATE (a_app)) ;
        
        editor = mlview_app_get_editor (a_app) ;
        g_return_if_fail (editor && MLVIEW_IS_EDITOR (editor)) ;

        mlview_editor_save_xml_document (editor);
}

static void 
mlview_app_save_as_menuitem_action_cb (GtkAction *a_action,
                                       MlViewApp *a_app)
{
        MlViewEditor *editor = NULL ;
        g_return_if_fail (a_action && a_app && PRIVATE (a_app)) ;
        
        editor = mlview_app_get_editor (a_app) ;
        g_return_if_fail (editor && MLVIEW_IS_EDITOR (editor)) ;
        
        mlview_editor_save_xml_document_as_interactive (editor);
}

static void 
mlview_app_close_menuitem_action_cb (GtkAction *a_action,
                                     MlViewApp *a_app)
{
        MlViewEditor *editor = NULL ;
        g_return_if_fail (a_action && a_app && PRIVATE (a_app)) ;
        
        editor = mlview_app_get_editor (a_app) ;
        g_return_if_fail (editor && MLVIEW_IS_EDITOR (editor)) ;

        mlview_editor_close_xml_document_interactive (editor) ;
}

static void 
mlview_app_quit_menuitem_action_cb (GtkAction *a_action,
                                    MlViewApp *a_app)
{
        MlViewEditor *editor = NULL ;
        struct MlViewWidgetsHandle * handle = NULL ;

        g_return_if_fail (a_action && a_app && PRIVATE (a_app)) ;
        
        handle = mlview_app_get_widgets_handle (a_app) ;
        g_return_if_fail (handle && handle->app_win) ;
        editor = mlview_app_get_editor (a_app) ;
        g_return_if_fail (editor && MLVIEW_IS_EDITOR (editor)) ;

        close_application (a_app) ;
}

static void 
mlview_app_undo_menuitem_action_cb (GtkAction *a_action,
                                    MlViewApp *a_app)
{
        MlViewEditor *editor = NULL ;
        g_return_if_fail (a_action && a_app && PRIVATE (a_app)) ;
        
        editor = mlview_app_get_editor (a_app) ;
        g_return_if_fail (editor && MLVIEW_IS_EDITOR (editor)) ;

        mlview_editor_undo (editor) ;
}

static void 
mlview_app_redo_menuitem_action_cb (GtkAction *a_action,
                                    MlViewApp *a_app)
{
        MlViewEditor *editor = NULL ;
        g_return_if_fail (a_action && a_app && PRIVATE (a_app)) ;
        
        editor = mlview_app_get_editor (a_app) ;
        g_return_if_fail (editor && MLVIEW_IS_EDITOR (editor)) ;

        mlview_editor_redo (editor) ;
}

static void 
mlview_app_new_view_on_doc_menuitem_action_cb (GtkAction *a_action,
                                               MlViewApp *a_app)
{
        MlViewEditor *editor = NULL ;
        g_return_if_fail (a_action && a_app && PRIVATE (a_app)) ;
        
        editor = mlview_app_get_editor (a_app) ;
        g_return_if_fail (editor && MLVIEW_IS_EDITOR (editor)) ;

        mlview_editor_create_new_view_on_current_document_interactive
                (editor);
}

/*
   NOT USED ATM
 *
static void 
mlview_app_find_node_menuitem_action_cb (GtkAction *a_action,
                                         MlViewApp *a_app)
{
        MlViewEditor *editor = NULL ;
        g_return_if_fail (a_action && a_app && PRIVATE (a_app)) ;
        
        editor = mlview_app_get_editor (a_app) ;
        g_return_if_fail (editor && MLVIEW_IS_EDITOR (editor)) ;

        mlview_editor_create_new_view_on_current_document_interactive
                (editor);
}
*/

static void 
mlview_app_rename_view_menuitem_action_cb (GtkAction *a_action,
                                           MlViewApp *a_app) 
{
        MlViewEditor *editor = NULL ;
        g_return_if_fail (a_action && a_app && PRIVATE (a_app)) ;
        
        editor = mlview_app_get_editor (a_app) ;
        g_return_if_fail (editor && MLVIEW_IS_EDITOR (editor)) ;

        mlview_editor_set_current_view_name_interactive (editor) ;
}

static void
mlview_app_associated_schemas_menuitem_action_cb (GtkAction *a_action,
                                                  MlViewApp *a_app)
{
        MlViewEditor *editor = NULL ;
        g_return_if_fail (a_action && a_app && PRIVATE (a_app)) ;
        
        editor = mlview_app_get_editor (a_app) ;
        g_return_if_fail (editor && MLVIEW_IS_EDITOR (editor)) ;
        
        mlview_editor_manage_associated_schemas (editor);
}

static void 
mlview_app_validate_dtd_menuitem_action_cb (GtkAction *a_action,
                                            MlViewApp *a_app)
{
        MlViewEditor *editor = NULL ;
        g_return_if_fail (a_action && a_app && PRIVATE (a_app)) ;
        
        editor = mlview_app_get_editor (a_app) ;
        g_return_if_fail (editor && MLVIEW_IS_EDITOR (editor)) ;

        mlview_editor_validate (editor) ;
}

static void 
mlview_app_apply_xslt_dtd_menuitem_action_cb (GtkAction *a_action,
                                              MlViewApp *a_app)
{
        MlViewEditor *editor = NULL ;
        g_return_if_fail (a_action && a_app && PRIVATE (a_app)) ;
        
        editor = mlview_app_get_editor (a_app) ;
        g_return_if_fail (editor && MLVIEW_IS_EDITOR (editor)) ;

        mlview_editor_xslt_transform_document_interactive 
                (editor) ;
}

static void 
mlview_app_about_menuitem_action_cb (GtkAction *a_action,
                                     MlViewApp *a_app)
{
        MlViewEditor *editor = NULL ;

        g_return_if_fail (a_action && a_app && PRIVATE (a_app)) ;
        
        editor = mlview_app_get_editor (a_app) ;
        g_return_if_fail (editor && MLVIEW_IS_EDITOR (editor)) ;

        display_about_dialog ();
}

static void 
mlview_app_edit_menu_action_cb (GtkAction *a_action,
                                MlViewApp *a_app)
{
        MlViewEditor *editor = NULL ;

        g_return_if_fail (a_action && GTK_IS_ACTION (a_action)
                          && a_app && PRIVATE (a_app)) ;

        editor = mlview_app_get_editor (a_app) ;
        g_return_if_fail (editor && MLVIEW_IS_EDITOR (editor)) ;

        mlview_editor_make_current_view_populate_application_edit_menu (editor) ;
}

/**
 *Make sure all the signals of the menu items
 *the right function callbacks are correctly
 *connected to the menu items.
 *@param a_menu_bar
 */
static enum MlViewStatus
init_menu_and_tool_bar (MlViewApp *a_app,
                        GladeXML *a_glade_xml)
{
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_val_if_fail (a_glade_xml && a_app,
                              MLVIEW_BAD_PARAM_ERROR) ;

        /*connect_menu_and_toolbar_signals (a_app, a_glade_xml) ;*/
        return status ;
}

static enum MlViewStatus
build_and_init_menus (MlViewApp *a_this,
                      GladeXML *a_glade_xml)
{
        gchar *file_path = NULL ;
        struct MlViewWidgetsHandle *widgets_handle= NULL ;
        GtkWidget *menubar = NULL, *toolbar = NULL ;
        MlViewAppContext *app_context = NULL ;

        GtkUIManager *ui_manager = NULL ;
        GtkActionGroup *action_group = NULL ;


        widgets_handle = mlview_app_get_widgets_handle (a_this) ;
        g_return_val_if_fail (widgets_handle, MLVIEW_ERROR) ;
        g_return_val_if_fail (widgets_handle->menu_bar_container, 
                              MLVIEW_ERROR) ;
        g_return_val_if_fail (widgets_handle->app_win,
                              MLVIEW_ERROR) ;

        action_group = gtk_action_group_new ("DocRequiredActions") ;
	gtk_action_group_set_translation_domain (action_group, GETTEXT_PACKAGE) ;

        gtk_action_group_add_actions (action_group, gv_doc_required_actions,
                                      sizeof (gv_doc_required_actions)/sizeof (GtkActionEntry),
                                      a_this) ;
        PRIVATE (a_this)->doc_required_action_group = action_group ;
        
        ui_manager = gtk_ui_manager_new () ;
        PRIVATE (a_this)->ui_manager = ui_manager ;
        gtk_ui_manager_insert_action_group (ui_manager, action_group, 0) ;

        action_group = gtk_action_group_new ("DocNotRequiredActions") ;
	gtk_action_group_set_translation_domain (action_group, GETTEXT_PACKAGE) ;

        gtk_action_group_add_actions (action_group, gv_doc_not_required_actions,
                                      sizeof (gv_doc_not_required_actions)/sizeof (GtkActionEntry),
                                      a_this) ;
        gtk_ui_manager_insert_action_group (ui_manager, action_group, 1) ;
        PRIVATE (a_this)->doc_not_required_action_group = action_group ;

        gtk_window_add_accel_group (GTK_WINDOW (widgets_handle->app_win),
                                    gtk_ui_manager_get_accel_group (ui_manager)) ;

        file_path = mlview_utils_locate_file ("main-menu-bar.xml") ;
        g_return_val_if_fail (file_path, MLVIEW_ERROR) ;

        PRIVATE (a_this)->main_menubar_merge_id = 
                gtk_ui_manager_add_ui_from_file (ui_manager, 
                                                 file_path,
                                                 NULL) ;
        g_return_val_if_fail (PRIVATE (a_this)->main_menubar_merge_id,
                              MLVIEW_ERROR) ;
        if (file_path) {
                g_free (file_path) ;
                file_path = NULL ;
        }

        file_path = mlview_utils_locate_file ("main-toolbar.xml") ;
        g_return_val_if_fail (file_path,
                              MLVIEW_ERROR) ;
        PRIVATE (a_this)->main_toolbar_merge_id = 
                gtk_ui_manager_add_ui_from_file (ui_manager, 
                                                 file_path,
                                                 NULL) ;
        if (file_path) {
                g_free (file_path) ;
                file_path = NULL ;
        }

        menubar = gtk_ui_manager_get_widget (ui_manager, "/MainMenubar") ;
        g_return_val_if_fail (menubar, MLVIEW_ERROR) ;
        gtk_widget_show_all (menubar) ;
        gtk_box_pack_end_defaults (GTK_BOX (widgets_handle->menu_bar_container),
                                   menubar) ;

        toolbar = gtk_ui_manager_get_widget (ui_manager, "/MainToolbar") ;
        g_return_val_if_fail (toolbar, MLVIEW_ERROR) ;
        gtk_widget_show_all (toolbar) ;
        gtk_box_pack_end_defaults (GTK_BOX (widgets_handle->toolbar_container),
                                   toolbar) ;

        app_context = mlview_app_get_application_context (a_this) ;
        g_return_val_if_fail (app_context, MLVIEW_ERROR) ;
        mlview_app_context_set_element (app_context, "MlViewUIManager",
                                        ui_manager) ;
        return MLVIEW_OK ;
}

static enum MlViewStatus
init_app_win (MlViewApp *a_app,
              GladeXML *a_glade_xml)
{
        MlViewAppContext *ctxt = NULL ;
        GtkWidget *editor_container = NULL ;
        gchar *mlview_icon_filename = NULL ;
        enum MlViewStatus status = MLVIEW_ERROR ;

        g_return_val_if_fail (a_glade_xml && a_app,
                              MLVIEW_BAD_PARAM_ERROR) ;

        /*
         *Build the widget handle datastructure
         */
        build_widgets_handle (a_app, a_glade_xml) ;

        /*
         *init the editor widget
         */
        status = init_editor (a_app, a_glade_xml) ;

        /*
         *init the menubar and toolbar
         *The 2 following functions must be merged.
         */
        status = init_menu_and_tool_bar (a_app, a_glade_xml) ;
        build_and_init_menus (a_app, a_glade_xml) ;

        g_return_val_if_fail (status == MLVIEW_OK, status) ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->editor,
                              MLVIEW_ERROR) ;

        ctxt = mlview_editor_get_app_context 
                (MLVIEW_EDITOR (PRIVATE (a_app)->widgets->editor)) ;
        g_return_val_if_fail (ctxt, MLVIEW_ERROR) ;

        g_object_set (G_OBJECT (PRIVATE (a_app)->widgets->app_win),
                      "allow-shrink", TRUE, NULL) ;
        
        /*
         * Add Drop Support to window
         */
        mlview_drop_manager_register_target (ctxt, GTK_WIDGET (PRIVATE (a_app)->widgets->app_win));
        
        /*
         *set the main window  and other mlview component 
	 *to the context. That way, components willing
	 *to access other components can get them from
	 *the context by calling mlview_app_context_get_element()
         */
        g_return_val_if_fail (PRIVATE (a_app)->widgets->app_win,
                              MLVIEW_ERROR) ;
        mlview_app_context_set_element
                (ctxt, "MlViewMainWindow",
                 PRIVATE (a_app)->widgets->app_win) ;
        mlview_app_context_set_element 
                (ctxt, "MlViewEditor",
                 PRIVATE (a_app)->widgets->editor) ;
        mlview_app_context_set_element
                (ctxt, "MlViewAppMainMenuBar",
                 PRIVATE (a_app)->widgets->main_menu_bar) ;
        mlview_app_context_set_element (ctxt, "MlViewApp",
                                        a_app) ;
        mlview_app_context_set_element (ctxt, "MlViewUIManager",
                                        PRIVATE (a_app)->ui_manager) ;


        mlview_icon_filename = gnome_program_locate_file
                (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
                 PACKAGE "/mlview-app-icon.png", TRUE,
                 NULL) ;
        g_return_val_if_fail (mlview_icon_filename, MLVIEW_ERROR) ;
        gtk_window_set_icon_from_file 
                (GTK_WINDOW (PRIVATE (a_app)->widgets->app_win),
                 mlview_icon_filename, NULL) ;
        gtk_window_set_default_icon_from_file 
                (mlview_icon_filename, NULL) ;

	/**
	 * Connect to usefull signal to give "life" to the
	 * application
	 */
        g_signal_connect (G_OBJECT (PRIVATE (a_app)->widgets->app_win),
                          "delete-event",
                          G_CALLBACK (delete_event_cb),
                          a_app) ;

        g_signal_connect (G_OBJECT (ctxt), 
                          "application-initialized",
                          G_CALLBACK (application_initialized_cb),
                          a_app) ;

        g_signal_connect (G_OBJECT (PRIVATE (a_app)->widgets->editor),
                          "last-view-removed",
                          G_CALLBACK (last_view_removed_cb),
                          a_app) ;

        g_signal_connect (G_OBJECT (PRIVATE (a_app)->widgets->editor),
                          "first-view-added",
                          G_CALLBACK (first_view_added_cb),
                          a_app) ;

        g_signal_connect (G_OBJECT (ctxt),
                          "document-name-changed",
                          G_CALLBACK (document_name_changed_cb),
                          a_app) ;

        g_signal_connect (G_OBJECT (ctxt), 
                          "view-swapped",
                          G_CALLBACK (view_swapped_cb),
                          a_app) ;

        g_signal_connect (G_OBJECT (ctxt),
                          "document-undo-state-changed",
                          G_CALLBACK (document_undo_state_changed_cb),
                          a_app) ;

        editor_container = NULL ;
        status = MLVIEW_OK ;
        /*
         *notify whomever listens to the event "the application has
         *been initialized".
         */
        mlview_app_context_notify_application_initialized (ctxt) ;
        return status ;
}

/**
 *populate the main container with the menu bar
 *a MlViewEditor widget.
 *@param a_glade_xml the glade xml object that contains
 *the mlview app graphical layout.
 */
static enum MlViewStatus
init_editor (MlViewApp *a_app,
             GladeXML *a_glade_xml)
{
        GtkWidget *mlview_editor = NULL ;
        GtkWidget *editor_container = NULL ;
        MlViewAppContext *ctxt = NULL ;

        g_return_val_if_fail (a_glade_xml && a_app, 
                              MLVIEW_ERROR) ;

        editor_container = glade_xml_get_widget (a_glade_xml,
                                                 "EditorContainer") ;
        g_return_val_if_fail (editor_container, MLVIEW_ERROR) ;
        
        /*create and pack the MlViewEditor widget*/
        mlview_editor = mlview_editor_new ("mlview:empty", NULL) ;
        g_return_val_if_fail (mlview_editor, MLVIEW_ERROR) ;


        ctxt = mlview_editor_get_app_context
                (MLVIEW_EDITOR (mlview_editor)) ;
        g_return_val_if_fail (ctxt, MLVIEW_ERROR) ;
        g_object_ref (G_OBJECT (ctxt)) ;
        /*
         *Pack the editor widget into the editor container
         */
        gtk_box_pack_start_defaults (GTK_BOX (editor_container),
                                     mlview_editor) ;
        PRIVATE (a_app)->widgets->editor = mlview_editor ;

        return MLVIEW_OK ;
}

static enum MlViewStatus
build_widgets_handle (MlViewApp *a_app,
                      GladeXML *a_glade_xml)
{
        g_return_val_if_fail (a_glade_xml 
                              && a_app
                              && PRIVATE (a_app)
                              && PRIVATE (a_app)->widgets,
                              MLVIEW_BAD_PARAM_ERROR) ;

        PRIVATE (a_app)->widgets->app_win = glade_xml_get_widget 
                (a_glade_xml, "AppWin") ;
        
        g_return_val_if_fail (PRIVATE (a_app)->widgets->app_win,
                              MLVIEW_BAD_PARAM_ERROR) ;

        PRIVATE (a_app)->widgets->menu_bar_container =
                glade_xml_get_widget (a_glade_xml,
                "MenuBarContainer") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->menu_bar_container,
                              MLVIEW_ERROR) ;

        PRIVATE (a_app)->widgets->toolbar_container = 
                glade_xml_get_widget (a_glade_xml, "ToolbarContainer") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->toolbar_container,
                              MLVIEW_ERROR) ;

        /*PRIVATE (a_app)->widgets->main_menu_bar = 
                glade_xml_get_widget (a_glade_xml,
                "MainMenuBar") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->main_menu_bar,
                              MLVIEW_ERROR) ;   
        PRIVATE (a_app)->widgets->file_menu =
                glade_xml_get_widget (a_glade_xml,
                                      "FileMenu") ;
        PRIVATE (a_app)->widgets->new_menu_item = 
                glade_xml_get_widget (a_glade_xml,
                                      "NewDocumentMenuItem") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->new_menu_item,
                              MLVIEW_ERROR) ;
        PRIVATE (a_app)->widgets->open_menu_item = 
                glade_xml_get_widget (a_glade_xml,
                                      "OpenMenuItem") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->open_menu_item,
                              MLVIEW_ERROR) ;
        PRIVATE (a_app)->widgets->save_menu_item = 
                glade_xml_get_widget (a_glade_xml,
                                      "SaveMenuItem") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->save_menu_item,
                              MLVIEW_ERROR) ;
        
        PRIVATE (a_app)->widgets->saveas_menu_item =
                glade_xml_get_widget (a_glade_xml,
                                      "SaveAsMenuItem") ;
        g_return_val_if_fail 
                (PRIVATE (a_app)->widgets->saveas_menu_item,
                 MLVIEW_ERROR) ;
        
        PRIVATE (a_app)->widgets->quit_menu_item = 
                glade_xml_get_widget (a_glade_xml,
                                      "QuitMenuItem") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->quit_menu_item,
                              MLVIEW_ERROR) ;
        
        PRIVATE (a_app)->widgets->close_menu_item = 
                glade_xml_get_widget (a_glade_xml,
                                      "CloseMenuItem") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->quit_menu_item,
                              MLVIEW_ERROR) ;

        PRIVATE (a_app)->widgets->action_menu =
                glade_xml_get_widget (a_glade_xml, "ActionMenu") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->action_menu,
                              MLVIEW_ERROR) ;
        
        PRIVATE (a_app)->widgets->main_toolbar = glade_xml_get_widget 
                (a_glade_xml, "MainToolbar") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->main_toolbar,
                              MLVIEW_ERROR) ;
       
        

        PRIVATE (a_app)->widgets->new_button = glade_xml_get_widget
                (a_glade_xml, "NewButton") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->new_button,
                              MLVIEW_ERROR) ;
        
        PRIVATE (a_app)->widgets->open_button = glade_xml_get_widget
                (a_glade_xml, "OpenButton") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->open_button,
                              MLVIEW_ERROR) ;

        PRIVATE (a_app)->widgets->close_button = glade_xml_get_widget
                (a_glade_xml, "CloseButton") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->close_button,
                              MLVIEW_ERROR) ;

        PRIVATE (a_app)->widgets->save_button = glade_xml_get_widget 
                (a_glade_xml, "SaveButton") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->save_button,
                              MLVIEW_ERROR) ;

        PRIVATE (a_app)->widgets->undo_button = glade_xml_get_widget 
                (a_glade_xml, "UndoButton") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->undo_button,
                              MLVIEW_ERROR) ;

        PRIVATE (a_app)->widgets->redo_button = glade_xml_get_widget 
                (a_glade_xml, "RedoButton") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->redo_button,
                              MLVIEW_ERROR) ;
        */

        return MLVIEW_OK ;
}

/****************
 *Public methods
 ***************/

/**
 *Getter of the application context 
 *associated to the current mlview application.
 *@param a_appname
 *@return the newly instanciated GtkWindow that
 *represents the mlview application.
 */
MlViewApp *
mlview_app_new (const guchar *a_appname)
{
        MlViewApp *result = NULL, *mlview_app = NULL;
        gchar *glade_file = NULL ;
        GladeXML *glade_xml = NULL ;

        mlview_app = g_try_malloc (sizeof (MlViewApp)) ;
        if (!mlview_app) {
                mlview_utils_trace_info ("Out of memory") ;
                return NULL ;
        }
        memset (mlview_app, 0, sizeof (MlViewApp)) ;        
        
        PRIVATE (mlview_app) = g_try_malloc (sizeof (MlViewAppPriv)) ;
        if (!PRIVATE (mlview_app)) {
                mlview_utils_trace_info ("Out of memory") ;
                goto end ;
        }
        memset (PRIVATE (mlview_app), 0, sizeof (MlViewAppPriv)) ;
        PRIVATE (mlview_app)->widgets = 
                g_try_malloc (sizeof (struct MlViewWidgetsHandle)) ;
        if (!PRIVATE (mlview_app)->widgets) {
                mlview_utils_trace_info ("Out of memory") ;
                goto end ;
        }
        memset (PRIVATE (mlview_app)->widgets, 0, 
                sizeof (struct MlViewWidgetsHandle)) ;
        glade_file = gnome_program_locate_file
                (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
                 PACKAGE "/mlview-main-app-win2.glade", TRUE,
                 NULL) ;
        g_return_val_if_fail (glade_file, NULL) ;

        glade_xml = glade_xml_new (glade_file,
                                   "AppWin",
                                   NULL) ;
        g_return_val_if_fail (glade_xml, NULL) ;        
        init_app_win (mlview_app, glade_xml) ;
        
        result = mlview_app ;
        /*
         *connect to MlViewEditor contextual
         *signals view and to enable/disable
         *editing.
         */
        mlview_app = NULL ;

 end:
        if (glade_xml) {
                g_object_unref (G_OBJECT (glade_xml)) ;
                glade_xml = NULL ;
        }
        if (mlview_app && PRIVATE (mlview_app) 
            && PRIVATE (mlview_app)
            && PRIVATE (mlview_app)->widgets) {
                g_free (PRIVATE (mlview_app)->widgets) ;
                PRIVATE (mlview_app)->widgets = NULL ;
        }
        if (mlview_app && PRIVATE (mlview_app) 
            && PRIVATE (mlview_app)) {
                g_free (PRIVATE (mlview_app)) ;
                PRIVATE (mlview_app) = NULL ;
        }        
        if (mlview_app) {
                g_free (mlview_app) ;
                mlview_app = NULL ;
        }
        return  result ;
}


MlViewAppContext *
mlview_app_get_application_context (MlViewApp *a_app)
{
        MlViewAppContext *ctxt = NULL ;

        g_return_val_if_fail (a_app, NULL) ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->editor,
                              NULL) ;
        ctxt = mlview_editor_get_app_context
                (MLVIEW_EDITOR (PRIVATE (a_app)->widgets->editor)) ;
        return ctxt ;
}

/**
 *Getter of the instance of MlViewEditor associated
 *to the current instance of GnomeApp.
 *
 *@param a_app the instance of GnomeApp to consider.
 *@return the instance of MlViewEditor associated to a_app or
 *NULL if nothing is associated.
 */
MlViewEditor *
mlview_app_get_editor (MlViewApp * a_app)
{
        MlViewEditor *result = NULL;

        g_return_val_if_fail (a_app 
                              && PRIVATE (a_app)->widgets->editor, 
                              NULL);
        result = MLVIEW_EDITOR (PRIVATE (a_app)->widgets->editor) ;
        return result;
}

/**
 *Returns the instance of #MlViewWidgetsHandle associated
 *to the current instance of #MlViewApp.
 *@param a_this the instance of #MlViewApp to consider.
 *@return a pointer to the widgets handle, NULL in case of
 *an error.
 */
struct MlViewWidgetsHandle *
mlview_app_get_widgets_handle (MlViewApp *a_this)
{
        g_return_val_if_fail (a_this, NULL) ;
        return PRIVATE (a_this)->widgets ;
}

/**
 *Sets the title of the main application window.
 *The title is built from the name of document being edited.
 *That document name must be passed as a parameter of this function.
 *This is a requirement of the GNOME HIG.
 *@param a_this the current instance of #MlViewApp.
 *@param a_document_name the name of the document being
 *@return MLVIEW_OK upon successful completion.
 */
enum MlViewStatus
mlview_app_set_main_window_title (MlViewApp *a_this,
                                  const gchar *a_document_name)
{
        gchar * title_string = NULL ;

        g_return_val_if_fail (a_this 
                              && PRIVATE (a_this)
                              && PRIVATE (a_this)->widgets
                              && PRIVATE (a_this)->widgets->app_win
                              && a_document_name, 
                              MLVIEW_BAD_PARAM_ERROR) ;

        title_string = g_strconcat (a_document_name,
                                    " - MlView",
                                    NULL) ;
        gtk_window_set_title (GTK_WINDOW (PRIVATE (a_this)->widgets->app_win),
                              (const gchar*)title_string) ;
        if (title_string) {
                g_free (title_string) ;
                title_string = NULL ;
        }
        return MLVIEW_OK ;
}

GtkUIManager * 
mlview_app_get_ui_manager (MlViewApp *a_this)
{
        g_return_val_if_fail (a_this && PRIVATE (a_this), NULL) ;

        return PRIVATE (a_this)->ui_manager ;
}
