/* gnome-db-entry-combo.c
 *
 * Copyright (C) 2003 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-entry-combo.h"
#include <libgnomedb/gnome-db-data-model.h>
#include <libgnomedb/gnome-db-server.h>
#include <libgnomedb/gnome-db-server-data-type.h>
#include <libgnomedb/gnome-db-data-handler.h>
#include <libgnomedb/gnome-db-query.h>
#include <libgnomedb/gnome-db-entity.h>
#include <libgnomedb/gnome-db-field.h>
#include <libgnomedb/gnome-db-qfield.h>
#include <libgnomedb/gnome-db-parameter.h>
#include <libgnomedb/gnome-db-renderer.h>
#include <libgnomedb/gnome-db-result-set.h>
#include <libgnomedb/libgnomedb.h>
#include <libgnomedb/utility.h>

static void gnome_db_entry_combo_class_init (GnomeDbEntryComboClass *class);
static void gnome_db_entry_combo_init (GnomeDbEntryCombo *wid);
static void gnome_db_entry_combo_dispose (GObject   *object);

static void gnome_db_entry_combo_set_property (GObject              *object,
					 guint                 param_id,
					 const GValue         *value,
					 GParamSpec           *pspec);
static void gnome_db_entry_combo_get_property (GObject              *object,
					 guint                 param_id,
					 GValue               *value,
					 GParamSpec           *pspec);

static void          choose_auto_default_value (GnomeDbEntryCombo *combo);
static void          combo_contents_changed_cb (GnomeDbCombo *entry, GnomeDbEntryCombo *combo);

static void          gnome_db_entry_combo_emit_signal (GnomeDbEntryCombo *combo);
static void          real_combo_block_signals (GnomeDbEntryCombo *wid);
static void          real_combo_unblock_signals (GnomeDbEntryCombo *wid);

/* GnomeDbDataEntry interface (value must be a GDA_VALUE_TYPE_LIST) */
static void            gnome_db_entry_combo_data_entry_init   (GnomeDbDataEntryIface *iface);
static void            gnome_db_entry_combo_set_value         (GnomeDbDataEntry *de, const GdaValue * value);
static GdaValue       *gnome_db_entry_combo_get_value         (GnomeDbDataEntry *de);
static void            gnome_db_entry_combo_set_value_orig    (GnomeDbDataEntry *de, const GdaValue * value);
static const GdaValue *gnome_db_entry_combo_get_value_orig    (GnomeDbDataEntry *de);
static void            gnome_db_entry_combo_set_value_default (GnomeDbDataEntry *de, const GdaValue * value);
static void            gnome_db_entry_combo_set_attributes    (GnomeDbDataEntry *de, guint attrs, guint mask);
static guint           gnome_db_entry_combo_get_attributes    (GnomeDbDataEntry *de);
static gboolean        gnome_db_entry_combo_expand_in_layout  (GnomeDbDataEntry *de);

/* signals */
enum
{
	LAST_SIGNAL
};

static gint gnome_db_entry_combo_signals[LAST_SIGNAL] = { };

/* properties */
enum
{
        PROP_0,
        PROP_SET_DEFAULT_IF_INVALID
};

/* ComboNode structures: there is one such structure for each GnomeDbParameter in the GnomeDbDataSetNode being
 * used.
 */
typedef struct {
	GnomeDbParameter  *param;
	GdaValue          *value;    /* we don't own the value, since it belongs to a GdaDataModel => don't free it */
	gint               position; /* column number in data model */
	GdaValue          *value_orig;
	GdaValue          *value_default;
} ComboNode;
#define COMBO_NODE(x) ((ComboNode*)(x))

/* Private structure */
struct  _GnomeDbEntryComboPriv {
        GtkWidget          *combo_entry;
	GSList             *combo_nodes; /* list of ComboNode structures */
	DataSetNodeInfo    *info;
	
	gboolean            data_valid;
	gboolean            null_forced;
	gboolean            default_forced;
	gboolean            null_possible;
	gboolean            default_possible;

	gboolean            show_actions;
	gboolean            set_default_if_invalid; /* use first entry when provided value is not found ? */
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

GType
gnome_db_entry_combo_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbEntryComboClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_entry_combo_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbEntryCombo),
			0,
			(GInstanceInitFunc) gnome_db_entry_combo_init
		};		

		static const GInterfaceInfo data_entry_info = {
			(GInterfaceInitFunc) gnome_db_entry_combo_data_entry_init,
			NULL,
			NULL
		};

		type = g_type_register_static (GNOME_DB_TYPE_ENTRY_SHELL, "GnomeDbEntryCombo", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_TYPE_DATA_ENTRY, &data_entry_info);
	}
	return type;
}

static void
gnome_db_entry_combo_data_entry_init (GnomeDbDataEntryIface *iface)
{
        iface->set_value_type = NULL;
        iface->get_value_type = NULL;
        iface->set_value = gnome_db_entry_combo_set_value;
        iface->get_value = gnome_db_entry_combo_get_value;
        iface->set_value_orig = gnome_db_entry_combo_set_value_orig;
        iface->get_value_orig = gnome_db_entry_combo_get_value_orig;
        iface->set_value_default = gnome_db_entry_combo_set_value_default;
        iface->set_attributes = gnome_db_entry_combo_set_attributes;
        iface->get_attributes = gnome_db_entry_combo_get_attributes;
        iface->get_handler = NULL;
        iface->expand_in_layout = gnome_db_entry_combo_expand_in_layout;
}


static void
gnome_db_entry_combo_class_init (GnomeDbEntryComboClass *class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = gnome_db_entry_combo_dispose;

	/* Properties */
        object_class->set_property = gnome_db_entry_combo_set_property;
        object_class->get_property = gnome_db_entry_combo_get_property;
        g_object_class_install_property (object_class, PROP_SET_DEFAULT_IF_INVALID,
					 g_param_spec_boolean ("set_default_if_invalid", NULL, NULL, FALSE,
                                                               (G_PARAM_READABLE | G_PARAM_WRITABLE)));
}

static void
real_combo_block_signals (GnomeDbEntryCombo *wid)
{
	g_signal_handlers_block_by_func (G_OBJECT (wid->priv->combo_entry),
					 G_CALLBACK (combo_contents_changed_cb), wid);
}

static void
real_combo_unblock_signals (GnomeDbEntryCombo *wid)
{
	g_signal_handlers_unblock_by_func (G_OBJECT (wid->priv->combo_entry),
					   G_CALLBACK (combo_contents_changed_cb), wid);
}


static void
gnome_db_entry_combo_emit_signal (GnomeDbEntryCombo *combo)
{
#ifdef debug_signal
	g_print (">> 'CONTENTS_MODIFIED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (combo), "contents_modified");
#ifdef debug_signal
	g_print ("<< 'CONTENTS_MODIFIED' from %s\n", __FUNCTION__);
#endif
}

static void
gnome_db_entry_combo_init (GnomeDbEntryCombo *combo)
{
	/* Private structure */
	combo->priv = g_new0 (GnomeDbEntryComboPriv, 1);
	combo->priv->combo_nodes = NULL;
	combo->priv->set_default_if_invalid = FALSE;
	combo->priv->combo_entry = NULL;
	combo->priv->data_valid = FALSE;
	combo->priv->null_forced = FALSE;
	combo->priv->default_forced = FALSE;
	combo->priv->null_possible = TRUE;
	combo->priv->default_possible = FALSE;
	combo->priv->show_actions = TRUE;
}

/**
 * gnome_db_entry_combo_new
 * @data_set: a #GnomeDbDataSet object
 * @node: a #GnomeDbDataSetNode structure, part of @data_set
 *
 * Creates a new #GnomeDbEntryCombo widget. The widget is a combo box which displays a
 * selectable list of items (the items come from the 'node->data_for_param' data model)
 *
 * The widget allows the simultaneuos selection of one or more values
 * (one for each 'node->params') while proposing potentially "more readable" choices.
 * 
 * Returns: the new widget
 */
GtkWidget *
gnome_db_entry_combo_new (GnomeDbDataSet *data_set, GnomeDbDataSetNode *node)
{
	GObject *obj;
	GnomeDbEntryCombo *combo;
	GSList *list;
	GSList *values;
	GtkWidget *entry;
	gboolean null_possible;

	g_return_val_if_fail (data_set && IS_GNOME_DB_DATA_SET (data_set), NULL);
	g_return_val_if_fail (node, NULL);
	g_return_val_if_fail (g_slist_find (data_set->nodes, node), NULL);
	g_return_val_if_fail (node->params, NULL);
	g_return_val_if_fail (node->data_for_param && IS_GNOME_DB_DATA_MODEL (node->data_for_param), NULL);

	obj = g_object_new (GNOME_DB_TYPE_ENTRY_COMBO, NULL);
	combo = GNOME_DB_ENTRY_COMBO (obj);

	g_object_ref (node->data_for_param);
	if (gnome_db_data_model_get_status (node->data_for_param) & 
	    (GNOME_DB_DATA_MODEL_CAN_BE_REFRESHED | GNOME_DB_DATA_MODEL_NEEDS_INIT_REFRESH))
		gnome_db_data_model_refresh (node->data_for_param, NULL);

	/* create the ComboNode structures, 
	 * and use the values provided by the parameters to display the correct row */
	null_possible = TRUE;
	values = NULL;
	list = node->params;
	while (list) {
		ComboNode *cnode = g_new0 (ComboNode, 1);
		
		cnode->param = GNOME_DB_PARAMETER (list->data);
		g_object_ref (cnode->param);
		cnode->value = gnome_db_parameter_get_value (cnode->param);
		cnode->position = GPOINTER_TO_INT (g_hash_table_lookup (node->pos_for_param, list->data));
		combo->priv->combo_nodes = g_slist_append (combo->priv->combo_nodes, cnode);

		values = g_slist_append (values, cnode->value);
		if (gnome_db_parameter_get_not_null (cnode->param))
			null_possible = FALSE;

		list = g_slist_next (list);
	}
	combo->priv->null_possible = null_possible;

	combo->priv->info = utility_data_set_node_info_create (data_set, node);

	/* create the combo box itself */
	entry = gnome_db_combo_new_with_model (GDA_DATA_MODEL (node->data_for_param), 
					       combo->priv->info->shown_n_cols, combo->priv->info->shown_cols_index);
	g_signal_connect (G_OBJECT (entry), "changed",
			  G_CALLBACK (combo_contents_changed_cb), combo);

	gnome_db_entry_shell_pack_entry (GNOME_DB_ENTRY_SHELL (combo), entry);
	gtk_widget_show (entry);
	combo->priv->combo_entry = entry;

	gnome_db_combo_set_values_ext (GNOME_DB_COMBO (entry), values, NULL);
	g_slist_free (values);
	gnome_db_combo_add_undef_choice (GNOME_DB_COMBO (entry), combo->priv->null_possible);

	return GTK_WIDGET (obj);
}

static void
choose_auto_default_value (GnomeDbEntryCombo *combo)
{
	gint pos = 0;
	if (combo->priv->null_possible)
		pos ++;
	
	gtk_combo_box_set_active (GTK_COMBO_BOX (combo->priv->combo_entry), pos);
}

static void
gnome_db_entry_combo_dispose (GObject *object)
{
	GnomeDbEntryCombo *combo;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_ENTRY_COMBO (object));

	combo = GNOME_DB_ENTRY_COMBO (object);

	if (combo->priv) {		
		if (combo->priv->info) {
			utility_data_set_node_info_destroy (combo->priv->info);
			combo->priv->info = NULL;
		}
				
		if (combo->priv->combo_nodes) {
			GSList *list= combo->priv->combo_nodes;
			while (list) {
				ComboNode *node = COMBO_NODE (list->data);
				g_object_unref (G_OBJECT (node->param));
				if (node->value)
					node->value = NULL; /* don't free that value since we have not copied it */
				if (node->value_orig)
					gda_value_free (node->value_orig);
				if (node->value_default)
					gda_value_free (node->value_default);
				g_free (node);
				list = g_slist_next (list);
			}
			g_slist_free (combo->priv->combo_nodes);
			combo->priv->combo_nodes = NULL;
		}

		g_free (combo->priv);
		combo->priv = NULL;
	}
	
	/* for the parent class */
	parent_class->dispose (object);
}

static void 
gnome_db_entry_combo_set_property (GObject              *object,
				   guint                 param_id,
				   const GValue         *value,
				   GParamSpec           *pspec)
{
	GnomeDbEntryCombo *combo = GNOME_DB_ENTRY_COMBO (object);
	if (combo->priv) {
		switch (param_id) {
		case PROP_SET_DEFAULT_IF_INVALID:
			if (combo->priv->set_default_if_invalid != g_value_get_boolean (value)) {
				guint attrs;

				combo->priv->set_default_if_invalid = g_value_get_boolean (value);
				attrs = gnome_db_data_entry_get_attributes (GNOME_DB_DATA_ENTRY (combo));

				if (combo->priv->set_default_if_invalid && (attrs & GNOME_DB_VALUE_DATA_NON_VALID)) 
					choose_auto_default_value (combo);
			}
			break;
		}
	}
}

static void
gnome_db_entry_combo_get_property (GObject              *object,
				   guint                 param_id,
				   GValue               *value,
				   GParamSpec           *pspec)
{
	GnomeDbEntryCombo *combo = GNOME_DB_ENTRY_COMBO (object);
	if (combo->priv) {
		switch (param_id) {
		case PROP_SET_DEFAULT_IF_INVALID:
			g_value_set_boolean (value, combo->priv->set_default_if_invalid);
			break;
		}
	}
}

static void
combo_contents_changed_cb (GnomeDbCombo *entry, GnomeDbEntryCombo *combo)
{
	if (gnome_db_combo_undef_selected (GNOME_DB_COMBO (combo->priv->combo_entry))) /* Set to NULL? */ {
		gnome_db_entry_combo_set_values (combo, NULL);
		gnome_db_entry_combo_emit_signal (combo);
	}
	else {
		GtkTreeIter iter;
		GSList *list;
		GtkTreeModel *model;

		combo->priv->null_forced = FALSE;
		combo->priv->default_forced = FALSE;
		combo->priv->data_valid = TRUE;
			
		/* updating the node's values */
		g_assert (gtk_combo_box_get_active_iter (GTK_COMBO_BOX (combo->priv->combo_entry), &iter));
		
		model = gtk_combo_box_get_model (GTK_COMBO_BOX (combo->priv->combo_entry));
		list = combo->priv->combo_nodes;
		while (list) {
			ComboNode *node = COMBO_NODE (list->data);

			gtk_tree_model_get (model, &iter, node->position, &(node->value), -1);
			/*g_print ("%s(): Set Combo Node value to %s\n", __FUNCTION__,
			  node->value ? gda_value_stringify (node->value) : "Null");*/
			list = g_slist_next (list);
		}
		
		g_signal_emit_by_name (G_OBJECT (combo), "status_changed");
		gnome_db_entry_combo_emit_signal (combo);
	}
}

/**
 * gnome_db_entry_combo_set_values
 * @combo: a #GnomeDbEntryCombo widet
 * @values: a list of #GdaValue values, or %NULL
 *
 * Sets the values of @combo to the specified ones. None of the
 * values provided in the list is modified.
 *
 * @values holds a list of #GdaValue values, one for each parameter that is present in the @node argument
 * of the gnome_db_entry_combo_new() function which created @combo.
 *
 * An error can occur when there is no corresponding value(s) to be displayed
 * for the provided values.
 *
 * If @values is %NULL, then the entry itself is set to NULL;
 *
 * Returns: TRUE if no error occured.
 */
gboolean
gnome_db_entry_combo_set_values (GnomeDbEntryCombo *combo, GSList *values)
{
	gboolean err = FALSE;
	gboolean allnull = TRUE;
	GSList *list;

	g_return_val_if_fail (combo && IS_GNOME_DB_ENTRY_COMBO (combo), FALSE);
	g_return_val_if_fail (combo->priv, FALSE);

	/* try to determine if all the values are NULL or of type GDA_VALUE_TYPE_NULL */
	list = values;
	while (list && allnull) {
		if (list->data && (gda_value_get_type ((GdaValue *)list->data) != GDA_VALUE_TYPE_NULL))
			allnull = FALSE;
		list = g_slist_next (list);
	}

	/* actual values settings */
	if (!allnull) {
		GtkTreeIter iter;
		GtkTreeModel *model;
		GSList *list;

		g_return_val_if_fail (g_slist_length (values) == g_slist_length (combo->priv->combo_nodes), FALSE);
		
		model = gtk_combo_box_get_model (GTK_COMBO_BOX (combo->priv->combo_entry));
		if (gnome_db_data_proxy_get_iter_from_values (GNOME_DB_DATA_PROXY (model), &iter, 
							      values, combo->priv->info->ref_cols_index)) {
			ComboNode *node;

			real_combo_block_signals (combo);
			gtk_combo_box_set_active_iter (GTK_COMBO_BOX (combo->priv->combo_entry), &iter);
			real_combo_unblock_signals (combo);

			/* adjusting the values */
			list = combo->priv->combo_nodes;
			while (list) {
				node = COMBO_NODE (list->data);
				gtk_tree_model_get (model, &iter, node->position, & (node->value), -1);
				list = g_slist_next (list);
			}
			
			combo->priv->null_forced = FALSE;
			combo->priv->default_forced = FALSE;
		}
		else
			/* values not found */
			err = TRUE;
	}
	else  { /* set to NULL */
		if (combo->priv->null_possible) {
			GSList *list;

			real_combo_block_signals (combo);
			gtk_combo_box_set_active (GTK_COMBO_BOX (combo->priv->combo_entry), -1);
			real_combo_unblock_signals (combo);

			/* adjusting the values */
			list = combo->priv->combo_nodes;
			while (list) {
				COMBO_NODE (list->data)->value = NULL;
				list = g_slist_next (list);
			}

			combo->priv->null_forced = TRUE;
			combo->priv->default_forced = FALSE;
		}
		else 
			err = TRUE;
	}

	combo->priv->data_valid = !err;
	g_signal_emit_by_name (G_OBJECT (combo), "status_changed");

	if (!err) 
		/* notify the status and contents changed */
		gnome_db_entry_combo_emit_signal (combo);

	return !err;
}

/**
 * gnome_db_entry_combo_get_values
 * @combo: a #GnomeDbEntryCombo widet
 *
 * Get the values stored within @combo. The returned values are a copy of the ones
 * within @combo, so they must be freed afterwards, the same for the list.
 *
 * Returns: a new list of values
 */
GSList *
gnome_db_entry_combo_get_values (GnomeDbEntryCombo *combo)
{
	GSList *list;
	GSList *retval = NULL;
	g_return_val_if_fail (combo && IS_GNOME_DB_ENTRY_COMBO (combo), NULL);
	g_return_val_if_fail (combo->priv, NULL);

	list = combo->priv->combo_nodes;
	while (list) {
		ComboNode *node = COMBO_NODE (list->data);
		
		if (node->value) 
			retval = g_slist_append (retval, gda_value_copy (node->value));
		else 
			retval = g_slist_append (retval, gda_value_new_null ());

		list = g_slist_next (list);
	}

	return retval;
}

/**
 * gnome_db_entry_combo_set_values_orig
 * @combo: a #GnomeDbEntryCombo widet
 * @values: a list of #GdaValue values
 *
 * Sets the original values of @combo to the specified ones. None of the
 * values provided in the list is modified.
 */
void
gnome_db_entry_combo_set_values_orig (GnomeDbEntryCombo *combo, GSList *values)
{
	GSList *list;

	g_return_if_fail (combo && IS_GNOME_DB_ENTRY_COMBO (combo));
	g_return_if_fail (combo->priv);

	/* rem: we don't care if the values can be set or not, we just try */
	gnome_db_entry_combo_set_values (combo, values);

	/* clean all the orig values */
	list = combo->priv->combo_nodes;
	while (list) {
		ComboNode *node = COMBO_NODE (list->data);
		if (node->value_orig) {
			gda_value_free (node->value_orig);
			node->value_orig = NULL;
		}
		
		list = g_slist_next (list);
	}

	if (values) {
		GSList *nodes;
		GSList *argptr;
		const GdaValue *arg_value;
		gboolean equal = TRUE;
		
		g_return_if_fail (g_slist_length (values) == g_slist_length (combo->priv->combo_nodes));
		
		/* 
		 * first make sure the value types are the same as for the data model 
		 */
		nodes = combo->priv->combo_nodes;
		argptr = values;
		while (argptr && nodes && equal) {
			GdaColumn *attrs;
			GdaValueType type=GDA_VALUE_TYPE_NULL;
			
			attrs = gda_data_model_describe_column (combo->priv->info->data_model, 
								COMBO_NODE (nodes->data)->position);
			arg_value = (GdaValue*) (argptr->data);
			
			if (arg_value)
				type = gda_value_get_type (arg_value);
			equal = (type == gda_column_get_gdatype (attrs));
			
			nodes = g_slist_next (nodes);
			argptr = g_slist_next (argptr);
		}
		
		/* 
		 * then, actual copy of the values
		 */
		if (equal) {
			nodes = combo->priv->combo_nodes;
			argptr = values;
			while (argptr && nodes && equal) {
				if (argptr->data)
					COMBO_NODE (nodes->data)->value_orig = gda_value_copy ((GdaValue*) (argptr->data));
				nodes = g_slist_next (nodes);
				argptr = g_slist_next (argptr);
			}
		} 
	}
}

/**
 * gnome_db_entry_combo_get_values_orig
 * @combo: a #GnomeDbEntryCombo widet
 *
 * Get the original values stored within @combo. The returned values are the ones
 * within @combo, so they must not be freed afterwards; the list has to be freed afterwards.
 *
 * Returns: a new list of values
 */
GSList *
gnome_db_entry_combo_get_values_orig (GnomeDbEntryCombo *combo)
{
	GSList *list;
	GSList *retval = NULL;
	gboolean allnull = TRUE;

	g_return_val_if_fail (combo && IS_GNOME_DB_ENTRY_COMBO (combo), NULL);
	g_return_val_if_fail (combo->priv, NULL);

	list = combo->priv->combo_nodes;
	while (list) {
		ComboNode *node = COMBO_NODE (list->data);

		if (node->value_orig && 
		    (gda_value_get_type (node->value_orig) != GDA_VALUE_TYPE_NULL))
			allnull = FALSE;
		
		retval = g_slist_append (retval, node->value_orig);
		
		list = g_slist_next (list);
	}

	if (allnull) {
		g_slist_free (retval);
		retval = NULL;
	}

	return retval;
}

/**
 * gnome_db_entry_combo_set_values_default
 * @combo: a #GnomeDbEntryCombo widet
 * @values: a list of #GdaValue values
 *
 * Sets the default values of @combo to the specified ones. None of the
 * values provided in the list is modified.
 */
void
gnome_db_entry_combo_set_values_default (GnomeDbEntryCombo *combo, GSList *values)
{
	g_return_if_fail (combo && IS_GNOME_DB_ENTRY_COMBO (combo));
	g_return_if_fail (combo->priv);
	TO_IMPLEMENT;
}


/* 
 * GnomeDbDataEntry Interface implementation 
 */

static void
gnome_db_entry_combo_set_value (GnomeDbDataEntry *iface, const GdaValue *value)
{
	GnomeDbEntryCombo *combo;

        g_return_if_fail (iface && IS_GNOME_DB_ENTRY_COMBO (iface));
        combo = GNOME_DB_ENTRY_COMBO (iface);
        g_return_if_fail (combo->priv);
        g_return_if_fail (!value ||
                          (value && (gda_value_isa (value, GDA_VALUE_TYPE_LIST) ||
                                     gda_value_isa (value, GDA_VALUE_TYPE_LIST))));

	TO_IMPLEMENT;
}

static GdaValue *
gnome_db_entry_combo_get_value (GnomeDbDataEntry *iface)
{
        GnomeDbEntryCombo *combo;
	
        g_return_val_if_fail (iface && IS_GNOME_DB_ENTRY_COMBO (iface), NULL);
        combo = GNOME_DB_ENTRY_COMBO (iface);
        g_return_val_if_fail (combo->priv, NULL);
	
	TO_IMPLEMENT;
	
	return NULL;
}

static void
gnome_db_entry_combo_set_value_orig (GnomeDbDataEntry *iface, const GdaValue * value)
{
        GnomeDbEntryCombo *combo;
	
        g_return_if_fail (iface && IS_GNOME_DB_ENTRY_COMBO (iface));
        combo = GNOME_DB_ENTRY_COMBO (iface);
        g_return_if_fail (combo->priv);

	TO_IMPLEMENT;
}

static const GdaValue *
gnome_db_entry_combo_get_value_orig (GnomeDbDataEntry *iface)
{
        GnomeDbEntryCombo *combo;
	
        g_return_val_if_fail (iface && IS_GNOME_DB_ENTRY_COMBO (iface), NULL);
        combo = GNOME_DB_ENTRY_COMBO (iface);
        g_return_val_if_fail (combo->priv, NULL);
                     
	TO_IMPLEMENT;

	return NULL;
}

static void
gnome_db_entry_combo_set_value_default (GnomeDbDataEntry *iface, const GdaValue * value)
{
        GnomeDbEntryCombo *combo;
	
        g_return_if_fail (iface && IS_GNOME_DB_ENTRY_COMBO (iface));
        combo = GNOME_DB_ENTRY_COMBO (iface);
        g_return_if_fail (combo->priv);
	
	TO_IMPLEMENT;
}


static void
gnome_db_entry_combo_set_attributes (GnomeDbDataEntry *iface, guint attrs, guint mask)
{
	GnomeDbEntryCombo *combo;

	g_return_if_fail (iface && IS_GNOME_DB_ENTRY_COMBO (iface));
	combo = GNOME_DB_ENTRY_COMBO (iface);
	g_return_if_fail (combo->priv);

	/* Setting to NULL */
	if (mask & GNOME_DB_VALUE_IS_NULL) {
		if ((mask & GNOME_DB_VALUE_CAN_BE_NULL) &&
		    !(attrs & GNOME_DB_VALUE_CAN_BE_NULL))
			g_return_if_reached ();
		if (attrs & GNOME_DB_VALUE_IS_NULL) {
			gnome_db_entry_combo_set_values (combo, NULL);
			
			/* if default is set, see if we can keep it that way */
			if (combo->priv->default_forced) {
				GSList *list;
				gboolean allnull = TRUE;
				list = combo->priv->combo_nodes;
				while (list && allnull) {
					if (COMBO_NODE (list->data)->value_default && 
					    (gda_value_get_type (COMBO_NODE (list->data)->value_default) != 
					     GDA_VALUE_TYPE_NULL))
						allnull = FALSE;
					list = g_slist_next (list);
				}

				if (!allnull)
					combo->priv->default_forced = FALSE;
			}

			gnome_db_entry_combo_emit_signal (combo);
			return;
		}
		else {
			combo->priv->null_forced = FALSE;
			gnome_db_entry_combo_emit_signal (combo);
		}
	}

	/* Can be NULL ? */
	if (mask & GNOME_DB_VALUE_CAN_BE_NULL)
		if (combo->priv->null_possible != (attrs & GNOME_DB_VALUE_CAN_BE_NULL) ? TRUE : FALSE) {
			combo->priv->null_possible = (attrs & GNOME_DB_VALUE_CAN_BE_NULL) ? TRUE : FALSE;
			gnome_db_combo_add_undef_choice (GNOME_DB_COMBO (combo->priv->combo_entry),
							 combo->priv->null_possible);		 
		}


	/* Setting to DEFAULT */
	if (mask & GNOME_DB_VALUE_IS_DEFAULT) {
		if ((mask & GNOME_DB_VALUE_CAN_BE_DEFAULT) &&
		    !(attrs & GNOME_DB_VALUE_CAN_BE_DEFAULT))
			g_return_if_reached ();
		if (attrs & GNOME_DB_VALUE_IS_DEFAULT) {
			GList *tmplist = NULL;
			GSList *list;
			
			list = combo->priv->combo_nodes;
			while (list) {
				tmplist = g_list_append (tmplist, COMBO_NODE (list->data)->value_default);
				list = g_slist_next (list);
			}
			gnome_db_entry_combo_set_values (combo, tmplist);
			g_list_free (tmplist);

			/* if NULL is set, see if we can keep it that way */
			if (combo->priv->null_forced) {
				GSList *list;
				gboolean allnull = TRUE;
				list = combo->priv->combo_nodes;
				while (list && allnull) {
					if (COMBO_NODE (list->data)->value_default && 
					    (gda_value_get_type (COMBO_NODE (list->data)->value_default) != 
					     GDA_VALUE_TYPE_NULL))
						allnull = FALSE;
					list = g_slist_next (list);
				}
				
				if (!allnull)
					combo->priv->null_forced = FALSE;
			}

			combo->priv->default_forced = TRUE;
			gnome_db_entry_combo_emit_signal (combo);
			return;
		}
		else {
			combo->priv->default_forced = FALSE;
			gnome_db_entry_combo_emit_signal (combo);
		}
	}

	/* Can be DEFAULT ? */
	if (mask & GNOME_DB_VALUE_CAN_BE_DEFAULT)
		combo->priv->default_possible = (attrs & GNOME_DB_VALUE_CAN_BE_DEFAULT) ? TRUE : FALSE;
	
	/* Modified ? */
	if (mask & GNOME_DB_VALUE_IS_UNCHANGED) {
		if (attrs & GNOME_DB_VALUE_IS_UNCHANGED) {
			GList *tmplist = NULL;
			GSList *list;
			
			list = combo->priv->combo_nodes;
			while (list) {
				tmplist = g_list_append (tmplist, COMBO_NODE (list->data)->value_orig);
				list = g_slist_next (list);
			}
				
			gnome_db_entry_combo_set_values (combo, tmplist);
			g_list_free (tmplist);
			combo->priv->default_forced = FALSE;
			gnome_db_entry_combo_emit_signal (combo);
		}
	}

	/* Actions buttons ? */
	if (mask & GNOME_DB_VALUE_ACTIONS_SHOWN) {
		GValue *gval;
		combo->priv->show_actions = (attrs & GNOME_DB_VALUE_ACTIONS_SHOWN) ? TRUE : FALSE;
		
		gval = g_new0 (GValue, 1);
		g_value_init (gval, G_TYPE_BOOLEAN);
		g_value_set_boolean (gval, combo->priv->show_actions);
		g_object_set_property (G_OBJECT (combo), "actions", gval);
		g_free (gval);
	}

	/* NON WRITABLE attributes */
	if (mask & GNOME_DB_VALUE_DATA_NON_VALID) 
		g_warning ("Can't force a GnomeDbDataEntry to be invalid!");

	if (mask & GNOME_DB_VALUE_HAS_VALUE_ORIG)
		g_warning ("Having an original value is not a write attribute on GnomeDbDataEntry!");

	g_signal_emit_by_name (G_OBJECT (combo), "status_changed");
}

static guint
gnome_db_entry_combo_get_attributes (GnomeDbDataEntry *iface)
{
	guint retval = 0;
	GnomeDbEntryCombo *combo;
	GSList *list;
	GSList *list2;
	gboolean isnull = TRUE;
	gboolean isunchanged = TRUE;
	gboolean orig_value_exists = FALSE;

	g_return_val_if_fail (iface && IS_GNOME_DB_ENTRY_COMBO (iface), 0);
	combo = GNOME_DB_ENTRY_COMBO (iface);
	g_return_val_if_fail (combo->priv, 0);

	list = combo->priv->combo_nodes;
	while (list) {
		gboolean changed = FALSE;

		/* NULL? */
		if (COMBO_NODE (list->data)->value &&
		    (gda_value_get_type (COMBO_NODE (list->data)->value) != GDA_VALUE_TYPE_NULL))
			isnull = FALSE;
		
		/* is unchanged */
		if (COMBO_NODE (list->data)->value_orig) {
			orig_value_exists = TRUE;
			
			if (COMBO_NODE (list->data)->value && 
			    (gda_value_get_type (COMBO_NODE (list->data)->value) == 
			     gda_value_get_type (COMBO_NODE (list->data)->value_orig))) {
				if (gda_value_get_type (COMBO_NODE (list->data)->value) == GDA_VALUE_TYPE_NULL) 
					changed = FALSE;
				else {
					if (gda_value_compare (COMBO_NODE (list->data)->value, 
							       COMBO_NODE (list->data)->value_orig))
						changed = TRUE;
				}
			}
			else
				changed = TRUE;
		}
		
		if (changed || 
		    (!orig_value_exists && !isnull))
			isunchanged = FALSE;
		
		list = g_slist_next (list);
	}

	if (isunchanged)
		retval = retval | GNOME_DB_VALUE_IS_UNCHANGED;

	if (isnull || combo->priv->null_forced)
		retval = retval | GNOME_DB_VALUE_IS_NULL;

	/* can be NULL? */
	if (combo->priv->null_possible) 
		retval = retval | GNOME_DB_VALUE_CAN_BE_NULL;
	
	/* is default */
	if (combo->priv->default_forced)
		retval = retval | GNOME_DB_VALUE_IS_DEFAULT;
	
	/* can be default? */
	if (combo->priv->default_possible)
		retval = retval | GNOME_DB_VALUE_CAN_BE_DEFAULT;
	

	/* actions shown */
	if (combo->priv->show_actions)
		retval = retval | GNOME_DB_VALUE_ACTIONS_SHOWN;

	/* data valid? */
	if (! combo->priv->data_valid)
		retval = retval | GNOME_DB_VALUE_DATA_NON_VALID;
	else {
		GSList *nodes;
		gboolean allnull = TRUE;
		
		nodes = combo->priv->combo_nodes;
 		while (nodes) {
			ComboNode *node = COMBO_NODE (nodes->data);

			/* all the nodes are NULL ? */
			if (node->value && (gda_value_get_type (node->value) != GDA_VALUE_TYPE_NULL))
				allnull = FALSE;

			nodes = g_slist_next (nodes);
		}

		if ((allnull && !combo->priv->null_possible) ||
		    (combo->priv->null_forced && !combo->priv->null_possible))
			retval = retval | GNOME_DB_VALUE_DATA_NON_VALID;
	}

	/* has original value? */
	list2 = gnome_db_entry_combo_get_values_orig (combo);
	if (list2) {
		retval = retval | GNOME_DB_VALUE_HAS_VALUE_ORIG;
		g_slist_free (list2);
	}

	return retval;
}


static gboolean
gnome_db_entry_combo_expand_in_layout (GnomeDbDataEntry *iface)
{
	GnomeDbEntryCombo *combo;

	g_return_val_if_fail (iface && IS_GNOME_DB_ENTRY_COMBO (iface), FALSE);
	combo = GNOME_DB_ENTRY_COMBO (iface);
	g_return_val_if_fail (combo->priv, FALSE);

	return FALSE;
}
