/* gnome-db-custom-layout.c
 *
 * Copyright (C) 2004 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-custom-layout.h"
#include "gnome-db-layout.h"
#include "gnome-db-ref-base.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-referer.h"
#include "gnome-db-query.h"
#include "gnome-db-target.h"
#include "gnome-db-field.h"
#include "gnome-db-data-set.h"
#include <string.h>

/* 
 * Main static functions 
 */
static void gnome_db_custom_layout_class_init (GnomeDbCustomLayoutClass * class);
static void gnome_db_custom_layout_init (GnomeDbCustomLayout * srv);
static void gnome_db_custom_layout_dispose (GObject   * object);
static void gnome_db_custom_layout_finalize (GObject   * object);

#ifdef debug
static void gnome_db_custom_layout_dump                     (GnomeDbCustomLayout *context, guint offset);
#endif

/* XML storage interface */
static void        gnome_db_custom_layout_xml_storage_init (GnomeDbXmlStorageIface *iface);
static gchar      *gnome_db_custom_layout_get_xml_id (GnomeDbXmlStorage *iface);
static xmlNodePtr  gnome_db_custom_layout_save_to_xml (GnomeDbXmlStorage *iface, GError **error);
static gboolean    gnome_db_custom_layout_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);

/* Referer interface */
static void        gnome_db_custom_layout_referer_init        (GnomeDbRefererIface *iface);
static gboolean    gnome_db_custom_layout_activate            (GnomeDbReferer *iface);
static void        gnome_db_custom_layout_deactivate          (GnomeDbReferer *iface);
static gboolean    gnome_db_custom_layout_is_active           (GnomeDbReferer *iface);
static GSList     *gnome_db_custom_layout_get_ref_objects     (GnomeDbReferer *iface);
static void        gnome_db_custom_layout_replace_refs        (GnomeDbReferer *iface, GHashTable *replacements);


/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/*
 * Structure to hold a Glade import spec
 */
typedef struct {
	gchar     *box_name;
	gboolean   show_actions;
	GSList    *dest_list; /* list of GnomeDbRefBase objects */
} GladeBox;
#define GLADE_BOX(x) ((GladeBox*)(x))

static GladeBox *glade_box_new (const gchar *name, gboolean show_actions);
static void      glade_box_destroy (GladeBox *box);


/*
 * Structure to hold parameters connections between GnomeDbDataWidget widgets
 */
typedef struct {
	GnomeDbRefBase *src_layout;
	GnomeDbRefBase *src_field;
	GnomeDbRefBase *dest_layout;
	GnomeDbRefBase *dest_field;
} LayoutConnect;
#define LAYOUT_CONNECT(x) ((LayoutConnect *)(x))
static LayoutConnect *layout_connect_new (GnomeDbCustomLayout *layout, GnomeDbDict *dict);
static void           layout_connect_destroy (GnomeDbCustomLayout *layout, LayoutConnect *lc);


/* private structure */
struct _GnomeDbCustomLayoutPrivate
{
	GnomeDbCustomLayoutType type;

	/* contents of the layout */
	union {
		/* sub layouts, for GNOME_DB_CUSTOM_LAYOUT_LAYOUT */
		struct {
			GSList           *children; /* list of GnomeDbCustomLayout objects, owned here */
			GSList           *connects; /* list of LayoutConnect structures */
		}                layout;
		
		struct {
			/* for all the GnomeDbWork* widgets */
			GnomeDbQuery          *query;       /* owned query, may be NULL */
			GnomeDbRefBase        *ref_query;      /* reference to another query, even 'query' or 'query_extra' */
			guint             mode;
			GnomeDbRefBase        *ref_modif;

			/* For GnomeDbMatrix only */
			GnomeDbQuery          *query_extra; /* owned query, may be NULL */
			GnomeDbRefBase        *ref_query_extra;/* reference to another query, even 'query' or 'query_extra' */
			GnomeDbRefBase        *ref_cols_target;
			GnomeDbRefBase        *ref_rows_target;
			GnomeDbMatrixType      view_type;
		}                work_iface;
	} contents;
	GSList          *referer_objects; /* list of all the GnomeDbRefBase objects owned by the GnomeDbCustomLayout */
	
	/* glade layout */
	gchar           *filename;
	gchar           *root_widget;
	GSList          *boxes; /* list of GladeBox structures */
};



/* module error */
GQuark gnome_db_custom_layout_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_custom_layout_error");
	return quark;
}


GType
gnome_db_custom_layout_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbCustomLayoutClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_custom_layout_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbCustomLayout),
			0,
			(GInstanceInitFunc) gnome_db_custom_layout_init
		};

		static const GInterfaceInfo xml_storage_info = {
			(GInterfaceInitFunc) gnome_db_custom_layout_xml_storage_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo referer_info = {
			(GInterfaceInitFunc) gnome_db_custom_layout_referer_init,
			NULL,
			NULL
		};
		
		type = g_type_register_static (GNOME_DB_TYPE_BASE, "GnomeDbCustomLayout", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_TYPE_XML_STORAGE, &xml_storage_info);
		g_type_add_interface_static (type, GNOME_DB_TYPE_REFERER, &referer_info);
	}
	return type;
}

static void 
gnome_db_custom_layout_xml_storage_init (GnomeDbXmlStorageIface *iface)
{
	iface->get_xml_id = gnome_db_custom_layout_get_xml_id;
	iface->save_to_xml = gnome_db_custom_layout_save_to_xml;
	iface->load_from_xml = gnome_db_custom_layout_load_from_xml;
}

static void
gnome_db_custom_layout_referer_init (GnomeDbRefererIface *iface)
{
	iface->activate = gnome_db_custom_layout_activate;
	iface->deactivate = gnome_db_custom_layout_deactivate;
	iface->is_active = gnome_db_custom_layout_is_active;
	iface->get_ref_objects = gnome_db_custom_layout_get_ref_objects;
	iface->replace_refs = gnome_db_custom_layout_replace_refs;
}

static void
gnome_db_custom_layout_class_init (GnomeDbCustomLayoutClass *class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = gnome_db_custom_layout_dispose;
	object_class->finalize = gnome_db_custom_layout_finalize;
	
	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_custom_layout_dump;
#endif
}

static void
gnome_db_custom_layout_init (GnomeDbCustomLayout *gnome_db_custom_layout)
{
	gnome_db_custom_layout->priv = g_new0 (GnomeDbCustomLayoutPrivate, 1);
}

/**
 * gnome_db_custom_layout_new
 * @dict: a #GnomeDbDict object
 *
 * Creates a new #GnomeDbCustomLayout object
 *
 * Returns: a new #GnomeDbCustomLayout object
 */
GObject *
gnome_db_custom_layout_new (GnomeDbDict *dict)
{
	GObject *obj;
	GnomeDbCustomLayout *gnome_db_custom_layout;
	guint id;

	g_return_val_if_fail (!dict || IS_GNOME_DB_DICT (dict), NULL);

	obj = g_object_new (GNOME_DB_TYPE_CUSTOM_LAYOUT, "dict", ASSERT_DICT (dict), NULL);
        gnome_db_custom_layout = GNOME_DB_CUSTOM_LAYOUT (obj);

	g_object_get (G_OBJECT (ASSERT_DICT (dict)), "layout_serial", &id, NULL);
	gnome_db_base_set_id (GNOME_DB_BASE (obj), id);

	gnome_db_dict_declare_layout (ASSERT_DICT (dict), gnome_db_custom_layout);

	return obj;
}

/**
 * gnome_db_custom_layout_get_layout_type
 * @layout: a #GnomeDbCustomLayout object
 *
 * Get the kind of custom layout @layout is.
 *
 * Returns:
 */
GnomeDbCustomLayoutType
gnome_db_custom_layout_get_layout_type (GnomeDbCustomLayout *layout)
{
	g_return_val_if_fail (layout && IS_GNOME_DB_CUSTOM_LAYOUT (layout), GNOME_DB_CUSTOM_LAYOUT_FORM);
	g_return_val_if_fail (layout->priv, GNOME_DB_CUSTOM_LAYOUT_FORM);

	return layout->priv->type;
}

static void
gnome_db_custom_layout_dispose (GObject *object)
{
	GnomeDbCustomLayout *cl;
	GSList *list;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_CUSTOM_LAYOUT (object));

	cl = GNOME_DB_CUSTOM_LAYOUT (object);

	/* contents related */
	switch (cl->priv->type) {
	case GNOME_DB_CUSTOM_LAYOUT_LAYOUT:
		if (cl->priv->contents.layout.children) {
			list = cl->priv->contents.layout.children;
			while (list) {
				g_object_unref (list->data);
				list = g_slist_next (list);
			}
			g_slist_free (cl->priv->contents.layout.children);
			cl->priv->contents.layout.children = NULL;
		}
		if (cl->priv->contents.layout.connects) {
			list = cl->priv->contents.layout.connects;
			while (list) {
				layout_connect_destroy (cl, LAYOUT_CONNECT (list->data));
				list = g_slist_next (list);
			}
			g_slist_free (cl->priv->contents.layout.connects);
			cl->priv->contents.layout.connects = NULL;
		}
		break;
	case GNOME_DB_CUSTOM_LAYOUT_GRID:
	case GNOME_DB_CUSTOM_LAYOUT_FORM:
	case GNOME_DB_CUSTOM_LAYOUT_MATRIX:
		if (cl->priv->contents.work_iface.query) {
			g_object_unref (cl->priv->contents.work_iface.query);
			cl->priv->contents.work_iface.query = NULL;
		}
		if (cl->priv->contents.work_iface.query_extra) {
			g_object_unref (cl->priv->contents.work_iface.query_extra);
			cl->priv->contents.work_iface.query_extra = NULL;
		}
		if (cl->priv->contents.work_iface.ref_query) {
			g_object_unref (cl->priv->contents.work_iface.ref_query);
			cl->priv->contents.work_iface.ref_query = NULL;
		}
		if (cl->priv->contents.work_iface.ref_query_extra) {
			g_object_unref (cl->priv->contents.work_iface.ref_query_extra);
			cl->priv->contents.work_iface.ref_query_extra = NULL;
		}
		if (cl->priv->contents.work_iface.ref_modif) {
			g_object_unref (cl->priv->contents.work_iface.ref_modif);
			cl->priv->contents.work_iface.ref_modif = NULL;
		}
		if (cl->priv->contents.work_iface.ref_cols_target) {
			g_object_unref (cl->priv->contents.work_iface.ref_cols_target);
			cl->priv->contents.work_iface.ref_cols_target = NULL;
		}
		if (cl->priv->contents.work_iface.ref_rows_target) {
			g_object_unref (cl->priv->contents.work_iface.ref_rows_target);
			cl->priv->contents.work_iface.ref_rows_target = NULL;
		}
		break;
	}

	/* Glade related */
	if (cl->priv->filename) {
		g_free (cl->priv->filename);
		cl->priv->filename = NULL;
	}

	if (cl->priv->root_widget) {
		g_free (cl->priv->root_widget);
		cl->priv->root_widget = NULL;
	}

	if (cl->priv->boxes) {
		list = cl->priv->boxes;		
		while (list) {
			glade_box_destroy (GLADE_BOX (list->data));
			list = g_slist_next (list);
		}
		g_slist_free (cl->priv->boxes);
	}

	if (cl->priv->referer_objects) {
		g_slist_free (cl->priv->referer_objects);
		cl->priv->referer_objects = NULL;
	}

	/* parent class */
	parent_class->dispose (object);
}


static void
gnome_db_custom_layout_finalize (GObject *object)
{
	GnomeDbCustomLayout *cl;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_CUSTOM_LAYOUT (object));

	cl = GNOME_DB_CUSTOM_LAYOUT (object);
	if (cl->priv) {
		g_free (cl->priv);
		cl->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


#ifdef debug
static void
gnome_db_custom_layout_dump (GnomeDbCustomLayout *cl, guint offset)
{
	gchar *str;
        guint i;
	GSList *list;
	
	g_return_if_fail (cl && IS_GNOME_DB_CUSTOM_LAYOUT (cl));

        /* string for the offset */
        str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

        /* dump */
        if (cl->priv) {
                g_print ("%s" D_COL_H1 "GnomeDbCustomLayout" D_COL_NOR " %p (id=%s) ",
                         str, cl, gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (cl)));
		switch (cl->priv->type) {
		case GNOME_DB_CUSTOM_LAYOUT_LAYOUT:
			g_print ("GNOME_DB_CUSTOM_LAYOUT_LAYOUT\n");
			break;
		case GNOME_DB_CUSTOM_LAYOUT_GRID:
			g_print ("GNOME_DB_CUSTOM_LAYOUT_GRID\n");
			break;
		case GNOME_DB_CUSTOM_LAYOUT_FORM:
			g_print ("GNOME_DB_CUSTOM_LAYOUT_FORM\n");
			break;
		case GNOME_DB_CUSTOM_LAYOUT_MATRIX:
			g_print ("GNOME_DB_CUSTOM_LAYOUT_MATRIX\n");
			break;
		}

		switch (cl->priv->type) {
		case GNOME_DB_CUSTOM_LAYOUT_LAYOUT:
			list = cl->priv->contents.layout.children;
			if (list)
				g_print ("%s" D_COL_H1 "* Sub layouts:\n" D_COL_NOR, str);
			else
				g_print ("%s" D_COL_H1 "* No sub layout.\n" D_COL_NOR, str);
			while (list) {
				gnome_db_base_dump (GNOME_DB_BASE (list->data), offset+5);
				list = g_slist_next (list);
			}
			list = cl->priv->contents.layout.connects;
			if (list)
				g_print ("%s" D_COL_H1 "* Connections:\n" D_COL_NOR, str);
			else
				g_print ("%s" D_COL_H1 "* No connections.\n" D_COL_NOR, str);
			while (list) {
				g_print ("%s - Source layout:\n", str);
				gnome_db_base_dump (GNOME_DB_BASE (LAYOUT_CONNECT (list->data)->src_layout), offset+5);
				g_print ("%s - Source field:\n", str);
				gnome_db_base_dump (GNOME_DB_BASE (LAYOUT_CONNECT (list->data)->src_field), offset+5);
				g_print ("%s - Destination layout:\n", str);
				gnome_db_base_dump (GNOME_DB_BASE (LAYOUT_CONNECT (list->data)->dest_layout), offset+5);
				g_print ("%s - Destination field:\n", str);
				gnome_db_base_dump (GNOME_DB_BASE (LAYOUT_CONNECT (list->data)->dest_field), offset+5);
				list = g_slist_next (list);
			}
			break;
		case GNOME_DB_CUSTOM_LAYOUT_GRID:
		case GNOME_DB_CUSTOM_LAYOUT_FORM:
		case GNOME_DB_CUSTOM_LAYOUT_MATRIX:
			if (cl->priv->contents.work_iface.query) {
				g_print ("%s" D_COL_H1 "* Query:\n" D_COL_NOR, str);
				gnome_db_base_dump (GNOME_DB_BASE (cl->priv->contents.work_iface.query), offset+5);
			}
			if (cl->priv->contents.work_iface.query_extra) {
				g_print ("%s" D_COL_H1 "* Query_extra:\n" D_COL_NOR, str);
				gnome_db_base_dump (GNOME_DB_BASE (cl->priv->contents.work_iface.query_extra), offset+5);
			}
			if (cl->priv->contents.work_iface.ref_query) {
				g_print ("%s" D_COL_H1 "* Ref Query:\n" D_COL_NOR, str);
				gnome_db_base_dump (GNOME_DB_BASE (cl->priv->contents.work_iface.ref_query), offset+5);
			}
			if (cl->priv->contents.work_iface.ref_query_extra) {
				g_print ("%s" D_COL_H1 "* Ref Query_extra:\n" D_COL_NOR, str);
				gnome_db_base_dump (GNOME_DB_BASE (cl->priv->contents.work_iface.ref_query), offset+5);
			}
			g_print ("%s" D_COL_H1 "* Mode:" D_COL_NOR " %d\n", str, cl->priv->contents.work_iface.mode);
			if (cl->priv->contents.work_iface.ref_modif) {
				g_print ("%s" D_COL_H1 "* Modif [table|target]:\n" D_COL_NOR, str);
				gnome_db_base_dump (GNOME_DB_BASE (cl->priv->contents.work_iface.ref_modif), offset+5);
			}
			break;
		}

		if (cl->priv->filename && cl->priv->root_widget) {
			g_print ("%s" D_COL_H1 "* Glade file:" D_COL_NOR " %s, root_widget=%s\n", 
				 str, cl->priv->filename, cl->priv->root_widget);
			list = cl->priv->boxes;
			while (list) {
				GSList *list2;
				g_print ("%s   " D_COL_H1 "-> box" D_COL_NOR " %s for:\n",
					 str, GLADE_BOX (list->data)->box_name);
				list2 = GLADE_BOX (list->data)->dest_list;
				while (list2) {
					gnome_db_base_dump (GNOME_DB_BASE (list2->data), offset+6);
					list2 = g_slist_next (list2);
				}
				g_print ("\n");
				list = g_slist_next (list);
			}
		}
		else 
			g_print ("%s" D_COL_H1 "* No Glade file.\n" D_COL_NOR, str);
	}
        else
                g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, cl);
}
#endif

/**
 * gnome_db_custom_layout_get_widget
 * @layout: a #GnomeDbCustomLayout object
 * @error: location to store error, or %NULL
 *
 * Creates a widget from the "specifications" stored within @layout.
 *
 * Returns: a new #GnomeDbWorkLayout widget, or %NULL if an error occured.
 */
GtkWidget *
gnome_db_custom_layout_get_widget (GnomeDbCustomLayout *layout, GError **error)
{
	g_return_val_if_fail (layout && IS_GNOME_DB_CUSTOM_LAYOUT (layout), NULL);
	g_return_val_if_fail (layout->priv, NULL);

	return gnome_db_layout_new (layout);
}

/**
 * gnome_db_custom_layout_get_glade_instance
 * @layout: a #GnomeDbCustomLayout object
 * @root_widget: a place to store a pointer to the root widget of the #GladeXml instance
 * @box_widgets: a place to store the GHashTable for the #GtkBox widgets
 * @error: location to store error, or %NULL
 *
 * Builds a #GladeXML object, and if there is no error, creates a new #GHashTable and stores
 * it in @box_widgets, and stores a pointer to the root widget into @root_widget.
 *
 * Returns: a new #GladeXML object, or %NULL if an error occured or no Glade
 * file is to be used in @layout.
 */
GladeXML *
gnome_db_custom_layout_get_glade_instance (GnomeDbCustomLayout *layout, GtkWidget **root_widget, 
				     GHashTable **box_widgets, GError **error)
{
	GladeXML *xml = NULL;
	GHashTable *hash;
	GSList *box_list;
	gboolean err = FALSE;
	
	g_return_val_if_fail (layout && IS_GNOME_DB_CUSTOM_LAYOUT (layout), NULL);
	g_return_val_if_fail (layout->priv, NULL);
	g_return_val_if_fail (root_widget, NULL);
	g_return_val_if_fail (box_widgets, NULL);

	/* chech */
	if (! gnome_db_referer_activate (GNOME_DB_REFERER (layout))) {
		g_set_error (error,
			     GNOME_DB_CUSTOM_LAYOUT_ERROR,
			     GNOME_DB_CUSTOM_LAYOUT_GLADEXML_ERROR,
			     _("Missing required objects!"));
		*root_widget = NULL;
		*box_widgets = NULL;
		return NULL;
	}

	/* making the GladeXML object */
	if (layout->priv->filename && *layout->priv->filename) {
		if (! g_file_test (layout->priv->filename, G_FILE_TEST_EXISTS))
			g_set_error (error,
				     GNOME_DB_CUSTOM_LAYOUT_ERROR,
				     GNOME_DB_CUSTOM_LAYOUT_GLADEXML_ERROR,
				     _("XML file '%s' does not exist"), layout->priv->filename);
		else {
			xml = glade_xml_new (layout->priv->filename, layout->priv->root_widget, NULL);
			if (!xml) 
				g_set_error (error,
					     GNOME_DB_CUSTOM_LAYOUT_ERROR,
					     GNOME_DB_CUSTOM_LAYOUT_GLADEXML_ERROR,
					     _("Can't use XML file '%s' with root widget '%s'"), 
					     layout->priv->filename, layout->priv->root_widget);
		}
	}

	if (!xml) {
		*root_widget = NULL;
		*box_widgets = NULL;
		return NULL;
	}

	/* creating the box_widgets hash table */
	hash = g_hash_table_new (NULL, NULL);
	if (!(*root_widget = glade_xml_get_widget (xml, layout->priv->root_widget))) {
		err = TRUE;
		g_set_error (error,
			     GNOME_DB_CUSTOM_LAYOUT_ERROR,
			     GNOME_DB_CUSTOM_LAYOUT_GLADEXML_ERROR,
			     _("Can't find widget named '%s'"), layout->priv->root_widget);
	}

	box_list = layout->priv->boxes;
	while (box_list && !err) {
		GtkWidget *box = glade_xml_get_widget (xml, GLADE_BOX (box_list->data)->box_name);

		if (box && GTK_IS_BOX (box)) {
			GSList *dest_list = GLADE_BOX (box_list->data)->dest_list;
			while (dest_list) {
				g_hash_table_insert (hash, 
						     gnome_db_ref_base_get_ref_object (GNOME_DB_REF_BASE (dest_list->data)), box);
				dest_list = g_slist_next (dest_list);
			}
			if (GLADE_BOX (box_list->data)->show_actions)
				g_object_set_data (G_OBJECT (box), "show_actions", GINT_TO_POINTER (TRUE));
		}
		else {
			err = TRUE;
			if (!box)
				g_set_error (error,
					     GNOME_DB_CUSTOM_LAYOUT_ERROR,
					     GNOME_DB_CUSTOM_LAYOUT_GLADEXML_ERROR,
					     _("Can't find widget named '%s'"), GLADE_BOX (box_list->data)->box_name);
			else
				g_set_error (error,
					     GNOME_DB_CUSTOM_LAYOUT_ERROR,
					     GNOME_DB_CUSTOM_LAYOUT_GLADEXML_ERROR,
					     _("Widget '%s' is not a GtkBox (is a %s)"), GLADE_BOX (box_list->data)->box_name,
					     G_OBJECT_TYPE_NAME (box));
		}

		box_list = g_slist_next (box_list);
	}

	if (!err)
		*box_widgets = hash;
	else {
		*box_widgets = NULL;
		g_hash_table_destroy (hash);
		g_object_unref (xml);
		xml = NULL;
	}

	return xml;
}

/**
 * gnome_db_custom_layout_get_exec_context
 * @layout: a #GnomeDbCustomLayout object
 *
 * Use this function to retreive a #GnomeDbDataSet object containing all the
 * #GnomeDbParameter parameters required to be able to 'run' a widget obtained using the
 * gnome_db_custom_layout_get_widget() function.
 *
 * Note: the parameters contained within the returned #GnomeDbDataSet won't be used by any
 * widget obtained using the gnome_db_custom_layout_get_widget(); accessing those parameters
 * is done using the methods of the #GnomeDbDataWidget interface.
 *
 * Returns: a new #GnomeDbDataSet object
 */
GnomeDbDataSet *
gnome_db_custom_layout_get_exec_context (GnomeDbCustomLayout *layout)
{
	GnomeDbDataSet *retval = NULL;

	g_return_val_if_fail (layout && IS_GNOME_DB_CUSTOM_LAYOUT (layout), NULL);
	g_return_val_if_fail (layout->priv, NULL);

	TO_IMPLEMENT;
	return retval;
}

/**
 * gnome_db_custom_layout_get_data_context
 * @layout: a #GnomeDbCustomLayout object
 *
 * Use this function to retreive a #GnomeDbDataSet object containing all the
 * #GnomeDbParameter parameters which get updated when a widget obtained using the
 * gnome_db_custom_layout_get_widget() is used.
 *
 * Note: the parameters contained within the returned #GnomeDbDataSet won't be used by any
 * widget obtained using the gnome_db_custom_layout_get_widget(); accessing those parameters
 * is done using the methods of the #GnomeDbDataWidget interface.
 *
 * Returns: a new #GnomeDbDataSet object
 */
GnomeDbDataSet *
gnome_db_custom_layout_get_data_context (GnomeDbCustomLayout *layout)
{
	GnomeDbDataSet *retval = NULL;

	g_return_val_if_fail (layout && IS_GNOME_DB_CUSTOM_LAYOUT (layout), NULL);
	g_return_val_if_fail (layout->priv, NULL);

	TO_IMPLEMENT;
	return retval;
}

/**
 * gnome_db_custom_layout_get_data
 * @layout: a #GnomeDbCustomLayout object
 * @error: place to store the error, or %NULL
 *
 * Retreive all the information stored in @layout to be able to create
 * #GnomeDbWorkLayout objects.
 *
 * The caller is responsible to call gnome_db_custom_layout_data_free() on the returned structure, without
 * trying to free its internals, or unref the objects pointer inside it.
 *
 * The returned #GnomeDbCustomLayoutData structure holds pointers to the referenced objects, and a check is made for
 * all the pointers which cannot be %NULL (if an error is found, then the function returns %NULL, and %error is set)
 *
 * Returns: a new #GnomeDbCustomLayoutData structure, to be free'ed by the caller using gnome_db_custom_layout_data_free()
 */
GnomeDbCustomLayoutData *
gnome_db_custom_layout_get_data (GnomeDbCustomLayout *layout, GError **error)
{
	GnomeDbCustomLayoutData *data;
	GnomeDbCustomLayoutConnect *cdata;
	GSList *list;
	gboolean err;

	g_return_val_if_fail (layout && IS_GNOME_DB_CUSTOM_LAYOUT (layout), NULL);
	g_return_val_if_fail (layout->priv, NULL);

	if (!gnome_db_referer_activate (GNOME_DB_REFERER (layout))) {
		g_set_error (error,
			     GNOME_DB_CUSTOM_LAYOUT_ERROR,
			     GNOME_DB_CUSTOM_LAYOUT_DATA_ERROR,
			     _("Cannot find some objects used by this GnomeDbCustomLayout object"));
		return NULL;
	}

	data = g_new0 (GnomeDbCustomLayoutData, 1);
	data->type = layout->priv->type;
	switch (layout->priv->type) {
	case GNOME_DB_CUSTOM_LAYOUT_LAYOUT:
		err = FALSE;
		list = layout->priv->contents.layout.connects;
		while (list) {
			cdata = g_new0 (GnomeDbCustomLayoutConnect, 1);
			cdata->src_layout = gnome_db_ref_base_get_ref_object (LAYOUT_CONNECT (list->data)->src_layout);
			cdata->src_field = gnome_db_ref_base_get_ref_object (LAYOUT_CONNECT (list->data)->src_field);
			cdata->dest_layout = gnome_db_ref_base_get_ref_object (LAYOUT_CONNECT (list->data)->dest_layout);
			cdata->dest_field = gnome_db_ref_base_get_ref_object (LAYOUT_CONNECT (list->data)->dest_field);
			data->contents.layout.connects = g_slist_append (data->contents.layout.connects, cdata);

			if (!g_slist_find (layout->priv->contents.layout.children, cdata->src_layout)) {
				g_set_error (error,
					     GNOME_DB_CUSTOM_LAYOUT_ERROR,
					     GNOME_DB_CUSTOM_LAYOUT_DATA_ERROR,
					     _("Can't find source layout among sub-layouts"));
				err = TRUE;
			}
			if (!g_slist_find (layout->priv->contents.layout.children, cdata->dest_layout)) {
				g_set_error (error,
					     GNOME_DB_CUSTOM_LAYOUT_ERROR,
					     GNOME_DB_CUSTOM_LAYOUT_DATA_ERROR,
					     _("Can't find destination layout among sub-layouts"));
				err = TRUE;
			}
			TO_IMPLEMENT;
			if (0 && !err) {
				/* make sure that:
				 * --> the cdata->src_field has a corresponding parameter in the potential data context
				 *     of a widget created using gnome_db_custom_layout_get_widget(cdata->src_layout)
				 * --> the cdata->dest_field has a corresponding parameter in the potential exec context
				 *     of a widget created using gnome_db_custom_layout_get_widget(cdata->dest_layout) 
				 */
				GnomeDbDataSet *context;

				context = gnome_db_custom_layout_get_data_context (cdata->src_layout);
				if (! gnome_db_data_set_find_parameter_for_field (context, cdata->src_field)) {
					g_set_error (error,
						     GNOME_DB_CUSTOM_LAYOUT_ERROR,
						     GNOME_DB_CUSTOM_LAYOUT_DATA_ERROR,
						     _("Invalid source field for layout connections"));
					err = TRUE;
				}
				g_object_unref (context);

				context = gnome_db_custom_layout_get_exec_context (cdata->dest_layout);
				if (! gnome_db_data_set_find_parameter_for_field (context, cdata->dest_field)) {
					g_set_error (error,
						     GNOME_DB_CUSTOM_LAYOUT_ERROR,
						     GNOME_DB_CUSTOM_LAYOUT_DATA_ERROR,
						     _("Invalid destination field for layout connections"));
					err = TRUE;
				}
				g_object_unref (context);
			}

			if (err) {
				gnome_db_custom_layout_data_free (data);
				return NULL;
			}

			list = g_slist_next (list);
		}
		data->contents.layout.children = layout->priv->contents.layout.children;
		break;
	case GNOME_DB_CUSTOM_LAYOUT_MATRIX:
		if (layout->priv->contents.work_iface.ref_query_extra)
			data->contents.work_iface.query_extra = 
				GNOME_DB_QUERY (gnome_db_ref_base_get_ref_object (layout->priv->contents.work_iface.ref_query_extra));
		if (layout->priv->contents.work_iface.ref_cols_target)
			data->contents.work_iface.cols_target = 
				GNOME_DB_TARGET (gnome_db_ref_base_get_ref_object (layout->priv->contents.work_iface.ref_cols_target));
		if (layout->priv->contents.work_iface.ref_rows_target)
			data->contents.work_iface.rows_target = 
				GNOME_DB_TARGET (gnome_db_ref_base_get_ref_object (layout->priv->contents.work_iface.ref_rows_target));
		data->contents.work_iface.view_type = layout->priv->contents.work_iface.view_type;
	case GNOME_DB_CUSTOM_LAYOUT_GRID:
	case GNOME_DB_CUSTOM_LAYOUT_FORM:
		data->contents.work_iface.mode = layout->priv->contents.work_iface.mode;
		if (layout->priv->contents.work_iface.ref_query)
			data->contents.work_iface.query = 
				GNOME_DB_QUERY (gnome_db_ref_base_get_ref_object (layout->priv->contents.work_iface.ref_query));
		if (layout->priv->contents.work_iface.ref_modif)
			data->contents.work_iface.modified = 
				gnome_db_ref_base_get_ref_object (layout->priv->contents.work_iface.ref_modif);
		break;
	}

	return data;
}

/**
 * gnome_db_custom_layout_data_free
 * @data: a #GnomeDbCustomLayoutData structure
 *
 * Ensures that the @data allocated through gnome_db_custom_layout_get_data() is de-allocated properly.
 */
void
gnome_db_custom_layout_data_free (GnomeDbCustomLayoutData *data)
{
	if (data->type == GNOME_DB_CUSTOM_LAYOUT_LAYOUT) {
		GSList *list = data->contents.layout.connects;
		while (list) {
			g_free (list->data);
			list = g_slist_next (list);
		}
		g_slist_free (data->contents.layout.connects);
	}

	g_free (data);
}

/* 
 * GnomeDbXmlStorage interface implementation
 */
static gchar *
gnome_db_custom_layout_get_xml_id (GnomeDbXmlStorage *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_CUSTOM_LAYOUT (iface), NULL);
	g_return_val_if_fail (GNOME_DB_CUSTOM_LAYOUT (iface)->priv, NULL);

	return g_strdup_printf ("CL%d", gnome_db_base_get_id (GNOME_DB_BASE (iface)));
}

static xmlNodePtr
gnome_db_custom_layout_save_to_xml (GnomeDbXmlStorage *iface, GError **error)
{
	xmlNodePtr node = NULL;
	GnomeDbCustomLayout *cl;
	GSList *list;
	GnomeDbBase *ref;
	gchar *str;
	const gchar *cstr;
	xmlNodePtr query_ref_node = NULL;

	g_return_val_if_fail (iface && IS_GNOME_DB_CUSTOM_LAYOUT (iface), NULL);
	g_return_val_if_fail (GNOME_DB_CUSTOM_LAYOUT (iface)->priv, NULL);

	cl = GNOME_DB_CUSTOM_LAYOUT (iface);

	/* main XML node */
	switch (cl->priv->type) {
	case GNOME_DB_CUSTOM_LAYOUT_LAYOUT:
		node = xmlNewNode (NULL, "GNOME_DB_CUSTOM_LAYOUT");
		break;
	case GNOME_DB_CUSTOM_LAYOUT_GRID:
		node = xmlNewNode (NULL, "GNOME_DB_CUSTOM_GRID");
		break;
	case GNOME_DB_CUSTOM_LAYOUT_FORM:
		node = xmlNewNode (NULL, "GNOME_DB_CUSTOM_FORM");
		break;
	case GNOME_DB_CUSTOM_LAYOUT_MATRIX:
		node = xmlNewNode (NULL, "GNOME_DB_CUSTOM_MATRIX");
		break;
	}
	
	str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (iface));
	xmlSetProp (node, "id", str);
	g_free (str);
	
	cstr = gnome_db_base_get_name (GNOME_DB_BASE (cl));
	if (cstr && *cstr)
		xmlSetProp (node, "name", cstr);
	cstr = gnome_db_base_get_description (GNOME_DB_BASE (cl));
	if (cstr && *cstr)
		xmlSetProp (node, "descr", cstr);

	/* contents */
	switch (cl->priv->type) {
	case GNOME_DB_CUSTOM_LAYOUT_LAYOUT:
		list = cl->priv->contents.layout.children;
		while (list) {
			xmlNodePtr child = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (list->data), error);
			if (!child)
				return NULL;
			xmlAddChild (node, child);
			list = g_slist_next (list);
		}
		break;
	case GNOME_DB_CUSTOM_LAYOUT_MATRIX:
		if (cl->priv->contents.work_iface.query_extra) {
			xmlNodePtr child = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (cl->priv->contents.work_iface.query_extra),
								       error);
			if (!child)
				return NULL;
			xmlAddChild (node, child);
		}
		ref = gnome_db_ref_base_get_ref_object (cl->priv->contents.work_iface.ref_query_extra);
		if (!ref) {
			g_set_error (error,
				     GNOME_DB_CUSTOM_LAYOUT_ERROR,
				     GNOME_DB_CUSTOM_LAYOUT_XML_SAVE_ERROR,
				     _("Can't find referenced query"));
			return NULL;
		}
		if (ref != (GnomeDbBase *) cl->priv->contents.work_iface.query_extra) {
			xmlNodePtr child = xmlNewChild (node, NULL, "GNOME_DB_QUERY_REF", NULL);
			xmlSetProp (child, "id_query", gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (ref)));
			query_ref_node = child;
		}

		ref = gnome_db_ref_base_get_ref_object (cl->priv->contents.work_iface.ref_rows_target);
		str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (ref));
		xmlSetProp (node, "rows_target", str);
		g_free (str);

		ref = gnome_db_ref_base_get_ref_object (cl->priv->contents.work_iface.ref_cols_target);
		str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (ref));
		xmlSetProp (node, "cols_target", str);
		g_free (str);

		switch (cl->priv->contents.work_iface.view_type) {
		case GNOME_DB_MATRIX_TABULAR_SYNTHETIC:
			str = "TS";
			break;
		case GNOME_DB_MATRIX_LIST_DETAILLED:
			str = "LD";
			break;
		case GNOME_DB_MATRIX_LIST_SYNTHETIC:
			str = "LS";
			break;
		default:
			g_assert_not_reached ();
			break;
		}
		xmlSetProp (node, "view_type", str);
	case GNOME_DB_CUSTOM_LAYOUT_GRID:
	case GNOME_DB_CUSTOM_LAYOUT_FORM:
		if (cl->priv->contents.work_iface.query) {
			xmlNodePtr child = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (cl->priv->contents.work_iface.query),
								       error);
			if (!child)
				return NULL;
			xmlAddChild (node, child);
		}
		ref = gnome_db_ref_base_get_ref_object (cl->priv->contents.work_iface.ref_query);
		if (!ref) {
			g_set_error (error,
				     GNOME_DB_CUSTOM_LAYOUT_ERROR,
				     GNOME_DB_CUSTOM_LAYOUT_XML_SAVE_ERROR,
				     _("Can't find referenced query"));
			return NULL;
		}
		if (ref != (GnomeDbBase *) cl->priv->contents.work_iface.query) {
			xmlNodePtr child;
			str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (ref));
			if (query_ref_node) {
				child = xmlNewNode (NULL, "GNOME_DB_QUERY_REF");
				xmlSetProp (child, "id_query", str);
				xmlAddPrevSibling (query_ref_node, child);
			}
			else {
				child = xmlNewChild (node, NULL, "GNOME_DB_QUERY_REF", NULL);
				xmlSetProp (child, "id_query", str);
			}
			g_free (str);
		}
		
		if (cl->priv->contents.work_iface.ref_modif) {
			ref = gnome_db_ref_base_get_ref_object (cl->priv->contents.work_iface.ref_modif);
			if (!ref) {
				g_set_error (error,
					     GNOME_DB_CUSTOM_LAYOUT_ERROR,
					     GNOME_DB_CUSTOM_LAYOUT_XML_SAVE_ERROR,
					     _("Can't find referenced modified target"));
				return NULL;
			}
			if (cl->priv->type == GNOME_DB_CUSTOM_LAYOUT_MATRIX)
				xmlSetProp (node, "modif_table", gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (ref)));
			else
				xmlSetProp (node, "modif_target", gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (ref)));
		}
		str = g_strdup_printf ("%d", cl->priv->contents.work_iface.mode);
		xmlSetProp (node, "mode", str);
		g_free (str);
		break;
	}

	/* Glade related */
	if (cl->priv->filename && cl->priv->root_widget && cl->priv->boxes) {
		xmlNodePtr glade;
		xmlNodePtr box, dest;
		GSList *dest_list;

		glade = xmlNewChild (node, NULL, "GNOME_DB_GLADE", NULL);
		xmlSetProp (glade, "filename", cl->priv->filename);
		xmlSetProp (glade, "root_widget", cl->priv->root_widget);
		list = cl->priv->boxes;
		while (list) {
			box =  xmlNewChild (glade, NULL, "GNOME_DB_GLADE_BOX", NULL);
			xmlSetProp (box, "box_name", GLADE_BOX (list->data)->box_name);
			xmlSetProp (box, "show_actions", GLADE_BOX (list->data)->show_actions ? "t" : "f");
			dest_list = GLADE_BOX (list->data)->dest_list;
			while (dest_list) {
				ref = gnome_db_ref_base_get_ref_object (GNOME_DB_REF_BASE (dest_list->data));
				if (!ref) {
					g_set_error (error,
						     GNOME_DB_CUSTOM_LAYOUT_ERROR,
						     GNOME_DB_CUSTOM_LAYOUT_XML_SAVE_ERROR,
						     _("Can't find referenced object for Glade box"));
					return NULL;
				}
				dest = xmlNewChild (box, NULL, "GNOME_DB_GLADE_DEST", NULL);
				xmlSetProp (dest, "object", gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (ref)));
				dest_list = g_slist_next (dest_list);
			}

			list = g_slist_next (list);
		}
	}


	/* Connects if applicable */
	if (cl->priv->type == GNOME_DB_CUSTOM_LAYOUT_LAYOUT) {
		if (cl->priv->contents.layout.connects) {
			list = cl->priv->contents.layout.connects;
			while (list) {
				xmlNodePtr lcnode;
				lcnode = xmlNewChild (node, NULL, "GNOME_DB_LAYOUT_CONNECT", NULL);

				xmlSetProp (lcnode, "src_layout", 
					    gnome_db_ref_base_get_ref_name (LAYOUT_CONNECT (list->data)->src_layout, NULL, NULL));
				xmlSetProp (lcnode, "src_field", 
					    gnome_db_ref_base_get_ref_name (LAYOUT_CONNECT (list->data)->src_field, NULL, NULL));
				xmlSetProp (lcnode, "dest_layout", 
					    gnome_db_ref_base_get_ref_name (LAYOUT_CONNECT (list->data)->dest_layout, NULL, NULL));
				xmlSetProp (lcnode, "dest_field", 
					    gnome_db_ref_base_get_ref_name (LAYOUT_CONNECT (list->data)->dest_field, NULL, NULL));
				list = g_slist_next (list);
			}
		}
	}

	return node;
}

static gboolean
gnome_db_custom_layout_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	GnomeDbCustomLayout *cl;
	gchar *prop;
	xmlNodePtr children;
	guint mode = 0;
	gboolean mode_found = FALSE;
	GnomeDbDict *dict;

	g_return_val_if_fail (iface && IS_GNOME_DB_CUSTOM_LAYOUT (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_CUSTOM_LAYOUT (iface)->priv, FALSE);
	g_return_val_if_fail (node, FALSE);

	cl = GNOME_DB_CUSTOM_LAYOUT (iface);
	dict = gnome_db_base_get_dict (GNOME_DB_BASE (cl));

	/* type of custom layout */
	if (!strcmp (node->name, "GNOME_DB_CUSTOM_LAYOUT")) {
		mode = GNOME_DB_CUSTOM_LAYOUT_LAYOUT;
		mode_found = TRUE;
	}
	if (!mode_found && !strcmp (node->name, "GNOME_DB_CUSTOM_GRID")) {
		mode = GNOME_DB_CUSTOM_LAYOUT_GRID;
		mode_found = TRUE;
	}
	if (!mode_found && !strcmp (node->name, "GNOME_DB_CUSTOM_FORM")) {
		mode = GNOME_DB_CUSTOM_LAYOUT_FORM;
		mode_found = TRUE;
	}
	if (!mode_found && !strcmp (node->name, "GNOME_DB_CUSTOM_MATRIX")) {
		mode = GNOME_DB_CUSTOM_LAYOUT_MATRIX;
		mode_found = TRUE;
	}

	if (!mode_found) {
		g_set_error (error,
			     GNOME_DB_CUSTOM_LAYOUT_ERROR,
			     GNOME_DB_CUSTOM_LAYOUT_XML_LOAD_ERROR,
			     _("XML Tag is not <GNOME_DB_CUSTOM_LAYOUT> or <GNOME_DB_CUSTOM_GRID> or "
			       "<GNOME_DB_CUSTOM_FORM> or <GNOME_DB_CUSTOM_MATRIX>"));
		return FALSE;
	}
	cl->priv->type = mode;

	/* name and description */
	prop = xmlGetProp (node, "name");
	if (prop) {
		gnome_db_base_set_name (GNOME_DB_BASE (cl), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "descr");
	if (prop) {
		gnome_db_base_set_description (GNOME_DB_BASE (cl), prop);
		g_free (prop);
	}

	if ((cl->priv->type == GNOME_DB_CUSTOM_LAYOUT_MATRIX) ||
	    (cl->priv->type == GNOME_DB_CUSTOM_LAYOUT_GRID) ||
	    (cl->priv->type == GNOME_DB_CUSTOM_LAYOUT_FORM)) {
		prop = xmlGetProp (node, "mode");
		if (prop) {
			cl->priv->contents.work_iface.mode = atoi (prop);
			g_free (prop);
		}

		if (cl->priv->type == GNOME_DB_CUSTOM_LAYOUT_MATRIX)
			prop = xmlGetProp (node, "modif_table");
		else
			prop = xmlGetProp (node, "modif_target");
		if (prop) {
			cl->priv->contents.work_iface.ref_modif = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
			gnome_db_ref_base_set_ref_name (cl->priv->contents.work_iface.ref_modif, 0, REFERENCE_BY_XML_ID, prop);
			g_free (prop);
			cl->priv->referer_objects = g_slist_prepend (cl->priv->referer_objects, 
								     cl->priv->contents.work_iface.ref_modif);
		}

		if (cl->priv->type == GNOME_DB_CUSTOM_LAYOUT_MATRIX) {
			prop = xmlGetProp (node, "rows_target");
			g_assert (prop); /* normally done through DTD */
			if (prop) {
				cl->priv->contents.work_iface.ref_rows_target = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));	
				gnome_db_ref_base_set_ref_name (cl->priv->contents.work_iface.ref_rows_target, 0, 
							  REFERENCE_BY_XML_ID, prop);
				g_free (prop);
			}
			prop = xmlGetProp (node, "cols_target");
			g_assert (prop); /* normally done through DTD */
			if (prop) {
				cl->priv->contents.work_iface.ref_cols_target = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));	
				gnome_db_ref_base_set_ref_name (cl->priv->contents.work_iface.ref_cols_target, 0, 
							  REFERENCE_BY_XML_ID, prop);
				g_free (prop);
			}
			prop = xmlGetProp (node, "view_type");
			if (prop) {
				if ((*prop == 'T') && (*(prop+1) == 'S'))
					cl->priv->contents.work_iface.view_type = GNOME_DB_MATRIX_TABULAR_SYNTHETIC;
				if ((*prop == 'L') && (*(prop+1) == 'S'))
					cl->priv->contents.work_iface.view_type = GNOME_DB_MATRIX_LIST_SYNTHETIC;
				if ((*prop == 'L') && (*(prop+1) == 'D'))
					cl->priv->contents.work_iface.view_type = GNOME_DB_MATRIX_LIST_DETAILLED;
				g_free (prop);
			}
		}
	}

	/* children nodes */
	children = node->children;
	while (children) {
		gboolean done = FALSE;

		/* a query */
		if (!strcmp (children->name, "GNOME_DB_QUERY")) {
			GnomeDbQuery *query = (GnomeDbQuery *) gnome_db_query_new (dict);
			if (!gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (query), children, error))
				return FALSE;
			if (! cl->priv->contents.work_iface.query) {
				cl->priv->contents.work_iface.query = query;
				cl->priv->contents.work_iface.ref_query = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
				gnome_db_ref_base_set_ref_object (cl->priv->contents.work_iface.ref_query, GNOME_DB_BASE (query));
				cl->priv->referer_objects = g_slist_prepend (cl->priv->referer_objects, 
									     cl->priv->contents.work_iface.ref_query);    
			}
			else {
				cl->priv->contents.work_iface.query_extra = query;
				cl->priv->contents.work_iface.ref_query_extra =  GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
				gnome_db_ref_base_set_ref_object (cl->priv->contents.work_iface.ref_query_extra, GNOME_DB_BASE (query));
				cl->priv->referer_objects = g_slist_prepend (cl->priv->referer_objects, 
									     cl->priv->contents.work_iface.ref_query_extra);   
			}
			done = TRUE;
		}

		/* a query reference */
		if (!done && !strcmp (children->name, "GNOME_DB_QUERY_REF")) {
			prop = xmlGetProp (children, "id_query");
			if (!prop) {
				g_set_error (error,
					     GNOME_DB_CUSTOM_LAYOUT_ERROR,
					     GNOME_DB_CUSTOM_LAYOUT_XML_LOAD_ERROR,
					     _("XML Tag <GNOME_DB_QUERY_REF> has no \"id_query\" property"));
				return FALSE;
			}
			if (! cl->priv->contents.work_iface.ref_query) {
				cl->priv->contents.work_iface.ref_query = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
				gnome_db_ref_base_set_ref_name (cl->priv->contents.work_iface.ref_query, 0, 
							  REFERENCE_BY_XML_ID, prop);
				cl->priv->referer_objects = g_slist_prepend (cl->priv->referer_objects, 
									     cl->priv->contents.work_iface.ref_query);	     
			}
			else {
				cl->priv->contents.work_iface.ref_query_extra = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
				gnome_db_ref_base_set_ref_name (cl->priv->contents.work_iface.ref_query_extra, 0, 
							  REFERENCE_BY_XML_ID, prop);
				cl->priv->referer_objects = g_slist_prepend (cl->priv->referer_objects, 
									     cl->priv->contents.work_iface.ref_query_extra);
			}
			g_free (prop);
			done = TRUE;
		}

		/* Glade */
		if (!done && !strcmp (children->name, "GNOME_DB_GLADE")) {
			xmlNodePtr boxes;

			prop = xmlGetProp (children, "filename");
			cl->priv->filename = prop;
			prop = xmlGetProp (children, "root_widget");
			cl->priv->root_widget = prop;
			
			boxes = children->children;
			while (boxes) {
				if (!strcmp (boxes->name, "GNOME_DB_GLADE_BOX")) {
					GladeBox *gb;
					xmlNodePtr dests;
					
					prop = xmlGetProp (boxes, "box_name");
					gb = glade_box_new (prop, TRUE);
					cl->priv->boxes = g_slist_prepend (cl->priv->boxes, gb);
					g_free (prop);
					prop = xmlGetProp (boxes, "show_actions");
					if (prop) {
						gb->show_actions = (*prop == 't');
						g_free (prop);
					}
					
					dests = boxes->children;
					while (dests) {
						if (!strcmp (dests->name, "GNOME_DB_GLADE_DEST")) {
							GnomeDbRefBase *ref = (GnomeDbRefBase *) gnome_db_ref_base_new (dict);
							prop = xmlGetProp (dests, "object");
							gnome_db_ref_base_set_ref_name (ref, 0, REFERENCE_BY_XML_ID, prop);
							g_free (prop);
							gb->dest_list = g_slist_prepend (gb->dest_list, ref);
							cl->priv->referer_objects = 
								g_slist_prepend (cl->priv->referer_objects, ref);
						}
						dests = dests->next;
					}
					gb->dest_list = g_slist_reverse (gb->dest_list);
				}
				boxes = boxes->next;
			}
			cl->priv->boxes = g_slist_reverse (cl->priv->boxes);
			
			done = TRUE;
		}

		/* connects */
		if (!done && !strcmp (children->name, "GNOME_DB_LAYOUT_CONNECT")) {
			LayoutConnect *lc;

			if (cl->priv->type != GNOME_DB_CUSTOM_LAYOUT_LAYOUT) {
				g_set_error (error,
					     GNOME_DB_CUSTOM_LAYOUT_ERROR,
					     GNOME_DB_CUSTOM_LAYOUT_XML_LOAD_ERROR,
					     _("This Custom Layout can't have a <GNOME_DB_LAYOUT_CONNECT> child"));
				return FALSE;
			}
			
			lc = layout_connect_new (cl, dict);
			prop = xmlGetProp (children, "src_layout");
			g_assert (prop); /* normally done through DTD */
			gnome_db_ref_base_set_ref_name (lc->src_layout, GNOME_DB_TYPE_CUSTOM_LAYOUT, REFERENCE_BY_XML_ID, prop);
			g_free (prop);

			prop = xmlGetProp (children, "src_field");
			g_assert (prop); /* normally done through DTD */
			gnome_db_ref_base_set_ref_name (lc->src_field, GNOME_DB_TYPE_FIELD, REFERENCE_BY_XML_ID, prop);
			g_free (prop);

			prop = xmlGetProp (children, "dest_layout");
			g_assert (prop); /* normally done through DTD */
			gnome_db_ref_base_set_ref_name (lc->dest_layout, GNOME_DB_TYPE_CUSTOM_LAYOUT, REFERENCE_BY_XML_ID, prop);
			g_free (prop);

			prop = xmlGetProp (children, "dest_field");
			g_assert (prop); /* normally done through DTD */
			gnome_db_ref_base_set_ref_name (lc->dest_field, GNOME_DB_TYPE_FIELD, REFERENCE_BY_XML_ID, prop);
			g_free (prop);

			cl->priv->contents.layout.connects = g_slist_append (cl->priv->contents.layout.connects, lc);

			done = TRUE;
		}

		/* sub layouts */
		if (!done && 
		    (!strcmp (children->name, "GNOME_DB_CUSTOM_LAYOUT") ||
		     !strcmp (children->name, "GNOME_DB_CUSTOM_GRID") ||
		     !strcmp (children->name, "GNOME_DB_CUSTOM_FORM") ||
		     !strcmp (children->name, "GNOME_DB_CUSTOM_MATRIX"))) {
			GnomeDbCustomLayout *scl = (GnomeDbCustomLayout *) gnome_db_custom_layout_new (dict);
			if (!gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (scl), children, error))
				return FALSE;
			cl->priv->contents.layout.children = g_slist_append (cl->priv->contents.layout.children, scl);
			done = TRUE;
		}
		
		children = children->next;
	}

	gnome_db_referer_activate (GNOME_DB_REFERER (iface));
	return TRUE;
}


/*
 * Functions dealing with GladeBox structures
 */
static GladeBox *
glade_box_new (const gchar *name, gboolean show_actions)
{
	GladeBox *gb = g_new0 (GladeBox, 1);
	gb->box_name = g_strdup (name);
	gb->show_actions = show_actions;

	return gb;
}

static void
glade_box_destroy (GladeBox *box)
{
	GSList *list;

	g_free (box->box_name);
	list = box->dest_list;
	while (list) {
		g_object_unref (list->data);
		list = g_slist_next (list);
	}
	g_slist_free (box->dest_list);
	g_free (box);
}

/* 
 * GnomeDbReferer interface implementation
 */
static gboolean
gnome_db_custom_layout_activate (GnomeDbReferer *iface)
{
	GSList *list;
	gboolean active = TRUE;

	g_return_val_if_fail (iface && IS_GNOME_DB_CUSTOM_LAYOUT (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_CUSTOM_LAYOUT (iface)->priv, FALSE);

	list = GNOME_DB_CUSTOM_LAYOUT (iface)->priv->referer_objects;
	while (list) {
		active = gnome_db_ref_base_activate (GNOME_DB_REF_BASE (list->data)) && active;
		list = g_slist_next (list);
	}

	return active;
}

static void
gnome_db_custom_layout_deactivate (GnomeDbReferer *iface)
{
	GSList *list;

	g_return_if_fail (iface && IS_GNOME_DB_CUSTOM_LAYOUT (iface));
	g_return_if_fail (GNOME_DB_CUSTOM_LAYOUT (iface)->priv);

	list = GNOME_DB_CUSTOM_LAYOUT (iface)->priv->referer_objects;
	while (list) {
		gnome_db_ref_base_deactivate (GNOME_DB_REF_BASE (list->data));
		list = g_slist_next (list);
	}
}

static gboolean
gnome_db_custom_layout_is_active (GnomeDbReferer *iface)
{
	GSList *list;
	gboolean active = TRUE;

	g_return_val_if_fail (iface && IS_GNOME_DB_CUSTOM_LAYOUT (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_CUSTOM_LAYOUT (iface)->priv, FALSE);

	list = GNOME_DB_CUSTOM_LAYOUT (iface)->priv->referer_objects;
	while (list && active) {
		active = gnome_db_ref_base_is_active (GNOME_DB_REF_BASE (list->data));
		list = g_slist_next (list);
	}

	return active;
}

static GSList *
gnome_db_custom_layout_get_ref_objects (GnomeDbReferer *iface)
{
	GSList *list = NULL;
	GnomeDbBase *base;

	g_return_val_if_fail (iface && IS_GNOME_DB_CUSTOM_LAYOUT (iface), NULL);
	g_return_val_if_fail (GNOME_DB_CUSTOM_LAYOUT (iface)->priv, NULL);

	list = GNOME_DB_CUSTOM_LAYOUT (iface)->priv->referer_objects;
	while (list) {
		base = gnome_db_ref_base_get_ref_object (GNOME_DB_REF_BASE (list->data));
		if (base)
			list = g_slist_append (list, base);
		list = g_slist_next (list);
	}

	return list;
}

static void
gnome_db_custom_layout_replace_refs (GnomeDbReferer *iface, GHashTable *replacements)
{
	GSList *list = NULL;

	g_return_if_fail (iface && IS_GNOME_DB_CUSTOM_LAYOUT (iface));
	g_return_if_fail (GNOME_DB_CUSTOM_LAYOUT (iface)->priv);

	list = GNOME_DB_CUSTOM_LAYOUT (iface)->priv->referer_objects;
	while (list) {
		gnome_db_ref_base_replace_ref_object (GNOME_DB_REF_BASE (list->data), replacements);
		list = g_slist_next (list);
	}
}


/*
 * Functions dealing with LayoutConnect structures
 */

static LayoutConnect *
layout_connect_new (GnomeDbCustomLayout *layout, GnomeDbDict *dict)
{
	LayoutConnect *lc;

	lc = g_new0 (LayoutConnect, 1);
	lc->src_layout = (GnomeDbRefBase *) gnome_db_ref_base_new (dict);
	lc->src_field = (GnomeDbRefBase *) gnome_db_ref_base_new (dict);
	lc->dest_layout = (GnomeDbRefBase *) gnome_db_ref_base_new (dict);
	lc->dest_field = (GnomeDbRefBase *) gnome_db_ref_base_new (dict);

	layout->priv->referer_objects = g_slist_prepend (layout->priv->referer_objects, lc->dest_field);
	layout->priv->referer_objects = g_slist_prepend (layout->priv->referer_objects, lc->dest_layout);
	layout->priv->referer_objects = g_slist_prepend (layout->priv->referer_objects, lc->src_field);
	layout->priv->referer_objects = g_slist_prepend (layout->priv->referer_objects, lc->src_layout);

	return lc;
}

static void
layout_connect_destroy (GnomeDbCustomLayout *layout, LayoutConnect *lc)
{
	layout->priv->referer_objects = g_slist_remove (layout->priv->referer_objects, lc->src_layout);
	layout->priv->referer_objects = g_slist_remove (layout->priv->referer_objects, lc->src_field);
	layout->priv->referer_objects = g_slist_remove (layout->priv->referer_objects, lc->dest_layout);
	layout->priv->referer_objects = g_slist_remove (layout->priv->referer_objects, lc->dest_field);

	g_object_unref (lc->src_layout);
	g_object_unref (lc->src_field);
	g_object_unref (lc->dest_layout);
	g_object_unref (lc->dest_field);
	
	g_free (lc);
}
