/* utility.c
 *
 * Copyright (C) 2003 - 2005 Vivien Malerba <malerba@gnome-db.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gnome-db-decl.h"
#include "utility.h"
#include <libgda/libgda.h>
#include "gnome-db-data-entry.h"
#include "gnome-db-result-set.h"
#include "gnome-db-parameter.h"
#include "gnome-db-server-data-type.h"
#include "gnome-db-server.h"
#include "gnome-db-query.h"
#include "gnome-db-renderer.h"
#include "gnome-db-entity.h"
#include "gnome-db-field.h"
#include "gnome-db-qfield.h"
#include "gnome-db-qf-field.h"
#include "gnome-db-data-model.h"
#include "gnome-db-data-proxy.h"


/**
 * utility_entry_build_actions_menu
 * @obj_data:
 * @attrs:
 * @function:
 *
 * Creates a GtkMenu widget which contains the possible actions on a data entry which 
 * attributes are @attrs. After the menu has been displayed, and when an action is selected,
 * the @function callback is called with the 'user_data' being @obj_data.
 *
 * The menu item which emits the signal has a "action" attribute which describes what action the
 * user has requested.
 *
 * Returns: the new menu
 */
GtkWidget *
utility_entry_build_actions_menu (GObject *obj_data, guint attrs, GCallback function)
{
	GtkWidget *menu, *mitem;
	gchar *str;
	gboolean nullact = FALSE;
	gboolean defact = FALSE;
	gboolean reset = FALSE;

	gboolean value_is_null;
	gboolean value_is_modified;
	gboolean value_is_default;

	menu = gtk_menu_new ();

	/* which menu items to make sensitive ? */
	value_is_null = attrs & GNOME_DB_VALUE_IS_NULL;
	value_is_modified = ! (attrs & GNOME_DB_VALUE_IS_UNCHANGED);
	value_is_default = attrs & GNOME_DB_VALUE_IS_DEFAULT;

	if ((attrs & GNOME_DB_VALUE_CAN_BE_NULL) && 
	    !(attrs & GNOME_DB_VALUE_IS_NULL))
		nullact = TRUE;
	if ((attrs & GNOME_DB_VALUE_CAN_BE_DEFAULT) && 
	    !(attrs & GNOME_DB_VALUE_IS_DEFAULT))
		defact = TRUE;
	if (!(attrs & GNOME_DB_VALUE_IS_UNCHANGED)) {
		if (attrs & GNOME_DB_VALUE_HAS_VALUE_ORIG) 
			reset = TRUE;
	}

	/* set to NULL item */
	str = g_strdup (_("Unset"));
	mitem = gtk_check_menu_item_new_with_label (str);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mitem),
					value_is_null);
	gtk_widget_show (mitem);
	g_object_set_data (G_OBJECT (mitem), "action", GUINT_TO_POINTER (GNOME_DB_VALUE_IS_NULL));
	g_signal_connect (G_OBJECT (mitem), "activate",
			  G_CALLBACK (function), obj_data);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), mitem);
	g_free (str);
	gtk_widget_set_sensitive (mitem, nullact);

	/* default value item */
	str = g_strdup (_("Set to default value"));
	mitem = gtk_check_menu_item_new_with_label (str);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mitem), 
					value_is_default);
	gtk_widget_show (mitem);
	g_object_set_data (G_OBJECT (mitem), "action", GUINT_TO_POINTER (GNOME_DB_VALUE_IS_DEFAULT));
	g_signal_connect (G_OBJECT (mitem), "activate",
			  G_CALLBACK (function), obj_data);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), mitem);
	g_free (str);
	gtk_widget_set_sensitive (mitem, defact);
		
	/* reset to original value item */
	str = g_strdup (_("Reset to original value"));
	mitem = gtk_check_menu_item_new_with_label (str);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mitem), 
					!value_is_modified);
	gtk_widget_show (mitem);
	g_object_set_data (G_OBJECT (mitem), "action", GUINT_TO_POINTER (GNOME_DB_VALUE_IS_UNCHANGED));
	g_signal_connect (G_OBJECT (mitem), "activate",
			  G_CALLBACK (function), obj_data);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), mitem);
	g_free (str);
	gtk_widget_set_sensitive (mitem, reset);

	return menu;
}


/**
 * utility_entry_build_info_colors_array
 * 
 * Creates an array of colors for the different states of an entry:
 *    Valid   <-> No special color
 *    Null    <-> Green
 *    Default <-> Blue
 *    Invalid <-> Red
 * Each status (except Valid) is represented by two colors for GTK_STATE_NORMAL and
 * GTK_STATE_PRELIGHT.
 *
 * Returns: a new array of 6 colors
 */
GdkColor **utility_entry_build_info_colors_array ()
{
	GdkColor **colors;
	GdkColor *color;
	
	colors = g_new0 (GdkColor *, 6);
	
	/* Green color */
	color = g_new0 (GdkColor, 1);
	gdk_color_parse (GNOME_DB_COLOR_NORMAL_NULL, color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[0] = color;
	
	color = g_new0 (GdkColor, 1);
	gdk_color_parse (GNOME_DB_COLOR_PRELIGHT_NULL, color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[1] = color;
	
	
	/* Blue color */
	color = g_new0 (GdkColor, 1);
	gdk_color_parse (GNOME_DB_COLOR_NORMAL_DEFAULT, color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[2] = color;
	
	color = g_new0 (GdkColor, 1);
	gdk_color_parse (GNOME_DB_COLOR_PRELIGHT_DEFAULT, color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[3] = color;
	
	
	/* Red color */
	color = g_new0 (GdkColor, 1);
	gdk_color_parse (GNOME_DB_COLOR_NORMAL_INVALID, color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[4] = color;
	
	color = g_new0 (GdkColor, 1);
	gdk_color_parse (GNOME_DB_COLOR_PRELIGHT_INVALID, color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[5] = color;

	return colors;
}


/**
 * utility_query_execute_modif
 * @query: the #GnomeDbQuery to be executed
 * @context: a #GnomeDbDataSet object
 * @ask_confirm_insert:
 * @ask_confirm_update:
 * @ask_confirm_delete:
 * @parent_window: a #GtkWindow object, or a #GtkWidget and the parent window will be found automatically
 * @user_cancelled: a place to store if the user cancelled the query if the choice was given, or %NULL
 * @query_error: a place to store if there was an error, or %NULL
 *
 * Executes @query and displays question and information dialogs if necessary.
 * If a user confirmation was required and the user cancelled the execution, then 
 * the returned value is FALSE.
 *
 * Returns: TRUE if the query was executed.
 */
gboolean
utility_query_execute_modif (GnomeDbQuery *query, GnomeDbDataSet *context,
			     gboolean ask_confirm_insert,
			     gboolean ask_confirm_update,
			     gboolean ask_confirm_delete,
			     GtkWidget *parent_window,
			     gboolean *user_cancelled,
			     gboolean *query_error)
{
	gchar *sql = NULL;
	GnomeDbQueryType qtype;
	gchar *confirm = NULL;
	gboolean do_execute = TRUE;
	gboolean allok = TRUE;
	GError *error = NULL;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), FALSE);
	
	/* find the GtkWindow object for @parent_window */
	while (parent_window && !GTK_IS_WINDOW (parent_window)) 
		parent_window = gtk_widget_get_parent (parent_window);

	sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (query), context, 0, &error);
	qtype = gnome_db_query_get_query_type (query);

	switch (qtype) {
	case GNOME_DB_QUERY_TYPE_INSERT:
		if (ask_confirm_insert)
			confirm = _("Execute the following insertion query ?");
		break;
	case GNOME_DB_QUERY_TYPE_UPDATE:
		if (ask_confirm_update)
			confirm = _("Execute the following update query ?");
		break;
	case GNOME_DB_QUERY_TYPE_DELETE:
		if (ask_confirm_delete)
			confirm = _("Execute the following deletion query ?");
		break;
	default:
		g_assert_not_reached ();
	}

	if (user_cancelled)
		*user_cancelled = FALSE;
	if (query_error)
		*query_error = FALSE;

	if (sql) {
		if (confirm) {
			GtkWidget *dlg;
			gint result;
			gchar *msg;
			
			msg = g_strdup_printf (_("<b><big>%s</big></b>\n"
						 "<small>The preferences require a confirmation for the "
						 "following query</small>\n\n%s"), confirm, sql);
			dlg = gtk_message_dialog_new (GTK_WINDOW (parent_window), 0,
						      GTK_MESSAGE_QUESTION,
						      GTK_BUTTONS_YES_NO, msg);
			g_free (msg);
			gtk_label_set_use_markup (GTK_LABEL (GTK_MESSAGE_DIALOG (dlg)->label), TRUE);
			result = gtk_dialog_run (GTK_DIALOG (dlg));
			gtk_widget_destroy (dlg);
			do_execute = (result == GTK_RESPONSE_YES);
			if (user_cancelled)
				*user_cancelled = !do_execute;
		}
			
		if (do_execute) {
			GnomeDbDict *dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));
				
#ifdef debug
			g_print ("MODIF SQL: %s\n", sql);
#endif
			gnome_db_server_do_query_as_data_model (gnome_db_dict_get_server (dict), sql, 
								GNOME_DB_SERVER_QUERY_SQL, &error);
			if (error) {
				GtkWidget *dlg;
				gchar *message;
				
				message = g_strdup (error->message);
				g_error_free (error);
				dlg = gtk_message_dialog_new (GTK_WINDOW (parent_window), 0,
							      GTK_MESSAGE_ERROR,
							      GTK_BUTTONS_CLOSE,
							      message);
				g_free (message);
				gtk_dialog_run (GTK_DIALOG (dlg));
				gtk_widget_destroy (dlg);
				allok = FALSE;

				if (query_error)
					*query_error = TRUE;
			}
		}
		else
			allok = FALSE;
		
		g_free (sql);
	}
	else {
		GtkWidget *dlg;
		gchar *message;
		
		if (error) {
			message = g_strdup_printf (_("The following error occurred while preparing the query:\n%s"),
						   error->message);
			g_error_free (error);
		}
		else
			message = g_strdup_printf (_("An unknown error occurred while preparing the query."));
		
		dlg = gtk_message_dialog_new (GTK_WINDOW (parent_window), 0,
					      GTK_MESSAGE_ERROR,
					      GTK_BUTTONS_CLOSE,
					      message);
		g_free (message);
		gtk_dialog_run (GTK_DIALOG (dlg));
		gtk_widget_destroy (dlg);
		allok = FALSE;

		if (query_error)
			*query_error = TRUE;
	}

	return allok;
}




static void compute_shown_columns_index (DataSetNodeInfo *info);
static void compute_ref_columns_index (DataSetNodeInfo *info);

/**
 * utility_data_set_node_info_create
 * @data_set:
 * @node:
 *
 * Creates a new DataSetNodeInfo structure and initializes it.
 *
 * Returns: the new structure (to be freed using utility_data_set_node_info_destroy())
 */
DataSetNodeInfo *
utility_data_set_node_info_create (GnomeDbDataSet *data_set, GnomeDbDataSetNode *node)
{
	DataSetNodeInfo *info;

	g_assert (data_set && IS_GNOME_DB_DATA_SET (data_set));
	g_assert (node && g_slist_find (data_set->nodes, node));
	g_assert (node->params);
	g_assert (node->data_for_param && IS_GNOME_DB_DATA_MODEL (node->data_for_param));

	info = g_new0 (DataSetNodeInfo, 1);

	g_object_ref (data_set);
	info->data_set = data_set;
	info->node = node;
	info->data_model = node->data_for_param;
	
	compute_shown_columns_index (info);
	compute_ref_columns_index (info);

	return info;
}

static void
compute_shown_columns_index (DataSetNodeInfo *info)
{
        gint ncols, nparams;
        gint *mask = NULL, masksize = 0;

        nparams = g_slist_length (info->node->params);
        g_return_if_fail (nparams > 0);
        ncols = gda_data_model_get_n_columns (GDA_DATA_MODEL (info->data_model));
        g_return_if_fail (ncols > 0);

        if (ncols > nparams) {
                /* we only want columns which are not parameter values */
                gint i, current = 0;

                masksize = ncols - nparams;
                mask = g_new0 (gint, masksize);
                for (i = 0; i < ncols ; i++) {
                        GSList *list = info->node->params;
                        gboolean found = FALSE;
                        while (list && !found) {
                                if (GPOINTER_TO_INT (g_hash_table_lookup (info->node->pos_for_param, list->data)) == i)
                                        found = TRUE;
                                else
                                        list = g_slist_next (list);
                        }
                        if (!found) {
                                mask[current] = i;
                                current ++;
                        }
                }
                masksize = current;
        }
        else {
                /* we want all the columns */
                gint i;

                masksize = ncols;
                mask = g_new0 (gint, masksize);
                for (i=0; i<ncols; i++) {
                        mask[i] = i;
                }
        }

        info->shown_n_cols = masksize;
        info->shown_cols_index = mask;
}

static void
compute_ref_columns_index (DataSetNodeInfo *info)
{
        gint ncols, nparams;
        gint *mask = NULL, masksize = 0;

        nparams = g_slist_length (info->node->params);
        g_return_if_fail (nparams > 0);
        ncols = gda_data_model_get_n_columns (GDA_DATA_MODEL (info->data_model));
        g_return_if_fail (ncols > 0);

        if (ncols > nparams) {
                /* we only want columns which are parameters values */
                gint i, current = 0;

                masksize = ncols - nparams;
                mask = g_new0 (gint, masksize);
                for (i=0; i<ncols ; i++) {
                        GSList *list = info->node->params;
                        gboolean found = FALSE;
                        while (list && !found) {
                                if (GPOINTER_TO_INT (g_hash_table_lookup (info->node->pos_for_param, list->data)) == i)
                                        found = TRUE;
                                else
                                        list = g_slist_next (list);
                        }
                        if (found) {
                                mask[current] = i;
                                current ++;
                        }
                }
                masksize = current;
        }
        else {
                /* we want all the columns */
                gint i;

                masksize = ncols;
                mask = g_new0 (gint, masksize);
                for (i=0; i<ncols; i++) {
                        mask[i] = i;
                }
        }

        info->ref_n_cols = masksize;
        info->ref_cols_index = mask;
}

/**
 * utility_data_set_node_info_destroy
 * @info:
 *
 * Destroys @info and cleans-up all its internals
 */
void
utility_data_set_node_info_destroy (DataSetNodeInfo *info)
{
	g_object_unref (info->data_set);

	if (info->shown_cols_index) 
		g_free (info->shown_cols_index);
	if (info->ref_cols_index) 
		g_free (info->ref_cols_index);

	g_free (info);
}

/**
 * utility_data_set_node_info_compute_values
 * @info:
 * @proxy:
 * @iter:
 *
 * Makes a list of the values in @proxy, at row @iter, which correspond to the parameters in the @info
 * structure.
 *
 * Returns: a new list which must be freed using g_list_free() (don't change or free the values in the list)
 */
GList *
utility_data_set_node_info_compute_values (DataSetNodeInfo *info, GnomeDbDataProxy *proxy, GtkTreeIter *iter)
{
	GList *values = NULL;
	GSList *list;
	GnomeDbDataModel *proxy_model;
	gint col;
	GdaValue *value;

	g_return_val_if_fail (info, NULL);
	g_return_val_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy), NULL);
	proxy_model = gnome_db_data_proxy_get_model (proxy);
	g_return_val_if_fail (proxy_model, NULL);

	/* list the values in proxy_model for each param in info->node->params */
	list = info->node->params;
	while (list) {
		col = gnome_db_data_model_get_column_at_param (proxy_model, info->data_set, list->data);
		gtk_tree_model_get (GTK_TREE_MODEL (proxy), iter, col, &value, -1);
		values = g_list_append (values, value);
		
		list = g_slist_next (list);
	}
	
	return values;
}

/**
 * utility_data_set_node_info_compute_values_ext
 * @info:
 * @proxy:
 * @iter:
 * 
 * Makes a list of the values in @proxy, at row @iter, which correspond to the parameters in the @info
 * structure, but also adding values to make it a complete valid row of node->data_for_param. Note: this feature
 * is not available on all the GnomeDbDataModel present in @proxy, so if it fails, then try 
 * utility_data_set_node_info_compute_values().
 *
 * Returns: a new list which must be freed using g_list_free() (don't change or free the values in the list)
 */
GList *
utility_data_set_node_info_compute_values_ext (DataSetNodeInfo *info, GnomeDbDataProxy *proxy, GtkTreeIter *iter)
{
	gint i, max, col, nb_cols;
	GList *values = NULL;
	GnomeDbDataModel *proxy_model;
	GdaValue *value;
	gboolean modified = FALSE;
	guint attributes;

	g_return_val_if_fail (info, NULL);
	g_return_val_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy), NULL);
	proxy_model = gnome_db_data_proxy_get_model (proxy);
	g_return_val_if_fail (proxy_model, NULL);
	
	if (!IS_GNOME_DB_RESULT_SET (proxy_model))
		return NULL;

	/* compute values */
	max = gda_data_model_get_n_columns (GDA_DATA_MODEL (info->node->data_for_param));
	nb_cols = gnome_db_data_proxy_get_n_columns (proxy);

	for (i = 0 ; (i < max) && !modified; i++) {
		col = gnome_db_result_set_find_column_ext (GNOME_DB_RESULT_SET (proxy_model),
							   info->node, i);
		g_assert (col >= 0);
		gtk_tree_model_get (GTK_TREE_MODEL (proxy), iter, col, &value, 
				    nb_cols + col, &attributes, -1);
		modified = ! (attributes & GNOME_DB_VALUE_IS_UNCHANGED);
		values = g_list_append (values, value);
	}

	/* if the proxy has been modified for the concerned columns, then this function can't be used */
	if (modified) {
		g_list_free (values);
		values = NULL;
	}

	return values;
}

/**
 * utility_data_set_node_info_compute_attributes
 * @info:
 * @proxy:
 * @iter:
 * @to_be_deleted: a place to store if the row is to be deleted, or %NULL
 *
 * Get the attributes of the values in @proxy, at row @iter, which correspond to the parameters in the @info
 *
 * Returns: the attributes
 */
guint
utility_data_set_node_info_compute_attributes (DataSetNodeInfo *info, GnomeDbDataProxy *proxy, GtkTreeIter *iter, 
					       gboolean *to_be_deleted)
{
	guint attributes = GNOME_DB_VALUE_IS_NULL | GNOME_DB_VALUE_CAN_BE_NULL |
		GNOME_DB_VALUE_IS_DEFAULT | GNOME_DB_VALUE_CAN_BE_DEFAULT |
		GNOME_DB_VALUE_IS_UNCHANGED | GNOME_DB_VALUE_HAS_VALUE_ORIG;
	gboolean to_del = TRUE, local_to_del;
	GSList *list;
	GnomeDbDataModel *proxy_model;
	gint col;
	gint offset;
	guint localattr;

	g_return_val_if_fail (info, 0);
	g_return_val_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy), 0);
	proxy_model = gnome_db_data_proxy_get_model (proxy);
	g_return_val_if_fail (proxy_model, 0);
	offset = gnome_db_data_proxy_get_n_columns (proxy);

	/* list the values in proxy_model for each param in info->node->params */
	list = info->node->params;
	while (list) {
		col = gnome_db_data_model_get_column_at_param (proxy_model, info->data_set, list->data);
		gtk_tree_model_get (GTK_TREE_MODEL (proxy), iter, 
				    PROXY_COL_TO_DELETE, &local_to_del,
				    offset + col, &localattr, -1);
		attributes &= localattr;
		to_del = to_del && local_to_del;
		
		list = g_slist_next (list);
	}

	if (to_be_deleted)
		*to_be_deleted = to_del;
	
	return attributes;
}
