/* gnome-db-base.h
 *
 * Copyright (C) 2003 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


/*
 * This object is the base object for most of Mergeant's objects, it provides
 * basic facilities:
 * - a reference to the GnomeDbDict object
 * - a unique id which is used to XML storing procedures
 * - some attributes such as name, description and owner of the object (only used
 *   for DBMS object which are derived from this class.
 */


#ifndef __GNOME_DB_BASE_H_
#define __GNOME_DB_BASE_H_

#include "gnome-db-decl.h"
#include <glib-object.h>
#include <libxml/tree.h>
#include "gnome-db-dict.h"

G_BEGIN_DECLS

#define GNOME_DB_BASE_TYPE          (gnome_db_base_get_type())
#define GNOME_DB_BASE(obj)          G_TYPE_CHECK_INSTANCE_CAST (obj, gnome_db_base_get_type(), GnomeDbBase)
#define GNOME_DB_BASE_CLASS(klass)  G_TYPE_CHECK_CLASS_CAST (klass, gnome_db_base_get_type (), GnomeDbBaseClass)
#define IS_GNOME_DB_BASE(obj)       G_TYPE_CHECK_INSTANCE_TYPE (obj, gnome_db_base_get_type ())


typedef struct _GnomeDbBase GnomeDbBase;
typedef struct _GnomeDbBaseClass GnomeDbBaseClass;
typedef struct _GnomeDbBasePrivate GnomeDbBasePrivate;

/* Properties:
 * name        type          read/write      description
 * ----------------------------------------------------------------
 * dict        pointer       RW              The GnomeDbDict object
 */

/* struct for the object's data */
struct _GnomeDbBase
{
	GObject            object;
	GnomeDbBasePrivate       *priv;
};

/* struct for the object's class */
struct _GnomeDbBaseClass
{
	GObjectClass            class;

	/* signals */
	void        (*changed)        (GnomeDbBase *base);
	void        (*id_changed)     (GnomeDbBase *base);
	void        (*name_changed)   (GnomeDbBase *base);
	void        (*descr_changed)  (GnomeDbBase *base);
	void        (*owner_changed)  (GnomeDbBase *base);
	void        (*nullified)      (GnomeDbBase *base);

	/* pure virtual functions */
	void        (*signal_changed) (GnomeDbBase *base, gboolean block_changed_signal);
#ifdef debug
	void        (*dump)           (GnomeDbBase *base, guint offset);
#endif
};

GType        gnome_db_base_get_type        (void);
GObject     *gnome_db_base_new             (void);

void         gnome_db_base_set_dict        (GnomeDbBase *base, GnomeDbDict *dict);
GnomeDbDict *gnome_db_base_get_dict        (GnomeDbBase *base);

void         gnome_db_base_set_id          (GnomeDbBase *base, guint id);
void         gnome_db_base_set_name        (GnomeDbBase *base, const gchar *name);
void         gnome_db_base_set_description (GnomeDbBase *base, const gchar *descr);
void         gnome_db_base_set_owner       (GnomeDbBase *base, const gchar *owner);

guint        gnome_db_base_get_id          (GnomeDbBase *base);
const gchar *gnome_db_base_get_name        (GnomeDbBase *base);
const gchar *gnome_db_base_get_description (GnomeDbBase *base);
const gchar *gnome_db_base_get_owner       (GnomeDbBase *base);

void         gnome_db_base_nullify         (GnomeDbBase *base); /* force the object to completely clean itself */
void         gnome_db_base_nullify_check   (GnomeDbBase *base); 
gboolean     gnome_db_base_nullified       (GnomeDbBase *base);

void         gnome_db_base_changed         (GnomeDbBase *base);
void         gnome_db_base_block_changed   (GnomeDbBase *base);
void         gnome_db_base_unblock_changed (GnomeDbBase *base);

#ifdef debug
void         gnome_db_base_dump            (GnomeDbBase *base, guint offset); /* dump contents on stdout */
#endif

#define      gnome_db_base_connect_nullify(base, callback, data) \
		(g_assert (!gnome_db_base_nullified ((GnomeDbBase*) (base))), g_signal_connect ((base), "nullified", (callback), (data)))

G_END_DECLS

#endif
