/* gnome-db-server.h
 *
 * Copyright (C) 2003 - 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


#ifndef __GNOME_DB_SERVER_H_
#define __GNOME_DB_SERVER_H_

#include <glib-object.h>
#include <libgda/libgda.h>
#include "gnome-db-decl.h"
#include "gnome-db-dict.h"
#include "gnome-db-server-info.h"

G_BEGIN_DECLS

#define GNOME_DB_SERVER_TYPE          (gnome_db_server_get_type())
#define GNOME_DB_SERVER(obj)          G_TYPE_CHECK_INSTANCE_CAST (obj, gnome_db_server_get_type(), GnomeDbServer)
#define GNOME_DB_SERVER_CLASS(klass)  G_TYPE_CHECK_CLASS_CAST (klass, gnome_db_server_get_type (), GnomeDbServerClass)
#define IS_GNOME_DB_SERVER(obj)       G_TYPE_CHECK_INSTANCE_TYPE (obj, gnome_db_server_get_type ())


/* error reporting */
extern GQuark gnome_db_server_error_quark (void);
#define GNOME_DB_SERVER_ERROR gnome_db_server_error_quark ()

enum
{
	GNOME_DB_SERVER_CONN_OPEN_ERROR,
	GNOME_DB_SERVER_DO_QUERY_ERROR,
	GNOME_DB_SERVER_XML_SAVE_ERROR,
	GNOME_DB_SERVER_XML_LOAD_ERROR,
	GNOME_DB_SERVER_META_DATA_UPDATE,
	GNOME_DB_SERVER_META_DATA_UPDATE_USER_STOPPED,
	GNOME_DB_SERVER_DATATYPE_ERROR,
	GNOME_DB_SERVER_FUNCTIONS_ERROR,
	GNOME_DB_SERVER_AGGREGATES_ERROR
};


typedef enum {
	GNOME_DB_SERVER_QUERY_SQL,
	GNOME_DB_SERVER_QUERY_XML
} GnomeDbServerQueryType;

typedef enum {
	GNOME_DB_SERVER_SELECT_OP,
	GNOME_DB_SERVER_INSERT_OP,
	GNOME_DB_SERVER_UPDATE_OP,
	GNOME_DB_SERVER_DELETE_OP,

	GNOME_DB_SERVER_DDL_OP,
	GNOME_DB_SERVER_UNKNOWN_OP
} GnomeDbServerOpMode;

/* struct to hold any information specific to the provider used */
struct _GnomeDbServerInfo {
	gchar         *provider_name; /* equal to the return of gda_connection_get_provider() */
	
	/*
	 * TRUE if all comparisons of names can be done on the lower case versions of the objects names
	 */
	gboolean       is_case_insensitive; 

	/*
	 * TRUE to suppose that there are implicit casts available for data types which have
	 * the same gda type 
	 */
	gboolean       implicit_data_types_casts;

	/*
	 * Used when looking for a data type from a SQL expression representing a value
	 */
	gint           value_nb_tests_types;
	gchar        **value_test_data_types; /* array of 'value_nb_tests_types' strings */
	GdaValueType  *value_test_gda_types;  /* array of 'value_nb_tests_types' GdaValueType */
};


/* struct for the object's data */
struct _GnomeDbServer
{
	GdaClient              object;
	GnomeDbServerPrivate       *priv;
};

/* struct for the object's class */
struct _GnomeDbServerClass
{
	GdaClientClass class;

	/* signals */
	void   (*conn_opened)               (GnomeDbServer *obj);
	void   (*conn_to_close)             (GnomeDbServer *obj);
	void   (*conn_closed)               (GnomeDbServer *obj);

	void   (*data_type_added)           (GnomeDbServer *obj, GnomeDbServerDataType *type);
	void   (*data_type_removed)         (GnomeDbServer *obj, GnomeDbServerDataType *type);
	void   (*data_type_updated)         (GnomeDbServer *obj, GnomeDbServerDataType *type);

	void   (*data_function_added)       (GnomeDbServer *obj, GnomeDbServerFunction *function);
	void   (*data_function_removed)     (GnomeDbServer *obj, GnomeDbServerFunction *function);
	void   (*data_function_updated)     (GnomeDbServer *obj, GnomeDbServerFunction *function);

	void   (*data_aggregate_added)      (GnomeDbServer *obj, GnomeDbServerAggregate *aggregate);
	void   (*data_aggregate_removed)    (GnomeDbServer *obj, GnomeDbServerAggregate *aggregate);
	void   (*data_aggregate_updated)    (GnomeDbServer *obj, GnomeDbServerAggregate *aggregate);

	void   (*data_update_started)       (GnomeDbServer *obj);
	void   (*update_progress)           (GnomeDbServer *obj, gchar * msg, guint now, guint total);
	void   (*data_update_finished)      (GnomeDbServer *obj);

	void   (*object_handler_updated)    (GnomeDbServer *obj);
};

guint                   gnome_db_server_get_type                  (void);
GObject                *gnome_db_server_new                       (GnomeDbDict *dict);

gboolean                gnome_db_server_set_datasource            (GnomeDbServer *srv, const gchar *datasource);
gchar                  *gnome_db_server_get_datasource            (GnomeDbServer *srv);
gboolean                gnome_db_server_set_user_name             (GnomeDbServer *srv, const gchar *username);
gchar                  *gnome_db_server_get_user_name             (GnomeDbServer *srv);
gboolean                gnome_db_server_set_user_password         (GnomeDbServer *srv, const gchar *password);
void                    gnome_db_server_reset                     (GnomeDbServer *srv);
GdaConnection          *gnome_db_server_get_gda_connection        (GnomeDbServer *srv);

gboolean                gnome_db_server_open_connect              (GnomeDbServer *srv, GError **error);
gboolean                gnome_db_server_conn_is_opened            (GnomeDbServer *srv);
void                    gnome_db_server_close_connect             (GnomeDbServer *srv);
void                    gnome_db_server_close_connect_no_warn     (GnomeDbServer *srv);
GnomeDbServerInfo      *gnome_db_server_get_server_info           (GnomeDbServer *srv);

GnomeDbDict            *gnome_db_server_get_dict                  (GnomeDbServer *srv);
GnomeDbServerOpMode     gnome_db_server_get_sql_op_mode           (GnomeDbServer *srv, const gchar *query);
GdaDataModel           *gnome_db_server_do_query_as_data_model    (GnomeDbServer *srv, const gchar *query,
		     						   GnomeDbServerQueryType type, GError **error);
gboolean                gnome_db_server_update_dbms_data          (GnomeDbServer *srv, GError **error);
void                    gnome_db_server_stop_update_dbms_data     (GnomeDbServer *srv);

GnomeDbDataHandler     *gnome_db_server_get_handler_by_name       (GnomeDbServer *srv, const gchar *name);
GnomeDbDataHandler     *gnome_db_server_get_handler_by_type       (GnomeDbServer *srv, GnomeDbServerDataType *type);
GnomeDbDataHandler     *gnome_db_server_get_handler_by_gda        (GnomeDbServer *srv, GdaValueType gda_type);
GSList                 *gnome_db_server_get_plugin_handlers       (GnomeDbServer *srv);

GnomeDbUser            *gnome_db_server_get_user_by_name          (GnomeDbServer *srv, const gchar *username);

GSList                 *gnome_db_server_get_data_types            (GnomeDbServer *srv);
GnomeDbServerDataType  *gnome_db_server_get_data_type_by_name     (GnomeDbServer *srv, const gchar *typename);
GnomeDbServerDataType  *gnome_db_server_get_data_type_by_xml_id   (GnomeDbServer *srv, const gchar *xml_id);
gboolean                gnome_db_server_declare_custom_data_type  (GnomeDbServer *srv, GnomeDbServerDataType *type);

GSList                 *gnome_db_server_get_functions             (GnomeDbServer *srv);
GSList                 *gnome_db_server_get_functions_by_name     (GnomeDbServer *srv, const gchar *funcname);
GnomeDbServerFunction  *gnome_db_server_get_function_by_name_arg  (GnomeDbServer *srv, const gchar *funcname, 
								   const GSList *argtypes);
GnomeDbServerFunction  *gnome_db_server_get_function_by_xml_id    (GnomeDbServer *srv, const gchar *xml_id);
GnomeDbServerFunction  *gnome_db_server_get_function_by_dbms_id   (GnomeDbServer *srv, const gchar *dbms_id);

GSList                 *gnome_db_server_get_aggregates            (GnomeDbServer *srv);
GSList                 *gnome_db_server_get_aggregates_by_name    (GnomeDbServer *srv, const gchar *aggname);
GnomeDbServerAggregate *gnome_db_server_get_aggregate_by_name_arg (GnomeDbServer *srv, const gchar *aggname, 
								   GnomeDbServerDataType *argtype);
GnomeDbServerAggregate *gnome_db_server_get_aggregate_by_xml_id   (GnomeDbServer *srv, const gchar *xml_id);
GnomeDbServerAggregate *gnome_db_server_get_aggregate_by_dbms_id  (GnomeDbServer *srv, const gchar *dbms_id);
 
GnomeDbDataHandler     *gnome_db_server_get_object_handler        (GnomeDbServer *srv, GObject *object);
void                    gnome_db_server_set_object_handler        (GnomeDbServer *srv, GObject *object, 
								   GnomeDbDataHandler *handler);
void                    gnome_db_server_unset_object_handler      (GnomeDbServer *srv, GObject *object);
gboolean                gnome_db_server_object_has_handler        (GnomeDbServer *srv, GObject *object);
void                    gnome_db_server_set_object_func_handler   (GnomeDbServer *srv, 
								   GnomeDbDataHandler *(*func)(GnomeDbServer *, GObject *));

/* Convenience function */
gchar                  *gnome_db_server_stringify_value           (GnomeDbServer *srv, const GdaValue * value);
gchar                  *gnome_db_server_escape_chars              (GnomeDbServer *srv, const gchar *string);
GdaDataModel           *gnome_db_server_get_gda_schema            (GnomeDbServer *srv, GdaConnectionSchema schema,
								   GdaParameterList *params);

#ifdef debug
void                    gnome_db_server_dump                      (GnomeDbServer *srv, gint offset);
#endif

/* internal use only */
void                    gnome_db_server_add_data_type             (GnomeDbServer *srv, GnomeDbServerDataType *datatype);

G_END_DECLS

#endif
