/* gnome-db-dict.h
 *
 * Copyright (C) 2003 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


/*
 * This object manages the objects used my mergeant (for tables, queries, etc),
 * but not what is specific to Mergeant's own GUI.
 */

#ifndef __GNOME_DB_DICT_H_
#define __GNOME_DB_DICT_H_

#include <glib-object.h>
#include "gnome-db-decl.h"
#include "gnome-db-enums.h"

G_BEGIN_DECLS

#define GNOME_DB_DICT_TYPE          (gnome_db_dict_get_type())
#define GNOME_DB_DICT(obj)          G_TYPE_CHECK_INSTANCE_CAST (obj, gnome_db_dict_get_type(), GnomeDbDict)
#define GNOME_DB_DICT_CLASS(klass)  G_TYPE_CHECK_CLASS_CAST (klass, gnome_db_dict_get_type (), GnomeDbDictClass)
#define IS_GNOME_DB_DICT(obj)       G_TYPE_CHECK_INSTANCE_TYPE (obj, gnome_db_dict_get_type ())


/* error reporting */
extern GQuark gnome_db_dict_error_quark (void);
#define GNOME_DB_DICT_ERROR gnome_db_dict_error_quark ()

enum {
	GNOME_DB_DICT_LOAD_FILE_NOT_EXIST_ERROR,
	GNOME_DB_DICT_FILE_LOAD_ERROR,
	GNOME_DB_DICT_FILE_SAVE_ERROR,
	GNOME_DB_DICT_PROPOSED_FILE
};


/* struct for the object's data */
struct _GnomeDbDict
{
	GObject              object;
	GnomeDbDictPrivate  *priv;
};

/* struct for the object's class */
struct _GnomeDbDictClass
{
	GObjectClass            class;

        /* signal the addition or removal of a query in the queries list */
        void (*query_added)      (GnomeDbDict * dict, GnomeDbQuery *new_query);
        void (*query_removed)    (GnomeDbDict * dict, GnomeDbQuery *old_query);
        void (*query_updated)    (GnomeDbDict * dict, GnomeDbQuery *query);

        /* signal the addition or removal of a graph in the graphs list */
        void (*graph_added)      (GnomeDbDict * dict, GnomeDbGraph *new_graph);
        void (*graph_removed)    (GnomeDbDict * dict, GnomeDbGraph *old_graph);
        void (*graph_updated)    (GnomeDbDict * dict, GnomeDbGraph *graph);

        /* signal the addition or removal of a custom layout in the layouts list */
        void (*layout_added)      (GnomeDbDict * dict, GnomeDbCustomLayout *new_layout);
        void (*layout_removed)    (GnomeDbDict * dict, GnomeDbCustomLayout *old_layout);
        void (*layout_updated)    (GnomeDbDict * dict, GnomeDbCustomLayout *layout);

	/* signal that a change in the whole dictionary has occured */
	void (*changed)          (GnomeDbDict * dict);
};

guint           gnome_db_dict_get_type            (void);
GObject        *gnome_db_dict_new                 (void);

guint           gnome_db_dict_get_id_serial       (GnomeDbDict *dict);
void            gnome_db_dict_set_id_serial       (GnomeDbDict *dict, guint value);

gchar          *gnome_db_dict_compute_xml_filename(GnomeDbDict *dict, const gchar *datasource, const gchar *app_id, GError **error);
void            gnome_db_dict_set_xml_filename    (GnomeDbDict *dict, const gchar *xmlfile);
const gchar    *gnome_db_dict_get_xml_filename    (GnomeDbDict *dict);
gboolean        gnome_db_dict_load_xml            (GnomeDbDict *dict, GError **error);
gboolean        gnome_db_dict_save_xml            (GnomeDbDict *dict, GError **error);
gboolean        gnome_db_dict_load_xml_file       (GnomeDbDict *dict, const gchar *xmlfile, GError **error);
gboolean        gnome_db_dict_save_xml_file       (GnomeDbDict *dict, const gchar *xmlfile, GError **error);

void            gnome_db_dict_declare_query       (GnomeDbDict *dict, GnomeDbQuery *query);
void            gnome_db_dict_assume_query        (GnomeDbDict *dict, GnomeDbQuery *query);
void            gnome_db_dict_unassume_query      (GnomeDbDict *dict, GnomeDbQuery *query);
GSList         *gnome_db_dict_get_queries         (GnomeDbDict *dict);
GnomeDbQuery        *gnome_db_dict_get_query_by_xml_id (GnomeDbDict *dict, const gchar *xml_id);

void            gnome_db_dict_declare_graph       (GnomeDbDict *dict, GnomeDbGraph *graph);
void            gnome_db_dict_assume_graph        (GnomeDbDict *dict, GnomeDbGraph *graph);
void            gnome_db_dict_unassume_graph      (GnomeDbDict *dict, GnomeDbGraph *graph);
GSList         *gnome_db_dict_get_graphs          (GnomeDbDict *dict, GnomeDbGraphType type_of_graphs);
GnomeDbGraph        *gnome_db_dict_get_graph_by_xml_id (GnomeDbDict *dict, const gchar *xml_id);
GnomeDbGraph        *gnome_db_dict_get_graph_for_object(GnomeDbDict *dict, GObject *obj);

void            gnome_db_dict_declare_layout       (GnomeDbDict *dict, GnomeDbCustomLayout *layout);
void            gnome_db_dict_assume_layout        (GnomeDbDict *dict, GnomeDbCustomLayout *layout);
void            gnome_db_dict_unassume_layout      (GnomeDbDict *dict, GnomeDbCustomLayout *layout);
GSList         *gnome_db_dict_get_layouts          (GnomeDbDict *dict);
GnomeDbCustomLayout *gnome_db_dict_get_layout_by_xml_id (GnomeDbDict *dict, const gchar *xml_id);


GnomeDbServer       *gnome_db_dict_get_server          (GnomeDbDict *dict);
GnomeDbDatabase     *gnome_db_dict_get_database        (GnomeDbDict *dict);

GSList         *gnome_db_dict_get_entities_fk_constraints (GnomeDbDict *dict, GnomeDbEntity *entity1, GnomeDbEntity *entity2,
						     gboolean entity1_has_fk);

#ifdef debug
void            gnome_db_dict_dump                (GnomeDbDict *dict);
#endif
G_END_DECLS


#endif
