/* gnome-db-canvas-entity.c
 *
 * Copyright (C) 2002 - 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-canvas.h"
#include "gnome-db-canvas-entity.h"
#include "gnome-db-canvas-field.h"
#include <libgnomedb/gnome-db-field.h>
#include <libgnomedb/gnome-db-table.h>
#include <libgnomedb/gnome-db-query.h>
#include <libgnomedb/gnome-db-entity.h>
#include <libgnomedb/gnome-db-target.h>

static void gnome_db_canvas_entity_class_init (GnomeDbCanvasEntityClass * class);
static void gnome_db_canvas_entity_init       (GnomeDbCanvasEntity * drag);
static void gnome_db_canvas_entity_dispose    (GObject   * object);
static void gnome_db_canvas_entity_finalize   (GObject   * object);

static void gnome_db_canvas_entity_set_property (GObject              *object,
					   guint                 param_id,
					   const GValue         *value,
					   GParamSpec           *pspec);
static void gnome_db_canvas_entity_get_property (GObject              *object,
					   guint                 param_id,
					   GValue               *value,
					   GParamSpec           *pspec);

enum
{
	PROP_0,
	PROP_ENTITY,
	PROP_TARGET,
	PROP_SCALE,
	PROP_MENU_FUNC
};

struct _GnomeDbCanvasEntityPrivate
{
	GnomeDbTarget           *target; /* NULL if the GnomeDbCanvasEntity is a generic entity */
	GnomeDbEntity           *entity;

	/* UI building information */
        GSList             *field_items; /* list of GnomeCanvasItem for the fields */
	gint                init_font_size;
        GnomeCanvasItem    *title;
	gdouble            *field_ypos; /* array for each field's Y position in this canvas group */
	GtkWidget        *(*popup_menu_func) (GnomeDbCanvasEntity *ce);

	/* presentation parameters */
        gdouble             x_text_space;
        gdouble             y_text_space;
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *entity_parent_class = NULL;

guint
gnome_db_canvas_entity_get_type (void)
{
	static GType type = 0;

        if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbCanvasEntityClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_canvas_entity_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbCanvasEntity),
			0,
			(GInstanceInitFunc) gnome_db_canvas_entity_init
		};		

		type = g_type_register_static (GNOME_DB_CANVAS_ITEM_TYPE, "GnomeDbCanvasEntity", &info, 0);
	}

	return type;
}

	

static void
gnome_db_canvas_entity_class_init (GnomeDbCanvasEntityClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	entity_parent_class = g_type_class_peek_parent (class);

	object_class->dispose = gnome_db_canvas_entity_dispose;
	object_class->finalize = gnome_db_canvas_entity_finalize;

	/* Properties */
	object_class->set_property = gnome_db_canvas_entity_set_property;
	object_class->get_property = gnome_db_canvas_entity_get_property;

	g_object_class_install_property
                (object_class, PROP_ENTITY,
                 g_param_spec_pointer ("entity", NULL, NULL, (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property
                (object_class, PROP_TARGET,
                 g_param_spec_pointer ("target", NULL, NULL, (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property 
		(object_class, PROP_SCALE,
		 g_param_spec_double ("scale", NULL, NULL, 0., G_MAXDOUBLE, 0., G_PARAM_WRITABLE));
	g_object_class_install_property 
		(object_class, PROP_MENU_FUNC,
                 g_param_spec_pointer ("popup_menu_func", "Popup menu function", 
				       "Function to create a popup menu on each GnomeDbCanvasEntity",  G_PARAM_WRITABLE));
}

static void
gnome_db_canvas_entity_init (GnomeDbCanvasEntity * entity)
{
	entity->priv = g_new0 (GnomeDbCanvasEntityPrivate, 1);
	entity->priv->entity = NULL;
	entity->priv->target = NULL;
	entity->priv->field_ypos = NULL;
	entity->priv->popup_menu_func = NULL;

	entity->priv->x_text_space = 3.;
	entity->priv->y_text_space = 3.;
}

static void clean_items (GnomeDbCanvasEntity *ce);
static void create_items (GnomeDbCanvasEntity *ce);
static void object_nullified_cb (GnomeDbEntity *ent, GnomeDbCanvasEntity *ce);
static void entity_changed_cb (GnomeDbEntity *ent, GnomeDbCanvasEntity *ce);

static void
gnome_db_canvas_entity_dispose (GObject   * object)
{
	GnomeDbCanvasEntity *ce;
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_CANVAS_ENTITY (object));

	ce = GNOME_DB_CANVAS_ENTITY (object);
	clean_items (ce);
	if (ce->priv->target) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (ce->priv->target),
						      G_CALLBACK (object_nullified_cb), ce);
		ce->priv->target = NULL;
	}
	if (ce->priv->entity) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (ce->priv->entity),
						      G_CALLBACK (object_nullified_cb), ce);
		g_signal_handlers_disconnect_by_func (G_OBJECT (ce->priv->entity),
						      G_CALLBACK (entity_changed_cb), ce);
		ce->priv->entity = NULL;
	}

	/* for the parent class */
	entity_parent_class->dispose (object);
}


static void
gnome_db_canvas_entity_finalize (GObject   * object)
{
	GnomeDbCanvasEntity *ce;
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_CANVAS_ENTITY (object));

	ce = GNOME_DB_CANVAS_ENTITY (object);
	if (ce->priv) {
		if (ce->priv->field_items)
			g_slist_free (ce->priv->field_items);
		if (ce->priv->field_ypos)
			g_free (ce->priv->field_ypos);

		g_free (ce->priv);
		ce->priv = NULL;
	}

	/* for the parent class */
	entity_parent_class->finalize (object);
}

static void 
gnome_db_canvas_entity_set_property    (GObject              *object,
				  guint                 param_id,
				  const GValue         *value,
				  GParamSpec           *pspec)
{
	GnomeDbCanvasEntity *ce;
	gpointer ptr;
	gdouble scale;
	GSList *list;
	PangoFontDescription *font_desc, *font_copy;

	ce = GNOME_DB_CANVAS_ENTITY (object);

	switch (param_id) {
	case PROP_TARGET:
		ptr = g_value_get_pointer (value);
		if (ptr == ce->priv->target)
			return;

		if (ce->priv->target) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (ce->priv->target),
							      G_CALLBACK (object_nullified_cb), ce);
			g_object_set (G_OBJECT (object), "entity", NULL, NULL);
			ce->priv->target = NULL;
		}
		if (ptr) {
			g_return_if_fail (IS_GNOME_DB_TARGET (ptr));
			ce->priv->target = GNOME_DB_TARGET (ptr);
			gnome_db_base_connect_nullify (ce->priv->target, G_CALLBACK (object_nullified_cb), ce);
			g_object_set (G_OBJECT (object), "entity", gnome_db_target_get_represented_entity (GNOME_DB_TARGET (ptr)), NULL);
		}
		break;
	case PROP_ENTITY:
		ptr = g_value_get_pointer (value);
		if (ptr == ce->priv->entity)
			return;

		if (ce->priv->entity) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (ce->priv->entity),
							      G_CALLBACK (object_nullified_cb), ce);
			g_signal_handlers_disconnect_by_func (G_OBJECT (ce->priv->entity),
							      G_CALLBACK (entity_changed_cb), ce);
			ce->priv->entity = NULL;
			clean_items (ce);
		}
		if (ptr) {
			g_return_if_fail (IS_GNOME_DB_ENTITY (ptr));
			ce->priv->entity = ptr;
			gnome_db_base_connect_nullify (ce->priv->entity, G_CALLBACK (object_nullified_cb), ce);
			/* Signals to keep the display up to date */
			g_signal_connect (G_OBJECT (ce->priv->entity), "changed",
					  G_CALLBACK (entity_changed_cb), ce);

			create_items (ce);
		}
		break;
	case PROP_SCALE:
		scale = g_value_get_double (value);
		list = ce->priv->field_items;
		while (list) {
			g_object_set (G_OBJECT (list->data), "scale", scale, NULL);
			list = g_slist_next (list);
		}
		g_object_get (G_OBJECT (ce->priv->title), "font-desc", &font_desc, NULL);
		font_copy = pango_font_description_copy (font_desc);
		pango_font_description_set_size (font_copy, scale * ce->priv->init_font_size);
		g_object_set (G_OBJECT (ce->priv->title), "font-desc", font_copy, NULL);
		pango_font_description_free (font_copy);
		break;
	case PROP_MENU_FUNC:
		ce->priv->popup_menu_func = (GtkWidget *(*) (GnomeDbCanvasEntity *ce)) g_value_get_pointer (value);
		create_items (ce);
		break;
	}
}

static void 
gnome_db_canvas_entity_get_property    (GObject              *object,
				    guint                 param_id,
				    GValue               *value,
				    GParamSpec           *pspec)
{
	TO_IMPLEMENT;
}

static void
object_nullified_cb (GnomeDbEntity *ent, GnomeDbCanvasEntity *ce)
{
	if ((ent == ce->priv->entity) && ce->priv->target) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (ce->priv->entity),
						      G_CALLBACK (object_nullified_cb), ce);
		g_signal_handlers_disconnect_by_func (G_OBJECT (ce->priv->entity),
						      G_CALLBACK (entity_changed_cb), ce);
		ce->priv->entity = NULL;
	}
	else 
		gtk_object_destroy (GTK_OBJECT (ce));
}

static void
entity_changed_cb (GnomeDbEntity *ent, GnomeDbCanvasEntity *ce)
{
	create_items (ce);
}

/* 
 * destroy any existing GnomeCanvasItem obejcts 
 */
static void 
clean_items (GnomeDbCanvasEntity *ce)
{
	/* destroy all the items in the group */
	while (GNOME_CANVAS_GROUP (ce)->item_list) 
		gtk_object_destroy (GTK_OBJECT (GNOME_CANVAS_GROUP (ce)->item_list->data));

	ce->priv->title = NULL;

	/* free the fields positions */
	if (ce->priv->field_ypos) {
		g_free (ce->priv->field_ypos);
		ce->priv->field_ypos = NULL;
	}
}

static int button_item_event (GnomeCanvasItem *ci, GdkEvent *event, GnomeDbCanvasEntity *ce);
static void field_item_destroy_cb (GnomeDbCanvasField *field, GnomeDbCanvasEntity *ce);

/*
 * create new GnomeCanvasItem objects
 */
static void 
create_items (GnomeDbCanvasEntity *ce)
{
	GnomeCanvasItem *item;
	const gchar *cstr, *cstr2;
	gchar *tmpstr = NULL;
	double x1, y1, x2, y2;
	GSList *list, *tmplist;
	gdouble x, sqsize, radius;
	gdouble title_text_height, title_text_width, total_width;
	gint fieldn;
	/* WARNING: the text items (GNOME_TYPE_CANVAS_TEXT) are first drawn without taking care of the real zoom factor =>
	 * we must take that into account when using the text sizes */
	gdouble scale = GNOME_CANVAS_ITEM (ce)->canvas->pixels_per_unit;

	if (!ce->priv->entity)
		return;

	clean_items (ce);

	/* Title of the Table or query */
	ce->priv->init_font_size = pango_font_description_get_size 
		(GTK_WIDGET (GNOME_CANVAS_ITEM (ce)->canvas)->style->font_desc);

	if (ce->priv->target)
		tmpstr = gnome_db_target_get_complete_name (ce->priv->target);
	else {
		cstr = gnome_db_base_get_name (GNOME_DB_BASE (ce->priv->entity));
		tmpstr = g_strdup (cstr);
	}
		
	item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (ce),
				      GNOME_TYPE_CANVAS_TEXT,
				      "font-desc", GTK_WIDGET (GNOME_CANVAS_ITEM (ce)->canvas)->style->font_desc,
				      "weight", PANGO_WEIGHT_BOLD,
				      "text", tmpstr,
				      "x", ce->priv->x_text_space, 
				      "y", ce->priv->y_text_space,
				      "fill_color", "black",
				      "justification", GTK_JUSTIFY_RIGHT, 
				      "anchor", GTK_ANCHOR_NORTH_WEST, 
				      NULL);
	g_free (tmpstr);
	gnome_canvas_item_get_bounds (item, &x1, &y1, &x2, &y2);
	ce->priv->title = item;

	/* Getting text metrics */
	title_text_height = (y2 - y1) * scale;
	title_text_width = (x2 - x1) * scale;
	total_width = title_text_width + 2 * ce->priv->x_text_space;

	/* fields */
	sqsize = title_text_height * 0.6;
	list = gnome_db_entity_get_fields (ce->priv->entity);

	ce->priv->field_ypos = g_new0 (gdouble, g_slist_length (list) + 1);
	ce->priv->field_ypos [0] = title_text_height + 3* ce->priv->y_text_space;

	tmplist = list;
	fieldn = 0;
	while (tmplist) {
		item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (ce),
					      GNOME_DB_CANVAS_FIELD_TYPE,
					      "x", 0.,
					      "y", ce->priv->field_ypos[fieldn],
					      "field", tmplist->data,
					      NULL);	
		ce->priv->field_items = g_slist_append (ce->priv->field_items, item);
		
		g_signal_connect (G_OBJECT (item), "destroy",
				  G_CALLBACK (field_item_destroy_cb), ce);

		gnome_canvas_item_get_bounds (item, &x1, &y1, &x2, &y2);
		ce->priv->field_ypos[fieldn+1] = y2;

		if (x2 - x1 > total_width)
			total_width = x2 - x1;

		tmplist = g_slist_next (tmplist);
		fieldn++;
	}
	g_slist_free (list);

	cstr = GNOME_DB_CANVAS_ENTITY_COLOR;
	if (IS_GNOME_DB_TABLE (ce->priv->entity))
		cstr = GNOME_DB_CANVAS_DB_TABLE_COLOR;
	if (IS_GNOME_DB_QUERY (ce->priv->entity))
		cstr = GNOME_DB_CANVAS_QUERY_COLOR;

	/* "button" to activate a popup menu the GnomeDbCanvasEntity  */
	if (ce->priv->popup_menu_func) {
		sqsize = title_text_height * 0.8;
		x = total_width + ce->priv->x_text_space;
		if (x - sqsize - 2*ce->priv->x_text_space < title_text_width)
			x = title_text_width + sqsize + 2*ce->priv->x_text_space;
		
		item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (ce),
				      GNOME_TYPE_CANVAS_RECT,
					      "x1", x - sqsize,
					      "y1", ce->priv->y_text_space,
					      "x2", x,
					      "y2", ce->priv->y_text_space + sqsize,
					      "fill_color", "white",
					      "outline_color", "black",
					      "width_units", 1.0,
					      NULL);
		gnome_canvas_item_raise_to_top (item);
		g_signal_connect (G_OBJECT (item),"event",
				  G_CALLBACK (button_item_event), ce);
		
		if (x - ce->priv->x_text_space > total_width)
			total_width = x - ce->priv->x_text_space;
		
		radius = sqsize * .2;
		item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (ce),
					      GNOME_TYPE_CANVAS_ELLIPSE,
					      "x1", x - sqsize/2. - radius,
					      "y1", ce->priv->y_text_space + sqsize/2. - radius,
					      "x2", x - sqsize/2. + radius,
					      "y2", ce->priv->y_text_space + sqsize/2. + radius,
					      "fill_color", cstr,
					      "outline_color", "black",
					      "width_units", 1.0,
					      NULL);
		gnome_canvas_item_raise_to_top (item);
		g_signal_connect (G_OBJECT (item),"event",
				  G_CALLBACK (button_item_event), ce);
	}

	/* Top little frame */
	item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (ce),
				      GNOME_TYPE_CANVAS_RECT,
				      "x1", (double) 0,
				      "y1", (double) 0,
				      "x2", total_width + 2. * ce->priv->x_text_space,
				      "y2", title_text_height + 2 * ce->priv->y_text_space,
				      "outline_color", "black",
				      "fill_color", cstr, 
				      "width_units", 1.0, 
				      NULL);
	gnome_canvas_item_lower_to_bottom (item);

	/* Outline frame */
	item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (ce),
				      GNOME_TYPE_CANVAS_RECT,
				      "x1", (double) 0,
				      "y1", (double) 0,
				      "x2", total_width + 2. * ce->priv->x_text_space,
				      "y2", ce->priv->field_ypos[fieldn] + 1,
				      "outline_color", "black",
				      "fill_color", "white",
				      "width_units", 1.0, NULL);
	gnome_canvas_item_lower_to_bottom (item);

	total_width += 2. * ce->priv->x_text_space;

	/* setting the fields' background width to be the same for all */
	tmplist = ce->priv->field_items;
	while (tmplist) {
		g_object_set (G_OBJECT (tmplist->data), "width", total_width, NULL);
		tmplist = g_slist_next (tmplist);
	}

	/* make sure the scale is correctly applied to text items */
	g_object_set (G_OBJECT (ce), "scale", scale, NULL);
}

static void
field_item_destroy_cb (GnomeDbCanvasField *field, GnomeDbCanvasEntity *ce)
{
	g_assert (g_slist_find (ce->priv->field_items, field));
	ce->priv->field_items = g_slist_remove (ce->priv->field_items, field);
}

static int 
button_item_event (GnomeCanvasItem *ci, GdkEvent *event, GnomeDbCanvasEntity *ce)
{
	gboolean done = TRUE;

	switch (event->type) {
	case GDK_BUTTON_PRESS:
		if (ce->priv->popup_menu_func) {
			GtkWidget *menu;
			menu = ce->priv->popup_menu_func (ce);
			gtk_menu_popup (GTK_MENU (menu), NULL, NULL,
					NULL, NULL, ((GdkEventButton *)event)->button,
					((GdkEventButton *)event)->time);
		}
		else
			done = FALSE;
		break;
	default:
		done = FALSE;
		break;
	}

	return done;	
}

/**
 * gnome_db_canvas_entity_get_field_item
 * @ce: a #GnomeDbCanvasEntity object
 * @field: a #GnomeDbField object
 *
 * Get the #GnomeDbCanvasField object representing @field
 * in @ce.
 *
 * Returns: the corresponding #GnomeDbCanvasField
 */
GnomeDbCanvasField *
gnome_db_canvas_entity_get_field_item (GnomeDbCanvasEntity *ce, GnomeDbField *field)
{
	GSList *fields;
	gint pos;

	g_return_val_if_fail (ce && IS_GNOME_DB_CANVAS_ENTITY (ce), NULL);
	g_return_val_if_fail (ce->priv, NULL);
	g_return_val_if_fail (ce->priv->entity, NULL);

	fields = gnome_db_entity_get_fields (ce->priv->entity);
	pos = g_slist_index (fields, field);
	g_return_val_if_fail (pos >= 0, NULL);

	return g_slist_nth_data (ce->priv->field_items, pos);
}


/**
 * gnome_db_canvas_entity_get_field_ypos
 * @ce: a #GnomeDbCanvasEntity object
 * @field: a #GnomeDbField object
 *
 * Get the Y position of the middle of the #GnomeDbCanvasField object representing @field
 * in @ce, in @ce's coordinates.
 *
 * Returns: the Y coordinate.
 */
gdouble
gnome_db_canvas_entity_get_field_ypos (GnomeDbCanvasEntity *ce, GnomeDbField *field)
{
	gint pos;

	g_return_val_if_fail (ce && IS_GNOME_DB_CANVAS_ENTITY (ce), 0.);
	g_return_val_if_fail (ce->priv, 0.);
	g_return_val_if_fail (ce->priv->entity, 0.);
	g_return_val_if_fail (ce->priv->field_ypos, 0.);

	pos = gnome_db_entity_get_field_index (ce->priv->entity, field);
	g_return_val_if_fail (pos >= 0, 0.);
	return (0.75 * ce->priv->field_ypos[pos+1] + 0.25 * ce->priv->field_ypos[pos]);
}
