/* gnome-db-server-function.c
 *
 * Copyright (C) 2003 - 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-server.h"
#include "gnome-db-server-function.h"
#include "gnome-db-server-data-type.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-data-handler.h"
#include "marshal.h"
#include <libgda/libgda.h>
#include <string.h>

/* 
 * Main static functions 
 */
static void gnome_db_server_function_class_init (GnomeDbServerFunctionClass * class);
static void gnome_db_server_function_init (GnomeDbServerFunction * srv);
static void gnome_db_server_function_dispose (GObject   * object);
static void gnome_db_server_function_finalize (GObject   * object);

static void gnome_db_server_function_set_property (GObject              *object,
					     guint                 param_id,
					     const GValue         *value,
					     GParamSpec           *pspec);
static void gnome_db_server_function_get_property (GObject              *object,
					     guint                 param_id,
					     GValue               *value,
					     GParamSpec           *pspec);

#ifdef debug
static void gnome_db_server_function_dump (GnomeDbServerFunction *func, guint offset);
#endif

static void        gnome_db_function_xml_storage_init (GnomeDbXmlStorageIface *iface);
static gchar      *gnome_db_function_get_xml_id (GnomeDbXmlStorage *iface);
static xmlNodePtr  gnome_db_function_save_to_xml (GnomeDbXmlStorage *iface, GError **error);
static gboolean    gnome_db_function_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	TEMPL_SIGNAL,
	LAST_SIGNAL
};

static gint gnome_db_server_function_signals[LAST_SIGNAL] = { 0 };

/* properties */
enum
{
	PROP_0,
	PROP
};


/* private structure */
struct _GnomeDbServerFunctionPrivate
{
	GnomeDbServer         *srv;
	gchar            *objectid;       /* unique id for the function */
	GnomeDbServerDataType *result_type;
	GSList           *arg_types;           /* list of GnomeDbServerDataType pointers */
};


/* module error */
GQuark gnome_db_server_function_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_server_function_error");
	return quark;
}


guint
gnome_db_server_function_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbServerFunctionClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_server_function_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbServerFunction),
			0,
			(GInstanceInitFunc) gnome_db_server_function_init
		};

		static const GInterfaceInfo xml_storage_info = {
			(GInterfaceInitFunc) gnome_db_function_xml_storage_init,
			NULL,
			NULL
		};
		
		type = g_type_register_static (GNOME_DB_BASE_TYPE, "GnomeDbServerFunction", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_XML_STORAGE_TYPE, &xml_storage_info);
	}
	return type;
}

static void 
gnome_db_function_xml_storage_init (GnomeDbXmlStorageIface *iface)
{
	iface->get_xml_id = gnome_db_function_get_xml_id;
	iface->save_to_xml = gnome_db_function_save_to_xml;
	iface->load_from_xml = gnome_db_function_load_from_xml;
}


static void
gnome_db_server_function_class_init (GnomeDbServerFunctionClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	gnome_db_server_function_signals[TEMPL_SIGNAL] =
		g_signal_new ("templ_signal",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbServerFunctionClass, templ_signal),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	class->templ_signal = NULL;

	object_class->dispose = gnome_db_server_function_dispose;
	object_class->finalize = gnome_db_server_function_finalize;

	/* Properties */
	object_class->set_property = gnome_db_server_function_set_property;
	object_class->get_property = gnome_db_server_function_get_property;
	g_object_class_install_property (object_class, PROP,
					 g_param_spec_pointer ("prop", NULL, NULL, (G_PARAM_READABLE | G_PARAM_WRITABLE)));

	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_server_function_dump;
#endif

}

static void
gnome_db_server_function_init (GnomeDbServerFunction * gnome_db_server_function)
{
	gnome_db_server_function->priv = g_new0 (GnomeDbServerFunctionPrivate, 1);
	gnome_db_server_function->priv->srv = NULL;
	gnome_db_server_function->priv->objectid = NULL;
	gnome_db_server_function->priv->result_type = NULL;
	gnome_db_server_function->priv->arg_types = NULL;
}


/**
 * gnome_db_server_function_new
 * @srv: a #GnomeDbServer object
 *
 * Creates a new GnomeDbServerFunction object
 *
 * Returns: the new object
 */
GObject*
gnome_db_server_function_new (GnomeDbServer *srv)
{
	GObject   *obj;
	GnomeDbServerFunction *gnome_db_server_function;

	g_return_val_if_fail (srv && GNOME_DB_SERVER (srv), NULL);

	obj = g_object_new (GNOME_DB_SERVER_FUNCTION_TYPE, "dict",
			    gnome_db_server_get_dict (srv), NULL);
	gnome_db_server_function = GNOME_DB_SERVER_FUNCTION (obj);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_server_function), 0);
	g_object_add_weak_pointer (G_OBJECT (srv), (gpointer) &(gnome_db_server_function->priv->srv));
	gnome_db_server_function->priv->srv = srv;

	return obj;
}


static void
gnome_db_server_function_dispose (GObject *object)
{
	GnomeDbServerFunction *gnome_db_server_function;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_SERVER_FUNCTION (object));

	gnome_db_server_function = GNOME_DB_SERVER_FUNCTION (object);
	if (gnome_db_server_function->priv) {
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));
		g_object_remove_weak_pointer (G_OBJECT (gnome_db_server_function->priv->srv), 
					      (gpointer) &(gnome_db_server_function->priv->srv));

		gnome_db_server_function_set_ret_type (gnome_db_server_function, NULL);
		gnome_db_server_function_set_arg_types (gnome_db_server_function, NULL);
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_server_function_finalize (GObject   * object)
{
	GnomeDbServerFunction *gnome_db_server_function;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_SERVER_FUNCTION (object));

	gnome_db_server_function = GNOME_DB_SERVER_FUNCTION (object);
	if (gnome_db_server_function->priv) {

		g_free (gnome_db_server_function->priv);
		gnome_db_server_function->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_server_function_set_property (GObject              *object,
			guint                 param_id,
			const GValue         *value,
			GParamSpec           *pspec)
{
	gpointer ptr;
	GnomeDbServerFunction *gnome_db_server_function;

	gnome_db_server_function = GNOME_DB_SERVER_FUNCTION (object);
	if (gnome_db_server_function->priv) {
		switch (param_id) {
		case PROP:
			/* FIXME */
			ptr = g_value_get_pointer (value);
			break;
		}
	}
}

static void
gnome_db_server_function_get_property (GObject              *object,
			guint                 param_id,
			GValue               *value,
			GParamSpec           *pspec)
{
	GnomeDbServerFunction *gnome_db_server_function;
	gnome_db_server_function = GNOME_DB_SERVER_FUNCTION (object);
	
	if (gnome_db_server_function->priv) {
		switch (param_id) {
		case PROP:
			/* FIXME */
			g_value_set_pointer (value, NULL);
			break;
		}	
	}
}

#ifdef debug
static void
gnome_db_server_function_dump (GnomeDbServerFunction *func, guint offset)
{
	gchar *str;
	GString *string;
	GSList *list;
	gboolean first = TRUE;
	gint i;

	g_return_if_fail (func && IS_GNOME_DB_SERVER_FUNCTION (func));
	g_return_if_fail (func->priv);
	
	/* string for the offset */
	str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

	/* return type */
	if (func->priv->result_type)
		g_print ("%sReturn data type: %s\n", str, gnome_db_server_data_type_get_sqlname (func->priv->result_type));
	else
		g_print ("%s" D_COL_ERR "No return type defined" D_COL_NOR "\n", str);
	
	/* arguments */
	string = g_string_new (" (");
	list = func->priv->arg_types;
	while (list) {
		if (first)
			first = FALSE;
		else
			g_string_append (string, ", ");
		g_string_append_printf (string, "%s", gnome_db_server_data_type_get_sqlname (GNOME_DB_SERVER_DATA_TYPE (list->data)));
		list = g_slist_next (list);
	}
	g_string_append (string, ")");
	g_print ("%sArguments: %s\n", str, string->str);
	g_string_free (string, TRUE);

	g_free (str);
}
#endif



/* GnomeDbXmlStorage interface implementation */
static gchar *
gnome_db_function_get_xml_id (GnomeDbXmlStorage *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_SERVER_FUNCTION (iface), NULL);
	g_return_val_if_fail (GNOME_DB_SERVER_FUNCTION (iface)->priv, NULL);

	return g_strdup_printf ("PR%s", GNOME_DB_SERVER_FUNCTION (iface)->priv->objectid);
}

static xmlNodePtr
gnome_db_function_save_to_xml (GnomeDbXmlStorage *iface, GError **error)
{
	xmlNodePtr node = NULL, subnode;
	GnomeDbServerFunction *func;
	gchar *str;
	GSList *list;

	g_return_val_if_fail (iface && IS_GNOME_DB_SERVER_FUNCTION (iface), NULL);
	g_return_val_if_fail (GNOME_DB_SERVER_FUNCTION (iface)->priv, NULL);

	func = GNOME_DB_SERVER_FUNCTION (iface);

	node = xmlNewNode (NULL, "GNOME_DB_FUNCTION");

	str = gnome_db_function_get_xml_id (iface);
	xmlSetProp (node, "id", str);
	g_free (str);
	xmlSetProp (node, "name", gnome_db_base_get_name (GNOME_DB_BASE (func)));
	xmlSetProp (node, "descr", gnome_db_base_get_description (GNOME_DB_BASE (func)));
	xmlSetProp (node, "owner", gnome_db_base_get_owner (GNOME_DB_BASE (func)));

	/* return type */
	if (func->priv->result_type) {
		subnode = xmlNewChild (node, NULL, "GNOME_DB_FUNC_PARAM", NULL);
		
		str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (func->priv->result_type));
		xmlSetProp (subnode, "type", str);
		g_free (str);
		xmlSetProp (subnode, "way", "out");
	}

	/* argument types */
	list = func->priv->arg_types;
	while (list) {
		subnode = xmlNewChild (node, NULL, "GNOME_DB_FUNC_PARAM", NULL);
		if (list->data) {
			str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (list->data));
			xmlSetProp (subnode, "type", str);
		}
		xmlSetProp (subnode, "way", "in");
		list = g_slist_next (list);
	}

	return node;
}

static gboolean
gnome_db_function_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	GnomeDbServerFunction *func;
	gchar *prop;
	gboolean pname = FALSE, pid = FALSE;
	xmlNodePtr subnode;
	GSList *argtypes = NULL;

	g_return_val_if_fail (iface && IS_GNOME_DB_SERVER_FUNCTION (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_SERVER_FUNCTION (iface)->priv, FALSE);
	g_return_val_if_fail (node, FALSE);

	func = GNOME_DB_SERVER_FUNCTION (iface);
	if (strcmp (node->name, "GNOME_DB_FUNCTION")) {
		g_set_error (error,
			     GNOME_DB_SERVER_FUNCTION_ERROR,
			     GNOME_DB_SERVER_FUNCTION_XML_LOAD_ERROR,
			     _("XML Tag is not <GNOME_DB_FUNCTION>"));
		return FALSE;
	}

	/* function's attributes */
	prop = xmlGetProp (node, "id");
	if (prop) {
		if ((*prop == 'P') && (*(prop+1)=='R')) {
			pid = TRUE;
			if (func->priv->objectid)
				g_free (func->priv->objectid);
			func->priv->objectid = g_strdup (prop+2);
		}
		g_free (prop);
	}

	prop = xmlGetProp (node, "name");
	if (prop) {
		pname = TRUE;
		gnome_db_base_set_name (GNOME_DB_BASE (func), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "descr");
	if (prop) {
		gnome_db_base_set_description (GNOME_DB_BASE (func), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "owner");
	if (prop) {
		gnome_db_base_set_owner (GNOME_DB_BASE (func), prop);
		g_free (prop);
	}
	
	/* arguments and return type */
	subnode = node->children;
	while (subnode) {
		if (!strcmp (subnode->name, "GNOME_DB_FUNC_PARAM")) {
			GnomeDbServerDataType *dt = NULL;
			prop = xmlGetProp (subnode, "type");
			if (prop) {
				dt = gnome_db_server_get_data_type_by_xml_id (func->priv->srv, prop);
				g_free (prop);
			}

			/* if (!dt) { */
/* 				g_set_error (error, */
/* 					     GNOME_DB_SERVER_FUNCTION_ERROR, */
/* 					     GNOME_DB_SERVER_FUNCTION_XML_LOAD_ERROR, */
/* 					     _("Can't find data type for function '%s'"),  */
/* 					     gnome_db_base_get_name (GNOME_DB_BASE (func))); */
/* 				return FALSE; */
/* 			} */
			
			prop = xmlGetProp (subnode, "way");
			if (prop) {
				if (*prop == 'o') {
					if (func->priv->result_type) {
						g_set_error (error,
							     GNOME_DB_SERVER_FUNCTION_ERROR,
							     GNOME_DB_SERVER_FUNCTION_XML_LOAD_ERROR,
							     _("More than one return type for function '%s'"), 
							     gnome_db_base_get_name (GNOME_DB_BASE (func)));
						return FALSE;
					}
					gnome_db_server_function_set_ret_type (func, dt);
				}
				else 
					argtypes = g_slist_prepend (argtypes, dt);
				g_free (prop);
			}
		}
		subnode = subnode->next;
	}

	argtypes = g_slist_reverse (argtypes);
	gnome_db_server_function_set_arg_types (func, argtypes);
	g_slist_free (argtypes);

	if (pname && pid)
		return TRUE;
	else {
		g_set_error (error,
			     GNOME_DB_SERVER_FUNCTION_ERROR,
			     GNOME_DB_SERVER_FUNCTION_XML_LOAD_ERROR,
			     _("Missing required attributes for <GNOME_DB_FUNCTION>"));
		return FALSE;
	}
}




/**
 * gnome_db_server_function_set_dbms_id
 * @func: a #GnomeDbServerFunction object
 * @id: the DBMS identifier
 *
 * Set the DBMS identifier of the function
 */
void
gnome_db_server_function_set_dbms_id (GnomeDbServerFunction *func, const gchar *id)
{
	g_return_if_fail (func && IS_GNOME_DB_SERVER_FUNCTION (func));
	g_return_if_fail (func->priv);
	g_return_if_fail (id && *id);

	if (func->priv->objectid)
		g_free (func->priv->objectid);
	func->priv->objectid = g_strdup (id);
}


/**
 * gnome_db_server_function_get_dbms_id
 * @func: a #GnomeDbServerFunction object
 *
 * Get the DBMS identifier of the function
 *
 * Returns: the function's id
 */
const gchar *
gnome_db_server_function_get_dbms_id (GnomeDbServerFunction *func)
{
	g_return_val_if_fail (func && IS_GNOME_DB_SERVER_FUNCTION (func), NULL);
	g_return_val_if_fail (func->priv, NULL);

	return func->priv->objectid;
}


/**
 * gnome_db_server_function_set_sqlname
 * @func: a #GnomeDbServerFunction object
 * @sqlname: 
 *
 * Set the SQL name of the data type.
 */
void
gnome_db_server_function_set_sqlname (GnomeDbServerFunction *func, const gchar *sqlname)
{
	g_return_if_fail (func && IS_GNOME_DB_SERVER_FUNCTION (func));
	g_return_if_fail (func->priv);

	gnome_db_base_set_name (GNOME_DB_BASE (func), sqlname);
}


/**
 * gnome_db_server_function_get_sqlname
 * @func: a #GnomeDbServerFunction object
 *
 * Get the DBMS's name of a data type.
 *
 * Returns: the name of the data type
 */
const gchar *
gnome_db_server_function_get_sqlname (GnomeDbServerFunction *func)
{
	g_return_val_if_fail (func && IS_GNOME_DB_SERVER_FUNCTION (func), NULL);
	g_return_val_if_fail (func->priv, NULL);

	return gnome_db_base_get_name (GNOME_DB_BASE (func));
}

static void nullified_data_type_cb (GnomeDbServerDataType *dt, GnomeDbServerFunction *func);

/**
 * gnome_db_server_function_set_arg_types
 * @func: a #GnomeDbServerFunction object
 * @arg_types: a list of #GnomeDbServerDataType objects or #NULL values ordered to represent the data types
 * of the function's arguments .
 *
 * Set the arguments types of a function
 */
void 
gnome_db_server_function_set_arg_types (GnomeDbServerFunction *func, const GSList *arg_types)
{
	GSList *list;

	g_return_if_fail (func && IS_GNOME_DB_SERVER_FUNCTION (func));
	g_return_if_fail (func->priv);

	if (func->priv->arg_types) {
		list = func->priv->arg_types;
		while (list) {
			if (list->data) {
				g_signal_handlers_disconnect_by_func (G_OBJECT (list->data), 
								      G_CALLBACK (nullified_data_type_cb),
								      func);
				g_object_unref (G_OBJECT (list->data));
			}
			list = g_slist_next (list);
		}
		g_slist_free (func->priv->arg_types);
	}

	func->priv->arg_types = g_slist_copy (arg_types);
	list = func->priv->arg_types;
	while (list) {
		if (list->data) {
			gnome_db_base_connect_nullify (list->data,
						 G_CALLBACK (nullified_data_type_cb), func);
			g_object_ref (G_OBJECT (list->data));
		}
		list = g_slist_next (list);
	}
}

/**
 * gnome_db_server_function_get_arg_types
 * @func: a #GnomeDbServerFunction object
 * 
 * To consult the list of arguments types (and number) of a function.
 *
 * Returns: a list of #GnomeDbServerDataType objects, the list MUST NOT be modified.
 */
const GSList *
gnome_db_server_function_get_arg_types (GnomeDbServerFunction *func)
{
	g_return_val_if_fail (func && IS_GNOME_DB_SERVER_FUNCTION (func), NULL);
	g_return_val_if_fail (func->priv, NULL);

	return func->priv->arg_types;
}

/**
 * gnome_db_server_function_set_ret_type
 * @func: a #GnomeDbServerFunction object
 * @dt: a #GnomeDbServerDataType object or #NULL
 *
 * Set the return type of a function
 */
void 
gnome_db_server_function_set_ret_type  (GnomeDbServerFunction *func, GnomeDbServerDataType *dt)
{
	g_return_if_fail (func && IS_GNOME_DB_SERVER_FUNCTION (func));
	g_return_if_fail (func->priv);
	if (dt)
		g_return_if_fail (dt && IS_GNOME_DB_SERVER_DATA_TYPE (dt));
	
	if (func->priv->result_type) { 
		g_signal_handlers_disconnect_by_func (G_OBJECT (func->priv->result_type), 
						      G_CALLBACK (nullified_data_type_cb), func);
		g_object_unref (G_OBJECT (func->priv->result_type));
	}

	func->priv->result_type = dt;
	if (dt) {
		gnome_db_base_connect_nullify (dt,
					 G_CALLBACK (nullified_data_type_cb), func);
		g_object_ref (G_OBJECT (dt));
	}
}

static void
nullified_data_type_cb (GnomeDbServerDataType *dt, GnomeDbServerFunction *func)
{
	gnome_db_base_nullify (GNOME_DB_BASE (func));
}

/**
 * gnome_db_server_function_get_ret_type
 * @func: a #GnomeDbServerFunction object
 * 
 * To consult the return type of a function.
 *
 * Returns: a #GnomeDbServerDataType object.
 */
GnomeDbServerDataType *
gnome_db_server_function_get_ret_type  (GnomeDbServerFunction *func)
{
	g_return_val_if_fail (func && IS_GNOME_DB_SERVER_FUNCTION (func), NULL);
	g_return_val_if_fail (func->priv, NULL);

	return func->priv->result_type;
}


/**
 * gnome_db_server_function_accepts_args
 * @func: a #GnomeDbServerFunction object
 * @arg_types: a list of #GnomeDbServerDataType objects or #NULL values, ordered
 *
 * Test if the proposed list of arguments (@arg_types) would be accepted by
 * the @func function.
 *
 * The non acceptance can be beause of data type incompatibilities or a wrong number
 * of data types.
 *
 * Returns: TRUE if accepted
 */
gboolean
gnome_db_server_function_accepts_args (GnomeDbServerFunction *func, const GSList *arg_types)
{
	GSList *arg = arg_types, *list;
	gboolean args_ok = TRUE;
	GnomeDbServerInfo *sinfo;

	g_return_val_if_fail (func && IS_GNOME_DB_SERVER_FUNCTION (func), FALSE);
	g_return_val_if_fail (func->priv, FALSE);

	sinfo = gnome_db_server_get_server_info (func->priv->srv);
	list = gnome_db_server_function_get_arg_types (func);
	
	if (g_slist_length (arg) != g_slist_length (list))
		return FALSE;
	
	while (GNOME_DB_FUNCTIONS_TEST_PARAMS_DO_TEST && arg && list && args_ok) {
		if (!sinfo || !sinfo->implicit_data_types_casts) {
			/* Strict tests */
			if (arg->data && list->data &&
			    (arg->data != list->data))
				args_ok = FALSE;
		}
		else {
			/* GdaValueType compatibility test */
			if (arg->data && list->data &&
			    (gnome_db_server_data_type_get_gda_type (GNOME_DB_SERVER_DATA_TYPE (arg->data)) !=
			     gnome_db_server_data_type_get_gda_type (GNOME_DB_SERVER_DATA_TYPE (list->data))))
				args_ok = FALSE;
		}
		
		arg = g_slist_next (arg);
		list = g_slist_next (list);
	}

	return args_ok;
}
