/* gnome-db-data-model.c
 *
 * Copyright (C) 2004 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-data-model.h"
#include "marshal.h"
#include "gnome-db-base.h"
#include "gnome-db-entity.h"


/* signals */
enum
{
	DATA_REFRESHED,
	LAST_SIGNAL
};

static gint gnome_db_data_model_signals[LAST_SIGNAL] = { 0 };

static void gnome_db_data_model_iface_init (gpointer g_class);

GType
gnome_db_data_model_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbDataModelIface),
			(GBaseInitFunc) gnome_db_data_model_iface_init,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) NULL,
			NULL,
			NULL,
			0,
			0,
			(GInstanceInitFunc) NULL
		};
		
		type = g_type_register_static (G_TYPE_INTERFACE, "GnomeDbDataModel", &info, 0);
		g_type_interface_add_prerequisite (type, GNOME_DB_BASE_TYPE);
		g_type_interface_add_prerequisite (type, GNOME_DB_ENTITY_TYPE);
		g_type_interface_add_prerequisite (type, GDA_TYPE_DATA_MODEL);
	}
	return type;
}


static void
gnome_db_data_model_iface_init (gpointer g_class)
{
	static gboolean initialized = FALSE;

	if (! initialized) {
		gnome_db_data_model_signals[DATA_REFRESHED] =
			g_signal_new ("data_refreshed",
				      GNOME_DB_DATA_MODEL_TYPE,
				      G_SIGNAL_RUN_FIRST,
				      G_STRUCT_OFFSET (GnomeDbDataModelIface, data_refreshed),
				      NULL, NULL,
				      marshal_VOID__VOID, G_TYPE_NONE,
				      0);
		initialized = TRUE;
	}
}

/**
 * gnome_db_data_model_copy
 * @iface: an object implementing the #GnomeDbDataModel interface
 * @replacements: a place to store the sub-objects correspondance, or %NULL
 *
 * This is a generic copy constructor.
 *
 * Returns: a new #GnomeDbDataModel object, or %NULL if the model can't be copied
 */
GObject *
gnome_db_data_model_copy (GnomeDbDataModel *iface, GHashTable *replacements)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_DATA_MODEL (iface), NULL);
	
	if (GNOME_DB_DATA_MODEL_GET_IFACE (iface)->copy)
		return (GNOME_DB_DATA_MODEL_GET_IFACE (iface)->copy) (iface, replacements);
	else
		return NULL;
}

/**
 * gnome_db_data_model_get_status
 * @iface: an object implementing the #GnomeDbDataModel interface
 *
 * Get the status of the data within @iface, as an ORed value of the #GnomeDbDataModelAttributes enumerated
 *
 * Returns: the status
 */
guint
gnome_db_data_model_get_status (GnomeDbDataModel *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_DATA_MODEL (iface), 0);
	
	if (GNOME_DB_DATA_MODEL_GET_IFACE (iface)->get_status)
		return (GNOME_DB_DATA_MODEL_GET_IFACE (iface)->get_status) (iface);
	else
		return 0;
}


/**
 * gnome_db_data_model_refresh
 * @iface: an object implementing the #GnomeDbDataModel interface
 * @error: place to store the error, or %NULL
 *
 * Refresh the data in @iface. If the data is statically defined, then nothing is done; if the contents
 * is defined as the result of a SELECT query's execution, then the query is executed again.
 *
 * Returns: TRUE if no error occured
 */
gboolean
gnome_db_data_model_refresh (GnomeDbDataModel *iface, GError **error)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_DATA_MODEL (iface), FALSE);
	
	if (GNOME_DB_DATA_MODEL_GET_IFACE (iface)->refresh)
		return (GNOME_DB_DATA_MODEL_GET_IFACE (iface)->refresh) (iface, error);
	return TRUE;
}

/**
 * gnome_db_data_model_get_params
 * @iface: an object implementing the #GnomeDbDataModel interface
 *
 * Get the #GnomeDbDataSet object which contains all the parameters which control the contents
 * of the data in the @iface object (the data itself is accessible using the 
 * #GdaDataModel interface).
 *
 * Returns: the #GnomeDbDataSet object for parameters
 */
GnomeDbDataSet *
gnome_db_data_model_get_params  (GnomeDbDataModel *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_DATA_MODEL (iface), NULL);
	
	if (GNOME_DB_DATA_MODEL_GET_IFACE (iface)->get_params)
		return (GNOME_DB_DATA_MODEL_GET_IFACE (iface)->get_params) (iface);
	return NULL;
}

/**
 * gnome_db_data_model_get_new_data_set
 * @iface: an object implementing the #GnomeDbDataModel interface
 *
 * Creates a new #GnomeDbDataSet object which can be used to retreive (or store)
 * individual data rows of the data in the @iface object. Using a #GnomeDbDataSet
 * object instead of having a direct access to the data through the #GdaDataModel object
 * has the advantage that the #GnomeDbDataSet can be manipulated as a single 'row' object.
 *
 * Returns: a new #GnomeDbDataSet object
 */
GnomeDbDataSet *
gnome_db_data_model_get_new_data_set (GnomeDbDataModel *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_DATA_MODEL (iface), NULL);
	
	if (GNOME_DB_DATA_MODEL_GET_IFACE (iface)->get_new_data_set) {
		GnomeDbDataSet *retval;

		retval = (GNOME_DB_DATA_MODEL_GET_IFACE (iface)->get_new_data_set) (iface);
		{
			gint i, col;
			GnomeDbParameter *param;
			gboolean allparamsok = TRUE;
			gboolean allcolsok = TRUE;
			
#ifdef debug_NO
			DEBUG_HEADER;
			gnome_db_base_dump (GNOME_DB_BASE (iface), 0);
			g_print ("Showing parameters for each column:\n");
#endif
			for (i = 0; i < gda_data_model_get_n_columns (GDA_DATA_MODEL (iface)); i++) {
				param = gnome_db_data_model_get_param_at_column (GNOME_DB_DATA_MODEL (iface), retval, i);
				if (!param)
					allparamsok = FALSE;
				col = gnome_db_data_model_get_column_at_param (GNOME_DB_DATA_MODEL (iface), retval, param);
				if (col != i)
					allcolsok = FALSE;
#ifdef debug_NO
				g_print ("Model column %d: => param=%p => col=%d\n", i, param, col);
#endif
			}
#ifdef debug_NO
			g_print ("\n");
#endif

			if (!allparamsok || !allcolsok)
				g_warning ("%s():\nThere is an incoherence in the generated GnomeDbDataSet\n"
					   "(Each column in the GnomeDbDataModel should correspond to a GnomeDbParameter\n"
					   "and that parameter should correspond to the same column number)\n", __FUNCTION__);
		}
		return retval;
	}
	return NULL;
}

/**
 * gnome_db_data_model_syn_data_set
 * @iface: an object implementing the #GnomeDbDataModel interface
 * @data_set: a #GnomeDbDataSet object obtained using gnome_db_data_model_get_data_set()
 * @row: the row in the data model to be synced.
 *
 * Changes the values stored in @data_set to reflect the actual data in @iface, at row @row.
 *
 * WARNING: the actual values of the data model are not modified by the changes made to the individual
 * values in @data_set. To modify the data in @iface, the changes MUST be made on the data model
 * itself.
 *
 * Returns: TRUE if no error occured
 */
gboolean
gnome_db_data_model_sync_data_set (GnomeDbDataModel *iface, GnomeDbDataSet *data_set, gint row)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_DATA_MODEL (iface), FALSE);
	
	if (GNOME_DB_DATA_MODEL_GET_IFACE (iface)->sync_data_set)
		return (GNOME_DB_DATA_MODEL_GET_IFACE (iface)->sync_data_set) (iface, data_set, row);
	return FALSE;
}

/**
 * gnome_db_data_model_get_param_at_column
 * @iface: an object implementing the #GnomeDbDataModel interface
 * @data_set: a #GnomeDbDataSet object obtained using gnome_db_data_model_get_data_set()
 * @param: a #GnomeDbParameter object present in @data_set->parameters
 *
 * Fetch the column number in @iface which corresponds to @param in the @data_set data set.
 * 
 * Returns: the column number, or -1 if an error occured
 */
GnomeDbParameter *
gnome_db_data_model_get_param_at_column (GnomeDbDataModel *iface, 
					 GnomeDbDataSet *data_set, gint column)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_DATA_MODEL (iface), NULL);

	if (GNOME_DB_DATA_MODEL_GET_IFACE (iface)->get_param_at_col)
		return (GNOME_DB_DATA_MODEL_GET_IFACE (iface)->get_param_at_col) (iface, data_set, column);
	return NULL;
}

/**
 * gnome_db_data_model_get_column_at_param
 * @iface: an object implementing the #GnomeDbDataModel interface
 * @data_set: a #GnomeDbDataSet object obtained using gnome_db_data_model_get_data_set()
 * @column: the requested column
 *
 * Fetch the #GbomeDbParameter in @data_set which corresponds to the column @column in the @data_set data set.
 * 
 * Returns: the parameter, or %NULL if an error occured
 */
gint
gnome_db_data_model_get_column_at_param (GnomeDbDataModel *iface, 
					 GnomeDbDataSet *data_set, GnomeDbParameter *param)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_DATA_MODEL (iface), -1);

	if (GNOME_DB_DATA_MODEL_GET_IFACE (iface)->get_col_at_param)
		return (GNOME_DB_DATA_MODEL_GET_IFACE (iface)->get_col_at_param) (iface, data_set, param);
	return -1;
}

/**
 * gnome_db_data_model_get_key_columns
 * @iface: an object implementing the #GnomeDbDataModel interface
 * @data_set: a #GnomeDbDataSet object obtained using gnome_db_data_model_get_data_set()
 *
 * Get a list of parameters which, taken together, hold values which are a key (unique and not NULL) for
 * the data held in @iface.
 *
 * Returns: a new list of #GnomeDbParameter objects, or %NULL
 */
GSList *
gnome_db_data_model_get_key_columns (GnomeDbDataModel *iface, GnomeDbDataSet *data_set)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_DATA_MODEL (iface), NULL);

	if (GNOME_DB_DATA_MODEL_GET_IFACE (iface)->get_key_columns)
		return (GNOME_DB_DATA_MODEL_GET_IFACE (iface)->get_key_columns) (iface, data_set);
	return NULL;
}


/**
 * gnome_db_data_model_get_row_from_values
 * @values: a list of #GdaValue values
 * @cols_index: an array of #gint containing the column number to match each value of @values
 *
 * Returns the first row where all the values in @values at the columns identified at
 * @cols_index match. If the row can't be identified, then returns -1;
 *
 * NOTE: the @cols_index array MUST contain a column index for each value in @values
 *
 * Returns: the requested row number, of -1 if not found
 */
gint
gnome_db_data_model_get_row_from_values (GnomeDbDataModel *model, GSList *values, gint *cols_index)
{
	gint row = -1;
	gint current_row, n_rows, n_cols;
	
	n_rows = gda_data_model_get_n_rows (GDA_DATA_MODEL (model));
	n_cols = gda_data_model_get_n_columns (GDA_DATA_MODEL (model));
	current_row = 0;

	while ((current_row < n_rows) && (row == -1)) {
		GSList *list;
		gboolean allequal = TRUE;
		const GdaValue *value;
		gint index;

		list = values;
		index = 0;
		while (list && allequal) {
			if (cols_index)
				g_return_val_if_fail (cols_index [index] < n_cols, FALSE);
			value = gda_data_model_get_value_at (GDA_DATA_MODEL (model), cols_index [index], current_row);

			if (gda_value_compare_ext ((GdaValue *) (list->data), value))
				allequal = FALSE;
			
			list = g_slist_next (list);
			index++;
		}
	
		if (allequal) 
			row = current_row;

		current_row++;
	}

	return row;
}
