/* gnome-db-condition.c
 *
 * Copyright (C) 2003 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "gnome-db-condition.h"
#include "gnome-db-query.h"
#include "gnome-db-join.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-renderer.h"
#include "gnome-db-referer.h"
#include "gnome-db-ref-base.h"
#include "gnome-db-qfield.h"
#include "gnome-db-qf-value.h"
#include "gnome-db-qf-field.h"

#include "gnome-db-query-parsing.h"

/* 
 * Main static functions 
 */
static void gnome_db_condition_class_init (GnomeDbConditionClass *class);
static void gnome_db_condition_init (GnomeDbCondition *condition);
static void gnome_db_condition_dispose (GObject *object);
static void gnome_db_condition_finalize (GObject *object);

static void gnome_db_condition_set_property (GObject              *object,
				       guint                 param_id,
				       const GValue         *value,
				       GParamSpec           *pspec);
static void gnome_db_condition_get_property (GObject              *object,
				       guint                 param_id,
				       GValue               *value,
				       GParamSpec           *pspec);

static gboolean condition_type_is_node (GnomeDbConditionType type);

static void nullified_object_cb (GObject *obj, GnomeDbCondition *cond);
static void nullified_parent_cb (GnomeDbCondition *parent, GnomeDbCondition *cond);
static void nullified_child_cb (GnomeDbCondition *child, GnomeDbCondition *cond);

static void child_cond_changed_cb (GnomeDbCondition *child, GnomeDbCondition *cond);
static void ops_ref_lost_cb (GnomeDbRefBase *refbase, GnomeDbCondition *cond);


/* XML storage interface */
static void        gnome_db_condition_xml_storage_init (GnomeDbXmlStorageIface *iface);
static gchar      *gnome_db_condition_get_xml_id (GnomeDbXmlStorage *iface);
static xmlNodePtr  gnome_db_condition_save_to_xml (GnomeDbXmlStorage *iface, GError **error);
static gboolean    gnome_db_condition_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);

/* Renderer interface */
static void            gnome_db_condition_renderer_init      (GnomeDbRendererIface *iface);
static GdaXqlItem     *gnome_db_condition_render_as_xql   (GnomeDbRenderer *iface, GnomeDbDataSet *context, GError **error);
static gchar          *gnome_db_condition_render_as_sql   (GnomeDbRenderer *iface, GnomeDbDataSet *context, guint options, GError **error);
static gchar          *gnome_db_condition_render_as_str   (GnomeDbRenderer *iface, GnomeDbDataSet *context);

/* Referer interface */
static void        gnome_db_condition_referer_init        (GnomeDbRefererIface *iface);
static gboolean    gnome_db_condition_activate            (GnomeDbReferer *iface);
static void        gnome_db_condition_deactivate          (GnomeDbReferer *iface);
static gboolean    gnome_db_condition_is_active           (GnomeDbReferer *iface);
static GSList     *gnome_db_condition_get_ref_objects     (GnomeDbReferer *iface);
static void        gnome_db_condition_replace_refs        (GnomeDbReferer *iface, GHashTable *replacements);

#ifdef debug
static void        gnome_db_condition_dump                (GnomeDbCondition *condition, guint offset);
#endif

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	LAST_SIGNAL
};

static gint gnome_db_condition_signals[LAST_SIGNAL] = { };

/* properties */
enum
{
	PROP_0,
	PROP_QUERY,
	PROP_JOIN
};


struct _GnomeDbConditionPrivate
{
	GnomeDbQuery         *query;
	GnomeDbJoin          *join; /* not NULL if cond is a join cond => more specific tests */
	GnomeDbConditionType  type;
	GnomeDbCondition     *cond_parent;
	GSList          *cond_children;
	GnomeDbRefBase       *ops[3]; /* references on GnomeDbQfield objects */
	gboolean         nullify_if_ref_lost; /* if TRUE, then commits suicide when one of the ops[] loses its ref to an object */

	gint            internal_transaction; /* > 0 to revent emission of the "changed" signal, when
					       * several changes must occur before the condition has a stable 
					       * structure again */
};

/* module error */
GQuark gnome_db_condition_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_condition_error");
	return quark;
}


guint
gnome_db_condition_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbConditionClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_condition_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbCondition),
			0,
			(GInstanceInitFunc) gnome_db_condition_init
		};
		
		static const GInterfaceInfo xml_storage_info = {
                        (GInterfaceInitFunc) gnome_db_condition_xml_storage_init,
                        NULL,
                        NULL
                };

                static const GInterfaceInfo renderer_info = {
                        (GInterfaceInitFunc) gnome_db_condition_renderer_init,
                        NULL,
                        NULL
                };

                static const GInterfaceInfo referer_info = {
                        (GInterfaceInitFunc) gnome_db_condition_referer_init,
                        NULL,
                        NULL
                };

		type = g_type_register_static (GNOME_DB_BASE_TYPE, "GnomeDbCondition", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_XML_STORAGE_TYPE, &xml_storage_info);
                g_type_add_interface_static (type, GNOME_DB_RENDERER_TYPE, &renderer_info);
                g_type_add_interface_static (type, GNOME_DB_REFERER_TYPE, &referer_info);
	}
	return type;
}

static void
gnome_db_condition_xml_storage_init (GnomeDbXmlStorageIface *iface)
{
        iface->get_xml_id = gnome_db_condition_get_xml_id;
        iface->save_to_xml = gnome_db_condition_save_to_xml;
        iface->load_from_xml = gnome_db_condition_load_from_xml;
}

static void
gnome_db_condition_renderer_init (GnomeDbRendererIface *iface)
{
        iface->render_as_xql = gnome_db_condition_render_as_xql;
        iface->render_as_sql = gnome_db_condition_render_as_sql;
        iface->render_as_str = gnome_db_condition_render_as_str;
	iface->is_valid = NULL;
}

static void
gnome_db_condition_referer_init (GnomeDbRefererIface *iface)
{
        iface->activate = gnome_db_condition_activate;
        iface->deactivate = gnome_db_condition_deactivate;
        iface->is_active = gnome_db_condition_is_active;
        iface->get_ref_objects = gnome_db_condition_get_ref_objects;
        iface->replace_refs = gnome_db_condition_replace_refs;
}

static void
gnome_db_condition_class_init (GnomeDbConditionClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = gnome_db_condition_dispose;
	object_class->finalize = gnome_db_condition_finalize;

	/* Properties */
	object_class->set_property = gnome_db_condition_set_property;
	object_class->get_property = gnome_db_condition_get_property;
	g_object_class_install_property (object_class, PROP_QUERY,
					 g_param_spec_pointer ("query", NULL, NULL, 
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property (object_class, PROP_JOIN,
					 g_param_spec_pointer ("join", NULL, NULL, 
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));

	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_condition_dump;
#endif
}

static void
gnome_db_condition_init (GnomeDbCondition *condition)
{
	gint i;
	condition->priv = g_new0 (GnomeDbConditionPrivate, 1);
	condition->priv->query = NULL;
	condition->priv->join = NULL;
	condition->priv->type = GNOME_DB_CONDITION_TYPE_UNKNOWN;
	condition->priv->cond_parent = NULL;
	condition->priv->cond_children = NULL;
	for (i=0; i<3; i++)
		condition->priv->ops[i] = NULL;
	condition->priv->nullify_if_ref_lost = TRUE; /* TODO: create a property for this ? */
	condition->priv->internal_transaction = FALSE;
}

/**
 * gnome_db_condition_new
 * @query: a #GnomeDbQuery object
 * @type: the condition type
 *
 * Creates a new #GnomeDbCondition object
 *
 * Returns: the newly created object
 */
GObject   *
gnome_db_condition_new (GnomeDbQuery *query, GnomeDbConditionType type)
{
	GObject *obj = NULL;
	GnomeDbDict *dict;
	gint i;
	GnomeDbCondition *cond;
	guint id;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));

	obj = g_object_new (GNOME_DB_CONDITION_TYPE, "dict", dict, NULL);
	g_object_get (G_OBJECT (query), "cond_serial", &id, NULL);
	gnome_db_base_set_id (GNOME_DB_BASE (obj), id);

	cond = GNOME_DB_CONDITION (obj);
	gnome_db_query_declare_condition (query, cond);
	for (i=0; i<3; i++) {
		cond->priv->ops[i] = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
		g_signal_connect (G_OBJECT (cond->priv->ops[i]), "ref_lost",
				  G_CALLBACK (ops_ref_lost_cb), cond);
	}

	cond->priv->type = type;

	g_object_set (G_OBJECT (cond), "query", query, NULL);

	return obj;
}

static void
ops_ref_lost_cb (GnomeDbRefBase *refbase, GnomeDbCondition *cond)
{
	if (cond->priv->nullify_if_ref_lost &&
	    ! cond->priv->internal_transaction) {
		cond->priv->nullify_if_ref_lost = FALSE;
		/* commit suicide */
		gnome_db_base_nullify (GNOME_DB_BASE (cond));
	}
}

/**
 * gnome_db_condition_new_copy
 * @orig: a #GnomeDbCondition to copy
 * @replacements: a hash table to store replacements, or %NULL
 *
 * This is a copy constructor
 *
 * Returns: the new object
 */
GObject *
gnome_db_condition_new_copy (GnomeDbCondition *orig, GHashTable *replacements)
{
	GObject *obj;
	GnomeDbQuery *query;
	GnomeDbCondition *newcond;
	GnomeDbDict *dict;
	GSList *list;
	gint i;
	guint id;

	g_return_val_if_fail (orig && IS_GNOME_DB_CONDITION (orig), NULL);
	g_return_val_if_fail (orig->priv, NULL);
	g_object_get (G_OBJECT (orig), "query", &query, NULL);
	g_return_val_if_fail (query, NULL);
	dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));

	obj = g_object_new (GNOME_DB_CONDITION_TYPE, "dict", dict, NULL);
	g_object_get (G_OBJECT (query), "cond_serial", &id, NULL);
	gnome_db_base_set_id (GNOME_DB_BASE (obj), id);

	newcond = GNOME_DB_CONDITION (obj);
	if (replacements)
		g_hash_table_insert (replacements, orig, newcond);

	for (i=0; i<3; i++) {
		newcond->priv->ops[i] = GNOME_DB_REF_BASE (gnome_db_ref_base_new_copy (orig->priv->ops[i]));
		g_signal_connect (G_OBJECT (newcond->priv->ops[i]), "ref_lost",
				  G_CALLBACK (ops_ref_lost_cb), newcond);
	}
	newcond->priv->type = orig->priv->type;

	g_object_set (G_OBJECT (newcond), "query", query, NULL);
	gnome_db_query_declare_condition (query, newcond);

	list = orig->priv->cond_children;
	while (list) {
		GObject *ccond;

		ccond = gnome_db_condition_new_copy (GNOME_DB_CONDITION (list->data), replacements);
		gnome_db_condition_node_add_child (newcond, GNOME_DB_CONDITION (ccond), NULL);
		g_object_unref (ccond);
		list = g_slist_next (list);
	}

	return obj;
}

/**
 * gnome_db_condition_new_from_sql
 * @query: a #GnomeDbQuery object
 * @sql_cond: a SQL statement representing a valid condition
 * @targets: location to store a list of targets used by the new condition (and its children), or %NULL
 * @error: location to store error, or %NULL
 *
 * Creates a new #GnomeDbCondition object, which references other objects of @query, 
 * from the @sql_cond statement.
 *
 * Returns: a new #GnomeDbCondition, or %NULL if there was an error in @sql_cond
 */
GObject *
gnome_db_condition_new_from_sql (GnomeDbQuery *query, const gchar *sql_cond, GSList **targets, GError **error)
{
	gchar *sql;
	sql_statement *result;
	GnomeDbCondition *newcond = NULL;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (sql_cond && *sql_cond, FALSE);
	
	sql = g_strdup_printf ("SELECT * FROM table WHERE %s", sql_cond);
	result = sql_parse_with_error (sql, error);
	if (result) {
		if (((sql_select_statement *)(result->statement))->where) {
			ParseData *pdata;
			
			pdata = parse_data_new (query);
			parse_data_compute_targets_hash (query, pdata);
			newcond = parsed_create_complex_condition (query, pdata, 
								   ((sql_select_statement *)(result->statement))->where,
								   TRUE, targets, error);
			parse_data_destroy (pdata);
		}
		else 
			g_set_error (error, GNOME_DB_JOIN_ERROR, GNOME_DB_JOIN_PARSE_ERROR,
				     _("No join condition found in '%s'"), sql_cond);
		sql_destroy (result);
	}
	else 
		if (error && !(*error))
			g_set_error (error, GNOME_DB_JOIN_ERROR, GNOME_DB_JOIN_PARSE_ERROR,
				     _("Error parsing '%s'"), sql_cond);
	g_free (sql);

	return G_OBJECT (newcond);
}

static void
gnome_db_condition_dispose (GObject *object)
{
	GnomeDbCondition *condition;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_CONDITION (object));

	condition = GNOME_DB_CONDITION (object);
	if (condition->priv) {
		gint i;

		condition->priv->internal_transaction ++;
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));

		if (condition->priv->cond_parent) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (condition->priv->cond_parent),
							      G_CALLBACK (nullified_parent_cb), condition);
			condition->priv->cond_parent = NULL;
		}

		if (condition->priv->query) {
			gnome_db_query_undeclare_condition (condition->priv->query, condition);
                        g_signal_handlers_disconnect_by_func (G_OBJECT (condition->priv->query),
                                                              G_CALLBACK (nullified_object_cb), condition);
                        condition->priv->query = NULL;
                }

		if (condition->priv->join) {
                        g_signal_handlers_disconnect_by_func (G_OBJECT (condition->priv->join),
                                                              G_CALLBACK (nullified_object_cb), condition);
                        condition->priv->join = NULL;
                }

		for (i=0; i<3; i++)
			if (condition->priv->ops[i]) {
				g_object_unref (condition->priv->ops[i]);
				condition->priv->ops[i] = NULL;
			}

		while (condition->priv->cond_children)
			nullified_child_cb (GNOME_DB_CONDITION (condition->priv->cond_children->data), condition);
		condition->priv->internal_transaction --;
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
nullified_object_cb (GObject *obj, GnomeDbCondition *cond)
{
	gnome_db_base_nullify (GNOME_DB_BASE (cond));
}

static void
nullified_parent_cb (GnomeDbCondition *parent, GnomeDbCondition *cond)
{
	g_assert (cond->priv->cond_parent == parent);
	g_signal_handlers_disconnect_by_func (G_OBJECT (parent),
					      G_CALLBACK (nullified_parent_cb) , cond);
	cond->priv->cond_parent = NULL;
	gnome_db_base_nullify (GNOME_DB_BASE (cond));
}

static void
nullified_child_cb (GnomeDbCondition *child, GnomeDbCondition *cond)
{
	g_assert (g_slist_find (cond->priv->cond_children, child));

	/* child signals disconnect */
	g_signal_handlers_disconnect_by_func (G_OBJECT (cond),
					      G_CALLBACK (nullified_parent_cb) , child);
	child->priv->cond_parent = NULL;

	/* parent signal disconnect */
	g_signal_handlers_disconnect_by_func (G_OBJECT (child),
					      G_CALLBACK (nullified_child_cb) , cond);
	g_signal_handlers_disconnect_by_func (G_OBJECT (child),
					      G_CALLBACK (child_cond_changed_cb) , cond);
	g_object_unref (G_OBJECT (child));
	cond->priv->cond_children = g_slist_remove (cond->priv->cond_children, child);
	if (! cond->priv->internal_transaction)
		gnome_db_base_changed (GNOME_DB_BASE (cond));
}


static void
gnome_db_condition_finalize (GObject   * object)
{
	GnomeDbCondition *condition;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_CONDITION (object));

	condition = GNOME_DB_CONDITION (object);
	if (condition->priv) {
		g_free (condition->priv);
		condition->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_condition_set_property (GObject              *object,
				 guint                 param_id,
				 const GValue         *value,
				 GParamSpec           *pspec)
{
	GnomeDbCondition *condition;
	gpointer ptr;

	condition = GNOME_DB_CONDITION (object);
	if (condition->priv) {
		switch (param_id) {
		case PROP_QUERY:
			ptr = g_value_get_pointer (value);
                        g_return_if_fail (ptr && IS_GNOME_DB_QUERY (ptr));

                        if (condition->priv->query) {
                                if (condition->priv->query == GNOME_DB_QUERY (ptr))
                                        return;
				
				gnome_db_query_undeclare_condition (condition->priv->query, condition);
                                g_signal_handlers_disconnect_by_func (G_OBJECT (condition->priv->query),
                                                                      G_CALLBACK (nullified_object_cb), 
								      condition);
                        }

                        condition->priv->query = GNOME_DB_QUERY (ptr);
			gnome_db_base_connect_nullify (ptr, G_CALLBACK (nullified_object_cb), condition);
			gnome_db_query_declare_condition (condition->priv->query, condition);
			break;
		case PROP_JOIN:
			ptr = g_value_get_pointer (value);
			if (ptr) {
				g_return_if_fail (IS_GNOME_DB_JOIN (ptr));
				g_return_if_fail (gnome_db_join_get_query (GNOME_DB_JOIN (ptr)) == condition->priv->query);
			}

                        if (condition->priv->join) {
                                if (condition->priv->join == GNOME_DB_JOIN (ptr))
                                        return;

                                g_signal_handlers_disconnect_by_func (G_OBJECT (condition->priv->join),
                                                                      G_CALLBACK (nullified_object_cb), 
								      condition);
				condition->priv->join = NULL;
                        }

			if (ptr) {
				condition->priv->join = GNOME_DB_JOIN (ptr);
				gnome_db_base_connect_nullify (ptr, G_CALLBACK (nullified_object_cb), condition);
			}
			break;
		}
	}
}

static void
gnome_db_condition_get_property (GObject              *object,
			   guint                 param_id,
			   GValue               *value,
			   GParamSpec           *pspec)
{
	GnomeDbCondition *condition;

	condition = GNOME_DB_CONDITION (object);
        if (condition->priv) {
                switch (param_id) {
                case PROP_QUERY:
			g_value_set_pointer (value, condition->priv->query);
                        break;
		case PROP_JOIN:
			g_value_set_pointer (value, condition->priv->join);
                        break;
                }
        }
}

static gboolean
condition_type_is_node (GnomeDbConditionType type)
{
	gboolean retval;

	switch (type) {
	case GNOME_DB_CONDITION_NODE_AND:
	case GNOME_DB_CONDITION_NODE_OR:
	case GNOME_DB_CONDITION_NODE_NOT:
		retval = TRUE;
		break;
	default:
		retval = FALSE;
		break;
	}
	return retval;
}

/**
 * gnome_db_condition_set_cond_type
 * @condition: a #GnomeDbCondition object
 * @type:
 *
 * Sets the kind of condition @condition represents. If @type implies a node condition and
 * @condition currently represents a leaf, or if @type implies a leaf condition and
 * @condition currently represents a node, then @condition is changed without any error.
 */
void
gnome_db_condition_set_cond_type (GnomeDbCondition *condition, GnomeDbConditionType type)
{
	g_return_if_fail (condition && IS_GNOME_DB_CONDITION (condition));
	g_return_if_fail (condition->priv);
	if (condition->priv->type == type)
		return;

	if (condition_type_is_node (condition->priv->type) != condition_type_is_node (type)) {
		/* FIXME: remove all the children and all the operators */
		TO_IMPLEMENT;
	}

	condition->priv->type = type;
	if (! condition->priv->internal_transaction)
		gnome_db_base_changed (GNOME_DB_BASE (condition));
}

/**
 * gnome_db_condition_get_cond_type
 * @condition: a #GnomeDbCondition object
 *
 * Get the type of @condition
 *
 * Returns: the type
 */
GnomeDbConditionType
gnome_db_condition_get_cond_type (GnomeDbCondition *condition)
{
	g_return_val_if_fail (condition && IS_GNOME_DB_CONDITION (condition), GNOME_DB_CONDITION_TYPE_UNKNOWN);
	g_return_val_if_fail (condition->priv, GNOME_DB_CONDITION_TYPE_UNKNOWN);

	return condition->priv->type;
}

/**
 * gnome_db_condition_get_children
 * @condition: a #GnomeDbCondition object
 *
 * Get a list of #GnomeDbCondition objects which are children of @condition
 *
 * Returns: a new list of #GnomeDbCondition objects
 */
GSList *
gnome_db_condition_get_children (GnomeDbCondition *condition)
{
	g_return_val_if_fail (condition && IS_GNOME_DB_CONDITION (condition), NULL);
	g_return_val_if_fail (condition->priv, NULL);

	if (condition->priv->cond_children)
		return g_slist_copy (condition->priv->cond_children);
	else
		return NULL;
}

/**
 * gnome_db_condition_get_parent
 * @condition: a #GnomeDbCondition object
 *
 * Get the #GnomeDbCondition object which is parent of @condition
 *
 * Returns: the parent object, or %NULL
 */
GnomeDbCondition *
gnome_db_condition_get_parent (GnomeDbCondition *condition)
{
	g_return_val_if_fail (condition && IS_GNOME_DB_CONDITION (condition), NULL);
	g_return_val_if_fail (condition->priv, NULL);

	return condition->priv->cond_parent;
}


/**
 * gnome_db_condition_get_child_by_xml_id
 * @condition: a #GnomeDbCondition object
 * @xml_id: the XML Id of the requested #GnomeDbCondition child
 *
 * Get a pointer to a #GnomeDbCondition child from its XML Id
 *
 * Returns: the #GnomeDbCondition object, or %NULL if not found
 */
GnomeDbCondition *
gnome_db_condition_get_child_by_xml_id (GnomeDbCondition *condition, const gchar *xml_id)
{
	TO_IMPLEMENT;
	return NULL;
}

/**
 * gnome_db_condition_is_ancestor
 * @condition: a #GnomeDbCondition object
 * @ancestor: a #GnomeDbCondition object
 *
 * Tests if @ancestor is an ancestor of @condition
 *
 * Returns: TRUE if @ancestor is an ancestor of @condition
 */
gboolean
gnome_db_condition_is_ancestor (GnomeDbCondition *condition, GnomeDbCondition *ancestor)
{
	g_return_val_if_fail (condition && IS_GNOME_DB_CONDITION (condition), FALSE);
	g_return_val_if_fail (condition->priv, FALSE);
	g_return_val_if_fail (ancestor && IS_GNOME_DB_CONDITION (ancestor), FALSE);
	g_return_val_if_fail (ancestor->priv, FALSE);
	
	if (condition->priv->cond_parent == ancestor)
		return TRUE;
	if (condition->priv->cond_parent)
		return gnome_db_condition_is_ancestor (condition->priv->cond_parent, ancestor);

	return FALSE;
}

/**
 * gnome_db_condition_is_leaf
 * @condition: a #GnomeDbCondition object
 *
 * Tells if @condition is a leaf condition (not AND, OR, NOT, etc)
 *
 * Returns: TRUE if @condition is a leaf condition
 */
gboolean
gnome_db_condition_is_leaf (GnomeDbCondition *condition)
{
	g_return_val_if_fail (condition && IS_GNOME_DB_CONDITION (condition), FALSE);
	g_return_val_if_fail (condition->priv, FALSE);

	switch (condition->priv->type) {
	case GNOME_DB_CONDITION_NODE_AND:
	case GNOME_DB_CONDITION_NODE_OR:
	case GNOME_DB_CONDITION_NODE_NOT:
		return FALSE;
	default:
		return TRUE;
	}
}

static gboolean gnome_db_condition_node_add_child_pos (GnomeDbCondition *condition, GnomeDbCondition *child, gint pos, GError **error);


/**
 * gnome_db_condition_node_add_child
 * @condition: a #GnomeDbCondition object
 * @child: a #GnomeDbCondition object
 * @error: location to store error, or %NULL
 *
 * Adds a child to @condition; this is possible only if @condition is a node type (AND, OR, etc)
 *
 * Returns: TRUE if no error occurred
 */
gboolean
gnome_db_condition_node_add_child (GnomeDbCondition *condition, GnomeDbCondition *child, GError **error)
{
	return gnome_db_condition_node_add_child_pos (condition, child, -1, error);
}

static gboolean
gnome_db_condition_node_add_child_pos (GnomeDbCondition *condition, GnomeDbCondition *child, gint pos, GError **error)
{
	g_return_val_if_fail (condition && IS_GNOME_DB_CONDITION (condition), FALSE);
	g_return_val_if_fail (condition->priv, FALSE);
	g_return_val_if_fail (child && IS_GNOME_DB_CONDITION (child), FALSE);
	g_return_val_if_fail (child->priv, FALSE);
	g_return_val_if_fail (!gnome_db_condition_is_leaf (condition), FALSE);

	if (child->priv->cond_parent == condition)
		return TRUE;

	g_object_ref (G_OBJECT (child));

	if (child->priv->cond_parent) 
		gnome_db_condition_node_del_child (child->priv->cond_parent, child);

	if (gnome_db_condition_is_ancestor (condition, child)) {
		g_set_error (error,
                             GNOME_DB_CONDITION_ERROR,
                             GNOME_DB_CONDITION_PARENT_ERROR,
			     _("Conditions hierarchy error"));
		return FALSE;
	}

	/* a NOT node can only have one child */
	if (condition->priv->cond_children && (condition->priv->type == GNOME_DB_CONDITION_NODE_NOT)) {
		g_set_error (error,
                             GNOME_DB_CONDITION_ERROR,
                             GNOME_DB_CONDITION_PARENT_ERROR,
			     _("A NOT node can only have one child"));
		return FALSE;
	}

	/* child part */
	child->priv->cond_parent = condition;
	gnome_db_base_connect_nullify (condition, G_CALLBACK (nullified_parent_cb), child);

	/* parent part */
	condition->priv->cond_children = g_slist_insert (condition->priv->cond_children, child, pos);
	gnome_db_base_connect_nullify (child, G_CALLBACK (nullified_child_cb), condition);
	g_signal_connect (G_OBJECT (child), "changed",
			  G_CALLBACK (child_cond_changed_cb), condition);

	if (! condition->priv->internal_transaction)
		gnome_db_base_changed (GNOME_DB_BASE (condition));

	return TRUE;
}

/* Forwards the "changed" signal from children to parent condition */
static void
child_cond_changed_cb (GnomeDbCondition *child, GnomeDbCondition *cond)
{
	if (! cond->priv->internal_transaction)
		gnome_db_base_changed (GNOME_DB_BASE (cond));
}

/**
 * gnome_db_condition_node_del_child
 * @condition: a #GnomeDbCondition object
 * @child: a #GnomeDbCondition object
 *
 * Removes a child from @condition; this is possible only if @condition is a node type (AND, OR, etc)
 */
void
gnome_db_condition_node_del_child (GnomeDbCondition *condition, GnomeDbCondition *child)
{
	g_return_if_fail (condition && IS_GNOME_DB_CONDITION (condition));
	g_return_if_fail (condition->priv);
	g_return_if_fail (child && IS_GNOME_DB_CONDITION (child));
	g_return_if_fail (child->priv);
	g_return_if_fail (child->priv->cond_parent == condition);
	g_return_if_fail (!gnome_db_condition_is_leaf (condition));

	nullified_child_cb (child, condition);
}

/**
 * gnome_db_condition_leaf_set_left_op
 * @condition: a #GnomeDbCondition object
 * @op: which oparetor is concerned
 * @field: a # GnomeDbQfield object
 *
 * Sets one of @condition's operators
 */
void
gnome_db_condition_leaf_set_operator (GnomeDbCondition *condition, GnomeDbConditionOperator op, GnomeDbQfield *field)
{
	GnomeDbQuery *query1, *query2;

	g_return_if_fail (condition && IS_GNOME_DB_CONDITION (condition));
	g_return_if_fail (condition->priv);
	g_return_if_fail (field && IS_GNOME_DB_QFIELD (field));
	g_return_if_fail (gnome_db_condition_is_leaf (condition));

	g_object_get (G_OBJECT (condition), "query", &query1, NULL);
	g_object_get (G_OBJECT (field), "query", &query2, NULL);
	g_return_if_fail (query1);
	g_return_if_fail (query1 == query2);

	gnome_db_ref_base_set_ref_object_type (condition->priv->ops[op], GNOME_DB_BASE (field), GNOME_DB_QFIELD_TYPE);
}

/**
 * gnome_db_condition_leaf_get_operator
 * @condition: a #GnomeDbCondition object
 * @op: which oparetor is concerned
 *
 * Get one of @condition's operators.
 *
 * Returns: the requested #GnomeDbQfield object
 */
GnomeDbQfield *
gnome_db_condition_leaf_get_operator  (GnomeDbCondition *condition, GnomeDbConditionOperator op)
{
	GnomeDbBase *base;
	g_return_val_if_fail (condition && IS_GNOME_DB_CONDITION (condition), NULL);
	g_return_val_if_fail (condition->priv, NULL);
	g_return_val_if_fail (gnome_db_condition_is_leaf (condition), NULL);

	gnome_db_ref_base_activate (condition->priv->ops[op]);
	base = gnome_db_ref_base_get_ref_object (condition->priv->ops[op]);
	if (base)
		return GNOME_DB_QFIELD (base);
	else
		return NULL;
}

/*
 * Analyses @field and the #GnomeDbTargets object used
 * if @target is not NULL, and if there is only one target used, then set it to that target
 *
 * Returns: 0 if no target is used
 *          1 if one target is used
 *          2 if more than one target is used
 */
static gint
qfield_uses_nb_target (GnomeDbQfield *field, GnomeDbTarget **target)
{
	gint retval = 0;
	GnomeDbTarget *t1 = NULL;

	if (!field) 
		retval = 0;
	else {
		if (IS_GNOME_DB_QF_FIELD (field)) {
			t1 = gnome_db_qf_field_get_target (GNOME_DB_QF_FIELD (field));
			retval = 1;
		}
		else {
			GSList *list = gnome_db_referer_get_ref_objects (GNOME_DB_REFERER (field));
			GSList *tmp;
			
			tmp = list;
			while (tmp) {
				if (IS_GNOME_DB_QF_FIELD (tmp->data)) {
					if (!t1) {
						t1 = gnome_db_qf_field_get_target (GNOME_DB_QF_FIELD (tmp->data));
						retval = 1;
					}
					else
						if (t1 != gnome_db_qf_field_get_target (GNOME_DB_QF_FIELD (tmp->data)))
							retval = 2;
				}
				tmp = g_slist_next (tmp);
			}
			g_slist_free (list);
		}
	}

	if (retval && target)
		*target = t1;
	
	return retval;
}

static gboolean
gnome_db_condition_represents_join_real (GnomeDbCondition *condition,
				   GnomeDbTarget **target1, GnomeDbTarget **target2, 
				   gboolean *is_equi_join, 
				   gboolean force_2_targets, gboolean force_fields)
{
	GnomeDbTarget *t1 = NULL, *t2 = NULL;
	gboolean retval = TRUE;
	gboolean is_equi;
	gint nb;

	if (gnome_db_condition_is_leaf (condition)) {
		GnomeDbQfield *field;

		field = gnome_db_condition_leaf_get_operator (condition, GNOME_DB_CONDITION_OP_LEFT);
		nb = qfield_uses_nb_target (field, &t1);
		if (nb == 2)
			retval = FALSE;
		if (force_fields && (nb == 0))
			retval = FALSE;

		if (retval) {
			field = gnome_db_condition_leaf_get_operator (condition, GNOME_DB_CONDITION_OP_RIGHT);
			nb = qfield_uses_nb_target (field, &t2);
			if (nb == 2)
				retval = FALSE;
			if (force_fields && (nb == 0))
				retval = FALSE;
		}
		
		if (retval && (condition->priv->type == GNOME_DB_CONDITION_LEAF_BETWEEN)) {
			if (force_fields)
				retval = FALSE;
			else {
				GnomeDbTarget *t2bis = NULL;
				field = gnome_db_condition_leaf_get_operator (condition, GNOME_DB_CONDITION_OP_RIGHT2);
				if (qfield_uses_nb_target (field, &t2bis) == 2)
					retval = FALSE;
				if (t2bis && (t2bis != t2))
					retval = FALSE;
			}
		}

		is_equi = (condition->priv->type == GNOME_DB_CONDITION_LEAF_EQUAL);
		if (force_fields && !is_equi)
			retval = FALSE;
	}
	else {
		gpointer tref[2] = {NULL, NULL};
		GSList *list = condition->priv->cond_children;
		is_equi = TRUE;

		while (list && retval) {
			GnomeDbTarget *tmp1 = NULL, *tmp2 = NULL;
			gboolean eqj = FALSE;

			retval = gnome_db_condition_represents_join_real (GNOME_DB_CONDITION (list->data), 
								    &tmp1, &tmp2, &eqj, FALSE, force_fields);
			if (retval) {
				if (tmp1) {
					gboolean pushed = FALSE;
					gint i = 0;
					while ((i<2) && !pushed) {
						if (! tref[i]) {
							tref[i] = tmp1;
							pushed = TRUE;
						}
						else 
							if (tref[i] == tmp1) pushed = TRUE;
						i++;
					}
					if (!pushed)
						retval = FALSE;
				}
				if (tmp2) {
					gboolean pushed = FALSE;
					gint i = 0;
					while ((i<2) && !pushed) {
						if (! tref[i]) {
							tref[i] = tmp2;
							pushed = TRUE;
						}
						else 
							if (tref[i] == tmp2) pushed = TRUE;
						i++;
					}
					if (!pushed)
						retval = FALSE;
				}

				is_equi = is_equi && eqj;
			}
			list = g_slist_next (list);
		}
		if (retval) {
			t1 = tref[0];
			t2 = tref[1];
		}

		is_equi = is_equi && (condition->priv->type == GNOME_DB_CONDITION_NODE_AND);
	}

	if (retval) {
		if (force_2_targets && (!t1 || !t2))
			retval = FALSE;
		else {
			if (force_fields && (t1 == t2))
				retval = FALSE;
			else {
				if (target1) *target1 = t1;
				if (target1) *target2 = t2;
				if (is_equi_join) *is_equi_join = is_equi;
			}
		}
	}
	
	return retval;
}


/**
 * gnome_db_condition_represents_join
 * @condition: a #GnomeDbCondition object
 * @target1: place to store one of the targets, or %NULL
 * @target2: place to store the other target, or %NULL
 * @is_equi_join: place to store if the join is an equi join
 *
 * Tells if @condition represents a join condition: it is a condition (within a #GnomeDbQuery object)
 * for which the only #GnomeDbQfField fields taking part in the condition are from two distincts
 * #GnomeDbTarget objects. Such conditions can be assigned to a #GnomeDbJoin object using the 
 * gnome_db_join_set_condition() or gnome_db_join_set_condition_from_fkcons() methods.
 *
 * Additionnaly, if @condition is a join condition, and if @target1 and @target2 are not %NULL
 * then they are set to point to the two #GnomeDbTarget objects taking part in the condition. In this
 * case @target1 and @target2 wil hold non %NULL values.
 *
 * In a similar way, if @is_equi_join is not %NULL, then it will be set to TRUE if the join
 * condition is an equi join (that is the only comparison operator is the equal sign and there are
 * only AND operators in the condition).
 *
 * If @condition is not a join condition, then @target1, @target2 and @is_equi_join are left
 * untouched.
 * 
 * Returns: TRUE if @condition is a join condition
 */
gboolean
gnome_db_condition_represents_join (GnomeDbCondition *condition,
			      GnomeDbTarget **target1, GnomeDbTarget **target2, gboolean *is_equi_join)
{
	g_return_val_if_fail (condition && IS_GNOME_DB_CONDITION (condition), FALSE);
	g_return_val_if_fail (condition->priv, FALSE);
	
	return gnome_db_condition_represents_join_real (condition, target1, target2, is_equi_join, TRUE, FALSE);
}

/**
 * gnome_db_condition_represents_join_strict
 * @condition: a #GnomeDbCondition object
 * @target1: place to store one of the targets, or %NULL
 * @target2: place to store the other target, or %NULL
 *
 * Tells if @condition represents a strict join condition: it is a join condition as defined for the 
 * gnome_db_condition_represents_join() method, but where the condition is either "target1.field1=target2.field2"
 * or a list of such conditions conjuncted by the AND operator.
 *
 * If @condition is not a join condition, then @target1 and @target2 are left
 * untouched.
 * 
 * Returns: TRUE if @condition is a strict join condition
 */
gboolean
gnome_db_condition_represents_join_strict (GnomeDbCondition *condition,
				     GnomeDbTarget **target1, GnomeDbTarget **target2)
{
	g_return_val_if_fail (condition && IS_GNOME_DB_CONDITION (condition), FALSE);
	g_return_val_if_fail (condition->priv, FALSE);
	
	return gnome_db_condition_represents_join_real (condition, target1, target2, NULL, TRUE, TRUE);
}

static GSList*
cond_get_main_sub_conditions (GnomeDbCondition *cond)
{
	GSList *retval = NULL;

	if (cond->priv->type == GNOME_DB_CONDITION_NODE_AND) {
		GSList *list;
		
		list = cond->priv->cond_children;
		while (list) {
			GSList *tmp = cond_get_main_sub_conditions (GNOME_DB_CONDITION (list->data));
			if (tmp)
				retval = g_slist_concat (retval, tmp);
			list = g_slist_next (list);
		}
	}
	else
		retval = g_slist_append (retval, cond);
	
	return retval;
}

/**
 * gnome_db_condition_get_main_conditions
 * @condition: a #GnomeDbCondition object
 *
 * Makes a list of all the conditions which
 * are always verified by @condition when it returns TRUE when evaluated.
 * Basically the returned list lists the atomic conditions which are AND'ed
 * together to form the complex @condition.
 *
 * Examples: if @condition is:
 * --> "A and B" then the list will contains {A, B}
 * --> "A and (B or C)" it will contain {A, B or C}
 * --> "A and (B and not C)", it will contain {A, B, not C}
 *
 * Returns: a new list of #GnomeDbCondition objects
 */
GSList *
gnome_db_condition_get_main_conditions (GnomeDbCondition *condition)
{
	g_return_val_if_fail (condition && IS_GNOME_DB_CONDITION (condition), FALSE);
	g_return_val_if_fail (condition->priv, FALSE);

	return cond_get_main_sub_conditions (condition);
}


static const gchar *condition_type_to_str (GnomeDbConditionType type);

#ifdef debug
static void
gnome_db_condition_dump (GnomeDbCondition *condition, guint offset)
{
	gchar *str;
	gint i;

	g_return_if_fail (condition && IS_GNOME_DB_CONDITION (condition));
	
        /* string for the offset */
        str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

        /* dump */
        if (condition->priv) {
		GSList *children;
		if (gnome_db_base_get_name (GNOME_DB_BASE (condition)))
			g_print ("%s" D_COL_H1 "GnomeDbCondition" D_COL_NOR " %s \"%s\" (%p, id=%d, query=%p) ",
				 str, condition_type_to_str (condition->priv->type), 
				 gnome_db_base_get_name (GNOME_DB_BASE (condition)), 
				 condition, gnome_db_base_get_id (GNOME_DB_BASE (condition)), condition->priv->query);
		else
			g_print ("%s" D_COL_H1 "GnomeDbCondition" D_COL_NOR " %s (%p, id=%d, query=%p) ",
				 str, condition_type_to_str (condition->priv->type), 
				 condition, gnome_db_base_get_id (GNOME_DB_BASE (condition)), condition->priv->query);
		if (gnome_db_condition_is_active (GNOME_DB_REFERER (condition)))
			g_print ("Active\n");
		else
			g_print (D_COL_ERR "Non active\n" D_COL_NOR);
		g_print ("\n");
		children = condition->priv->cond_children;
		while (children) {
			gnome_db_condition_dump (GNOME_DB_CONDITION (children->data), offset+5);
			children = g_slist_next (children);
		}
		for (i=0; i<3; i++)
			if (condition->priv->ops[i]) {
				g_print ("%s" D_COL_H1 "Field %d:\n" D_COL_NOR, str, i);
				gnome_db_base_dump (GNOME_DB_BASE (condition->priv->ops[i]), offset+5);
			}
	}
        else
                g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, condition);
}
#endif

/* 
 * GnomeDbXmlStorage interface implementation
 */

static const gchar *
condition_type_to_str (GnomeDbConditionType type)
{
	switch (type) {
	case GNOME_DB_CONDITION_NODE_AND:
		return "AND";
	case GNOME_DB_CONDITION_NODE_OR:
		return "OR";
        case GNOME_DB_CONDITION_NODE_NOT:
		return "NOT";
        case GNOME_DB_CONDITION_LEAF_EQUAL:
		return "EQ";
        case GNOME_DB_CONDITION_LEAF_DIFF:
		return "NE";
        case GNOME_DB_CONDITION_LEAF_SUP:
		return "SUP";
        case GNOME_DB_CONDITION_LEAF_SUPEQUAL:
		return "ESUP";
        case GNOME_DB_CONDITION_LEAF_INF:
		return "INF";
        case GNOME_DB_CONDITION_LEAF_INFEQUAL:
		return "EINF";
        case GNOME_DB_CONDITION_LEAF_LIKE:
		return "LIKE";
	case GNOME_DB_CONDITION_LEAF_SIMILAR:
		return "SIMI";
        case GNOME_DB_CONDITION_LEAF_REGEX:
		return "REG";
        case GNOME_DB_CONDITION_LEAF_REGEX_NOCASE:
		return "CREG";
        case GNOME_DB_CONDITION_LEAF_NOT_REGEX:
		return "NREG";
        case GNOME_DB_CONDITION_LEAF_NOT_REGEX_NOCASE:
		return "CNREG";
        case GNOME_DB_CONDITION_LEAF_IN:
		return "IN";
        case GNOME_DB_CONDITION_LEAF_BETWEEN:
		return "BTW";
	default:
		return "???";
	}
}

static GnomeDbConditionType
condition_str_to_type (const gchar *type)
{
	switch (*type) {
	case 'A':
		return GNOME_DB_CONDITION_NODE_AND;
	case 'O':
		return GNOME_DB_CONDITION_NODE_OR;
	case 'N':
		if (!strcmp (type, "NOT"))
			return GNOME_DB_CONDITION_NODE_NOT;
		else {
			if (!strcmp (type, "NE"))
				return GNOME_DB_CONDITION_LEAF_DIFF;
			else
				return GNOME_DB_CONDITION_LEAF_NOT_REGEX;
		}
	case 'E':
		if (!strcmp (type, "EQ"))
			return GNOME_DB_CONDITION_LEAF_EQUAL;
		else if (strcmp (type, "ESUP"))
			return GNOME_DB_CONDITION_LEAF_SUPEQUAL;
		else
			return GNOME_DB_CONDITION_LEAF_INFEQUAL;
	case 'S':
		if (type[1] == 'I')
			return GNOME_DB_CONDITION_LEAF_SIMILAR;
		else
			return GNOME_DB_CONDITION_LEAF_SUP;

	case 'I':
		if (!strcmp (type, "INF"))
			return GNOME_DB_CONDITION_LEAF_INF;
		else
			return GNOME_DB_CONDITION_LEAF_IN;
	case 'L':
		return GNOME_DB_CONDITION_LEAF_LIKE;
	case 'R':
		return GNOME_DB_CONDITION_LEAF_REGEX;
	case 'C':
		switch (type[1]) {
		case 'R':
			return GNOME_DB_CONDITION_LEAF_REGEX_NOCASE;
		case 'N':
			return GNOME_DB_CONDITION_LEAF_NOT_REGEX_NOCASE;
		default:
			return GNOME_DB_CONDITION_TYPE_UNKNOWN;
		}
	case 'B':
		return GNOME_DB_CONDITION_LEAF_BETWEEN;
	default:
		return GNOME_DB_CONDITION_TYPE_UNKNOWN;
	}
}

static gchar *
gnome_db_condition_get_xml_id (GnomeDbXmlStorage *iface)
{
        gchar *q_xml_id, *xml_id;

        g_return_val_if_fail (iface && IS_GNOME_DB_CONDITION (iface), NULL);
        g_return_val_if_fail (GNOME_DB_CONDITION (iface)->priv, NULL);

        q_xml_id = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (GNOME_DB_CONDITION (iface)->priv->query));
        xml_id = g_strdup_printf ("%s:C%d", q_xml_id, gnome_db_base_get_id (GNOME_DB_BASE (iface)));
        g_free (q_xml_id);

        return xml_id;
}


static xmlNodePtr
gnome_db_condition_save_to_xml (GnomeDbXmlStorage *iface, GError **error)
{
        xmlNodePtr node = NULL;
	GnomeDbCondition *cond;
        gchar *str;
	GnomeDbBase *base;
	GSList *list;

        g_return_val_if_fail (iface && IS_GNOME_DB_CONDITION (iface), NULL);
        g_return_val_if_fail (GNOME_DB_CONDITION (iface)->priv, NULL);

        cond = GNOME_DB_CONDITION (iface);

        node = xmlNewNode (NULL, "GNOME_DB_COND");

        str = gnome_db_condition_get_xml_id (iface);
        xmlSetProp (node, "id", str);
        g_free (str);

        xmlSetProp (node, "type", condition_type_to_str (cond->priv->type));

	base = gnome_db_ref_base_get_ref_object (cond->priv->ops[GNOME_DB_CONDITION_OP_LEFT]);
	if (base) {
		str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (base));
		xmlSetProp (node, "l_op", str);
		g_free (str);
	}

	base = gnome_db_ref_base_get_ref_object (cond->priv->ops[GNOME_DB_CONDITION_OP_RIGHT]);
	if (base) {
		str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (base));
		xmlSetProp (node, "r_op", str);
		g_free (str);
	}

	base = gnome_db_ref_base_get_ref_object (cond->priv->ops[GNOME_DB_CONDITION_OP_RIGHT2]);
	if (base) {
		str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (base));
		xmlSetProp (node, "r_op2", str);
		g_free (str);
	}

	/* sub conditions */
	list = cond->priv->cond_children;
	while (list) {
		xmlNodePtr sub = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (list->data), error);
		if (sub)
                        xmlAddChild (node, sub);
                else {
                        xmlFreeNode (node);
                        return NULL;
                }
		list = g_slist_next (list);
	}

        return node;
}

static gboolean
gnome_db_condition_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	GnomeDbCondition *cond;
        gchar *prop;
        gboolean id = FALSE;
	xmlNodePtr children;

        g_return_val_if_fail (iface && IS_GNOME_DB_CONDITION (iface), FALSE);
        g_return_val_if_fail (GNOME_DB_CONDITION (iface)->priv, FALSE);
        g_return_val_if_fail (node, FALSE);

	cond = GNOME_DB_CONDITION (iface);

	if (strcmp (node->name, "GNOME_DB_COND")) {
                g_set_error (error,
                             GNOME_DB_CONDITION_ERROR,
                             GNOME_DB_CONDITION_XML_LOAD_ERROR,
                             _("XML Tag is not <GNOME_DB_COND>"));
                return FALSE;
        }

	prop = xmlGetProp (node, "id");
        if (prop) {
                gchar *ptr, *tok;
                ptr = strtok_r (prop, ":", &tok);
                ptr = strtok_r (NULL, ":", &tok);
                if (strlen (ptr) < 2) {
                        g_set_error (error,
                                     GNOME_DB_CONDITION_ERROR,
                                     GNOME_DB_CONDITION_XML_LOAD_ERROR,
                                     _("Wrong 'id' attribute in <GNOME_DB_COND>"));
                        return FALSE;
                }
                gnome_db_base_set_id (GNOME_DB_BASE (cond), atoi (ptr+1));
		id = TRUE;
                g_free (prop);
        }

	prop = xmlGetProp (node, "type");
        if (prop) {
		cond->priv->type = condition_str_to_type (prop);
		if (cond->priv->type == GNOME_DB_CONDITION_TYPE_UNKNOWN) {
			g_set_error (error,
                                     GNOME_DB_CONDITION_ERROR,
                                     GNOME_DB_CONDITION_XML_LOAD_ERROR,
                                     _("Wrong 'type' attribute in <GNOME_DB_COND>"));
                        return FALSE;
		}
                g_free (prop);
        }

	prop = xmlGetProp (node, "l_op");
	if (prop) {
		gnome_db_ref_base_set_ref_name (cond->priv->ops[GNOME_DB_CONDITION_OP_LEFT], GNOME_DB_QFIELD_TYPE,
					  REFERENCE_BY_XML_ID, prop);
		g_free (prop);
	}


	prop = xmlGetProp (node, "r_op");
	if (prop) {
		gnome_db_ref_base_set_ref_name (cond->priv->ops[GNOME_DB_CONDITION_OP_RIGHT], GNOME_DB_QFIELD_TYPE,
					  REFERENCE_BY_XML_ID, prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "r_op2");
	if (prop) {
		gnome_db_ref_base_set_ref_name (cond->priv->ops[GNOME_DB_CONDITION_OP_RIGHT2], GNOME_DB_QFIELD_TYPE,
					  REFERENCE_BY_XML_ID, prop);
		g_free (prop);
	}

	/* children nodes */
	children = node->children;
	while (children) {
		if (!strcmp (children->name, "GNOME_DB_COND")) {
			GnomeDbCondition *scond;

			scond = GNOME_DB_CONDITION (gnome_db_condition_new (cond->priv->query, GNOME_DB_CONDITION_NODE_AND));
			if (gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (scond), children, error)) {
				gnome_db_condition_node_add_child (cond, scond, NULL);
				g_object_unref (G_OBJECT (scond));
			}
			else
				return FALSE;
                }

		children = children->next;
	}

	if (!id) {
		g_set_error (error,
			     GNOME_DB_CONDITION_ERROR,
			     GNOME_DB_CONDITION_XML_LOAD_ERROR,
			     _("Missing Id attribute in <GNOME_DB_COND>"));
		return FALSE;
        }

        return TRUE;
}


/*
 * GnomeDbRenderer interface implementation
 */
static GdaXqlItem *
gnome_db_condition_render_as_xql (GnomeDbRenderer *iface, GnomeDbDataSet *context, GError **error)
{
        GdaXqlItem *node = NULL;

        g_return_val_if_fail (iface && IS_GNOME_DB_CONDITION (iface), NULL);
        g_return_val_if_fail (GNOME_DB_CONDITION (iface)->priv, NULL);

        TO_IMPLEMENT;
        return node;
}

static gchar *
gnome_db_condition_render_as_sql (GnomeDbRenderer *iface, GnomeDbDataSet *context, guint options, GError **error)
{
        gchar *retval = NULL, *str;
	GString *string;
        GnomeDbCondition *cond;
	gboolean is_node = FALSE;
	gchar *link = NULL;
	GnomeDbBase *ops[3];
	gint i;
	gboolean pprint = options & GNOME_DB_RENDERER_EXTRA_PRETTY_SQL;

        g_return_val_if_fail (iface && IS_GNOME_DB_CONDITION (iface), NULL);
        g_return_val_if_fail (GNOME_DB_CONDITION (iface)->priv, NULL);
        cond = GNOME_DB_CONDITION (iface);
	if (!gnome_db_condition_activate (GNOME_DB_REFERER (cond))) {
		g_set_error (error,
			     GNOME_DB_CONDITION_ERROR,
			     GNOME_DB_CONDITION_RENDERER_ERROR,
			     _("Condition is not active"));
		return NULL;
	}

	for (i=0; i<3; i++)
		ops[i] = gnome_db_ref_base_get_ref_object (cond->priv->ops[i]);

	/* testing for completeness */
	switch (cond->priv->type) {
	case GNOME_DB_CONDITION_NODE_NOT:
		if (g_slist_length (cond->priv->cond_children) != 1) {
			g_set_error (error,
				     GNOME_DB_CONDITION_ERROR,
				     GNOME_DB_CONDITION_RENDERER_ERROR,
				     _("Condition operator 'NOT' must have one argument"));
			return NULL;
		}
		break;
	case GNOME_DB_CONDITION_NODE_AND:
	case GNOME_DB_CONDITION_NODE_OR:
		if (g_slist_length (cond->priv->cond_children) == 0) {
			g_set_error (error,
				     GNOME_DB_CONDITION_ERROR,
				     GNOME_DB_CONDITION_RENDERER_ERROR,
				     _("Condition must have at least one argument"));
			return NULL;
		}
		break;
	case GNOME_DB_CONDITION_LEAF_EQUAL:
	case GNOME_DB_CONDITION_LEAF_DIFF:
	case GNOME_DB_CONDITION_LEAF_SUP:
        case GNOME_DB_CONDITION_LEAF_SUPEQUAL:
        case GNOME_DB_CONDITION_LEAF_INF:
        case GNOME_DB_CONDITION_LEAF_INFEQUAL:
        case GNOME_DB_CONDITION_LEAF_LIKE:
	case GNOME_DB_CONDITION_LEAF_SIMILAR:
        case GNOME_DB_CONDITION_LEAF_REGEX:
        case GNOME_DB_CONDITION_LEAF_REGEX_NOCASE:
        case GNOME_DB_CONDITION_LEAF_NOT_REGEX:
        case GNOME_DB_CONDITION_LEAF_NOT_REGEX_NOCASE:
	case GNOME_DB_CONDITION_LEAF_IN:
		if (!ops[GNOME_DB_CONDITION_OP_LEFT] || !ops[GNOME_DB_CONDITION_OP_RIGHT]) {
			g_set_error (error,
				     GNOME_DB_CONDITION_ERROR,
				     GNOME_DB_CONDITION_RENDERER_ERROR,
				     _("Condition must have two arguments"));
			return NULL;
		}
		break;
        case GNOME_DB_CONDITION_LEAF_BETWEEN:
		if (!ops[GNOME_DB_CONDITION_OP_LEFT] || !ops[GNOME_DB_CONDITION_OP_RIGHT] ||
		    !ops[GNOME_DB_CONDITION_OP_RIGHT2]) {
			g_set_error (error,
				     GNOME_DB_CONDITION_ERROR,
				     GNOME_DB_CONDITION_RENDERER_ERROR,
				     _("Condition 'BETWEEN' must have three arguments"));
			return NULL;
		}
	default:
		break;
	}

	/* actual rendering */
	string = g_string_new ("");
	switch (cond->priv->type) {
	case GNOME_DB_CONDITION_NODE_AND:
		is_node = TRUE;
		link = " AND ";
		break;
	case GNOME_DB_CONDITION_NODE_OR:
		is_node = TRUE;
		link = " OR ";
		break;
	case GNOME_DB_CONDITION_NODE_NOT:
		is_node = TRUE;
		link = "NOT ";
		break;
	case GNOME_DB_CONDITION_LEAF_EQUAL:
		link = "=";
		if (IS_GNOME_DB_QF_VALUE (ops[GNOME_DB_CONDITION_OP_RIGHT]) &&
		    gnome_db_qf_value_is_value_null (GNOME_DB_QF_VALUE (ops[GNOME_DB_CONDITION_OP_RIGHT]), context) &&
		    ! gnome_db_qf_value_is_parameter (GNOME_DB_QF_VALUE (ops[GNOME_DB_CONDITION_OP_RIGHT])))
			link = " IS ";
		break;
	case GNOME_DB_CONDITION_LEAF_DIFF:
		link = "!=";
		if (IS_GNOME_DB_QF_VALUE (ops[GNOME_DB_CONDITION_OP_RIGHT]) &&
		    gnome_db_qf_value_is_value_null (GNOME_DB_QF_VALUE (ops[GNOME_DB_CONDITION_OP_RIGHT]), context) &&
		    ! gnome_db_qf_value_is_parameter (GNOME_DB_QF_VALUE (ops[GNOME_DB_CONDITION_OP_RIGHT])))
			link = " IS NOT ";
		break;
	case GNOME_DB_CONDITION_LEAF_SUP:
		link = " > ";
		break;
        case GNOME_DB_CONDITION_LEAF_SUPEQUAL:
		link = " >= ";
		break;
        case GNOME_DB_CONDITION_LEAF_INF:
		link = " < ";
		break;
        case GNOME_DB_CONDITION_LEAF_INFEQUAL:
		link = " <= ";
		break;
        case GNOME_DB_CONDITION_LEAF_LIKE:
		link = " LIKE ";
		break;
        case GNOME_DB_CONDITION_LEAF_SIMILAR:
		link = " SIMILAR TO ";
		break;
        case GNOME_DB_CONDITION_LEAF_REGEX:
		link = " ~ ";
		break;
        case GNOME_DB_CONDITION_LEAF_REGEX_NOCASE:
		link = " ~* ";
		break;
        case GNOME_DB_CONDITION_LEAF_NOT_REGEX:
		link = " !~ ";
		break;
        case GNOME_DB_CONDITION_LEAF_NOT_REGEX_NOCASE:
		link = " !~* ";
		break;
	case GNOME_DB_CONDITION_LEAF_IN:
		link = " IN ";
		break;
        case GNOME_DB_CONDITION_LEAF_BETWEEN:
		link = " BETWEEN ";
		break;
	default:
		break;
	}

	if (link) {
		if (is_node) {
			if (cond->priv->type != GNOME_DB_CONDITION_NODE_NOT) { /* type = AND or OR */
				gboolean first = TRUE;
				GSList *list;

				list = cond->priv->cond_children;
				while (list) {
					if (first) 
						first = FALSE;
					else {
						if (pprint && ! cond->priv->join)
							g_string_append (string, "\n\t");
						g_string_append_printf (string, "%s", link);
					}

					if ((cond->priv->type == GNOME_DB_CONDITION_NODE_AND) &&
					    (GNOME_DB_CONDITION (list->data)->priv->type == GNOME_DB_CONDITION_NODE_OR))
						g_string_append (string, "(");

					str = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (list->data), context, 
									 options, error);
					if (!str) {
						g_string_free (string, TRUE);
						return NULL;
					}
					g_string_append (string, str);
					g_free (str);

					if ((cond->priv->type == GNOME_DB_CONDITION_NODE_AND) &&
					    (GNOME_DB_CONDITION (list->data)->priv->type == GNOME_DB_CONDITION_NODE_OR))
						g_string_append (string, ")");
					list = g_slist_next (list);
				}
			}
			else { /* type = NOT */
				g_string_append_printf (string, "%s", link);
				str = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (cond->priv->cond_children->data), 
								 context, options, error);
				if (!str) {
					g_string_free (string, TRUE);
					return NULL;
				}
				g_string_append (string, "(");
				g_string_append (string, str);
				g_string_append (string, ")");
				g_free (str);
			}
		}
		else {
			str = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (ops[GNOME_DB_CONDITION_OP_LEFT]), context, 
							 options, error);
			if (!str) {
				g_string_free (string, TRUE);
				return NULL;
			}
			g_string_append (string, str);
			g_free (str);
			g_string_append_printf (string, "%s", link);
			str = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (ops[GNOME_DB_CONDITION_OP_RIGHT]), context, 
							 options, error);
			if (!str) {
				g_string_free (string, TRUE);
				return NULL;
			}
			g_string_append (string, str);
			g_free (str);
		}
	}

	if (cond->priv->type == GNOME_DB_CONDITION_LEAF_BETWEEN) {
		str = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (ops[GNOME_DB_CONDITION_OP_RIGHT2]), context, 
						 options, error);
		if (!str) {
			g_string_free (string, TRUE);
			return NULL;
		}
		g_string_append_printf (string, " AND %s", str);
		g_free (str);
	}

	retval = string->str;
	g_string_free (string, FALSE);

        return retval;
}

static gchar *
gnome_db_condition_render_as_str (GnomeDbRenderer *iface, GnomeDbDataSet *context)
{
        gchar *str = NULL;

        g_return_val_if_fail (iface && IS_GNOME_DB_CONDITION (iface), NULL);
        g_return_val_if_fail (GNOME_DB_CONDITION (iface)->priv, NULL);

        str = gnome_db_condition_render_as_sql (iface, context, 0, NULL);
        if (!str)
                str = g_strdup ("???");
        return str;
}


/*
 * GnomeDbReferer interface implementation
 */
static gboolean
gnome_db_condition_activate (GnomeDbReferer *iface)
{
	gboolean activated = TRUE;
	gint i;
	GSList *list;
	GnomeDbCondition *cond;

        g_return_val_if_fail (iface && IS_GNOME_DB_CONDITION (iface), FALSE);
        g_return_val_if_fail (GNOME_DB_CONDITION (iface)->priv, FALSE);
	cond = GNOME_DB_CONDITION (iface);

	for (i=0; i<3; i++) {
		if (!gnome_db_ref_base_activate (cond->priv->ops[i]))
			activated = FALSE;
	}
	list = cond->priv->cond_children;
	while (list) {
		if (!gnome_db_condition_activate (GNOME_DB_REFERER (list->data)))
			activated = FALSE;

		list = g_slist_next (list);
	}

	return activated;
}

static void
gnome_db_condition_deactivate (GnomeDbReferer *iface)
{
	gint i;
	GSList *list;
	GnomeDbCondition *cond;

        g_return_if_fail (iface && IS_GNOME_DB_CONDITION (iface));
        g_return_if_fail (GNOME_DB_CONDITION (iface)->priv);
	cond = GNOME_DB_CONDITION (iface);

	for (i=0; i<3; i++)
		gnome_db_ref_base_deactivate (cond->priv->ops[i]);
	list = cond->priv->cond_children;
	while (list) {
		gnome_db_condition_deactivate (GNOME_DB_REFERER (list->data));
		list = g_slist_next (list);
	}
}

static gboolean
gnome_db_condition_is_active (GnomeDbReferer *iface)
{
	gboolean activated = TRUE;
	gint i;
	GSList *list;
	GnomeDbCondition *cond;

        g_return_val_if_fail (iface && IS_GNOME_DB_CONDITION (iface), FALSE);
        g_return_val_if_fail (GNOME_DB_CONDITION (iface)->priv, FALSE);
	cond = GNOME_DB_CONDITION (iface);

	for (i=0; i<3; i++) {
		if (!gnome_db_ref_base_is_active (cond->priv->ops[i]))
			activated = FALSE;
	}
	list = cond->priv->cond_children;
	while (list && activated) {
		activated = gnome_db_condition_is_active (GNOME_DB_REFERER (list->data));
		list = g_slist_next (list);
	}

	return activated;
}

static GSList *
gnome_db_condition_get_ref_objects (GnomeDbReferer *iface)
{
        GSList *list = NULL;
	gint i;
	
        g_return_val_if_fail (iface && IS_GNOME_DB_CONDITION (iface), NULL);
        g_return_val_if_fail (GNOME_DB_CONDITION (iface)->priv, NULL);

	for (i=0; i<3; i++) {
                GnomeDbBase *base = gnome_db_ref_base_get_ref_object (GNOME_DB_CONDITION (iface)->priv->ops[i]);
                if (base)
                        list = g_slist_append (list, base);
        }

        return list;
}

/**
 * gnome_db_condition_get_ref_objects_all
 * @cond: a #GnomeDbCondition object
 *
 * Get a complete list of the objects referenced by @cond, 
 * including its descendants (unlike the gnome_db_referer_get_ref_objects()
 * function applied to @cond).
 *
 * Returns: a new list of referenced objects
 */
GSList *
gnome_db_condition_get_ref_objects_all (GnomeDbCondition *cond)
{
        GSList *list = NULL, *children;
	gint i;
	
        g_return_val_if_fail (cond && IS_GNOME_DB_CONDITION (cond), NULL);
        g_return_val_if_fail (cond->priv, NULL);

	for (i=0; i<3; i++) {
		if (cond->priv->ops[i]) {
			GnomeDbBase *base = gnome_db_ref_base_get_ref_object (cond->priv->ops[i]);
			if (base)
				list = g_slist_append (list, base);
		}
        }

	children = cond->priv->cond_children;
	while (children) {
		GSList *clist = gnome_db_condition_get_ref_objects_all (GNOME_DB_CONDITION (children->data));
		if (clist)
			list = g_slist_concat (list, clist);
		children = g_slist_next (children);
	}

        return list;
}

static void
gnome_db_condition_replace_refs (GnomeDbReferer *iface, GHashTable *replacements)
{
	gint i;
	GnomeDbCondition *cond;
	GSList *list;

        g_return_if_fail (iface && IS_GNOME_DB_CONDITION (iface));
        g_return_if_fail (GNOME_DB_CONDITION (iface)->priv);

        cond = GNOME_DB_CONDITION (iface);
        if (cond->priv->query) {
                GnomeDbQuery *query = g_hash_table_lookup (replacements, cond->priv->query);
                if (query) {
			gnome_db_query_undeclare_condition (cond->priv->query, cond);
                        g_signal_handlers_disconnect_by_func (G_OBJECT (cond->priv->query),
                                                              G_CALLBACK (nullified_object_cb), cond);
                        cond->priv->query = query;
			gnome_db_base_connect_nullify (query, G_CALLBACK (nullified_object_cb), cond);
			gnome_db_query_declare_condition (query, cond);
                }
        }

	if (cond->priv->join) {
                GnomeDbJoin *join = g_hash_table_lookup (replacements, cond->priv->join);
                if (join) {
                        g_signal_handlers_disconnect_by_func (G_OBJECT (cond->priv->join),
                                                              G_CALLBACK (nullified_object_cb), cond);
                        cond->priv->join = join;
			gnome_db_base_connect_nullify (join, G_CALLBACK (nullified_object_cb), cond);
                }
        }

	/* references in operators */
	for (i=0; i<3; i++)
                gnome_db_ref_base_replace_ref_object (cond->priv->ops[i], replacements);

	/* references in children's list */
	list = cond->priv->cond_children;
	while (list) {
		GnomeDbCondition *sub = g_hash_table_lookup (replacements, list->data);
		if (sub) {
			i = g_slist_position (cond->priv->cond_children, list);
			cond->priv->internal_transaction ++;
			gnome_db_condition_node_del_child (cond, GNOME_DB_CONDITION (list->data));
			cond->priv->internal_transaction --;
			gnome_db_condition_node_add_child_pos (cond, sub, i, NULL);
		}
		else
			list = g_slist_next (list);
	}

	/* references in children themselves */
	list = cond->priv->cond_children;
	while (list) {
		gnome_db_condition_replace_refs (GNOME_DB_REFERER (list->data), replacements);
		list = g_slist_next (list);
	}
}
