/* -*- tab-width: 4; c-basic-offset: 4 -*- */

using System;
using System.Collections.Generic;
using Gtk;
using Cairo;

namespace iogrind
{
	class MainClass
	{
		static void PrintHelp ()
		{
			Console.WriteLine ("Usage: iogrind-gui [options] --fsmodel=<fsmodel> [--warm <warm files...>] [--exec] <file1.prf> [<file2.prf>]\n" +
							   "  iogrind simulates I/O profiles generated from applications\n" +
							   "  --help                    show command-line documentation\n" +
							   "  --console                 don't activate the GUI\n" +
							   "  --fsmodel=<image.xml.gz>  use the specified file-system model\n" +
							   "  --disksim=<model.parv>    use a different disksim disk model\n" +
							   "  --warm <files>            optional - set of files to pre-warm the page cache with\n" +
							   "  --exec <profiles>         --exec terminates list of --warm files, and is optional\n");
		}

		static void PrintData (string profile, SimulationData sd)
		{
			Console.WriteLine (profile + " executed in " +
							   Utils.PrettyTime (sd.Elapsed));

			Console.WriteLine ("time\tpercent\tsize\tmap");

			List<SimFile> sorted = sd.FilesSortedByIO();
			foreach (SimFile simFile in sorted) {
				if (simFile.TotalTime <= 0)
					continue;

					Console.WriteLine (Utils.PrettyTime (simFile.TotalTime) + "\t" +
									   Utils.Percentage (simFile.Percentage) + "\t" +
									   Utils.PrettyBytes (simFile.TotalBytes) + "\t" +
									   simFile.FileNameAtOpen);
			}

			Console.WriteLine ("Summary:");

			Console.WriteLine ("   executed in " + Utils.PrettyTime (sd.Elapsed));
			Console.WriteLine ("   size read: " + sd.TotalBytes / 1024 + "kb");
			Console.WriteLine ("   avg I/O B/W: " + Utils.PrettyBW (sd.TotalBytes, sd.Elapsed));
		}

        static void DoReadTest (string diskProfile, long startBlock)
        {
            int i, read_sects;
            double time = 0.0;
            DiskSim sim = new DiskSim (diskProfile);
            read_sects = 64 * 1024 / 512;
            for (i = 0; i < 25; i++) {
                SimEvent ev = new SimEvent();
                DiskModel.IORequest req;
                req = new DiskModel.TestRequest (sim, ev,
                                                 startBlock + i * read_sects,
                                                 read_sects);
                sim.SubmitTestRequest (req, time);
                sim.Wait (ref time);
            }
        }
		
		public static void Main (string[] args)
		{
			string fsName = null;
            string diskProfile = null;
			List<string> preWarm = new List<string>();
			List<string> profiles = new List<string>();
			bool   warming = false;
            long   readTest = -1;
			bool   gui = true;

			foreach (string s in args) {
				if (s == "--help" || s == "-h") {
					PrintHelp();
					return;
				} else if (s == "--console")
					gui = false;
				else if (s.StartsWith("--fsmodel="))
					fsName = s.Substring (10);
				else if (s.StartsWith("--disksim="))
					diskProfile = s.Substring (10);
				else if (s.StartsWith("--readtest=")) {
                    int dummy;
					readTest = Utils.ReadHex (s, 11, out dummy);
                    Console.WriteLine ("Read test at {0:x}", readTest);
                } else if (s.StartsWith("--warm"))
					warming = true;
				else if (s.StartsWith("--exec"))
					warming = false;
				else {
					if (warming)
						preWarm.Add (s);
					else
						profiles.Add(s);
				}
			}
			if (diskProfile == null)
                diskProfile = "simple.parv";
            if (readTest > 0) {
                DoReadTest (diskProfile, readTest);
                return;
            }
            if (profiles.Count == 0) {
                System.Console.WriteLine("No profile name(s)");
				PrintHelp();
				return;
            }
			if (fsName == null) {
                System.Console.WriteLine("No file-system model name(s)");
				PrintHelp();
				return;
            }

            StatusFileStream strm;
			if (gui) {
				Application.Init ();
                strm = new GuiStatusFileStream (fsName);
            } else
                strm = new StatusFileStream (fsName);

			FSData fsData = new FSData (strm, fsName);

			Simulator sim = new Simulator (fsData, diskProfile);

			System.Console.WriteLine ("warming ...\n");
			foreach (string profile in preWarm) {
				// discard output
				sim.Simulate (new Trace.Data (profile));
			}

			System.Console.WriteLine ("simulating ...\n");
			foreach (string profile in profiles) {
				Trace.Data traceData = new Trace.Data (profile);
				SimulationData sd = sim.Simulate (traceData);

				if (gui)
					new MainWindow (sd, profile);
				else
					PrintData (profile, sd);
			}
			if (gui) {
				foreach (Gtk.Window window in MainWindow.AllWindows)
					window.ShowAll ();
				Application.Run ();
			}
        }
    }
}
