/* -*- tab-width: 4; c-basic-offset: 4 -*- */

using System;
using System.Xml;
using System.Threading;
using System.Collections.Generic;
using FS;

namespace FS {
    public struct Block {
        long start;
        long length;
        public long Start {
            get { return this.start; }
            set { this.start = value; }
        }
        public long End {
            get { return this.start + (this.length > 0 ? this.length : 1); }
        }
        public long Length {
            get { return length < 0 ? 1 : this.length; }
            set { this.length = value; }
        }
        public bool IsIndirect {
            get { return this.length < 0; }
            set { this.length = -1; }
        }
        public Block Copy()
        {
            Block blk = new Block();
            blk.start = this.start;
            blk.length = this.length;
            return blk;
        }
        public bool Contains (long blkId)
        {
            return blkId >= start && blkId < start + length;
        }
    }

    public class File
    {
        public Directory Parent;
        public string Name;
        public long INodeId;
        public long INodeBlock;
        public long Size;
        public List<Block> Blocks;

        public string GetPath()
        {
            return (Parent != null ? (Parent.GetPath() + "/" + Name) : "");
        }

        protected void AccumulMinMax (ref long min, ref long max, long block)
        {
            min = Math.Min (min, block);
            max = Math.Max (max, block);
        }

        internal protected virtual void MinMaxCalc (ref long min, ref long max)
        {
            AccumulMinMax (ref min, ref max, this.INodeBlock);
            if (this.Blocks != null) {
                foreach (Block blk in this.Blocks) {
                    AccumulMinMax (ref min, ref max, blk.Start);
                    AccumulMinMax (ref min, ref max, blk.End);
                }
            }
        }

        public virtual void GetMinMax (out long min, out long max)
        {
            min = Int64.MaxValue;
            max = 0;
            MinMaxCalc (ref min, ref max);
        }

        protected internal File ()
        { // used by Copy etc.
        }
    	public File (FSData fsd, XmlTextReader reader,
                     Directory parent, bool readBlocks)
        {
            this.Parent = parent;
            this.Name = reader ["name"];
            this.Size = Utils.ReadHex (reader ["size"]);
            if (parent != null) {
                this.INodeId = Utils.ReadHex (reader ["inode"]);
                this.INodeBlock = Utils.ReadHex (reader ["iblk"]);
            }
            if (readBlocks)
                ReadBlocks (fsd, reader);
        }
        void SyntaxError (string blocks, int offset)
        {
            Console.WriteLine ("** Syntax error in '" + blocks + "' at offset {0}", offset);
        }
        
        internal protected void ReadBlocks (FSData fsd, XmlTextReader reader)
        {
            if (reader.IsEmptyElement)
                return;
            reader.Read();
            reader.MoveToContent();
            bool stripBlks;
            if ((stripBlks = (reader.Name == "blks"))) {
                reader.Read();
                if (reader.MoveToContent() == XmlNodeType.EndElement)
                    return; // empty <blks></blks>
            }
// Example:
//          0x4c-0x57, i0x58, 0x5a-0x158, i0x159, 0x15b-0x24f
            string blocks = reader.Value;
            
            if (blocks != null && blocks.Length > 0) {
                int i, len = blocks.Length;
                
                // it is worth getting the correct list size.
                int blkCount = 1;
                for (i = 0; i < len; i++) {
                    if (blocks[i] == ',')
                        blkCount++;
                }
                this.Blocks = new List<Block> (blkCount);

//                Console.WriteLine ("Parse '" + blocks + "'");

                for (i = 0; i < len;) {
                    Block blk = new Block();
                    blk.Length = 1;

                    // loose whitespace
                    while (Char.IsWhiteSpace (blocks [i]) && i < len)
                        i++;
                    
                    if (i < len && blocks[i] == 'i') {
                        blk.IsIndirect = true;
                        i++;
                    }
                    blk.Start = Utils.ReadHex (blocks, i, out i);
                    if (i < len) {
                        if (blocks[i] == '-') {
                            i++;
                            blk.Length = (Utils.ReadHex (blocks, i, out i) - blk.Start) + 1;
                        }
                    }
                    if (i < len && blocks[i] != ',') {
                        SyntaxError (blocks, i);
                        return;
                    } else
                        i++;
                        
                    this.Blocks.Add (blk);
//                    Console.WriteLine (blocks + " -> {0}0x{1:x}-0x{2:x}",
//                                       blk.IsIndirect ? "i" : "",
//                                       blk.Start, blk.End - 1);
                }
            }

            if (stripBlks)
                reader.Read();
        }

        protected internal void CopyFields (File tmpl, Directory newParent)
        {
            this.Parent = newParent;
            if (newParent != null)
                newParent.AddChild (this);
            this.Name = tmpl.Name;
            this.INodeId = tmpl.INodeId;
            this.INodeBlock = tmpl.INodeBlock;
            this.Size = tmpl.Size;
            if (tmpl.Blocks != null) {
                this.Blocks = new List<Block>(tmpl.Blocks.Count);
                foreach (Block b in tmpl.Blocks)
                    this.Blocks.Add (b.Copy());
            } else
                this.Blocks = null;
        }
        public virtual File Copy (Directory newParent)
        {
            File fi = new File();
            fi.CopyFields (this, newParent);
            return fi;
        }
    }

    public class Special : File
    {
        public int Type;
        public Special (FSData fsd, XmlTextReader reader, Directory parent)
            : base (fsd, reader, parent, false)
        {
            this.Type = (int)Utils.ReadHex (reader ["type"]);
            ReadBlocks (fsd, reader);
        }
    }

    public class SymLink : File
    {
        public string LinkTarget;
        SymLink() : base() {}
        public SymLink (FSData fsd, XmlTextReader reader, Directory parent)
            : base (fsd, reader, parent, false)
        {
            this.LinkTarget = reader["link"];
            if (this.LinkTarget == null || this.LinkTarget == "")
                Console.WriteLine ("Serious error reading link " + this.Name + " child of " +
                                   parent.GetPath());
            ReadBlocks (fsd, reader);
        }
        public long TargetBlock {
            get {
                if (this.Blocks != null)
                    return this.Blocks[0].Start;
                else
                    return -1;
            }
        }
        public override File Copy (Directory newParent)
        {
            SymLink fi = new SymLink();
            fi.LinkTarget = this.LinkTarget;
            fi.CopyFields (this, newParent);
            return fi;
        }
    }
    public class Directory : File
    {
        // helps accelerate block based search hugely.
        public long MinBlock;
        public long MaxBlock;
        public long BlockExtent {
            get { return MaxBlock - MinBlock; }
        }

        public override void GetMinMax (out long min, out long max)
        {
            min = this.MinBlock;
            max = this.MaxBlock;
        }

        public List<File> Children; // sort order is important 

        Directory() : base() {}
        public Directory (FSData fsd, XmlTextReader reader, Directory parent)
            : base (fsd, reader, parent, true)
        {
            bool finishedDir = reader.IsEmptyElement;

            this.MinBlock = Int64.MaxValue;
            this.MaxBlock = 0;

            while (!finishedDir) 
    	    {
                reader.Read();
                XmlNodeType type = reader.NodeType;
                
                File item = null;
                switch (reader.Name) {
                case "dir":
                    if (type != XmlNodeType.EndElement)
                        item = new Directory (fsd, reader, this);
                    else
                        finishedDir = true;
                    break;
                case "file":
                    if (type != XmlNodeType.EndElement)
                        item = new File (fsd, reader, this, true);
                    break;
                case "symlink":
                    if (type != XmlNodeType.EndElement)
                        item = new SymLink (fsd, reader, this);
                    break;
                case "special":
                    if (type != XmlNodeType.EndElement)
                        item = new Special (fsd, reader, this);
                    break;
                default:
                    Console.WriteLine ("Error - unknown element '" + reader.Name + "' at "
                                       + reader.LineNumber + ":" + reader.LinePosition);
                    return;
                }
                if (item != null) {
                    if (this.Children == null) // at least . & .. required.
                        this.Children = new List<File>(4);
                    this.Children.Add (item);
                }
            }

            // calc min/max blocks
            MinMaxCalc (ref this.MinBlock, ref this.MaxBlock);
        }

        internal protected override void MinMaxCalc (ref long min, ref long max)
        {
            base.MinMaxCalc (ref min, ref max);
            if (this.Children != null) {
                foreach (File fi in this.Children) {
                    if (fi is Directory) {
                        min = Math.Min (min, (fi as Directory).MinBlock);
                        max = Math.Max (max, (fi as Directory).MaxBlock);
                    } else
                        fi.MinMaxCalc (ref min, ref max);
                }
            }
        }
        
        // For fast lookup ...
        public Dictionary<string,File> ChildHash;
        public File Lookup (string name)
        {
            if (this.ChildHash != null)
                Console.WriteLine ("FIXME: no fast hash yet ...");

            if (name == ".")
                return this;
            if (name == "..")
                return this.Parent != null ? this.Parent : this;

            if (this.Children == null)
                return null;

            // Fine for smaller directories ...
            foreach (File f in Children) {
                if (f.Name == name)
                    return f;
            }
            return null;
        }

        public void CopyChildren (Directory tmpl)
        {
            if (tmpl.Children == null)
                this.Children = null;
            else {
                this.Children = new List<File>(tmpl.Children.Count);
                foreach (File fi in tmpl.Children)
                    this.Children.Add (fi.Copy (this));
            }
        }

        public void AddChild (File fi)
        {
            if (this.Children == null)
                this.Children = new List<File>(1);
            this.Children.Add (fi);
        }

        public override File Copy (Directory newParent)
        {
            Directory fi = new Directory();
            fi.CopyFields (this, newParent);
            return fi;
        }
    } // end Directory
} // end namespace

public class FSData {
    string imageName;
    Directory root;

    public int BlockSize;
    public long MaxBlock    { get { return this.root.MaxBlock; } }
    public string ImageName { get { return this.imageName; } }
    public Directory Root   { get { return this.root; } }
        
    public void Dump (int depth, File file)
    {
        for (int i = 0; i < depth; i++)
            System.Console.Write (" ");
        System.Console.WriteLine (file.Name);
        if (file is Directory) {
            Directory dir = (Directory) file;
            if (dir.Children != null) {
                foreach (File f in dir.Children)
                    Dump (depth + 1, f);
            }
        }
    }

    // May need to show a progress bar ...
    public FSData (StatusFileStream fs, string fileName)
    {
        System.IO.Stream str = fs;
        if (fileName.EndsWith (".gz"))
            str = new ICSharpCode.SharpZipLib.GZip.GZipInputStream (fs);
        XmlTextReader reader = new XmlTextReader (str);
        reader.XmlResolver = null; // no dtd.
        reader.WhitespaceHandling = WhitespaceHandling.None;
        reader.Read();
        this.imageName = reader["name"];
        this.BlockSize = (int)Utils.ReadHex (reader ["blksize"]);
        reader.Read();

        fs.PerformIO( delegate { this.root = new Directory (this, reader, null); } );
    }

    // empty fsdata - for transfering items
    public FSData (FSData templ)
    {
        this.imageName = "subset of " + templ.imageName;
        this.BlockSize = templ.BlockSize;
        this.root = (Directory) templ.root.Copy (null);
    }

    void PrintBlocks (System.IO.StreamWriter s, List<Block> blks)
    {
        bool bFirst = true;
        foreach (Block b in blks) {
            if (bFirst)
                bFirst = false;
            else
                s.Write(", ");
            if (b.IsIndirect)
                s.Write("i");
            s.Write ("0x{0:x}", b.Start);
            if (b.Length > 1)
                s.Write ("-0x{0:x}", b.End - 1);
        }
    }

    void WriteFile (System.IO.StreamWriter s, File f)
    {
        string nodeType = "file";
        if (f is Directory)
            nodeType = "dir";
        else if (f is SymLink)
            nodeType = "symlink";
        else if (f is Special)
            nodeType = "special";
        s.Write ("<" + nodeType + " name=\"" + f.Name + "\" ");
        if (f is Special)
            s.Write ("type=\"0x{0:x}\"", (f as Special).Type);
        else
            s.Write ("size=\"0x{0:x}\"", f.Size);
        if (f is SymLink)
            s.Write (" link=\"" + ((f as SymLink).LinkTarget) + "\"");
        s.Write (" inode=\"0x{0:x}\" iblk=\"0x{1:x}\"",
                 f.INodeId, f.INodeBlock);

        if (!(f is Directory)) {
            if (f.Blocks == null)
                s.Write ("/>\n");
            else {
                s.Write (">");
                PrintBlocks (s, f.Blocks);
                s.Write ("</" + nodeType + ">\n");
            }
        } else {
            Directory dir = (Directory) f;
            s.Write ("><blks>");
            PrintBlocks (s, f.Blocks);
            s.Write ("</blks>\n");
            if (dir.Children != null) {
                foreach (File child in dir.Children)
                    WriteFile (s, child);
            }
            s.Write ("</" + nodeType + ">\n");
        }
    }

    public void Write (string fileName)
    {
        System.IO.FileStream fileStrm = new System.IO.FileStream (fileName,
                                                                  System.IO.FileMode.Create);
        System.IO.Stream strm = fileStrm;
        if (fileName.EndsWith (".gz"))
            strm = new ICSharpCode.SharpZipLib.GZip.GZipOutputStream (fileStrm);

        System.IO.StreamWriter s = new System.IO.StreamWriter (strm);
        s.WriteLine ("<root name=\"" + this.imageName + "\" blksize=\"0x{0:x}\">",
                     this.BlockSize);
        WriteFile (s, this.Root);
        s.WriteLine ("</root>");
        s.Close();
    }
}
