/* -*- tab-width: 4; c-basic-offset: 4 -*- */

using System;
using Gdk;
using Gtk;
using Pango;
using Cairo;
using System.Collections.Generic;
using System.Diagnostics;

/* This is just a brainless class that has the OnSetScrollAdjustments handler
 * defined, so that we can stick it inside a GtkScrolledWindow.  All the actual
 * brains are in the AddrView class.
 */
public class ViewScrollable : Gtk.DrawingArea {
	public ViewScrollable ()
	{
		AddEvents((int) EventMask.PointerMotionMask
				  | (int) EventMask.ScrollMask
				  | (int) EventMask.EnterNotifyMask
				  | (int) EventMask.LeaveNotifyMask);
	}

	int width;
	int height;

	public void SetSizes( int width, int height )
	{
		this.width = width;
		this.height = height;
	}

	protected override void OnSizeRequested (ref Gtk.Requisition req)
	{
		req.Width = this.width;;
		req.Height = this.height;
	}

	protected override void OnSetScrollAdjustments (Gtk.Adjustment hadj, Gtk.Adjustment vadj)
	{
		/* nothing; just a placeholder */
	}
}


public class AddrView : Gtk.Table {
	public enum ZoomMode {
		FIT,		/* View will re-scale to fit if the window is resized */
		ZOOMED		/* View will retain its zoom factor if the window is resized */
	}

	ZoomMode zoomMode;

	double pixelsPerVMPage;
	double pixelsPerTimeUnit;
	bool   viewAsBands;
    bool   viewWorkingSet;

	PosSpan selectedRegion;
    AddrMouseOver mouseOverHelper;

	void SetScale (double tm, double addr, bool haveAnchor, int anchorX, int anchorY)
	{
		Layout oldLayout;
		LayoutSettings settings;
		double oldPixelsPerTimeUnit, oldPixelsPerVMPage;

		zoomMode = ZoomMode.ZOOMED;

		oldPixelsPerTimeUnit = pixelsPerTimeUnit;
		oldPixelsPerVMPage = pixelsPerVMPage;

        pixelsPerTimeUnit = tm;
		settings.PixelsPerTimeUnit = tm;
        pixelsPerVMPage = addr;
		settings.PixelsPerVMPage = addr;
		settings.Bands = viewAsBands;

		oldLayout = this.layout;

		this.layout = new Layout (sd, settings);

		if (oldLayout != null) {
			double xRel, yRel;

			if (haveAnchor) {
				xRel = (double) anchorX / drawArea.Allocation.Width;
				yRel = (double) anchorY / drawArea.Allocation.Height;
			} else {
				xRel = 0.5;
				yRel = 0.5;
			}

			ComputeCenterZoomOffsets (oldPixelsPerTimeUnit, pixelsPerTimeUnit,
										 oldLayout.ViewTimeRange.end,
										 drawArea.Allocation.Width,
										 tmScrollOffset,
										 xRel,
										 out tmScrollOffset);

			ComputeCenterZoomOffsets (oldPixelsPerVMPage, pixelsPerVMPage,
										 oldLayout.ViewAddressRange.end,
										 drawArea.Allocation.Height,
										 vmScrollOffset,
										 yRel,
										 out vmScrollOffset);
		} else {
			tmScrollOffset = 0;
			vmScrollOffset = 0;
		}

		UpdateScrollbarValues ();
		RenderVisibleContent ();
//		Console.WriteLine ("set new scale to (time = {0}, address = {0})", tm, addr);
	}

	void ComputeCenterZoomOffsets (double oldZoom, double newZoom,
								   long oldViewArea,
								   int allocation,
								   long scrollOffset,
								   double anchor,
								   out long offset)
	{
		double newViewArea;
		double viewCenter;

		if (oldViewArea < allocation)
			viewCenter = anchor * oldViewArea / oldZoom;
		else
			viewCenter = (scrollOffset + anchor * allocation) / oldZoom;

		newViewArea = oldViewArea / oldZoom * newZoom; /* just an estimate */

		if (newViewArea < allocation)
			offset = 0;
		else {
			offset = (long) (viewCenter * newZoom - anchor * allocation + 0.5);
			if (offset < 0)
				offset = 0;
		}
	}

	void UpdateScrollbarValues ()
	{
		Gtk.Adjustment hadj, vadj;

		updatingScrollbars = true;

		hadj = scrolledWindow.Hadjustment;
		vadj = scrolledWindow.Vadjustment;

		hadj.Lower = this.layout.ViewTimeRange.start;
		hadj.Upper = this.layout.ViewTimeRange.end;
		hadj.Value = tmScrollOffset;
		hadj.PageSize = System.Math.Min (this.layout.ViewTimeRange.end, drawArea.Allocation.Width);

		if (hadj.Value < hadj.Lower)
			hadj.Value = hadj.Lower;
		else if (hadj.Value > hadj.Upper - hadj.PageSize)
			hadj.Value = hadj.Upper - hadj.PageSize;

		hadj.PageIncrement = drawArea.Allocation.Width / 2;
		hadj.StepIncrement = TMScrollStepPixels;

		vadj.Lower = this.layout.ViewAddressRange.start;
		vadj.Upper = this.layout.ViewAddressRange.end;
		vadj.Value = vmScrollOffset;
		vadj.PageSize = System.Math.Min (this.layout.ViewAddressRange.end, drawArea.Allocation.Height);

		if (vadj.Value < vadj.Lower)
			vadj.Value = vadj.Lower;
		else if (vadj.Value > vadj.Upper - vadj.PageSize)
			vadj.Value = vadj.Upper - vadj.PageSize;

		vadj.PageIncrement = drawArea.Allocation.Height / 2;
		vadj.StepIncrement = VMScrollStepPixels;

		hadj.Change ();
		vadj.Change ();

		updatingScrollbars = false;
	}

	void ZoomFit (int width, int height)
	{
        Console.WriteLine ("width " + width + " height " + height);
        Console.WriteLine ("layout " + this.layout.TimeRange.end + " - " + 
                           this.layout.AddressRange.end);
		SetScale ((double) width / this.layout.TimeRange.end,
				  (double) height / this.layout.AddressRange.end,
				  false, 0, 0);
		zoomMode = ZoomMode.FIT;
	}

	void ZoomFitWindow ()
	{
		ZoomFit (scrolledWindow.Allocation.Width - 2 * scrolledWindow.Style.Xthickness,
				 scrolledWindow.Allocation.Height - 2 * scrolledWindow.Style.Ythickness);
	}

	const double VMZoomMultiplier = 1.1;
	const double TMZoomMultiplier = 1.1;

	const long VMScrollStepPixels = 64;
	const long TMScrollStepPixels = 64;

	public AddrView (SimulationData sd) :
		base (2, 3, false)
	{
		if (sd == null)
			throw new ArgumentNullException ("simulator");

		this.sd = sd;
		selectedRegion = new PosSpan (0, sd.Elapsed);

        zoomMode = ZoomMode.FIT;

		ColumnSpacing = 6;
		RowSpacing = 6;
		viewAsBands = false;
        viewWorkingSet = false;

		cursorActive = false;

		CreateVMWidgets ();
		CreateTMWidgets ();
		CreateFitWidget ();
		CreateDrawArea ();
		CreateStatusBar ();

        this.mouseOverHelper = new AddrMouseOver (this);

		UpdateSelection ();
	}

	void CreateVMWidgets ()
	{
		Gtk.Label label;
		Gtk.VBox box;

		box = new Gtk.VBox (false, 6);
		Attach (box,
			0, 1,
			1, 2,
			0, Gtk.AttachOptions.Expand | Gtk.AttachOptions.Fill,
			0, 0);

		label = new Gtk.Label ("VM address (Shift+wheel to zoom)");
		label.Angle = 90;
		box.PackStart (label, false, false, 0);

		GUtils.AddButton (Gtk.Stock.ZoomOut, box,
                          delegate { SetScale (pixelsPerTimeUnit, pixelsPerVMPage / VMZoomMultiplier, false, 0, 0); });
		GUtils.AddButton (Gtk.Stock.ZoomIn, box,
                          delegate { SetScale (pixelsPerTimeUnit, pixelsPerVMPage * VMZoomMultiplier, false, 0, 0); });

		box.ShowAll ();
	}

	void CreateTMWidgets ()
	{
		Gtk.Label label;
		Gtk.HBox box;

		box = new Gtk.HBox (false, 6);
		Attach (box,
			1, 2,
			0, 1,
			Gtk.AttachOptions.Expand | Gtk.AttachOptions.Fill, 0,
			0, 0);

		label = new Gtk.Label ("Time (Ctrl+wheel to zoom)");
		box.PackStart (label, false, false, 0);

		GUtils.AddButton (Gtk.Stock.ZoomOut, box,
                          delegate { SetScale (pixelsPerTimeUnit / TMZoomMultiplier, pixelsPerVMPage, false, 0, 0); });
		GUtils.AddButton (Gtk.Stock.ZoomIn, box,
                          delegate { SetScale (pixelsPerTimeUnit * TMZoomMultiplier, pixelsPerVMPage, false, 0, 0); });

		elapsedDisplay = new Gtk.Label ("");
		box.PackStart (elapsedDisplay, false, false, 0);

		Gtk.ToggleButton tg = new Gtk.CheckButton("expand addresses");
		tg.Active = !this.viewAsBands;
		tg.Toggled += delegate {
			this.viewAsBands = !tg.Active;
			ZoomFitWindow ();
		};
		box.PackStart (tg, false, false, 0);

		tg = new Gtk.CheckButton("working set");
		tg.Active = this.viewWorkingSet;
		tg.Toggled += delegate {
            Console.WriteLine ("do we have thunks ? " + sd.Thunks.Count);
            if (tg.Active && sd.HasNoThunks) {
                Gtk.MessageDialog error;
                error = new Gtk.MessageDialog (null, Gtk.DialogFlags.Modal,
                                               Gtk.MessageType.Warning,
                                               Gtk.ButtonsType.Ok, true,
                                               "<b>Error</b> - the trace you generated doesn't seem to contain working set " +
                                               " information - please re-capture the trace with --enable-working-set=yes");
                error.DeleteEvent += delegate { error.Destroy(); };
                error.Response += delegate { error.Destroy(); };
                error.ShowAll();
                tg.Active = false;
            } else {
                this.viewWorkingSet = tg.Active;
                ZoomFitWindow ();
            }
		};
		box.PackStart (tg, false, false, 0);

		box.ShowAll ();
	}

	void CreateFitWidget ()
	{
		Gtk.Button button;

		button = new Gtk.Button ();
		button.Image = new Gtk.Image (Gtk.Stock.ZoomFit, IconSize.SmallToolbar);
		button.Clicked += delegate { ZoomFitWindow (); };
		button.ShowAll();
		Attach (button,
				0, 1,
				0, 1,
				Gtk.AttachOptions.Fill, Gtk.AttachOptions.Fill,
				0, 0);
	}

	void CreateDrawArea ()
	{
		Gtk.Adjustment hadj, vadj;

		drawArea = new ViewScrollable ();

		scrolledWindow = new Gtk.ScrolledWindow (null, null);
		scrolledWindow.SetPolicy (Gtk.PolicyType.Automatic, Gtk.PolicyType.Automatic);
		scrolledWindow.ShadowType = Gtk.ShadowType.In;
		scrolledWindow.Add (drawArea);

		Attach (scrolledWindow,
				1, 2,
				1, 2,
				Gtk.AttachOptions.Fill,
				Gtk.AttachOptions.Fill,
				0, 0);

		drawArea.SizeAllocated += DrawAreaSizeAllocatedCallback;
		drawArea.Realized += DrawAreaRealizedCallback;
		drawArea.Mapped += DrawAreaMappedCallback;
		drawArea.ExposeEvent += DrawAreaExposeEventCallback;
		drawArea.ScrollEvent += DrawAreaScrollEventCallback;
		drawArea.EnterNotifyEvent += DrawAreaEnterNotifyEventCallback;
		drawArea.LeaveNotifyEvent += DrawAreaLeaveNotifyEventCallback;
		drawArea.MotionNotifyEvent += DrawAreaMotionEventCallback;

		hadj = scrolledWindow.Hadjustment;
		vadj = scrolledWindow.Vadjustment;

		hadj.ValueChanged += AdjustmentValueChangedCallback;
		vadj.ValueChanged += AdjustmentValueChangedCallback;

		scrolledWindow.ShowAll ();
	}

	void AdjustmentValueChangedCallback (object o, EventArgs args)
	{
		Gtk.Adjustment hadj, vadj;

		if (updatingScrollbars)
			return;

		hadj = scrolledWindow.Hadjustment;
		vadj = scrolledWindow.Vadjustment;

		vmScrollOffset = (long) vadj.Value;
		tmScrollOffset = (long) hadj.Value;

		RenderVisibleContent ();
	}

	Gtk.Statusbar statusBar;
	void SetStatusPosition (long time, long vOffset,
							SimMap simMap)
	{
		/* FIXME: really need to have separate (labels?) packed in here */
		string mapStr = "";
		if (simMap != null)
			mapStr = " Map: " + simMap.FileName +
				" - " + Utils.Percentage (simMap.Percentage);
		string addrStr = "";
		long   addr = this.layout.GetAddressAtOffset (vOffset);
		if (addr >= 0)
			addrStr = " Addr: " + String.Format ("0x{0:x08}", addr);
		string str = "Time: " + Utils.PrettyTime(time) + addrStr + mapStr;
		statusBar.Push (0, str);
	}

	void CreateStatusBar ()
	{
		statusBar = new Statusbar();
		statusBar.Show();
		Attach (statusBar,
				0, 2,
				2, 3,
				Gtk.AttachOptions.Fill,
				Gtk.AttachOptions.Fill,
				0, 0);
	}

	void DrawAreaSizeAllocatedCallback (object obj, SizeAllocatedArgs args)
	{
		if (zoomMode == ZoomMode.FIT) {
			LayoutSettings settings;

			/* Hack; use a meaningless zoom just to obtain the extents */

			settings.PixelsPerTimeUnit = 1.0;
			settings.PixelsPerVMPage = 1.0;
			settings.Bands = viewAsBands;

			this.layout = new Layout (sd, settings);

			ZoomFitWindow ();
		} else {
			UpdateScrollbarValues ();
		}

		Console.WriteLine ("rendering visible content in size_allocate");
		RenderVisibleContent ();
		Console.WriteLine ("end size allocate: visibleContent = {0}", (visibleContent != null) ? true : false);
	}

	void SetEmptyCursor ()
	{
		Pixbuf pixbuf;

		pixbuf = new Pixbuf (Gdk.Colorspace.Rgb, true, 8, 1, 1);
		pixbuf.Fill (0x0);

		drawArea.GdkWindow.Cursor = new Gdk.Cursor (Display, pixbuf, 0, 0);
	}

	void RenderVisibleContent ()
	{
		Cairo.Context cr;

		if (!this.drawArea.IsMapped || this.layout == null)
			return;

		if (this.visibleContent == null
			|| visibleContentWidth != drawArea.Allocation.Width
			|| visibleContentHeight != drawArea.Allocation.Height) {
			visibleContentWidth = drawArea.Allocation.Width;
			visibleContentHeight = drawArea.Allocation.Height;

			this.visibleContent = new Gdk.Pixmap (drawArea.GdkWindow, visibleContentWidth, visibleContentHeight);
		}

		if (this.layout.TimeRange.end <= 0 || this.layout.AddressRange.end <= 0)
		{
			Console.WriteLine ("Degenerate time (" + this.layout.TimeRange.end +
							   ") address (" + this.layout.AddressRange.end + ")");
		}

		using (cr = Gdk.CairoHelper.Create (this.visibleContent)) {
			DrawBackground (cr);
			DrawMaps (cr);
            if (this.viewWorkingSet)
                DrawWorkingSet (cr);
			DrawEvents (cr);
		}

		this.drawArea.QueueDraw ();
	}

	void DrawAreaRealizedCallback (object o, EventArgs args)
	{
		SetEmptyCursor ();
	}

	void DrawAreaMappedCallback (object o, EventArgs args)
	{
		RenderVisibleContent ();
	}

	void DrawAreaEnterNotifyEventCallback (object o, EnterNotifyEventArgs args)
	{
		cursorActive = true;
		cursorX = (int) args.Event.X;
		cursorY = (int) args.Event.Y;

		RequestDrawCursor (cursorX, cursorY, cursorX, cursorY);

		args.RetVal = true;
	}

	void DrawAreaLeaveNotifyEventCallback (object o, LeaveNotifyEventArgs args)
	{
		cursorActive = false;

		RequestDrawCursor (cursorX, cursorY, cursorX, cursorY);

		args.RetVal = true;
	}

    uint mouseoverTimeout;
    EventMotion mouseoverEvent;

	bool MouseoverEventCallback ()
	{
		double posx = this.mouseoverEvent.X / pixelsPerTimeUnit;
		double posy = this.mouseoverEvent.Y / pixelsPerVMPage;

//		Console.WriteLine ("Pos " + posx + " " + posy + " time: "
//						   + Utils.PrettyTime ((long)posx) + " addr " + posy);
		SimMap simMap = null;
		Layout.Map m = new Layout.Map();
		if (this.layout.GetMapAt ((long) mouseoverEvent.X, (long)mouseoverEvent.Y, ref m))
			simMap = m.SimMap;

		SetStatusPosition ((long) posx, (long) posy, simMap);

		return false;
	}

	void DrawAreaMotionEventCallback (object obj, MotionNotifyEventArgs args)
	{
		int cursorNewX, cursorNewY;

		// Set cursor to drag / hand if ctrl pressed ?
		this.mouseoverEvent = new EventMotion (args.Event);

		if (mouseoverTimeout != 0)
			GLib.Source.Remove (mouseoverTimeout);
		mouseoverTimeout = GLib.Timeout.Add (10, new GLib.TimeoutHandler (MouseoverEventCallback));

		cursorNewX = (int) args.Event.X;
		cursorNewY = (int) args.Event.Y;

		RequestDrawCursor (cursorX, cursorY, cursorNewX, cursorNewY);
		cursorX = cursorNewX;
		cursorY = cursorNewY;

		args.RetVal = false;
	}

	public struct Rect {
		public long x, y, w, h;

		public static bool Intersect (Rect a, Rect b, out Rect dest)
		{
			bool retval;

			dest.x = System.Math.Max (a.x, b.x);
			dest.y = System.Math.Max (a.y, b.y);
			dest.w = System.Math.Min (a.x + a.w, b.x + b.w) - dest.x;
			dest.h = System.Math.Min (a.y + a.h, b.y + b.h) - dest.y;

			if (dest.w > 0 && dest.h > 0)
				retval = true;
			else {
				dest.w = 0;
				dest.h = 0;
				retval = false;
			}

			return retval;
		}
	}

	void FillClippedRectangle (Cairo.Context cr,
							   long x, long y, long w, long h)
	{
		Rect r, a, dest;

		/* Translate to window coordinates */

		r.x = x - tmScrollOffset;
		r.y = y - vmScrollOffset;
		r.w = w;
		r.h = h;

		/* Clip into the visible area */

		a.x = 0;
		a.y = 0;
		a.w = visibleContentWidth;
		a.h = visibleContentHeight;

		if (Rect.Intersect (r, a, out dest)) {
			cr.Rectangle (dest.x, dest.y, dest.w, dest.h);
			cr.Fill ();
		}
	}

	void DrawBackground (Cairo.Context cr)
	{
		cr.Color = new Cairo.Color (1, 1, 1, 1);
		cr.Rectangle (0, 0, visibleContentWidth, visibleContentHeight);
		cr.Fill ();
	}

	void DrawMaps (Cairo.Context cr)
	{
		int i;
		Rect viewArea;

		viewArea.x = 0;
		viewArea.y = 0;
		viewArea.w = visibleContentWidth;
		viewArea.h = visibleContentHeight;

		for (i = 0; i < this.layout.Maps.Length; i++) {
			Rect rect, dest;

			cr.Color = this.layout.GetColor (this.layout.Maps[i]);
			FillClippedRectangle (cr,
								  this.layout.Maps[i].tm.start,
								  this.layout.Maps[i].vm.start,
								  this.layout.Maps[i].tm.end - this.layout.Maps[i].tm.start,
								  this.layout.Maps[i].vm.end - this.layout.Maps[i].vm.start);

			rect.x = this.layout.Maps[i].tm.start - tmScrollOffset;
			rect.y = this.layout.Maps[i].vm.start - vmScrollOffset;
			rect.w = this.layout.Maps[i].tm.end - this.layout.Maps[i].tm.start;
			rect.h = this.layout.Maps[i].vm.end - this.layout.Maps[i].vm.start;

			if (Rect.Intersect (rect, viewArea, out dest)) {
				/* Now, draw the text */

				TextExtents extents;

				cr.Color = new Cairo.Color (0, 0, 0, 1);

				cr.SetFontSize (16);
				extents = cr.TextExtents (this.layout.Maps[i].Label);
				if (extents.Height <= dest.h) {
					cr.MoveTo (dest.x, dest.y + extents.Height);
					cr.ShowText (this.layout.Maps[i].Label);
				}

				/* Sigh, pango-sharp in SLED 10 doesn't yet have Pango.CairoHelper...
				Pango.Layout layout;
				Pango.Rectangle inkRect, logRect;

				layout = Pango.CairoHelper.CreateLayout (cr);
				this.layout.SetText (this.layout.Maps[i].Label);
				this.layout.GetPixelExtents (out inkRect, out logRect);

				if (inkRect.Y + inkRect.Height <= dest.h) {
					cr.MoveTo (dest.x - inkRect.X, dest.y - inkRect.Y);
					Pango.CairoHelper.ShowLayout (cr, layout);
				}
				*/
			}
		}
	}

	void DrawEvents (Cairo.Context cr)
	{
		int i;

		for (i = 0; i < this.layout.Events.Length; i++) {
			cr.Color = this.layout.GetColor (this.layout.Events[i]);
			FillClippedRectangle (cr,
								  this.layout.Events[i].tm.start,
								  this.layout.Events[i].vm.start,
								  this.layout.Events[i].tm.end - this.layout.Events[i].tm.start,
								  this.layout.Events[i].vm.end - this.layout.Events[i].vm.start);
		}
	}

	void DrawWorkingSet (Cairo.Context cr)
	{
        cr.Color = new Cairo.Color (0, 0, 0, 1);

		foreach (SimEvent sime in sd.Thunks)
        {
            Trace.Thunk thunk = (Trace.Thunk) sime.Detail;
            foreach (Trace.ThunkSpan span in thunk.SpanEnds) {
                SimEvent start = sd.Thunks[(int)span.StartId];
                long tm_start = (long)(start.StartTime * this.layout.settings.PixelsPerTimeUnit);
                long tm_end = (long)(sime.EndTime * this.layout.settings.PixelsPerTimeUnit) + 1;
                PosSpan screenSpan = this.layout.GetOffsetAtAddress (span.Region);
                long vm_start = (long)(screenSpan.start * this.layout.settings.PixelsPerVMPage);
                long vm_end = (long)(screenSpan.end * this.layout.settings.PixelsPerVMPage) + 1;
                FillClippedRectangle (cr, tm_start, vm_start, tm_end - tm_start,
                                      vm_end - vm_start);
            }
		}
	}

	void RequestDrawCursor (int oldX, int oldY, int newX, int newY)
	{
		Gdk.Rectangle r;

		r.X = oldX;
		r.Y = 0;
		r.Width = 1;
		r.Height = drawArea.Allocation.Height;
		drawArea.GdkWindow.InvalidateRect (r, false);

		r.X = 0;
		r.Y = oldY;
		r.Width = drawArea.Allocation.Width;
		r.Height = 1;
		drawArea.GdkWindow.InvalidateRect (r, false);

		r.X = newX;
		r.Y = 0;
		r.Width = 1;
		r.Height = drawArea.Allocation.Height;
		drawArea.GdkWindow.InvalidateRect (r, false);

		r.X = 0;
		r.Y = newY;
		r.Width = drawArea.Allocation.Width;
		r.Height = 1;
		drawArea.GdkWindow.InvalidateRect (r, false);
	}

	void DrawCursor (Cairo.Context cr)
	{
		if (!cursorActive)
			return;

		cr.Color = new Cairo.Color (0, 0, 0, 1);
		cr.LineWidth = 1.0;

		cr.MoveTo (cursorX + 0.5, 0);
		cr.LineTo (cursorX + 0.5, drawArea.Allocation.Height);
		cr.Stroke ();

		cr.MoveTo (0, cursorY + 0.5);
		cr.LineTo (drawArea.Allocation.Width, cursorY + 0.5);
		cr.Stroke ();
	}

	void DrawAreaExposeEventCallback (object obj, ExposeEventArgs args)
	{
		Cairo.Context cr;
		Gdk.Rectangle[] rectangles;

//		Console.WriteLine ("start expose: visibleContent = {0}", (visibleContent != null) ? true : false);

		rectangles = args.Event.Region.GetRectangles ();
		foreach (Gdk.Rectangle r in rectangles) {
			drawArea.GdkWindow.DrawDrawable (drawArea.Style.BlackGC,
											 visibleContent,
											 r.X, r.Y,
											 r.X, r.Y,
											 r.Width, r.Height);
		}

		using (cr = Gdk.CairoHelper.Create (drawArea.GdkWindow)) {
			DrawCursor (cr);
		}
//		Console.WriteLine ("end expose");

		args.RetVal = false;
	}

	void DrawAreaScrollEventCallback (object obj, ScrollEventArgs args)
	{
		bool consumed = false;

		if ((args.Event.State & Gdk.ModifierType.ShiftMask) != 0) {
			switch (args.Event.Direction) {
			case Gdk.ScrollDirection.Up:
				SetScale (pixelsPerTimeUnit, pixelsPerVMPage * VMZoomMultiplier,
						  true, (int) args.Event.X, (int) args.Event.Y);
				consumed = true;
				break;
			case Gdk.ScrollDirection.Down:
				SetScale (pixelsPerTimeUnit, pixelsPerVMPage / VMZoomMultiplier,
						  true, (int) args.Event.X, (int) args.Event.Y);
				consumed = true;
				break;
			}
		}

		if ((args.Event.State & Gdk.ModifierType.ControlMask) != 0) {
			switch (args.Event.Direction) {
			case Gdk.ScrollDirection.Up:
				SetScale (pixelsPerTimeUnit * TMZoomMultiplier, pixelsPerVMPage,
						  true, (int) args.Event.X, (int) args.Event.Y);
				consumed = true;
				break;

			case Gdk.ScrollDirection.Down:
				SetScale (pixelsPerTimeUnit / TMZoomMultiplier, pixelsPerVMPage,
						  true, (int) args.Event.X, (int) args.Event.Y);
				consumed = true;
				break;
			}
		}

		args.RetVal = consumed;
	}

	void UpdateSelection ()
	{
		elapsedDisplay.Text =
			"Elapsed : " +
			Utils.PrettyTime( selectedRegion.end - selectedRegion.start);
	}

	SimulationData sd;

	public Layout layout;
	Gtk.Label     elapsedDisplay;

	Gtk.ScrolledWindow scrolledWindow;
	ViewScrollable drawArea;
	Gdk.Pixmap visibleContent;
	int visibleContentWidth, visibleContentHeight;

	long vmScrollOffset;
	long tmScrollOffset;

	int cursorX;
	int cursorY;
	bool cursorActive;

	bool updatingScrollbars;
}

class AddrMouseOver : MouseOverHelper {
    AddrView view;
    public AddrMouseOver (AddrView view) : base (view)
    {
        this.view = view;
    }
    public override object GetItemAt (long x, long y)
    {
        Layout.Event e;
        if (view.layout.GetEventAt (x, y, out e)
            && e.ev.Detail != null
            && e.ev.Detail is Trace.TraceEvent)
            return e.ev;
        return null;
    }
    public override void PopulatePopup (Gtk.Window popup, object obj)
    {
        SimEvent e = (SimEvent) obj;
        Trace.Stack.Frame frame = (e.Detail as Trace.TraceEvent).Frame;
		popup.Add (new BackTraceView (frame));
    }
}
