/* -*- tab-width: 4; c-basic-offset: 4 -*- */

using System;
using System.IO;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Globalization;

internal class LineParser {
    int curPos;
    string line;
    bool IsWhite (char c) { return c == ' ' || c == '\t'; }
    string NextToken()
    {
        if (this.curPos >= this.line.Length)
            return null;

        bool in_quote = this.line[this.curPos] == '\'';
        if (in_quote)
            this.curPos++;

        int i;
        for (i = this.curPos; i < this.line.Length; i++) {
            if ((!in_quote && IsWhite (this.line[i])) ||
                ( in_quote && this.line[i] == '\'' ))
                break;
        }
        int len = i - this.curPos;
        string ret = this.line.Substring (this.curPos, len);
        this.curPos = i;
        if (in_quote) this.curPos++; // skip quote
        // skip whitespace
        for (; this.curPos < this.line.Length; this.curPos++) {
            if (!IsWhite(this.line[this.curPos]))
                break;
        }
        return ret;
    }
    public string [] ParseString (string line)
    {
        this.line = line;
        this.curPos = 0;
        List<string> elements = new List<string>();
        string tok;
        do {
            tok = NextToken();
            if (tok != null)
                elements.Add (tok);
        } while (tok != null);
        return elements.ToArray();
    }
}

namespace Trace {

public class Stack {
    Stack<Frame> frames;

    public class Frame {
        string name;
        Frame parent;
        Dictionary<string,Frame> children;

        public Frame (string name, Frame parent)
        {
            this.name = name;
            this.parent = parent;
            if (parent != null)
                parent.Add (this);
        }
        public Frame Lookup (string name)
        {
            if (this.children == null ||
                !this.children.ContainsKey(name))
                return null;
            return this.children[name];
        }
        public void Add (Frame child)
        {
            if (this.children == null)
                this.children = new Dictionary<string,Frame>();
            this.children.Add (child.name, child);
        }
        public string Name {
            get { return this.name; }
        }
        public Frame Parent {
            get { return this.parent; }
        }

		public string DisplayName {
			get {
				int colon_pos;
				string filename;
				string func;

				colon_pos = name.IndexOf (':');
				if (colon_pos == -1) {
					filename = name;
					func = "";
				} else {
					filename = name.Substring (0, colon_pos);
					func = name.Substring (colon_pos + 1);
				}

				filename = Utils.PathBaseName (filename);

				if (func == "")
					return filename;
				else
					return String.Format ("{0} - {1}", func, filename);
			}
		}
    }
    public Stack()
    {
        this.frames = new Stack<Frame>();
        this.frames.Push (new Frame ("<system>", null));
    }
    public void Pop(int depth)
    {
//        System.Console.WriteLine ("Pop " + depth);
        if (depth >= this.frames.Count) {
            System.Console.WriteLine ("Error: pop too many frames "
                                      + depth + " vs. " + this.frames.Count);
            depth = this.frames.Count - 1;
        }
        for (; depth > 0; depth--)
            this.frames.Pop();
    }
    public void Push (string name)
    {
//        System.Console.WriteLine ("Push " + name);
        Frame frame = this.frames.Peek().Lookup (name);
        if (frame == null)
            frame = new Frame (name, this.frames.Peek());
        this.frames.Push (frame);
/*        {
            Frame[] here = this.frames.ToArray();
            for (int i = 0; i < here.Length; i++)
                System.Console.WriteLine ("\t" + here[i].Name);
                } */
    }
    public Stack.Frame Current {
        get {
//            Console.WriteLine ("event in " + this.frames.Peek().Name);
            return this.frames.Peek();
        }
    }
};

public class Process
{
    ArrayList ioEvents;
    Thunk     lastThunk;
    public Stack Stk;

    public Process()
    {
        this.ioEvents = new ArrayList (1024);
        this.Stk = new Stack();
        Thunk (0);
    }
    public ArrayList IOEvents {
        get { return this.ioEvents; }
    }
    public void AddEvent (object ev)
    {
        this.ioEvents.Add (ev);
    }
    public void Thunk (int id)
    {
        this.lastThunk = new Thunk();
        this.lastThunk.Id = id;
        this.ioEvents.Add (this.lastThunk);
    }
    public void AddThunkSpan (ThunkSpan span)
    {
        this.lastThunk.SpanEnds.Add (span);
    }
}

public enum PageTouchKind { LOAD, STORE, TRACEDEBUG };
public interface TraceEvent {
    Stack.Frame Frame { get; }
}
  
public class PageTouch : MemRegion, TraceEvent
{
    public PageTouchKind Kind;

    Stack.Frame frame;
    public Stack.Frame Frame {
        get { return this.frame; }
    }
    public PageTouch (PageTouchKind kind, Stack stk)
    {
        this.Kind = kind;
        this.frame = stk.Current;
    }

    public override string ToString()
    {
        return "pagetouch: " + this.Kind;
    }
}

public enum MapEventKind { MAP, UNMAP };
public class MapEvent : MemRegion, TraceEvent {
    public MapEventKind Kind; 
    public string       FileName;
    public long         FileOffset;

    Stack.Frame frame;
    public Stack.Frame Frame {
        get { return this.frame; }
    }
    public MapEvent (MapEventKind kind, long start, long end,
                     string fileName, long offset, Stack stk)
        : base (start, end)
    {
        this.Kind = kind;
        this.FileName = String.Intern (fileName);
        this.frame = stk.Current;
        this.FileOffset = offset;
    }

    public override string ToString()
    {
        return "map: " + this.FileName + " t:" + this.Kind;
    }
}

public struct ThunkSpan {
    public MemRegion Region;
    public long StartId;
}

public class Thunk {
    public long Id;
    public List<ThunkSpan> SpanEnds;
    public Thunk()
    {
        Id = 0;
        SpanEnds = new List<ThunkSpan>();
    }
    
}

public enum IOEventType : int {
    Open     = 0,
    Create   = 1,
    Truncate = 2,
    Close    = 5,

    Seek       = 10,
    ReadBlock  = 11,
    ReadDEnts  = 12,
    ReadAhead  = 13,
    WriteBlock = 14,

    ReadINode = 21,
    Chdir     = 31,

    // flags
    Direct      = 0x100,
    Synchronous = 0x200,
};

public class IOEvent : TraceEvent {
    public int FileHandle;   // IO happens on either file handles
    public string FileName;  // or filenames ...
    Stack.Frame frame;
    IOEventType type;
    public Stack.Frame Frame {
        get { return this.frame; }
    }
    public IOEvent (IOEventType type, string fileName, int handle, Stack stk)
    {
        this.type = type;
        this.frame = stk.Current;
        this.FileName = fileName == null ? null : String.Intern (fileName);
        this.FileHandle = handle;
    }
    public IOEvent (IOEventType type, string fileName, Stack stk)
        : this (type, fileName, -1, stk) {}

    public IOEventType Type { get { return type; } }

    public override string ToString()
    {
        return "ioe: '" + this.FileName + "'(" + this.FileHandle + ")" +
            " t:" + this.type;
    }
}

public class OpenEvent : IOEvent {

    // eg. Open: 'name' [] = 3
    public OpenEvent (IOEventType type, string[] elements, Stack stk)
        : base (type, elements[1], Int32.Parse (elements[4]), stk)
    {   // TODO: store flags elements[2]
    }
    public bool Failed { get { return this.FileHandle < 0; } }
    public int  Result   { get { return this.FileHandle; } }
    public bool IsDirect { get { return (this.Type & IOEventType.Direct) != 0; }}
    public bool IsSync   { get { return (this.Type & IOEventType.Synchronous) != 0; }}
}

public class BlockEvent : IOEvent {
    public long Start;
    public long Length;
    public BlockEvent (IOEventType type, int fd,
                       long start, long length, Stack stk)
        : base (type, null, fd, stk)
    {
        this.Start = start;
        this.Length = length;
    }
}

public class SeekEvent : IOEvent {
    public long Offset;
    public SeekEvent (int fd, long offset, Stack stk)
        : base (IOEventType.Seek, null, fd, stk)
    {
        this.Offset = offset;
    }
}

public class Data
{		
    List<Process> processes;

    // FIXME: hack ...
    public List<Process> Processes {
        get { return this.processes; }
    }

    static string UnEscapeFilename (string name)
    {
        return name.Trim('\'');
    }

    void AddEvent (int processId, object ev)
    {
        this.processes[processId].AddEvent (ev);
    }

    public Data(string filename)
    {
        FileStream fs = new FileStream(filename, FileMode.Open);
        StreamReader a = new StreamReader(fs);

        string str;
        int curProcess = 0;
        Stack stk;

        this.processes = new List<Process>(1);
        this.processes.Add (new Process());

        LineParser parse = new LineParser();
        for (str = a.ReadLine(); str != null; str = a.ReadLine())
        {
            // Skip valgrind headers ==<pid>== foo baa
            if (str.Length <= 0 || str.StartsWith("=="))
                continue;
            
            stk = this.processes[curProcess].Stk;

            string[] elements = parse.ParseString (str);
            char type = str[0];
            switch (elements[0]) {
            // comments & debugging 
            case "#":
                break;
            case "T:": { // for debugging layout etc.
                PageTouch ioe = new PageTouch(PageTouchKind.TRACEDEBUG, stk);
                if (elements.Length > 1) {
                    ioe.Start = Utils.ReadHex (elements[1]);
                    ioe.End = ioe.Start + 4; // small read.
                }
                AddEvent (curProcess, ioe);
                break;
            }
            // Stack Frames
            case "pop:":
                stk.Pop (Int32.Parse (elements[1]));
                break;
            case "fn:":
                stk.Push (elements[1]);
                break;

            // Memory mapping / page touch events
            case "L:":
            case "S:":
                PageTouch ioe = new PageTouch(type == 'L' ? PageTouchKind.LOAD : PageTouchKind.STORE, stk);
                ioe.Start = Utils.ReadHex (elements[1]);
                ioe.End = ioe.Start + 4; // small read.
                AddEvent (curProcess, ioe);
                break;
            case "MI:": // initial memory map
            case "M:":
            case "U:":
                AddEvent (curProcess, new MapEvent (type == 'M' ? MapEventKind.MAP : MapEventKind.UNMAP,
                                                    Utils.ReadHex (elements[1]), Utils.ReadHex (elements[2]),
                                                    elements.Length > 3 ? UnEscapeFilename (elements[3]) : "",
                                                    elements.Length > 4 ? Utils.ReadHex (elements[4]) : 0,
                                                    stk));
                break;
            case "A:":
            case "F:":
                AddEvent (curProcess, new MapEvent (type == 'A' ? MapEventKind.MAP : MapEventKind.UNMAP,
                                                    Utils.ReadHex (elements[1]), Utils.ReadHex (elements[2]),
                                                    "<heap>", 0, stk));
                break;
                
                // Explicit progamatic I/O events
            case "Open:":
                AddEvent (curProcess, new OpenEvent (IOEventType.Open, elements, stk));
                break;
            case "Creat:":
                AddEvent (curProcess, new OpenEvent (IOEventType.Create, elements, stk));
                break;
            case "Trunc:":
                AddEvent (curProcess, new OpenEvent (IOEventType.Truncate, elements, stk));
                break;
            case "RdInode:":
                AddEvent (curProcess, new IOEvent (IOEventType.ReadINode, elements[1], stk));
                break;
            case "BlockRw:":
                IOEventType et;
                switch (elements[4][1]) {
                case 'r': et = IOEventType.ReadBlock; break;
                case 'A': et = IOEventType.ReadAhead; break;
                case 'w': et = IOEventType.WriteBlock; break;
                case 'D': et = IOEventType.ReadDEnts; break;
                default:
                    Console.WriteLine ("Unknown block I/O flag " + elements[4]);
                    et = IOEventType.ReadBlock;
                    break;
                }
                BlockEvent bev = new BlockEvent (et,
                                                 Int32.Parse (elements[1]),   // fd
                                                 Utils.ReadHex (elements[2]), // start
                                                 Int32.Parse (elements[3]),   // len
                                                 stk);
                AddEvent (curProcess, bev);
                break;
            case "Seek:":
                SeekEvent sev = new SeekEvent (Int32.Parse (elements[1]),   // fd
                                               Int64.Parse (elements[1]),   // offset
                                               stk);
                AddEvent (curProcess, sev);
                break;
            case "thunk:":
                this.processes[curProcess].Thunk (Int32.Parse (elements[1]));
                break;
            case "thread:":
                int threadIdx = Int32.Parse (elements[1]);
                if (threadIdx < 1) // thread 0 - bootstrapping
                    threadIdx = 1;
                // FIXME: handle discontiguous thread ids ?
                while (threadIdx > this.processes.Count)
                    this.processes.Add (new Process());
                curProcess = threadIdx - 1;
                break;
            case "cwd:":
                AddEvent (curProcess, new IOEvent (IOEventType.Chdir, elements[1], stk));
                break;
            case "page:": {
                ThunkSpan span = new ThunkSpan();
                long start = Utils.ReadHex (elements[1]);
                long end = Utils.ReadHex (elements[2]);
                span.Region = new MemRegion (start, end);
                span.StartId = Int32.Parse (elements[3]);
                processes[curProcess].AddThunkSpan (span);
                break;
            }
            default:
                Console.WriteLine("Unknown event '" + str + "'");
                break;
            }
        }
    }
}

} // end namespace Trace
