/* -*- tab-width: 4; c-basic-offset: 4 -*- */

using System;
using Trace;
using System.Collections;
using System.Collections.Generic;

public interface SimCost
{
	long TotalTime  { get; set; }
	long TotalBytes { get; set; }
}

// Used for logging low-level I/O block events
public class SimBlk
{
    public SimEvent Ev;
    public string   Name;
    public long     FsBlkStart;
    public int      FsBlkLen;

    // Merely to improve display readability
    public bool IsSequence {
        get { return FsBlkStart < 0; }
    }
    public string BlockName {
        get { return this.Name != null ? this.Name : "<null>"; }
    }
    public string LocationStr {
        get {
            if (!IsSequence)
                return String.Format ("0x{0:x}", this.FsBlkStart);
            else
                return "<seq>";
        }
    }
    public string TimeStr {
        get {
            if (!IsSequence)
                return Utils.PrettyTime (this.Ev.Elapsed);
            else
                return "";
        }
    }
}

public class SimEvent
{
    public Object Detail;
    public long   StartTime;
    public long   EndTime;
    
    public bool IsPageTouch
    {   get { return this.Detail is PageTouch; } }
    public bool IsThunk
    {   get { return this.Detail is Thunk; } }
    public MapEvent GetAsMap()
    {   return (MapEvent) this.Detail; }
    public PageTouch GetAsPageTouch()
    {   return (PageTouch) this.Detail; }
    public PosSpan Times {
        get { return new PosSpan( this.StartTime, this.EndTime ); }
    }
    public long Elapsed {
        get { return this.EndTime - this.StartTime; }
    }
    public bool Defunct {
        get { return this.StartTime < 0; }
    }
    public void SetDefunct() {
        this.StartTime = this.EndTime = -1;
    }
    public override string ToString()
    {
        string ret = "SimEvent: " + this.StartTime
            + "->" + this.EndTime;
        if (Detail == null)
            ret += " <null>";
        else
            ret += " " + Detail.ToString();
        return ret;
    }
}

public class SimMap {
    public MapEvent Map;
    public long     MapTime;
    public MapEvent Unmap;
    public long     UnmapTime;
    public SimFile  File;
    public PosSpan  MapDuration {
        get { return new PosSpan( MapTime, UnmapTime ); }
    }
    public string FileName { get { return File.FileNameAtOpen; } }
    // We may want to do per-map accounting later
    public long   TotalTime  {
        get { return File.TotalTime; }
        set { File.TotalTime = value; }
    }
    public long   TotalBytes {
        get { return File.TotalBytes; }
        set { File.TotalBytes = value; }
    }
    public double Percentage {
        get { return File.Percentage; }
        set { File.Percentage = value; }
    }
}

public class SimFile {
    public string  FileNameAtOpen;
    public FS.File File;
    public long    TotalTime;
    public long    TotalBytes;
    public double  Percentage;
    public SimFile (FSModel fsModel, string fileName)
    {
        this.FileNameAtOpen = fileName;
        // FIXME: get this from the file handle instead ! ...
        this.File = fsModel.GetFileFromPath (fileName);
    }
}

public class SimulationData {
    public class IOEvent {
		public long   TotalTime;
		public long   TotalBytes;
        public IOEvent Detail;
        public string FileName {
            get { return Detail.FileName; }
        }
        public IOEventType Type {
            get { return Detail.Type; }
        }
    }

    public class StackFrame {
        public Trace.Stack.Frame Detail;
        public long        TotalTime;
        public long        TotalBytes;
        
        StackFrame              parent;
        public List<StackFrame> Children;
        public StackFrame Parent {
            get { return this.parent; }
            set {
                this.parent = value;
                if (this.parent != null)
                {
                    if (this.parent.Children == null)
                        this.parent.Children = new List<StackFrame>();
                    this.parent.Children.Add (this);
                }
            }
        }
    }

    // Tree of stack frames
    StackFrame stackRoot;

    // List of all events [etc.] <with timestamps>
    public List<SimEvent> allEvents;

    // List of all map events
    public List<SimMap> allMaps;

    public long     TotalTime;
    public long     TotalBytes;

    public long     PageSize;
    public FSData   FileSystem;

    public SimulationData (FSData fsData)
    {
        this.allEvents = new List<SimEvent>();
        this.allMaps = new List<SimMap>();
        this.blocks = new List<SimBlk>();
        this.frameMap = new Dictionary<Trace.Stack.Frame, StackFrame>();
        this.allFiles = new Dictionary<string, SimFile>();
        this.TotalTime = 0;
        this.PageSize = 0;
        this.FileSystem = fsData;
    }

    public void Add (SimEvent sime)
    {
        this.allEvents.Add (sime);
    }

    public void AddMap (FSModel fsModel, SimMap simMap)
    {
        this.allMaps.Add (simMap);
        simMap.File = FindFile (fsModel, simMap.Map.FileName);
    }

    public void CalcTime ()
    {
        this.TotalTime = 0;
        foreach (SimEvent sime in this.allEvents) {
            this.TotalTime += sime.Elapsed;
            if (sime.Elapsed < 0)
                Console.WriteLine ("Error - I/O simulation filed to provide " +
                                   "timings for event: " + sime);
        }
    }

	public void CalcPercentages ()
	{
		foreach (SimFile simFile in this.allFiles.Values)
			simFile.Percentage = (double)simFile.TotalTime/this.TotalTime;
	}

    public List<SimFile> FilesSortedByIO()
	{
		List<SimFile> sorted = new List<SimFile>();

		foreach (SimFile simFile in this.allFiles.Values)
			sorted.Add (simFile);
		
		sorted.Sort(
            delegate (SimFile a, SimFile b)
                { return b.Percentage.CompareTo(a.Percentage); }
                );

		return sorted;
	}

    // List of all files involved in I/O
    // NB. a single SimFile may have many names.
    public Dictionary<string, SimFile> allFiles;

    public SimFile FindFile (FSModel fsModel, string fileName)
    {
        SimFile simFile = null;

        if (this.allFiles.ContainsKey (fileName))
            return allFiles [fileName];

        string flatName = Utils.PathNormalize (fileName);
        if (flatName != fileName && this.allFiles.ContainsKey (flatName))
            simFile = this.allFiles[flatName];
        else {
            simFile = new SimFile (fsModel, flatName);
            // FIXME: assign simFile.File !! ?! 
            allFiles.Add (flatName, simFile);
        }

        if (flatName != fileName)
            allFiles.Add (fileName, simFile);
            
        return simFile;
    }

    public StackFrame StackRoot {
        get { return stackRoot; }
    }

    Dictionary<Trace.Stack.Frame, StackFrame> frameMap;

    StackFrame FindFrame (Trace.Stack.Frame frame)
    {
        if (frame == null)
            return null;
        if (!this.frameMap.ContainsKey (frame))
        {
            StackFrame ret = new StackFrame();
            ret.Parent = FindFrame (frame.Parent);
            if (ret.Parent == null) {
                if (stackRoot != null)
                    Console.WriteLine ("Serious stack frame building error" + frame.Name);
                else
                    stackRoot = ret;
            }
            ret.Detail = frame;
            this.frameMap.Add (frame, ret);
            return ret;
        }
        else
            return this.frameMap [frame];
    }

    public void AccumulateFrame (Trace.Stack.Frame frame, long elapsed, long bytesRead)
    {
        StackFrame sf = FindFrame (frame);
        sf.TotalTime += elapsed;
		sf.TotalBytes += bytesRead;
    }

    public long Elapsed {
        get { return this.TotalTime; }
    }

    public List<SimEvent> Thunks;
    public void AddThunk (SimEvent thunk)
    {
        if (this.Thunks == null)
            this.Thunks = new List<SimEvent>();
        this.Thunks.Add (thunk);
    }

    List<SimBlk> blocks;
    public List<SimBlk> Blocks {
        get { return this.blocks; }
    }

    public void AddBlocks (SimEvent ev, string name, long start, int count)
    {
        SimBlk blk = new SimBlk();
        blk.Ev = ev;
        blk.Name = name;
        blk.FsBlkStart = start;
        blk.FsBlkLen = count;

        this.blocks.Add (blk);
    }
}

// linux/include/linux/sched.hxx - task_struct
public enum ProcessState { UnRunnable, Runnable, Stopped };
public class SimProcess
{
    Simulator sim;
    DiskModel disk;
    FSModel   fsModel;

    public ProcessState State;
    System.Threading.AutoResetEvent Start, Stop;
    bool ioWait;

    Trace.Process tp;
    public SimulationData sd;

    public PageCache PageCache { get { return sim.PageCache; } }
    public long CurrentTime { get { return sim.CurrentTime; } }
    public double CurrentTimeAsDouble { get { return sim.CurrentTimeAsDouble; } }

    public SimulationData Data {
        get { return this.sd; }
    }
    public bool IOWait {
        get { return this.ioWait; }
    }

    // FIXME: move these elsewhere - system bits ...
    // tick tock ...
	long    totalBytes;
	long    pageSize;

	// The first touch of a page has some, quantifiable cost pwrt. swap etc.
	void simulatePageAllocate (PageTouch e, SimEvent sime)
	{
        sime.SetDefunct();
	}

    static bool WarnedOnce = false;
    static int  testPageBlock;
	void simulatePageTouch (SimEvent sime, PageTouch e)
    {
		// For debugging / validation
		if (e.Kind == PageTouchKind.TRACEDEBUG) {
            this.disk.ReadBlocks (this, sime, testPageBlock++, 1);
			return;
		}

		// FIXME - we really need a simple file system
		// indirection here ...
		if (e.Kind == PageTouchKind.STORE) {
			simulatePageAllocate (e, sime);
			return;
		}

        SimMap simMap = LookupMap (e);
        if (simMap == null) {
            if (!WarnedOnce)
                Console.WriteLine ("Error: no map " + e);
            WarnedOnce = true;
            sime.SetDefunct();
            return;
        }

//        Console.WriteLine ("page hit at {0:x08} in " +
//                           (meme != null ? meme.FileName : "foo"),
//                           e.Start);

		if (simMap == null ||
            simMap.FileName == "<heap>") {
			// Stack & Heap
			simulatePageAllocate (e, sime);
			return;
		} else if (simMap.File.File == null) {
            if (simMap.FileName != "<no-name>") // FIXME: vs. heap ?
                Console.WriteLine ("Unknown map - with no file backing " +
                                   simMap.FileName);
            return;
        }

        long timeBefore = CurrentTime;
        long bytesBefore = totalBytes;
		long offset = 0;
		do {
            this.fsModel.MMapRead (sime, simMap.File.File,
                                   simMap.Map.FileOffset +
                                   (e.Start - simMap.Map.Start) + offset);
			offset += this.PageCache.PageSize;
		} while (offset < e.Size);

		simMap.TotalTime += CurrentTime - timeBefore;
		simMap.TotalBytes += totalBytes - bytesBefore;
    }

    string GetEventFName (SimEvent sime)
    {
        string fname = null;
        if (sime.Detail is IOEvent) {
            IOEvent ioe = (IOEvent) sime.Detail;
            fname = ioe.FileName;
            if (fname == null)
                fname = this.fsModel.GetFilePath (ioe.FileHandle);
        } else if (sime.Detail is PageTouch) {
            PageTouch e = (PageTouch) sime.Detail;
            SimMap simMap = LookupMap (e);
            if (simMap != null)
                fname = simMap.FileName;
        }
        return fname;
    }

    static int FIXME_warnings = 0;
	void simulateIOEvent (SimEvent sime, IOEvent ioe)
    {
        long timeBefore = CurrentTime;
        long bytesBefore = totalBytes;

        switch (ioe.Type) {
        case IOEventType.Open:
        case IOEventType.Create:
        case IOEventType.Truncate: {
            OpenEvent oe = (OpenEvent) ioe;
            this.fsModel.Open (sime, oe.FileName, oe.Result,
                               oe.IsDirect, oe.IsSync);
            break;
        }
        case IOEventType.Chdir:
            this.fsModel.Chdir (sime, ioe.FileName);
            break;
        case IOEventType.Close:
            this.fsModel.Close (sime, ioe.FileHandle);
            break;
        case IOEventType.ReadINode:
            this.fsModel.Stat (sime, ioe.FileName);
            break;
        case IOEventType.Seek:
            this.fsModel.Seek (sime, ioe.FileHandle, (ioe as SeekEvent).Offset);
            break;
        case IOEventType.ReadDEnts:
        case IOEventType.ReadBlock: {
            BlockEvent be = (BlockEvent) ioe;
            this.fsModel.Read (sime, ioe.FileHandle, be.Start, (int)be.Length);
            // FIXME: chain to simulatePageTouch for address 'be.Start' ...
            break;
        }

        case IOEventType.WriteBlock: {
            BlockEvent be = (BlockEvent) ioe;
            this.fsModel.Write (sime, ioe.FileHandle, be.Start, be.Length);
            // FIXME: chain to simulatePageTouch for address 'be.Start' ...
            break;
        }
        case IOEventType.ReadAhead: {
            BlockEvent be = (BlockEvent) ioe;
            this.fsModel.ReadAhead (sime, ioe.FileHandle, be.Start, be.Length);
            break;
        }
        default:
            Console.WriteLine ("Unknown I/O event " + ioe.Type + " to simulate... ");
            break;
        }

        // FIXME: we should canonicalize names here ...
        // FIXME: we ~should handle mmaps here ?
        string fname = GetEventFName (sime);

        if (fname != null) {
            SimFile simFile = sd.FindFile (fsModel, fname);
            simFile.TotalTime += CurrentTime - timeBefore;
            simFile.TotalBytes += totalBytes - bytesBefore;
        } else if (FIXME_warnings++ == 0)
            Console.WriteLine ("Error: file-system out of sync with trace: " +
                               "can't resolve " + ioe);
    }

	void simulateThunk (SimEvent sime, Thunk thunk)
    {
        this.sd.AddThunk (sime);
    }

	// FIXME: yes we need some tree structure here ?
	// C5::TreeSet ? or SortedDictionary ?
	List<SimMap> curMemMaps; // live maps
	
	SimMap LookupMap(MemRegion r)
	{
		foreach (SimMap i in curMemMaps) {
            MapEvent meme = i.Map;
			if (meme.Overlaps(r))
				return i;
		}
		return null;
	}
	
	void simulateMaps (SimEvent sime, MapEvent m)
	{
		if (m.Kind == MapEventKind.MAP) {
            SimMap simMap = new SimMap();
            simMap.Map = m;
            simMap.MapTime = CurrentTime;

            // FIXME: should unmap any overlapping stuff, and re-map it.
            //        this is what the kernel does. For now, we just
            //        merge any overlapping bits into the 1st map.
            //        ~all shlib loads do an unmap/remap on load ...

/*          SimMap pair;
            if ((pair = LookupMap (m)) != null) {
                pair.Map = new MapEvent ();
                if (m.Start <= pair.Start) {
                    Console.WriteLine ("Error: broken re-mapping assumption");
                } else {
                    pair.End = m.Start;
                }
            } */

			curMemMaps.Add (simMap);
            sd.AddMap (this.fsModel, simMap);
		} else {
            SimMap simMap = LookupMap (m);

			if (simMap == null || simMap.Map == null)
				Console.WriteLine("Error: failed to remove memory map!");
			else {
                simMap.Unmap = m;
                simMap.UnmapTime = CurrentTime;
                // same address range ...
				System.Diagnostics.Debug.Assert (simMap.Map.IsEqualTo (simMap.Unmap));
				curMemMaps.Remove (simMap);
			}
		}
	}

    void UnmapAtExit()
    {
		foreach (SimMap s in curMemMaps)
            s.UnmapTime = CurrentTime;
        curMemMaps.Clear();
    }

    void PerformWork()
    {
        ArrayList ioevents = this.tp.IOEvents;

		this.sd = new SimulationData (this.fsModel.fsData);

		this.totalBytes = 0;
		this.curMemMaps = new List<SimMap>();

		foreach (Object o in ioevents) {

            // The scheduler schedules us...
            this.Stop.Set();
            this.Start.WaitOne();

            SimEvent sime = new SimEvent();
            sime.Detail = o;
            sime.StartTime = CurrentTime;
            sime.EndTime = CurrentTime;

            // Process events ...
			if (o is PageTouch)
                simulatePageTouch (sime, (PageTouch) o);
			else if (o is MapEvent)
				simulateMaps (sime, (MapEvent) o);
            else if (o is IOEvent)
				simulateIOEvent (sime, (IOEvent) o);
            else if (o is Thunk)
                simulateThunk (sime, (Thunk) o);
            else
                sime.SetDefunct();
                
			if (!sime.Defunct)
				this.sd.Add (sime);
		}

        UnmapAtExit();
        this.sd.TotalBytes = totalBytes;
        this.sd.PageSize = pageSize;
        this.sd.CalcTime();
		this.sd.CalcPercentages();

        this.State = ProcessState.Stopped;
        this.Stop.Set();
    }

    public void RunSlice()
    {
        this.Start.Set();
        // process performs it's labours ...
        this.Stop.WaitOne();
    }
    public void IOSleep()
    {
        this.ioWait = true;
        this.Stop.Set();
        // hand back to the scheduler
        this.Start.WaitOne();
    }
    public void IOWakeup (SimEvent sime, long bytesRead)
    {
        sime.EndTime = CurrentTime;

        if (sime.Detail is TraceEvent)
            this.sd.AccumulateFrame (((TraceEvent) sime.Detail).Frame, sime.Elapsed, bytesRead);

		this.totalBytes += bytesRead;
        
        this.ioWait = false;
        // will wake up @ next scheduler iteration
    }

    public SimProcess (Simulator sim, FSData fsData, DiskModel disk, Trace.Process tp)
    {
        this.sim = sim;
        this.disk = disk;
        this.fsModel = new FSModel (this, fsData, disk);

        this.tp = tp; 
		this.pageSize = 4096;
        this.State = ProcessState.UnRunnable;
        this.Start = new System.Threading.AutoResetEvent(false);
        this.Stop = new System.Threading.AutoResetEvent(false);

        System.Threading.Thread thread = new System.Threading.Thread (
            new System.Threading.ThreadStart (PerformWork));
        thread.Start();
    }

    public void AddBlocks (SimEvent ev, long fsBlkStart, int fsBlkCount)
    {
        this.sd.AddBlocks (ev, GetEventFName (ev), fsBlkStart, fsBlkCount);
    }

    public void AddNotice (SimEvent ev)
    {
        this.sd.AddBlocks (ev, GetEventFName (ev), -1, 0);
    }
}

public class Simulator
{
	DiskModel disk;
    FSData    fsData;
    PageCache pageCache;

    // Global time ... in microseconds
    double currentTime;

    public PageCache PageCache { get { return this.pageCache; } }
    public long CurrentTime { get { return (long)this.currentTime; } }
    public double CurrentTimeAsDouble { get { return this.currentTime; } }

	public Simulator (FSData fsData, string diskProfile)
	{
        this.fsData = fsData;
        this.pageCache = new PageCache (4096);

        if (diskProfile == null)
            this.disk = new DiskModelSimple ();
        else
            this.disk = new DiskSim (diskProfile);

        long sectSize = (this.fsData.MaxBlock
                         * (this.fsData.BlockSize / disk.SectorSize));
        if (!this.disk.CheckMaxSector (sectSize))
        {
            Console.WriteLine ("Error - disk model too small to accomodate filesystem");
            Console.WriteLine ("\t" + sectSize + " blocks required ("
                               + Utils.PrettyBytes (this.fsData.MaxBlock * this.fsData.BlockSize) + ")");
            System.Environment.Exit (1);
        }
	}

    List<SimProcess> procs;

	public SimulationData Simulate (Trace.Data td)
	{
        this.procs = new List<SimProcess>();
        int numAlive = 0;

        foreach (Process tp in td.Processes) {
            SimProcess sp = new SimProcess (this, this.fsData, this.disk, tp);
            // FIXME: need to have 'exec' hintsto defer Runnable ...
            sp.State = ProcessState.Runnable;
            numAlive++;
            this.procs.Add (sp);
        }

        // We implement all this with threads, to make the various
        // synchronous methods easy to implement. However we force
        // the order of the thread execution to our bidding.

        currentTime = 0.0;
        while (numAlive > 0) {
            numAlive = 0;
            int numIOWait = 0;

            // spin each process until each hits an I/O wait ...
            foreach (SimProcess sp in procs) {
                if (sp.State == ProcessState.Runnable) {
                    if (!sp.IOWait)
                        sp.RunSlice();
                    else
                        numIOWait++;
                    numAlive++;
                }
            }
            if (numIOWait > 0) {
                // now we know we have at least one I/O event queued
                // per process - hanging around in 'SimProcess::IOSleep'

                // Wait for the I/O to complete ...
                disk.Wait(ref this.currentTime);
            }
        }

        // FIXME: hack ! ...
        return procs[0].Data;
	}
}
