/* Tree View/Editable Cells
 *
 * This demo demonstrates the use of editable cells in a GtkTreeView. If
 * you're new to the GtkTreeView widgets and associates, look into
 * the GtkListStore example first.
 *
 */

#include <gtkmm.h>

class CellItem_Product
{
public:
  CellItem_Product();
  CellItem_Product(const CellItem_Product& src);
  ~CellItem_Product();
  CellItem_Product& operator=(const CellItem_Product& src);

  int m_number;
  Glib::ustring m_product;
  bool m_editable;
};

class Example_TreeView_EditableCells : public Gtk::Window
{
public:
  Example_TreeView_EditableCells();
  virtual ~Example_TreeView_EditableCells();

protected:
  //signal handlers:
  virtual void on_button_add_clicked();
  virtual void on_button_remove_clicked();
  virtual void on_cell_edited_number(const Glib::ustring& path_string, const Glib::ustring& new_text);
  virtual void on_cell_edited_product(const Glib::ustring& path_string, const Glib::ustring& new_text);

  virtual void create_model();
  virtual void add_columns();
  virtual void add_items();
  virtual void liststore_add_item(const CellItem_Product& foo);

  //Member widgets:
  Gtk::VBox m_VBox;
  Gtk::ScrolledWindow m_ScrolledWindow;
  Gtk::Label m_Label;
  Gtk::TreeView m_TreeView;
  Glib::RefPtr<Gtk::ListStore> m_refListStore;
  Gtk::HBox m_HBox;
  Gtk::Button m_Button_Add, m_Button_Remove;

  typedef std::vector<CellItem_Product> type_vecItems;
  type_vecItems m_vecItems;

  struct ModelColumns : public Gtk::TreeModelColumnRecord
  {
    Gtk::TreeModelColumn<int>           number;
    Gtk::TreeModelColumn<Glib::ustring> product;
    Gtk::TreeModelColumn<bool>          editable;

    ModelColumns() { add(number); add(product); add(editable); }
  };

  const ModelColumns m_columns;
};


CellItem_Product::CellItem_Product()
{
  m_number = 0;
  m_editable = false;
}

CellItem_Product::CellItem_Product(const CellItem_Product& src)
{
  operator=(src);
}

CellItem_Product::~CellItem_Product()
{
}

CellItem_Product& CellItem_Product::operator=(const CellItem_Product& src)
{
  m_number = src.m_number;
  m_product = src.m_product;
  m_editable = src.m_editable;

  return *this;
}


//Called by DemoWindow;
Gtk::Window* do_treeview_editable_cells()
{
  return new Example_TreeView_EditableCells();
}


Example_TreeView_EditableCells::Example_TreeView_EditableCells()
: m_VBox(false, 5),
  m_Label("Shopping list (you can edit the cells!)"),
  m_HBox(true, 4),
  m_Button_Add("Add item"),
  m_Button_Remove("Remove item")
{
  set_title("Shopping List");
  set_border_width(5);
  set_default_size(320, 200);

  add(m_VBox);
  m_VBox.pack_start(m_Label, Gtk::SHRINK);

  m_ScrolledWindow.set_shadow_type(Gtk::SHADOW_ETCHED_IN);
  m_ScrolledWindow.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
  m_VBox.pack_start(m_ScrolledWindow);

  /* create model */
  create_model();

  /* create tree view */
  m_TreeView.set_model(m_refListStore);
  m_TreeView.set_rules_hint();
  Glib::RefPtr<Gtk::TreeSelection> refTreeSelection = m_TreeView.get_selection();
  refTreeSelection->set_mode(Gtk::SELECTION_SINGLE);

  add_columns();
  m_ScrolledWindow.add(m_TreeView);

  /* some buttons */
  m_VBox.pack_start(m_HBox, Gtk::SHRINK);

  m_HBox.pack_start(m_Button_Add);
  m_Button_Add.signal_clicked().connect(
    SigC::slot(*this, &Example_TreeView_EditableCells::on_button_add_clicked));

  m_HBox.pack_start(m_Button_Remove);
  m_Button_Remove.signal_clicked().connect(
    SigC::slot(*this, &Example_TreeView_EditableCells::on_button_remove_clicked));

  show_all();
}

Example_TreeView_EditableCells::~Example_TreeView_EditableCells()
{
}

void Example_TreeView_EditableCells::add_items()
{
  CellItem_Product foo;

  foo.m_number = 3;
  foo.m_product = "bottles of coke";
  foo.m_editable = true;
  m_vecItems.push_back(foo);

  foo.m_number = 5;
  foo.m_product = "packages of noodles";
  foo.m_editable = true;
  m_vecItems.push_back(foo);

  foo.m_number = 2;
  foo.m_product = "packages of chocolate chip cookies";
  foo.m_editable = true;
  m_vecItems.push_back(foo);

  foo.m_number = 1;
  foo.m_product = "can vanilla ice cream";
  foo.m_editable = true;
  m_vecItems.push_back(foo);

  foo.m_number = 6;
  foo.m_product = "eggs";
  foo.m_editable = true;
  m_vecItems.push_back(foo);
}

void Example_TreeView_EditableCells::create_model()
{
  m_refListStore = Gtk::ListStore::create(m_columns);

  /* add items */
  add_items();

  std::for_each(
      m_vecItems.begin(), m_vecItems.end(),
      SigC::slot(*this, &Example_TreeView_EditableCells::liststore_add_item));
}

void Example_TreeView_EditableCells::liststore_add_item(const CellItem_Product& foo)
{
  Gtk::TreeRow row = *(m_refListStore->append());

  row[m_columns.number]   = foo.m_number;
  row[m_columns.product]  = foo.m_product;
  row[m_columns.editable] = foo.m_editable;
}

void Example_TreeView_EditableCells::add_columns()
{
  /* number column */
  {
    Gtk::CellRendererText* pRenderer = Gtk::manage( new Gtk::CellRendererText() );
    pRenderer->signal_edited().connect(
        SigC::slot(*this, &Example_TreeView_EditableCells::on_cell_edited_number));

    gint cols_count = m_TreeView.insert_column(-1, "Number", *pRenderer);
    Gtk::TreeViewColumn* pColumn = m_TreeView.get_column(cols_count-1);

    pColumn->add_attribute(pRenderer->property_text(), m_columns.number);
    pColumn->add_attribute(pRenderer->property_editable(), m_columns.editable);
  }

  /* product column */
  {
    Gtk::CellRendererText* pRenderer = Gtk::manage( new Gtk::CellRendererText() );
    pRenderer->signal_edited().connect(
        SigC::slot(*this, &Example_TreeView_EditableCells::on_cell_edited_product));

    gint cols_count = m_TreeView.insert_column(-1, "Product", *pRenderer);
    Gtk::TreeViewColumn* pColumn = m_TreeView.get_column(cols_count-1);

    pColumn->add_attribute(pRenderer->property_text(), m_columns.product);
    pColumn->add_attribute(pRenderer->property_editable(), m_columns.editable);
  }
}


void Example_TreeView_EditableCells::on_button_add_clicked()
{
  CellItem_Product foo;
  foo.m_number = 0;
  foo.m_product = "Description here";
  foo.m_editable = true;
  m_vecItems.push_back(foo);

  liststore_add_item(foo);
}

void Example_TreeView_EditableCells::on_button_remove_clicked()
{
  Glib::RefPtr<Gtk::TreeSelection> refSelection = m_TreeView.get_selection();

  if(const Gtk::TreeIter iter = refSelection->get_selected())
  {
    Gtk::TreePath path = m_refListStore->get_path(iter);

    std::vector<int> indices = path.get_indices();
    if(indices.size())
    {
      //Remove item from ListStore:
      m_refListStore->erase(iter);

      //Remove item from vecItems.
      int index = indices[0];
      if(index < m_vecItems.size())
        m_vecItems.erase(m_vecItems.begin() + index);
    }
  }
}

void Example_TreeView_EditableCells::on_cell_edited_number(
    const Glib::ustring& path_string, const Glib::ustring& new_text)
{
  //TODO: Why doesn't the path(strng) constructor work?
  //Using it causes the new text to be put always in the first row.
  //Gtk::TreePath path(path_string);
  GtkTreePath *gpath = gtk_tree_path_new_from_string(path_string.c_str());
  Gtk::TreePath path(gpath);

  std::vector<int> indices = path.get_indices();
  if(!indices.empty())
  {
    int i = indices[0];
    m_vecItems[i].m_number = atoi(new_text.c_str());

    Gtk::TreeRow row = *(m_refListStore->get_iter(path));
    row[m_columns.number] = atoi(new_text.c_str());
  }
}

void Example_TreeView_EditableCells::on_cell_edited_product(
    const Glib::ustring& path_string, const Glib::ustring& new_text)
{
  //TODO: Why doesn't the path(strng) constructor work?
  //Using it causes the new text to be put always in the first row.
  //Gtk::TreePath path(path_string);
  GtkTreePath *gpath = gtk_tree_path_new_from_string(path_string.c_str());
  Gtk::TreePath path(gpath);

  std::vector<int> indices = path.get_indices();
  if(!indices.empty())
  {
    int i = indices[0];
    m_vecItems[i].m_product = new_text;

    Gtk::TreeRow row = *(m_refListStore->get_iter(path));
    row[m_columns.product] = new_text;
  }
}

