// -*- c++ -*-
#ifndef _GLIBMM_VALUE_H
#define _GLIBMM_VALUE_H
/* $Id: value.h,v 1.13 2002/04/23 15:57:19 daniel Exp $ */

/* Copyright 2002 Free Software Foundation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <glib-object.h>
#include <glibmm/refptr.h>
#include <glibmm/ustring.h>


namespace Glib
{

class ObjectBase;


class ValueBase
{
public:
  /** Initializes the GValue, but without a type.  You have to
   * call init() before using the set(), get(), or reset() methods.
   */
  ValueBase();

  ValueBase(const ValueBase& other);
  ValueBase& operator=(const ValueBase& other);

  ~ValueBase();

  /** Setup the GValue for storing the specified @a type.
   * The contents will be initialized to the default value for this type.
   * Note that init() should never be called twice.
   *
   * init() is not implemented as constructor, to avoid the necessity
   * to implement a forward constructor in each derived class.
   */
  void init(GType type);

  /** Reset contents to the default value of its type.
   */
  void reset();

  GValue*       gobj()       { return &gobject_; }
  const GValue* gobj() const { return &gobject_; }

protected:
  GValue gobject_;
};


class ValueBase_Boxed : public ValueBase
{
public:
  static GType value_type() G_GNUC_CONST;

protected:
  void  set_boxed(const void* data);
  void* get_boxed() const; // doesn't copy
};


class ValueBase_Object : public ValueBase
{
public:
  static GType value_type() G_GNUC_CONST;

protected:
  void set_object(Glib::ObjectBase* data);
  Glib::ObjectBase* get_object() const;
  Glib::RefPtr<Glib::ObjectBase> get_object_copy() const;
};


class ValueBase_Enum : public ValueBase
{
public:
  typedef gint CType;
  static GType value_type() G_GNUC_CONST;

protected:
  void set_enum(int data);
  int  get_enum() const;
};


class ValueBase_Flags : public ValueBase
{
public:
  typedef guint CType;
  static GType value_type() G_GNUC_CONST;

protected:
  void set_flags(unsigned int data);
  unsigned int get_flags() const;
};


class ValueBase_Pointer : public ValueBase
{
public:
  typedef gpointer CType;
  static GType value_type() G_GNUC_CONST;

protected:
  void  set_pointer(void* data);
  void* get_pointer() const;
};


class ValueBase_String : public ValueBase
{
public:
  typedef const gchar* CType;
  static GType value_type() G_GNUC_CONST;

protected:
  void set_cstring(const char* data);
  const char* get_cstring() const; // never returns 0
};


/** Glib::Value<> is specialized for almost any type:
 *
 * @li Basic types like @c int, @c char, @c bool, etc., also <tt>void*<tt>.
 * @li Glib::ustring and std::string.
 * @li Pointer types, which are assumed to be Glib::Object pointers.
 * @li Glib::RefPtr<> pointer types, also assumed to be Glib::Object pointers.
 * @li All flags and enum types used by gtkmm.
 *
 * If a type doesn't fit into any of these categories, it is assumed to
 * be a Boxed Type wrapper class.  This won't work for user-defined flags
 * and enum types, which means the distinct Value_Flags<> and Value_Enum<>
 * templates have to be used instead.
 *
 * Value_Pointer<> can be used for any pointer type, it doesn't have to be
 * registered in the GObject type system.  No attempt is made to manage the
 * memory associated with the pointer, you must take care of that yourself.
 */
template <class T>
class Value : public ValueBase_Boxed
{
public:
  typedef T                           CppType;
  typedef typename T::BaseObjectType* CType;

  static GType value_type() { return T::get_type(); }

  void set(const CppType& data) { set_boxed(data.gobj()); }
  CppType get() const           { return CppType(static_cast<CType>(get_boxed())); }
};


/** Partial specialization for pointers to Glib::Object.
 */
template <class T>
class Value<T*> : public ValueBase_Object
{
public:
  typedef T*                          CppType;
  typedef typename T::BaseObjectType* CType;

  static GType value_type() { return T::get_base_type(); }

  void set(CppType data) { set_object(data); }
  CppType get() const    { return dynamic_cast<T*>(get_object()); }
};

/** Partial specialization for pointers to const Glib::Object.
 */
template <class T>
class Value<const T*> : public ValueBase_Object
{
public:
  typedef const T*                    CppType;
  typedef typename T::BaseObjectType* CType;

  static GType value_type() { return T::get_base_type(); }

  void set(CppType data) { set_object(const_cast<T*>(data)); }
  CppType get() const    { return dynamic_cast<T*>(get_object()); }
};


/** Partial specialization for RefPtr<> to Glib::Object.
 */
template <class T>
class Value< Glib::RefPtr<T> > : public ValueBase_Object
{
public:
  typedef Glib::RefPtr<T>             CppType;
  typedef typename T::BaseObjectType* CType;

  static GType value_type() { return T::get_base_type(); }

  void set(const CppType& data) { set_object(data.operator->()); }
  CppType get() const           { return Glib::RefPtr<T>::cast_dynamic(get_object_copy()); }
};

/** Partial specialization for RefPtr<> to const Glib::Object.
 */
template <class T>
class Value< Glib::RefPtr<const T> > : public ValueBase_Object
{
public:
  typedef Glib::RefPtr<const T>       CppType;
  typedef typename T::BaseObjectType* CType;

  static GType value_type() { return T::get_base_type(); }

  void set(const CppType& data) { set_object(const_cast<T*>(data.operator->())); }
  CppType get() const           { return Glib::RefPtr<T>::cast_dynamic(get_object_copy()); }
};

} // namespace Glib


/* Include generated specializations of Glib::Value<> for fundamental types:
 */
#define _GLIBMM_VALUE_H_INCLUDE_VALUE_BASICTYPES_H
#include <glibmm/value_basictypes.h>
#undef _GLIBMM_VALUE_H_INCLUDE_VALUE_BASICTYPES_H


namespace Glib
{

/** Dummy specialization to prevent accidental use with char*.
 * Since gtkmm often creates temporary Glib::Value instances, you
 * would soon end up accessing a string which no longer exists.
 * @see Value<std::string> and Value<Glib::ustring> specializations.
 */
template <>
class Value<char*>
{};

/** Dummy specialization to prevent accidental use with const char*.
 * Since gtkmm often creates temporary Glib::Value instances, you
 * would soon end up accessing a string which no longer exists.
 * @see Value<std::string> and Value<Glib::ustring> specializations.
 */
template <>
class Value<const char*>
{};

/** Specialization for strings.
 */
template <>
class Value<std::string> : public ValueBase_String
{
public:
  typedef std::string CppType;

  void set(const std::string& data);
  std::string get() const { return get_cstring(); }
};

/** Specialization for UTF-8 strings.
 */
template <>
class Value<Glib::ustring> : public ValueBase_String
{
public:
  typedef Glib::ustring CppType;

  void set(const Glib::ustring& data);
  Glib::ustring get() const { return get_cstring(); }
};


/** Base class of Glib::Value<T> specializations for enum types.
 */
template <class T>
class Value_Enum : public ValueBase_Enum
{
public:
  typedef T CppType;

  void set(CppType data) { set_enum(data); }
  CppType get() const    { return CppType(get_enum()); }
};

/** Base class of Glib::Value<T> specializations for flags types.
 */
template <class T>
class Value_Flags : public ValueBase_Flags
{
public:
  typedef T CppType;

  void set(CppType data) { set_flags(data); }
  CppType get() const    { return CppType(get_flags()); }
};


/** Dummy class to catch accidental use with non-pointer types.
 * @see Value_Pointer<T*> and Value_Pointer<const T*> specializations.
 */
template <class T>
class Value_Pointer
{};

/** Specialization for pointers to instances of any type.
 */
template <class T>
class Value_Pointer<T*> : public ValueBase_Pointer
{
public:
  typedef T* CppType;

  void set(CppType data) { set_pointer(data); }
  CppType get() const    { return static_cast<T*>(get_pointer()); }
};

/** Specialization for pointers to const instances of any type.
 */
template <class T>
class Value_Pointer<const T*> : public ValueBase_Pointer
{
public:
  typedef const T* CppType;

  void set(CppType data) { set_pointer(const_cast<T*>(data)); }
  CppType get() const    { return static_cast<T*>(get_pointer()); }
};

} // namespace Glib


#endif /* _GLIBMM_VALUE_H */

