/* gok-word-complete.c
*
* Copyright 2001,2002 Sun Microsystems, Inc.,
* Copyright 2001,2002 University Of Toronto
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Library General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Library General Public License for more details.
*
* You should have received a copy of the GNU Library General Public
* License along with this library; if not, write to the
* Free Software Foundation, Inc., 59 Temple Place - Suite 330,
* Boston, MA 02111-1307, USA.
*/

/*
* To use this thing:
* - Call "gok_wordcomplete_open". If it returns TRUE then you're ready to go.
* - Call "gok_wordcomplete_on (TRUE)" to turn it on.
* - Call "gok_wordcomplete_predict" to make the word predictions.
* - Call "gok_wordcomplete_close" when you're done. 
* - To add a word, call "gok_wordcomplete_add_new_word".
*
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "gok-word-complete.h"
#include "gok-keyslotter.h"
#include "main.h"
#include "gok-log.h"
#include "gok-modifier.h"
#include "gok-data.h"
#include "gok-keyboard.h"


/* the word, or part word, that is getting completed */
static gchar *word_part = NULL;

static GokOutput* m_pOutputListStart;

/**
* gok_wordcomplete_open
* 
* Opens and initializes the word completor engine.
*
* returns: TRUE if it was opend OK, FALSE if not.
**/
gboolean gok_wordcomplete_open (gchar *directory)
{
	m_pOutputListStart = NULL;
	
	gok_wordcomplete_end_word();
	
	gok_wordcomplete_on (gok_data_get_wordcomplete());
	
	return (WordCompleteOpen(directory));
}

/**
* gok_wordcomplete_close
* 
* Closes the word completor engine.
*
* returns: void
**/
void gok_wordcomplete_close ()
{
	if (m_pOutputListStart == NULL)
	{
		gok_output_delete_all (m_pOutputListStart);
	}

	WordCompleteClose();
}

/**
* gok_wordcomplete_on
* 
* Turns on (or off) the word completor.
* This adds (or removes) the word completion keys to every keyboard that supports 
* word completion.
*
* returns: void.
**/
void gok_wordcomplete_on (gboolean bOnOff)
{
	gok_log_enter();
	gok_keyslotter_on(bOnOff, KEYTYPE_WORDCOMPLETE);
	gok_log_leave();
}

/**
* gok_wordcomplete_change_number_predictions
*
* @Number: Number of prediction keys.
*
* Changes the number of word prediction keys displayed on the keyboard. This
* should be called after the user has selected a new number from the settings
* dialog.
*
* returns: void
**/
void gok_wordcomplete_change_number_predictions (int Number)
{
	gok_keyslotter_change_number_predictions( Number, KEYTYPE_WORDCOMPLETE );
}

/**
* gok_wordcomplete_add_prediction_keys
*
* @pKeyboard: Pointer to the keyboard that gets the new prediction keys.
*
* Adds a row of prediction keys to the given keyboard.
*
* returns: TRUE if the prediction keys were added, FALSE if not.
**/
gboolean gok_wordcomplete_add_prediction_keys (GokKeyboard* pKeyboard)
{
	return gok_keyslotter_add_prediction_keys (pKeyboard, KEYTYPE_WORDCOMPLETE);
}

/**
* gok_wordcomplete_predict
* 
* Makes a prediction. If the currently displayed keyboard is showing prediction
* keys then they are filled in with the predictions.
*
* returns: The number of words predicted.
**/
int gok_wordcomplete_predict (const gunichar letter)
{
	GokKeyboard* pKeyboard;
	GokKey* pKey;
	gint count;
	gint length;
	gint statusPrediction = 0;
	gchar* pWordsPredicted;
	gchar* token;
	gchar separators[] = " ,\t\n";

	/* is word completion turned on? */
	if (gok_data_get_wordcomplete() == FALSE)
	{
		return 0;
	}
	
	/* get the currently displayed keyboard */
	pKeyboard = gok_main_get_current_keyboard();
	
	/* does it support word completion? */
	if (gok_keyboard_get_supports_wordcomplete (pKeyboard) == FALSE)
	{
		return 0;
	}

	/* clear the current word predictions */
	gok_wordcomplete_clear_keys();

	if (word_part)
		length = g_utf8_strlen (word_part, -1);
	else
		length = 0;


	/* is this character a backspace, delete/rubout, or replace-char? */
	if ((letter == 0x08) || (letter == 0x7f) || (letter == 0xfffd)) 
	{
		if (length > 0) 
		{
			--length;
			* g_utf8_offset_to_pointer (word_part, length) = '\0';
		}
	}
	else	/* add the letter to the current part word, subject to conditions below */
	{
	        gchar *tmp, utf8char[7]; /* min 6, plus NULL */

		/* 
		 * Is this whitespace, empty, punctuation, or nonprintable? Also catches NULL/0 
		 * we make exception for U+00a0 (nonbreak space) so 'words' containing 
		 * space can be handled. 
		 */
		if (!g_unichar_validate (letter) ||  /* if invalid, */
		    ((g_unichar_ispunct (letter) || !g_unichar_isgraph (letter)) /* or punct/non-graphical, */
		     && (letter != 0x00a0)) || /* but not nbspace */
		    (letter == 0x2f)) /* or if destructive backspace */
		{
			/* reset the part word */
			gok_wordcomplete_end_word ();		
			return 0;
		}
		
		utf8char [g_unichar_to_utf8 (letter, utf8char)] = '\0';
		if (word_part) 
		{
			tmp = g_strconcat (word_part, utf8char, NULL);
			g_free (word_part);
			word_part = tmp;
		}
		else
		{
			word_part = g_strdup (utf8char);
		}
	}

	/* make the prediction */
	if (word_part) {
		statusPrediction = WordCompletePredict (word_part, 
							gok_data_get_num_predictions(), 
							&pWordsPredicted);
	}
	if (statusPrediction != 0)
	{
		return 0;
	}
	
	/* fill in the word completion keys */
	count = 0;
	pKey = pKeyboard->pKeyFirst;
	token = strtok (pWordsPredicted, separators);
	while (token != NULL)
	{
		/* make sure we're not going over our maximum predictions */
		count++;
		if (count > gok_data_get_num_predictions())
		{
			break;
		}
		
		/* get the next word completion key on the keyboard */
		while (pKey != NULL)
		{
			if (pKey->Type == KEYTYPE_WORDCOMPLETE)
			{
				gok_key_set_button_label (pKey, token);
				gok_key_add_label (pKey, token, MODIFIER_NORMAL);
								
				pKey = pKey->pKeyNext;
				break;
			}
			pKey = pKey->pKeyNext;
		}
		if (pKey == NULL)
		{
			break;
		}
		
		/* get the next word predicted */
		token = strtok (NULL, separators);
	}
	
	/* change the font size for the word completion keys */
	if (count > 0)
	{
		gok_keyboard_calculate_font_size_group (pKeyboard, FONT_SIZE_GROUP_WORDCOMPLETE, TRUE);
	}
	
	return count;
}

/**
* gok_wordcomplete_end_word
* 
* Resets the part word buffer.
*
* returns: void
**/
void gok_wordcomplete_end_word ()
{
	g_free (word_part);
	word_part = NULL;
}

/**
* gok_wordcomplete_clear_keys
* 
* Clears the word completion keys on the current keyboard.
* (The keys are still displayed, they just have no label or output.)
*
* returns: void
**/
void gok_wordcomplete_clear_keys ()
{
	GokKeyboard* pKeyboard;
	GokKey* pKey;
	
	/* get the currently displayed keyboard */
	pKeyboard = gok_main_get_current_keyboard();
	
	/* clear the current word predictions */
	pKey = pKeyboard->pKeyFirst;
	while (pKey != NULL)
	{
		if (pKey->Type == KEYTYPE_WORDCOMPLETE)
		{
			gok_key_set_button_label (pKey, "");
			gok_key_add_label (pKey, "", MODIFIER_NORMAL);
			gok_key_set_output (pKey, 0, NULL, 0);
		}
		pKey = pKey->pKeyNext;
	}
}

/**
* gok_wordcomplete_add_new_word
* 
* Adds a new word to the predictor dictionary.
*
* returns: TRUE if the word was added to the dictionary, FALSE if not.
**/
gboolean gok_wordcomplete_add_new_word (const gchar* pWord)
{
        return WordCompleteAddNewWord (pWord);
}

/**
* gok_wordcomplete_increment_word_frequency
* 
* Increments the frequency of a word in the dictionary.
*
* returns: TRUE if the word's frequency was incremented, FALSE if not.
**/
gboolean gok_wordcomplete_increment_word_frequency (const gchar* pWord)
{
	return WordCompleteIncrementFrequency (pWord);
}

/**
* gok_wordcomplete_get_output
* 
* Gets the output for a word prediction key.
*
* @pKey: Pointer to the word completion key that will be output.
*
* returns: A pointer to the output.
**/
GokOutput* gok_wordcomplete_get_output (GokKey* pKey)
{
	GokOutput* pOutputNew;
	GokOutput* pOutputListEnd;
	gchar* LabelText;
	
	g_assert (pKey != NULL);

	/* make sure this is a word completion key */
	if (pKey->Type != KEYTYPE_WORDCOMPLETE)
	{
		gok_log_x ("Hey, this is not a word completion key!\n");
		return NULL;
	}
	
	if (m_pOutputListStart != NULL)
	{
		gok_output_delete_all (m_pOutputListStart);
	}
	
	/* subtract the part word from the output */
	LabelText = gok_key_get_label (pKey);
	if (LabelText == NULL)
	{
		return NULL;
	}
	
	m_pOutputListStart = NULL;
	pOutputListEnd = NULL;

	if (word_part && (g_utf8_strlen (word_part, -1) < g_utf8_strlen (LabelText, -1)))
	{
		gchar *cp;
                /* 
		 * note that this won't match if the decompositions don't match. Should we
		 * decompose to canonical first ? 
		 */ 
		cp = strstr (LabelText, word_part); 
		/* sanity check: is our word_part found in our label? */
		if (cp) {
			gint offset = strlen (word_part); /* in bytes, not chars */
			pOutputNew = gok_output_new (OUTPUT_KEYSTRING, cp + offset, SPI_KEY_STRING);
			if (m_pOutputListStart == NULL)
			{
				m_pOutputListStart = pOutputNew;
				pOutputListEnd = pOutputNew;
			}
			else
			{
				pOutputListEnd->pOutputNext = pOutputNew;
				pOutputListEnd = pOutputNew;
			}
		}
		else {
			/* if not then we are doing a 'replace', handle that via EditableText. */
			g_warning ("Sorry, string replacement is not yet implemented.");
		}
	}

	return m_pOutputListStart;
}

