/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *	Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <bonobo/bonobo-i18n.h>
#include <libgda/gda-config.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtkpaned.h>
#include <gtk/gtkstock.h>
#include <gtk/gtktable.h>
#include <gtk/gtktogglebutton.h>
#include "gnome-db-defs.h"
#include "gnome-db-shell-settings.h"
#include "gnome-db-shortcut.h"

#define PARENT_TYPE GTK_TYPE_VBOX

struct _GnomeDbShellSettingsPrivate {
	GHashTable *loaded_widgets;
	GnomeDbShell *shell;
	GtkWidget *toolbar;
	GtkWidget *notebook;

	/* General tab widgets */
	GtkWidget *general_save_window_pos;
};

static void gnome_db_shell_settings_class_init (GnomeDbShellSettingsClass *klass);
static void gnome_db_shell_settings_init       (GnomeDbShellSettings *settings,
						GnomeDbShellSettingsClass *klass);
static void gnome_db_shell_settings_finalize   (GObject *object);

static void shortcut_button_clicked_cb (GnomeDbShortcut *shortcut,
					const gchar *label,
					const gchar *tooltip,
					gpointer user_data);

static GObjectClass *parent_class = NULL;

/*
 * Private functions
 */

static void
add_components_to_shortcut (GnomeDbShellSettings *settings)
{
	GList *components;

	g_return_if_fail (GNOME_DB_IS_SHELL_SETTINGS (settings));

	/* add the 'General' tab */
	gnome_db_shortcut_append (GNOME_DB_SHORTCUT (settings->priv->toolbar),
				  _("General"), _("General"),
				  GTK_STOCK_PREFERENCES,
				  GTK_SIGNAL_FUNC (shortcut_button_clicked_cb),
				  settings);

	/* add the other available components */
	components = gda_config_get_component_list (
		"repo_ids.has('IDL:GNOME/Database/UIShellComponent:1.0')");
	if (components) {
		GList *node;
		GdaComponentInfo *comp_info;
		gchar *str;

		for (node = g_list_first (components);
		     node != NULL;
		     node = g_list_next (node)) {
			comp_info = (GdaComponentInfo *) node->data;
			str = g_strdup_printf ("%s/%s",
					       GNOME_DB_SHELL_CONFIG_SECTION_LOAD,
					       comp_info->id);
			if (gda_config_has_key (str)) {
				GdaParameter *param;
				gchar *shell_icon;

				param = gda_parameter_list_find (comp_info->properties,
								 "shell-icon");
				shell_icon = gda_value_stringify (gda_parameter_get_value (param));
				gnome_db_shortcut_append (
					GNOME_DB_SHORTCUT (settings->priv->toolbar),
					comp_info->description,
					comp_info->id,
					shell_icon,
					GTK_SIGNAL_FUNC (shortcut_button_clicked_cb),
					settings);

				g_free (shell_icon);
			}
			g_free (str);
		}
		gda_config_free_component_list (components);
	}
}

/*
 * Callbacks
 */

static void
shortcut_button_clicked_cb (GnomeDbShortcut *shortcut,
			    const gchar *label,
			    const gchar *tooltip,
			    gpointer user_data)
{
}

/*
 * GnomeDbShellSettings class implementation
 */

static void
gnome_db_shell_settings_class_init (GnomeDbShellSettingsClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gnome_db_shell_settings_finalize;
}

static void
gnome_db_shell_settings_init (GnomeDbShellSettings *settings,
			      GnomeDbShellSettingsClass *klass)
{
	GtkWidget *paned;
	GtkWidget *toolbar;
	GtkWidget *table;

	g_return_if_fail (GNOME_DB_IS_SHELL_SETTINGS (settings));

	/* allocate internal structure */
	settings->priv = g_new0 (GnomeDbShellSettingsPrivate, 1);
	settings->priv->shell = NULL;
	settings->priv->loaded_widgets = g_hash_table_new (g_str_hash, g_str_equal);

	/* set up widgets */
	paned = gnome_db_new_hpaned_widget ();
	gtk_box_pack_start (GTK_BOX (settings), paned, TRUE, TRUE, 0);

	settings->priv->toolbar = gnome_db_shortcut_new ();
	gtk_paned_add1 (GTK_PANED (paned), settings->priv->toolbar);
	add_components_to_shortcut (settings);
	gtk_widget_show (settings->priv->toolbar);

	settings->priv->notebook = gnome_db_new_notebook_widget ();
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (settings->priv->notebook), FALSE);
	gtk_notebook_set_show_border (GTK_NOTEBOOK (settings->priv->notebook), FALSE);
	gtk_paned_add2 (GTK_PANED (paned), settings->priv->notebook);

	/* create the default configuration tab */
	table = gnome_db_new_table_widget (10, 4, FALSE);

	settings->priv->general_save_window_pos = gnome_db_new_check_button_widget (
		_("Save window position"));
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON (settings->priv->general_save_window_pos),
		gda_config_get_boolean (GNOME_DB_SHELL_CONFIG_SAVE_WINDOW_POS));
	gtk_table_attach (GTK_TABLE (table), settings->priv->general_save_window_pos,
			  0, 1, 0, 1, GTK_FILL, GTK_FILL, 2, 2);

	gtk_notebook_append_page (GTK_NOTEBOOK (settings->priv->notebook),
				  table, gtk_label_new (_("General")));
	g_hash_table_insert (settings->priv->loaded_widgets, g_strdup (_("General")), table);
}

static void
gnome_db_shell_settings_finalize (GObject *object)
{
	GnomeDbShellSettings *settings = (GnomeDbShellSettings *) object;

	g_return_if_fail (GNOME_DB_IS_SHELL_SETTINGS (settings));

	/* free memory */
	g_hash_table_destroy (settings->priv->loaded_widgets);
	settings->priv->loaded_widgets = NULL;

	if (GNOME_DB_IS_SHELL (settings->priv->shell)) {
		bonobo_object_unref (BONOBO_OBJECT (settings->priv->shell));
		settings->priv->shell = NULL;
	}

	g_free (settings->priv);
	settings->priv = NULL;

	parent_class->finalize (object);
}

GType
gnome_db_shell_settings_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbShellSettingsClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_shell_settings_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbShellSettings),
			0,
			(GInstanceInitFunc) gnome_db_shell_settings_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbShellSettings", &info, 0);
	}
	return type;
}

/**
 * gnome_db_shell_settings_new
 */
GtkWidget *
gnome_db_shell_settings_new (GnomeDbShell *shell)
{
	GnomeDbShellSettings *settings;

	settings = g_object_new (GNOME_DB_TYPE_SHELL_SETTINGS, NULL);
	if (GNOME_DB_IS_SHELL (shell)) {
		bonobo_object_ref (BONOBO_OBJECT (shell));
		settings->priv->shell = shell;
	}

	return GTK_WIDGET (settings);
}
