/* GNOME-DB Components
 * Copyright (C) 2000-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *	Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <glib/gstrfuncs.h>
#include <gtk/gtkframe.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkvpaned.h>
#include <libgnomedb/gnome-db-grid.h>
#include <libgnomedb/gnome-db-util.h>
#include "command-dialog.h"
#include "components-connection.h"
#include "components-sql-view.h"

struct _ComponentsSqlViewPrivate {
	/* the connections we're using */
	GdaConnection *cnc;

	GList *current_recsets;

	/* internal widgets */
	GtkWidget *paned;
	GtkWidget *text;
	GtkWidget *grid;
};

static void components_sql_view_class_init (ComponentsSqlViewClass *klass);
static void components_sql_view_init       (ComponentsSqlView *sql_view,
					    ComponentsSqlViewClass *klass);
static void components_sql_view_finalize   (GObject *object);

static GObjectClass *parent_class = NULL;
static gchar *sql_keywords[] = {
	" BY ",
	" CREATE DATABASE ",
	" CREATE TABLE ",
	" DELETE ",
	" FROM ",
	" ORDER ",
	" SELECT ",
	" UPDATE ",
	NULL
};

/*
 * Callbacks
 */

static void
parse_chars (gchar *text, gchar **end_ptr, gchar **tag, gboolean start)
{
	gint i;
	gchar *next_token;

	*tag = NULL;
	*end_ptr = NULL;

	/* SQL keywords */
	for (i = 0; sql_keywords[i] != NULL; i++) {
		if (!g_ascii_strncasecmp (text, sql_keywords[i], strlen (sql_keywords[i]))) {
			*end_ptr = text + strlen (sql_keywords[i]);
			*tag = "sql";
			return;
		}
	}

	/* not at the start of a TAG, find the next one */
	for (i = 0; sql_keywords[i] != NULL; i++) {
		next_token = strstr (text, sql_keywords[i]);
		if (next_token) {
			if (*end_ptr)
				*end_ptr = (*end_ptr < next_token) ? *end_ptr : next_token;
			else
				*end_ptr = next_token;
		}
	}
}

static void
text_changed_cb (GtkTextBuffer *buffer, gpointer user_data)
{
	GtkTextIter start_iter, next_iter, tmp_iter;
	gchar *text, *start_ptr, *end_ptr, *tag;
	ComponentsSqlView *sql_view = (ComponentsSqlView *) user_data;

	g_return_if_fail (COMPONENTS_IS_SQL_VIEW (sql_view));

	/* parse the text buffer to higlight SQL keywords */
	gtk_text_buffer_get_iter_at_offset (buffer, &start_iter, 0);
	next_iter = start_iter;
	while (gtk_text_iter_forward_line (&next_iter)) {
		gboolean start = TRUE;

		start_ptr = text = gtk_text_iter_get_text (&start_iter, &next_iter);
		do {
			parse_chars (start_ptr, &end_ptr, &tag, start);
			start = FALSE;
			if (end_ptr) {
				tmp_iter = start_iter;
				gtk_text_iter_forward_chars (&tmp_iter, end_ptr - start_ptr);
			}
			else
				tmp_iter = next_iter;

			if (tag) {
				gtk_text_buffer_apply_tag_by_name (
					buffer, tag, &start_iter, &tmp_iter);
			}

			start_iter = tmp_iter;
			start_ptr = end_ptr;
		} while (end_ptr);

		g_free (text);
		start_iter = next_iter;
	}
}

/*
 * ComponentsSqlView class implementation
 */

static void
components_sql_view_class_init (ComponentsSqlViewClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = components_sql_view_finalize;
}

static void
components_sql_view_init (ComponentsSqlView *sql_view, ComponentsSqlViewClass *klass)
{
	GtkWidget *scroll;
	GtkWidget *frame;

	/* allocate our private structure */
	sql_view->priv = g_new0 (ComponentsSqlViewPrivate, 1);
	sql_view->priv->cnc = NULL;

	frame = gnome_db_new_frame_widget (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_OUT);
	gtk_box_pack_start (GTK_BOX (sql_view), frame, 1, 1, 0);

	sql_view->priv->paned = gnome_db_new_vpaned_widget ();
	gtk_container_add (GTK_CONTAINER (frame), sql_view->priv->paned);

	/* create the SQL buffer */
	scroll = gnome_db_new_scrolled_window_widget ();
	gtk_container_set_border_width (GTK_CONTAINER (scroll), 3);

	sql_view->priv->text = gnome_db_new_text_widget (NULL);
	gtk_text_buffer_create_tag (gtk_text_view_get_buffer (sql_view->priv->text), "sql",
				    "foreground", "blue",
				    "weight", PANGO_WEIGHT_BOLD,
				    NULL);
	g_signal_connect (G_OBJECT (gtk_text_view_get_buffer (sql_view->priv->text)),
			  "changed", G_CALLBACK (text_changed_cb), sql_view);
	gtk_container_add (GTK_CONTAINER (scroll), sql_view->priv->text);
	gtk_paned_add1 (GTK_PANED (sql_view->priv->paned), scroll);

	/* create the grid widget */
	sql_view->priv->grid = gnome_db_grid_new ();
	gtk_widget_show (sql_view->priv->grid);

	gtk_paned_add2 (GTK_PANED (sql_view->priv->paned), sql_view->priv->grid);

	/* initialize private data */
	sql_view->priv->current_recsets = NULL;
}

static void
components_sql_view_finalize (GObject *object)
{
	ComponentsSqlView *sql_view = (ComponentsSqlView *) object;

	g_return_if_fail (COMPONENTS_IS_SQL_VIEW (sql_view));

	/* free memory */
	g_list_foreach (sql_view->priv->current_recsets, (GFunc) g_object_unref, NULL);
	g_list_free (sql_view->priv->current_recsets);
	sql_view->priv->current_recsets = NULL;

	if (GDA_IS_CONNECTION (sql_view->priv->cnc)) {
		components_connection_close (sql_view->priv->cnc);
		sql_view->priv->cnc = NULL;
	}

	g_free (sql_view->priv);
	sql_view->priv = NULL;

	parent_class->finalize (object);
}

GType
components_sql_view_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (ComponentsSqlViewClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) components_sql_view_class_init,
			NULL,
			NULL,
			sizeof (ComponentsSqlView),
			0,
			(GInstanceInitFunc) components_sql_view_init
		};
		type = g_type_register_static (GTK_TYPE_VBOX, "ComponentsSqlView", &info, 0);
	}

	return type;
}

GtkWidget *
components_sql_view_new (GdaConnection *cnc)
{
	ComponentsSqlView *sql_view;

	sql_view = g_object_new (COMPONENTS_TYPE_SQL_VIEW, NULL);
	if (GDA_IS_CONNECTION (cnc))
		components_sql_view_set_connection (sql_view, cnc);

	return GTK_WIDGET (sql_view);
}

GtkWidget *
components_sql_view_new_from (const gchar *name,
			      const gchar *user,
			      const gchar *password)
{
	GtkWidget *sql_view;
	GdaConnection *cnc;

	/* first try to open the connection */
	cnc = components_connection_open (name, user, password, FALSE, FALSE);
	if (!GDA_IS_CONNECTION (cnc))
		return NULL;

	sql_view = components_sql_view_new (cnc);

	/* we unref the connection, since we internally ref it in
	   components_sql_view_set_connection */
	g_object_unref (G_OBJECT (cnc));
}

void
components_sql_view_cut_clipboard (ComponentsSqlView *sql_view)
{
	g_return_if_fail (COMPONENTS_IS_SQL_VIEW (sql_view));
	gnome_db_text_cut_clipboard (GTK_TEXT_VIEW (sql_view->priv->text));
}

void
components_sql_view_copy_clipboard (ComponentsSqlView *sql_view)
{
	g_return_if_fail (COMPONENTS_IS_SQL_VIEW (sql_view));
	gnome_db_text_copy_clipboard (GTK_TEXT_VIEW (sql_view->priv->text));
}

void
components_sql_view_paste_clipboard (ComponentsSqlView *sql_view)
{
	g_return_if_fail (COMPONENTS_IS_SQL_VIEW (sql_view));
	gnome_db_text_paste_clipboard (GTK_TEXT_VIEW (sql_view->priv->text));
}

GdaConnection *
components_sql_view_get_connection (ComponentsSqlView *sql_view)
{
	g_return_val_if_fail (COMPONENTS_IS_SQL_VIEW (sql_view), NULL);
	return sql_view->priv->cnc;
}

void
components_sql_view_set_connection (ComponentsSqlView *sql_view, GdaConnection *cnc)
{
	g_return_if_fail (COMPONENTS_IS_SQL_VIEW (sql_view));

	if (GDA_IS_CONNECTION (cnc))
		g_object_ref (G_OBJECT (cnc));
	if (GDA_IS_CONNECTION (sql_view->priv->cnc))
		g_object_unref (G_OBJECT (sql_view->priv->cnc));

	sql_view->priv->cnc = cnc;

	gnome_db_text_clear (GTK_TEXT_VIEW (sql_view->priv->text));
	gnome_db_grid_set_model (GNOME_DB_GRID (sql_view->priv->grid), NULL);
}

void
components_sql_view_open_command (ComponentsSqlView *sql_view)
{
	gchar *cmd;

	cmd = command_dialog_open ();
	if (cmd) {
		gnome_db_text_set_text (GTK_TEXT_VIEW (sql_view->priv->text),
					cmd, strlen (cmd));
		g_free (cmd);
	}
}

void
components_sql_view_run_current (ComponentsSqlView *sql_view)
{
	gchar *sql_txt;
	GdaCommand *cmd;
	GList *recset_list;

	g_return_if_fail (COMPONENTS_IS_SQL_VIEW (sql_view));

	if (!GDA_IS_CONNECTION (sql_view->priv->cnc))
		return;

	sql_txt = gnome_db_text_get_text (GTK_TEXT_VIEW (sql_view->priv->text));
	if (!sql_txt || strlen (sql_txt) <= 0)
		return;

	/* execute the command */
	cmd = gda_command_new (sql_txt, GDA_COMMAND_TYPE_SQL, 0);
	recset_list = gda_connection_execute_command (sql_view->priv->cnc, cmd, NULL);
	if (recset_list != NULL) {
		/* free previous list of recordsets */
		g_list_foreach (sql_view->priv->current_recsets, (GFunc) g_object_unref, NULL);
		g_list_free (sql_view->priv->current_recsets);

		sql_view->priv->current_recsets = recset_list;

		/* show the first recordset */
		gnome_db_grid_set_model (
			GNOME_DB_GRID (sql_view->priv->grid),
			GDA_DATA_MODEL (sql_view->priv->current_recsets->data));
	}
}
