/* GNOME DB components libary
 * Copyright (C) 1999 Rodrigo Moya
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtksignal.h>
#include <bonobo/bonobo-object.h>
#include "gnome-db-control.h"
#include "gnome-db-control-corba.h"

#define PARENT_TYPE BONOBO_X_OBJECT_TYPE

struct _GnomeDbControlPrivate {
	BonoboControl*     bonobo_control;
	GtkWidget*         widget;
	BonoboPropertyBag* properties;
	BonoboUIComponent* ui_component;

	gpointer           user_data;
	BonoboUIVerb*      verbs;
	gchar*             app_prefix;
	gchar*             ui_xml_file;
	gpointer           cb_user_data;
};

enum {
	GET_PROPERTY,
	SET_PROPERTY,
	SET_OWNER,
	UNSET_OWNER,
	DO_PRINT,
	DO_CUT_CLIPBOARD,
	DO_COPY_CLIPBOARD,
	DO_PASTE_CLIPBOARD,
	LAST_SIGNAL
};

static guint control_signals[LAST_SIGNAL] = { 0, };

/*
 * Callbacks
 */
static void
control_activated_cb (BonoboControl *bonobo_control,
                      gboolean activate,
                      GnomeDbControl *control)
{
	if (activate)
		gnome_db_control_activate(control);
	else
		gnome_db_control_deactivate(control);
}

static void
get_prop_cb (BonoboPropertyBag *bag, BonoboArg *arg, guint arg_id, gpointer user_data)
{
	GnomeDbControl *control = (GnomeDbControl *) user_data;

	g_return_if_fail (GNOME_DB_IS_CONTROL (control));
}

static void
set_prop_cb (BonoboPropertyBag *bag, const BonoboArg *arg, guint arg_id, gpointer user_data)
{
	GnomeDbControl *control = (GnomeDbControl *) user_data;

	g_return_if_fail (GNOME_DB_IS_CONTROL (control));
}

static void
widget_destroyed_cb (GtkWidget *w, GnomeDbControl *control)
{
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	bonobo_object_unref(BONOBO_OBJECT(control));
}

/*
 * GnomeDbControl object
 */
static void
gnome_db_control_destroy (GtkObject *object)
{
	GtkObjectClass *parent_class;
	GnomeDbControl *control = (GnomeDbControl *) object;

	g_return_if_fail(GNOME_DB_IS_CONTROL(control));

	if (control->priv->app_prefix != NULL)
		g_free (control->priv->app_prefix);
	if (control->priv->ui_xml_file != NULL)
		g_free (control->priv->ui_xml_file);

	bonobo_object_unref(BONOBO_OBJECT(control->priv->bonobo_control));
	bonobo_object_unref(BONOBO_OBJECT(control->priv->ui_component));
	g_free((gpointer) control->priv);

	parent_class = gtk_type_class(PARENT_TYPE);
	if (parent_class && parent_class->destroy)
		parent_class->destroy(object);
}

static void
gnome_db_control_class_init (GnomeDbControlClass *klass)
{
	GtkObjectClass* object_class = GTK_OBJECT_CLASS(klass);
	POA_GNOME_Database_UIControl__epv *epv = &klass->epv;
	
	control_signals[GET_PROPERTY] =
		gtk_signal_new("get_property",
		               GTK_RUN_FIRST,
		               object_class->type,
		               GTK_SIGNAL_OFFSET(GnomeDbControlClass, get_property),
		               gtk_marshal_NONE__POINTER_POINTER,
		               GTK_TYPE_NONE, 2, GTK_TYPE_POINTER, GTK_TYPE_STRING);
	control_signals[SET_PROPERTY] =
		gtk_signal_new("set_property",
		               GTK_RUN_FIRST,
		               object_class->type,
		               GTK_SIGNAL_OFFSET(GnomeDbControlClass, set_property),
		               gtk_marshal_NONE__POINTER_POINTER_POINTER,
		               GTK_TYPE_NONE, 3, GTK_TYPE_POINTER, GTK_TYPE_STRING,
		               GTK_TYPE_POINTER);
	control_signals[SET_OWNER] =
		gtk_signal_new("set_owner",
			       GTK_RUN_FIRST,
			       object_class->type,
			       GTK_SIGNAL_OFFSET (GnomeDbControlClass, set_owner),
			       gtk_signal_default_marshaller,
			       GTK_TYPE_NONE, 0);
	control_signals[UNSET_OWNER] =
		gtk_signal_new("unset_owner",
			       GTK_RUN_FIRST,
			       object_class->type,
			       GTK_SIGNAL_OFFSET (GnomeDbControlClass, unset_owner),
			       gtk_signal_default_marshaller,
			       GTK_TYPE_NONE, 0);
	control_signals[DO_PRINT] =
		gtk_signal_new("do_print",
		               GTK_RUN_FIRST,
		               object_class->type,
		               GTK_SIGNAL_OFFSET(GnomeDbControlClass, do_print),
		               gtk_signal_default_marshaller,
		               GTK_TYPE_NONE, 0);
	control_signals[DO_CUT_CLIPBOARD] =
		gtk_signal_new("do_cut_clipboard",
		               GTK_RUN_FIRST,
		               object_class->type,
		               GTK_SIGNAL_OFFSET(GnomeDbControlClass, do_cut_clipboard),
		               gtk_signal_default_marshaller,
		               GTK_TYPE_NONE, 0);
	control_signals[DO_COPY_CLIPBOARD] =
		gtk_signal_new("do_copy_clipboard",
		               GTK_RUN_FIRST,
		               object_class->type,
		               GTK_SIGNAL_OFFSET(GnomeDbControlClass, do_copy_clipboard),
		               gtk_signal_default_marshaller,
		               GTK_TYPE_NONE, 0);
	control_signals[DO_PASTE_CLIPBOARD] =
		gtk_signal_new("do_paste_clipboard",
		               GTK_RUN_FIRST,
		               object_class->type,
		               GTK_SIGNAL_OFFSET(GnomeDbControlClass, do_paste_clipboard),
		               gtk_signal_default_marshaller,
		               GTK_TYPE_NONE, 0);

	gtk_object_class_add_signals(object_class, control_signals, LAST_SIGNAL);

	object_class->destroy = gnome_db_control_destroy;
	klass->get_property = NULL;
	klass->set_property = NULL;
	klass->do_print = NULL;
	klass->do_cut_clipboard = NULL;
	klass->do_copy_clipboard = NULL;
	klass->do_paste_clipboard = NULL;

	epv->setOwner = impl_GNOME_Database_UIControl_setOwner;
	epv->unsetOwner = impl_GNOME_Database_UIControl_unsetOwner;
	epv->getVersion = impl_GNOME_Database_UIControl_getVersion;
	epv->getAuthors = impl_GNOME_Database_UIControl_getAuthors;
	epv->run = impl_GNOME_Database_UIControl_run;
	epv->doPrint = impl_GNOME_Database_UIControl_doPrint;
	epv->doCutClipboard = impl_GNOME_Database_UIControl_doCutClipboard;
	epv->doCopyClipboard = impl_GNOME_Database_UIControl_doCopyClipboard;
	epv->doPasteClipboard = impl_GNOME_Database_UIControl_doPasteClipboard;

}

static void
gnome_db_control_init (GnomeDbControl *control)
{
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	control->priv = g_new0(GnomeDbControlPrivate, 1);
	
	control->priv->bonobo_control = NULL;
	control->priv->widget = NULL;
	control->priv->properties = NULL;
	control->priv->ui_component = NULL;
	control->priv->user_data = NULL;
}

GtkType
gnome_db_control_get_type (void)
{
	static GtkType db_control_type = 0;
	
	if (!db_control_type) {
		GtkTypeInfo db_control_info = {
			"GnomeDbControl",
			sizeof (GnomeDbControl),
			sizeof (GnomeDbControlClass),
			(GtkClassInitFunc) gnome_db_control_class_init,
			(GtkObjectInitFunc) gnome_db_control_init,
			(GtkArgSetFunc) NULL,
			(GtkArgSetFunc) NULL
		};
		db_control_type = bonobo_x_type_unique(
			PARENT_TYPE,
			POA_GNOME_Database_UIControl__init, NULL,
			GTK_STRUCT_OFFSET (GnomeDbControlClass, epv),
			&db_control_info);
	}
	return db_control_type;
}

/**
 * gnome_db_control_construct
 */
GnomeDbControl *
gnome_db_control_construct (GnomeDbControl *control, GtkWidget *w)
{
	CORBA_Environment ev;
	
	g_return_val_if_fail(GNOME_DB_IS_CONTROL(control), NULL);
	g_return_val_if_fail(GTK_IS_WIDGET(w), NULL);

	control->priv->widget = w;
	gtk_signal_connect(GTK_OBJECT(control->priv->widget),
	                   "destroy",
	                   GTK_SIGNAL_FUNC(widget_destroyed_cb),
	                   (gpointer) control);

	control->priv->bonobo_control = bonobo_control_new(control->priv->widget);
	gtk_signal_connect(GTK_OBJECT(control->priv->bonobo_control),
	                   "activate",
	                   GTK_SIGNAL_FUNC(control_activated_cb),
	                   (gpointer) control);
	bonobo_object_add_interface(BONOBO_OBJECT(control),
	                            BONOBO_OBJECT(control->priv->bonobo_control));

	/* create property bag */
	control->priv->properties = bonobo_property_bag_new(
		(BonoboPropertyGetFn) get_prop_cb,
		(BonoboPropertySetFn) set_prop_cb,
		(gpointer) control);
	bonobo_control_set_properties(control->priv->bonobo_control,
	                              control->priv->properties);

	/* create UI component */
	//control->priv->ui_component = bonobo_ui_component_new_default ();
	//bonobo_control_set_ui_component(control->priv->bonobo_control,
	//                                control->priv->ui_component);

	return bonobo_object_construct(
		BONOBO_OBJECT(control),
		bonobo_object_corba_objref (BONOBO_OBJECT (control)));
}

/**
 * gnome_db_control_new
 * @w: the widget to be wrapped by the control
 *
 * Creates a new #GnomeDbControl object, which is an instance of the
 * Bonobo::Control and GNOME::Database::UIControl CORBA interfaces.
 *
 * When calling this function, you must specify an already created
 * #GtkWidget, which is the one that will be used for creating the
 * Bonobo control. You should use this class if you intend to write your
 * own Bonobo controls.
 *
 * Returns: the newly created #GnomeDbControl object.
 */
GnomeDbControl *
gnome_db_control_new (GtkWidget *w)
{
	GnomeDbControl* control;

	g_return_val_if_fail(GTK_IS_WIDGET(w), NULL);
	
	control = GNOME_DB_CONTROL(gtk_type_new(gnome_db_control_get_type()));
	return gnome_db_control_construct(control, w);
}

/**
 * gnome_db_control_set_ui
 * @control: a #GnomeDbControl object
 * @app_prefix: the prefix where your application data is installed. This is
 * used by the underlying Bonobo code, which searches for the given file in
 * that directory
 * @ui_xml_file: name of the file in @app_prefix that contains the XML
 * description of the component's UI
 * @verbs: list of verbs and corresponding actions
 * @user_data: data to be passed to the verb callback functions
 *
 * Sets the UI for the given #GnomeDbControl object. This is done by
 * specifying a file that contains the description of the UI (see Bonobo
 * documentation) and a set of verbs and their corresponding actions.
 */
void
gnome_db_control_set_ui (GnomeDbControl *control,
			 const gchar *app_prefix,
			 const gchar *ui_xml_file,
			 BonoboUIVerb *verbs,
			 gpointer user_data)
{
	g_return_if_fail (GNOME_DB_IS_CONTROL (control));

	if (control->priv->app_prefix != NULL)
		g_free (control->priv->app_prefix);
	control->priv->app_prefix = g_strdup (app_prefix);

	if (control->priv->ui_xml_file != NULL)
		g_free (control->priv->ui_xml_file);
	control->priv->ui_xml_file = g_strdup (ui_xml_file);

	control->priv->verbs = verbs;
	control->priv->cb_user_data = user_data;
}

/**
 * gnome_db_control_activate
 * @control: a #GnomeDbControl object
 *
 * Activates the given #GnomeDbControl object. This means that the UI for
 * this control is merged with the UI of the container.
 */
void
gnome_db_control_activate (GnomeDbControl *control)
{
	BonoboUIComponent *uic;

	g_return_if_fail (GNOME_DB_IS_CONTROL (control));

	uic = bonobo_control_get_ui_component (control->priv->bonobo_control);
	if (uic != NULL) {
		Bonobo_UIContainer remote_ui_container;

		remote_ui_container = bonobo_control_get_remote_ui_container (
			control->priv->bonobo_control);
		bonobo_ui_component_set_container (uic, remote_ui_container);
		bonobo_object_release_unref (remote_ui_container, NULL);

		bonobo_ui_component_add_verb_list_with_data (
			uic, control->priv->verbs, control->priv->cb_user_data);

		bonobo_ui_component_freeze (uic, NULL);
		bonobo_ui_util_set_ui (uic,
				       control->priv->app_prefix,
				       control->priv->ui_xml_file,
				       g_get_prgname);
		bonobo_ui_component_thaw (uic, NULL);
	}
}

/**
 * gnome_db_control_deactivate
 * @control: a #GnomeDbControl object
 *
 * Deactivates the given #GnomeDbControl object, which means that the
 * control's UI is removed from the container's.
 */
void
gnome_db_control_deactivate (GnomeDbControl *control)
{
	BonoboUIComponent* uic;
	
	g_return_if_fail (GNOME_DB_IS_CONTROL (control));
	
	uic = bonobo_control_get_ui_component (control->priv->bonobo_control);
	if (uic) {
		bonobo_ui_component_rm (uic, "/", NULL);
		bonobo_ui_component_unset_container (uic);
	}
}

/**
 * gnome_db_control_set_status
 * @control: a #GnomeDbControl object
 * @msg: message to show on the status bar
 *
 * Displays the given message on the container's status bar. If you use
 * #GnomeDbWindow for your containers, you automatically have a status
 * bar on your window, and, with this function, you can set the message
 * to be displayed on that status bar from the components.
 */
void
gnome_db_control_set_status (GnomeDbControl *control, const gchar *msg)
{
	BonoboUIComponent* uic;
	
	g_return_if_fail(GNOME_DB_CONTROL(control));
	
	uic = bonobo_control_get_ui_component(control->priv->bonobo_control);
	if (uic)
		bonobo_ui_component_set_status(uic, msg, NULL);
}

/**
 * gnome_db_control_get_user_data
 * @control: a #GnomeDbControl object
 *
 * Gets the user data associated with the given #GnomeDbControl object. This
 * data is user-specific, and can be used to add context-sensitive data
 * to each instance of the #GnomeDbControl class.
 *
 * Returns: a pointer to the data associated with the control
 */
gpointer
gnome_db_control_get_user_data (GnomeDbControl *control)
{
	g_return_val_if_fail(GNOME_DB_IS_CONTROL(control), NULL);
	return control->priv->user_data;
}

/**
 * gnome_db_control_set_user_data
 * @control: a #GnomeDbControl object
 *
 * Associates user data with the given #GnomeDbControl object. This data
 * can then be retrieved by using the #gnome_db_control_get_user_data
 * function.
 */
void
gnome_db_control_set_user_data (GnomeDbControl *control, gpointer user_data)
{
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	control->priv->user_data = user_data;
}

/**
 * gnome_db_control_get_widget
 * @control: a #GnomeDbControl object
 *
 * Gets the GtkWidget being wrapped by the given #GnomeDbControl. The returned
 * value for this function should be the same pointer than the one used when
 * creating the #GnomeDbControl (that is, in the call to #gnome_db_control_new).
 *
 * Returns: the GtkWidget used by the control.
 */
GtkWidget *
gnome_db_control_get_widget (GnomeDbControl *control)
{
	g_return_val_if_fail(GNOME_DB_IS_CONTROL(control), NULL);
	return control->priv->widget;
}

/**
 * gnome_db_control_get_bonobo_control
 * @control: a #GnomeDbControl object
 *
 * Gets the BonoboControl object from the given #GnomeDbControl object.
 * You may find this function interesting in cases where you need "low level"
 * access to the underlying Bonobo control.
 *
 * Returns: a BonoboControl.
 */
BonoboControl *
gnome_db_control_get_bonobo_control (GnomeDbControl *control)
{
	g_return_val_if_fail(GNOME_DB_IS_CONTROL(control), NULL);
	return control->priv->bonobo_control;
}

/**
 * gnome_db_control_get_ui_container
 * @control: a #GnomeDbControl object
 *
 * Returns the Bonobo_UIContainer for the given #GnomeDbControl object.
 *
 * Returns: the Bonobo_UIContainer for this #GnomeDbControl
 */
Bonobo_UIContainer
gnome_db_control_get_ui_container (GnomeDbControl *control)
{
	g_return_val_if_fail(GNOME_DB_IS_CONTROL(control), CORBA_OBJECT_NIL);
	return bonobo_ui_component_get_container(control->priv->ui_component);
}
