/* Dia -- an diagram creation/manipulation program
 * Copyright (C) 1998 Alexander Larsson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <assert.h>
#include <math.h>
#include <string.h>

#include "intl.h"
#include "object.h"
#include "orth_conn.h"
#include "diarenderer.h"
#include "attributes.h"
#include "arrows.h"
#include "properties.h"
#include "stereotype.h"
#include "uml.h"

#include "pixmaps/realizes.xpm"

typedef struct _Realizes Realizes;

struct _Realizes {
  OrthConn orth;

  Point text_pos;
  Alignment text_align;
  real text_width;

  Color text_color;
  Color line_color;
  
  char *name;
  char *stereotype; /* excluding << and >> */
  char *st_stereotype; /* including << and >> */
};


#define REALIZES_WIDTH 0.1
#define REALIZES_TRIANGLESIZE 0.8
#define REALIZES_DASHLEN 0.4
#define REALIZES_FONTHEIGHT 0.8

static DiaFont *realize_font = NULL;

static real realizes_distance_from(Realizes *realize, Point *point);
static void realizes_select(Realizes *realize, Point *clicked_point,
			      DiaRenderer *interactive_renderer);
static ObjectChange* realizes_move_handle(Realizes *realize, Handle *handle,
					  Point *to, ConnectionPoint *cp,
					  HandleMoveReason reason, ModifierKeys modifiers);
static ObjectChange* realizes_move(Realizes *realize, Point *to);
static void realizes_draw(Realizes *realize, DiaRenderer *renderer);
static Object *realizes_create(Point *startpoint,
				 void *user_data,
				 Handle **handle1,
				 Handle **handle2);
static void realizes_destroy(Realizes *realize);
static DiaMenu *realizes_get_object_menu(Realizes *realize,
					 Point *clickedpoint);

static PropDescription *realizes_describe_props(Realizes *realizes);
static void realizes_get_props(Realizes * realizes, GPtrArray *props);
static void realizes_set_props(Realizes * realizes, GPtrArray *props);

static Object *realizes_load(ObjectNode obj_node, int version,
			     const char *filename);

static void realizes_update_data(Realizes *realize);

static ObjectTypeOps realizes_type_ops =
{
  (CreateFunc) realizes_create,
  (LoadFunc)   realizes_load,/*using_properties*/     /* load */
  (SaveFunc)   object_save_using_properties,      /* save */
  (GetDefaultsFunc)   NULL, 
  (ApplyDefaultsFunc) NULL
};

ObjectType realizes_type =
{
  "UML - Realizes",   /* name */
  0,                      /* version */
  (char **) realizes_xpm,  /* pixmap */
  
  &realizes_type_ops,      /* ops */
  NULL,                 /* pixmap_file */
  0                     /* default_user_data */
};

static ObjectOps realizes_ops = {
  (DestroyFunc)         realizes_destroy,
  (DrawFunc)            realizes_draw,
  (DistanceFunc)        realizes_distance_from,
  (SelectFunc)          realizes_select,
  (CopyFunc)            object_copy_using_properties,
  (MoveFunc)            realizes_move,
  (MoveHandleFunc)      realizes_move_handle,
  (GetPropertiesFunc)   object_create_props_dialog,
  (ApplyPropertiesFunc) object_apply_props_from_dialog,
  (ObjectMenuFunc)      realizes_get_object_menu,
  (DescribePropsFunc)   realizes_describe_props,
  (GetPropsFunc)        realizes_get_props,
  (SetPropsFunc)        realizes_set_props
};

static PropDescription realizes_props[] = {
  ORTHCONN_COMMON_PROPERTIES,
  PROP_STD_LINE_COLOUR_OPTIONAL, 
  PROP_STD_TEXT_COLOUR_OPTIONAL, 
  { "name", PROP_TYPE_STRING, PROP_FLAG_VISIBLE,
    N_("Name:"), NULL, NULL },
  { "stereotype", PROP_TYPE_STRING, PROP_FLAG_VISIBLE,
    N_("Stereotype:"), NULL, NULL },  
  PROP_DESC_END
};

static PropDescription *
realizes_describe_props(Realizes *realizes)
{
  if (realizes_props[0].quark == 0) {
    prop_desc_list_calculate_quarks(realizes_props);
  }
  return realizes_props;
}

static PropOffset realizes_offsets[] = {
  ORTHCONN_COMMON_PROPERTIES_OFFSETS,
  { "line_colour", PROP_TYPE_COLOUR, offsetof(Realizes, line_color) },
  { "text_colour", PROP_TYPE_COLOUR, offsetof(Realizes, text_color) },
  { "name", PROP_TYPE_STRING, offsetof(Realizes, name) },
  { "stereotype", PROP_TYPE_STRING, offsetof(Realizes, stereotype) },
  { NULL, 0, 0 }
};

static void
realizes_get_props(Realizes * realizes, GPtrArray *props)
{
  object_get_props_from_offsets(&realizes->orth.object,
                                realizes_offsets,props);
}

static void
realizes_set_props(Realizes *realizes, GPtrArray *props)
{
  object_set_props_from_offsets(&realizes->orth.object, 
                                realizes_offsets, props);
  g_free(realizes->st_stereotype);
  realizes->st_stereotype = NULL;
  realizes_update_data(realizes);
}


static real
realizes_distance_from(Realizes *realize, Point *point)
{
  OrthConn *orth = &realize->orth;
  return orthconn_distance_from(orth, point, REALIZES_WIDTH);
}

static void
realizes_select(Realizes *realize, Point *clicked_point,
		  DiaRenderer *interactive_renderer)
{
  orthconn_update_data(&realize->orth);
}

static ObjectChange*
realizes_move_handle(Realizes *realize, Handle *handle,
		     Point *to, ConnectionPoint *cp,
		     HandleMoveReason reason, ModifierKeys modifiers)
{
  ObjectChange *change;
  assert(realize!=NULL);
  assert(handle!=NULL);
  assert(to!=NULL);
  
  change = orthconn_move_handle(&realize->orth, handle, to, cp, reason, modifiers);
  realizes_update_data(realize);

  return change;
}

static ObjectChange*
realizes_move(Realizes *realize, Point *to)
{
  orthconn_move(&realize->orth, to);
  realizes_update_data(realize);

  return NULL;
}

static void
realizes_draw(Realizes *realize, DiaRenderer *renderer)
{
  DiaRendererClass *renderer_ops = DIA_RENDERER_GET_CLASS (renderer);
  OrthConn *orth = &realize->orth;
  Point *points;
  int n;
  Point pos;
  Arrow arrow;
  
  points = &orth->points[0];
  n = orth->numpoints;
  
  renderer_ops->set_linewidth(renderer, REALIZES_WIDTH);
  renderer_ops->set_linestyle(renderer, LINESTYLE_DASHED);
  renderer_ops->set_dashlength(renderer, REALIZES_DASHLEN);
  renderer_ops->set_linejoin(renderer, LINEJOIN_MITER);
  renderer_ops->set_linecaps(renderer, LINECAPS_BUTT);

  arrow.type = ARROW_HOLLOW_TRIANGLE;
  arrow.width = REALIZES_TRIANGLESIZE;
  arrow.length = REALIZES_TRIANGLESIZE;
  renderer_ops->draw_polyline_with_arrows(renderer, points, n,
					   REALIZES_WIDTH,
					   &realize->line_color,
					   &arrow, NULL);

  renderer_ops->set_font(renderer, realize_font, REALIZES_FONTHEIGHT);
  pos = realize->text_pos;
  
  if (realize->st_stereotype != NULL && realize->st_stereotype[0] != '\0') {
    renderer_ops->draw_string(renderer,
			       realize->st_stereotype,
			       &pos, realize->text_align,
			       &realize->text_color);

    pos.y += REALIZES_FONTHEIGHT;
  }
  
  if (realize->name != NULL && realize->name[0] != '\0') {
    renderer_ops->draw_string(renderer,
			       realize->name,
			       &pos, realize->text_align,
			       &realize->text_color);
  }
  
}

static void
realizes_update_data(Realizes *realize)
{
  OrthConn *orth = &realize->orth;
  Object *obj = &orth->object;
  int num_segm, i;
  Point *points;
  Rectangle rect;
  PolyBBExtras *extra;

  orthconn_update_data(orth);
  
  realize->text_width = 0.0;

  realize->stereotype = remove_stereotype_from_string(realize->stereotype);
  if (!realize->st_stereotype) {
    realize->st_stereotype =  string_to_stereotype(realize->stereotype);
  }

  if (realize->name)
    realize->text_width = dia_font_string_width(realize->name, realize_font,
					    REALIZES_FONTHEIGHT);
  if (realize->stereotype)
    realize->text_width = MAX(realize->text_width,
			      dia_font_string_width(realize->stereotype,
						realize_font,
						REALIZES_FONTHEIGHT));

  extra = &orth->extra_spacing;
  
  extra->start_trans = REALIZES_WIDTH/2.0 + REALIZES_TRIANGLESIZE;
  extra->start_long = 
    extra->middle_trans = 
    extra->end_trans = 
    extra->end_long = REALIZES_WIDTH/2.0;

  orthconn_update_boundingbox(orth);
  
  /* Calc text pos: */
  num_segm = realize->orth.numpoints - 1;
  points = realize->orth.points;
  i = num_segm / 2;
  
  if ((num_segm % 2) == 0) { /* If no middle segment, use horizontal */
    if (realize->orth.orientation[i]==VERTICAL)
      i--;
  }

  switch (realize->orth.orientation[i]) {
  case HORIZONTAL:
    realize->text_align = ALIGN_CENTER;
    realize->text_pos.x = 0.5*(points[i].x+points[i+1].x);
    realize->text_pos.y = points[i].y;
    if (realize->name)
      realize->text_pos.y -=
        dia_font_descent(realize->name,realize_font, REALIZES_FONTHEIGHT);
    break;
  case VERTICAL:
    realize->text_align = ALIGN_LEFT;
    realize->text_pos.x = points[i].x + 0.1;
    realize->text_pos.y = 0.5*(points[i].y+points[i+1].y);
    if (realize->name)
      realize->text_pos.y -=
        dia_font_descent(realize->name, realize_font, REALIZES_FONTHEIGHT);
    break;
  }

  /* Add the text recangle to the bounding box: */
  rect.left = realize->text_pos.x;
  if (realize->text_align == ALIGN_CENTER)
    rect.left -= realize->text_width/2.0;
  rect.right = rect.left + realize->text_width;
  rect.top = realize->text_pos.y;
  if (realize->name)
    rect.top -= dia_font_ascent(realize->name,realize_font, REALIZES_FONTHEIGHT);
  rect.bottom = rect.top + 2*REALIZES_FONTHEIGHT;

  rectangle_union(&obj->bounding_box, &rect);
}

static ObjectChange *
realizes_add_segment_callback(Object *obj, Point *clicked, gpointer data)
{
  ObjectChange *change;
  change = orthconn_add_segment((OrthConn *)obj, clicked);
  realizes_update_data((Realizes *)obj);
  return change;
}

static ObjectChange *
realizes_delete_segment_callback(Object *obj, Point *clicked, gpointer data)
{
  ObjectChange *change;
  change = orthconn_delete_segment((OrthConn *)obj, clicked);
  realizes_update_data((Realizes *)obj);
  return change;
}


static DiaMenuItem object_menu_items[] = {
  { N_("Add segment"), realizes_add_segment_callback, NULL, 1 },
  { N_("Delete segment"), realizes_delete_segment_callback, NULL, 1 },
  ORTHCONN_COMMON_MENUS,
};

static DiaMenu object_menu = {
  "Realizes",
  sizeof(object_menu_items)/sizeof(DiaMenuItem),
  object_menu_items,
  NULL
};

static DiaMenu *
realizes_get_object_menu(Realizes *realize, Point *clickedpoint)
{
  OrthConn *orth;

  orth = &realize->orth;
  /* Set entries sensitive/selected etc here */
  object_menu_items[0].active = orthconn_can_add_segment(orth, clickedpoint);
  object_menu_items[1].active = orthconn_can_delete_segment(orth, clickedpoint);
  orthconn_update_object_menu(orth, clickedpoint, &object_menu_items[2]);
  return &object_menu;
}

static Object *
realizes_create(Point *startpoint,
	       void *user_data,
  	       Handle **handle1,
	       Handle **handle2)
{
  Realizes *realize;
  OrthConn *orth;
  Object *obj;
  PolyBBExtras *extra;

  if (realize_font == NULL) {
    realize_font = 
      dia_font_new_from_style (DIA_FONT_MONOSPACE, REALIZES_FONTHEIGHT);
  }
  
  realize = g_malloc0(sizeof(Realizes));
  orth = &realize->orth;
  obj = &orth->object;
  extra = &orth->extra_spacing;

  obj->type = &realizes_type;

  obj->ops = &realizes_ops;

  orthconn_init(orth, startpoint);

  realize->text_color = color_black;
  realize->line_color = attributes_get_foreground();

  realize->name = NULL;
  realize->stereotype = NULL;
  realize->st_stereotype = NULL;
  realize->text_width = 0;

  extra->start_trans = REALIZES_WIDTH/2.0 + REALIZES_TRIANGLESIZE;
  extra->start_long = 
    extra->middle_trans = 
    extra->end_trans = 
    extra->end_long = REALIZES_WIDTH/2.0;

  realizes_update_data(realize);
  
  *handle1 = orth->handles[0];
  *handle2 = orth->handles[orth->numpoints-2];
  return &realize->orth.object;
}

static void
realizes_destroy(Realizes *realize)
{
  g_free(realize->name);
  g_free(realize->stereotype);
  g_free(realize->st_stereotype);
  orthconn_destroy(&realize->orth);
}

static Object *
realizes_load(ObjectNode obj_node, int version, const char *filename)
{
  return object_load_using_properties(&realizes_type,
                                      obj_node,version,filename);
}
