import sys, logging, urlparse

import gobject, gtk
import bigboard.accounts as accounts

import bigboard.globals as globals
import libbig.logutil
from libbig.logutil import log_except
import bigboard.libbig.gutil as gutil

_logger = logging.getLogger("bigboard.AccountsDialog")

# this class will probably be removed since we are changing the layot of the accounts dialog
class AccountEditor(gtk.VBox):
    def __init__(self, *args, **kwargs):
        if 'account' in kwargs:
            self.__account = kwargs['account']
            del kwargs['account']
        else:
            raise Error("must provide account to AccountEditor")
        
        super(AccountEditor, self).__init__(*args, **kwargs)

        self.__username_entry = gtk.Entry()
        self.__password_entry = gtk.Entry()
        self.__password_entry.set_visibility(False)

        self.__username_entry.connect('changed',
                                      self.__on_username_entry_changed)
        self.__password_entry.connect('changed',
                                      self.__on_password_entry_changed)

        hbox = gtk.HBox(spacing=10)
        label = gtk.Label("Email")
        label.set_alignment(0.0, 0.5)
        hbox.pack_start(label)
        hbox.pack_end(self.__username_entry, False)
        self.pack_start(hbox)

        hbox = gtk.HBox(spacing=10)
        label = gtk.Label("Password")
        label.set_alignment(0.0, 0.5)
        hbox.pack_start(label)    
        hbox.pack_end(self.__password_entry, False)
        self.pack_start(hbox)

        self.show_all()

        self.__on_account_changed(self.__account)
        self.__changed_id = self.__account.connect('changed', self.__on_account_changed)

        self.__password_entry.set_activates_default(True)

        self.connect('destroy', self.__on_destroy)

    def __on_destroy(self, self2):
        self.__account.disconnect(self.__changed_id)

    def __on_account_changed(self, account):
        self.__username_entry.set_text(account.get_username())
        self.__password_entry.set_text(account.get_password())

    def __on_username_entry_changed(self, entry):
        text = entry.get_text()
        accounts.get_accounts().save_account_changes(self.__account,
                                                     { 'username' : text })

    def __on_password_entry_changed(self, entry):
        text = entry.get_text()
        accounts.get_accounts().save_account_changes(self.__account,
                                                     { 'password' : text })

class Dialog(gtk.Window):
    @log_except(_logger)      
    def __init__(self, *args, **kwargs):
        super(Dialog, self).__init__(*args, **kwargs)        
        
        self.set_title('Accounts')
        self.set_position(gtk.WIN_POS_CENTER)

        self.__notebook = gtk.Notebook()
        self.__notebook.set_border_width(5)

        self.__outer_existing_accounts_vbox = gtk.VBox() 
        self.__outer_existing_accounts_vbox.set_border_width(5)

        self.__outer_new_account_vbox = gtk.VBox() 
        self.__outer_new_account_vbox.set_border_width(5)

        # make tabs take an equal amount of space
        self.__notebook.set_property("homogeneous", True)
        self.__notebook.append_page(self.__outer_existing_accounts_vbox, gtk.Label("Existing Accounts"))
        self.__notebook.append_page(self.__outer_new_account_vbox, gtk.Label("Add Account"))
        # make tabs expand horizontally
        self.__notebook.set_tab_label_packing(self.__outer_new_account_vbox, True, True, gtk.PACK_START)

        self.__existing_accounts_vbox = gtk.VBox()
        self.__outer_existing_accounts_vbox.pack_start(self.__existing_accounts_vbox, False, False) 
        self.__new_account_vbox = gtk.VBox() 
        self.__outer_new_account_vbox.pack_start(self.__new_account_vbox, False, False) 

        self.add(self.__notebook)

        self.connect('delete-event', self.__on_delete_event)

        # stuff below gets added to the "Existing Accounts" tab

        # self.__no_accounts_label = gtk.Label("No existing accounts")
        # self.__no_accounts_label.size_allocate(gtk.gdk.Rectangle(width=0, height=0)) 
        # self.__existing_accounts_vbox.pack_start(self.__no_accounts_label)

        self.__model = gtk.ListStore(gobject.TYPE_STRING)
        self.__accounts_combo = gtk.ComboBox(self.__model)
        self.__existing_accounts_vbox.pack_start(self.__accounts_combo, True, False)
        textrender = gtk.CellRendererText()
        self.__accounts_combo.pack_start(textrender, True)
        self.__accounts_combo.add_attribute(textrender, 'text', 0)
        self.__accounts_combo.connect('notify::active', self.__on_edited_account_changed)

        # TODO: don't display the dropdown and password entry box if there are no accounts to edit 
        self.__password_entry = gtk.Entry()
        self.__password_entry.set_visibility(False) # this sets a password mode
        self.__password_entry.connect('changed',
                                      self.__on_account_settings_changed)
        self.__password_entry.connect('activate',
                                      self.__on_account_settings_applied)

        password_vbox = gtk.VBox(spacing=2)
        self.__existing_accounts_label = gtk.Label()
        password_vbox.pack_start(self.__existing_accounts_label, True, False)    
        password_vbox.pack_end(self.__password_entry, False, False)
        self.__existing_accounts_vbox.pack_start(password_vbox, padding=5)

        box_with_buttons = gtk.HBox()         
        self.__check_box = gtk.CheckButton(label="Enabled")
        box_with_buttons.pack_start(self.__check_box, False, False, padding=5)  
        self.__existing_accounts_vbox.pack_start(box_with_buttons, False, False, padding=5)

        self.__check_box.connect('toggled',
                                 self.__on_account_settings_changed)

        self.__undo_button = gtk.Button(stock=gtk.STOCK_UNDO)
        self.__undo_button.connect('clicked',
                                   self.__on_account_settings_reset)
        self.__undo_button.set_sensitive(False)                     
        self.__undo_button.size_allocate(gtk.gdk.Rectangle(width=100))               

        self.__apply_button = gtk.Button(stock=gtk.STOCK_APPLY)
        self.__apply_button.connect('clicked',
                                   self.__on_account_settings_applied)
        self.__apply_button.set_sensitive(False)

        box_with_buttons.set_spacing(5)
        box_with_buttons.set_homogeneous(False)
        box_with_buttons.pack_end(self.__apply_button, expand=False, fill=False)
        box_with_buttons.pack_end(self.__undo_button, expand=False, fill=False)

        remove_button_box = gtk.HBox()
        self.__remove_button = gtk.Button(label="Remove Account")
        remove_image = gtk.Image()
        remove_image.set_from_stock(gtk.STOCK_REMOVE, gtk.ICON_SIZE_BUTTON)
        self.__remove_button.set_image(remove_image) 
        self.__remove_button.set_sensitive(False)
        self.__remove_button.connect('clicked',
                                   self.__on_account_remove_clicked)
        remove_button_box.pack_end(self.__remove_button, expand=False, fill=False)

        self.__remove_link_box = gtk.HBox()
        self.__remove_link = Link()
        self.__remove_link.set_text("Remove account online")
        self.__remove_link.set_enabled(False)
        self.__remove_link.connect("clicked", self.__open_account_page)
        self.__remove_link_box.pack_end(self.__remove_link, expand=False, fill=False)

        self.__existing_accounts_vbox.pack_start(remove_button_box, False, False, padding=5)
        self.__existing_accounts_vbox.pack_start(self.__remove_link_box, False, False)   

        # stuff below gets added to the "Add Account" tab

        self.__account_kinds_model = gtk.ListStore(gobject.TYPE_STRING)
        self.__account_kinds_combo = gtk.ComboBox(self.__account_kinds_model)
        self.__new_account_vbox.pack_start(self.__account_kinds_combo, True, False)
        account_kinds_textrender = gtk.CellRendererText()
        self.__account_kinds_combo.pack_start(account_kinds_textrender, True)
        self.__account_kinds_combo.add_attribute(account_kinds_textrender, 'text', 0)
        for kind in accounts.ALL_ACCOUNT_KINDS:
            if not kind.get_provided_by_server():
                self.__account_kinds_model.append([kind.get_name()])
            
        # Setting padding to 1 here is a hack to get the content of both tabs to be aligned,
        # we'll need to change this when it will become possible to add new account types to
        # the existing dialog, for example, by adding stocks that require these account types
        padding = 1  
        if len(self.__account_kinds_model) > 0:
            self.__account_kinds_combo.set_active(0) 
            padding = 5

        self.__username_entry = gtk.Entry()
        self.__username_entry.connect('changed',
                                      self.__on_new_username_changed)
        self.__username_entry.connect('activate',
                                      self.__on_new_username_added)

        username_vbox = gtk.VBox(spacing=2)
        self.__username_label = gtk.Label("Username:")
        self.__username_label.set_alignment(0.0, 0.5)
        username_vbox.pack_start(self.__username_label, False, False)    
        username_vbox.pack_end(self.__username_entry, False, False)
        self.__new_account_vbox.pack_start(username_vbox, padding=padding)

        self.__add_button_box = gtk.HBox()
        self.__add_button = gtk.Button(label="Add Account")
        add_image = gtk.Image()
        add_image.set_from_stock(gtk.STOCK_ADD, gtk.ICON_SIZE_BUTTON)
        self.__add_button.set_image(add_image) 
        self.__add_button.set_sensitive(False)
        self.__add_button.connect('clicked',
                                   self.__on_new_username_added)
        self.__add_button_box.pack_end(self.__add_button, expand=False, fill=False)
        self.__new_account_vbox.pack_start(self.__add_button_box, False, False, padding=padding)

        self.__add_link_box = gtk.HBox()
        self.__add_link = Link()
        self.__add_link.connect("clicked", self.__open_account_page)
        if len(self.__account_kinds_model) > 0:
            self.__account_kinds_combo.set_active(0)
            self.__add_link.set_text("Add other accounts online")
            self.__add_link_box.pack_end(self.__add_link, expand=False, fill=False)
            self.__outer_new_account_vbox.pack_end(self.__add_link_box, False, False)
        else:  
            if len(accounts.ALL_ACCOUNT_KINDS) == 1 and \
               accounts.KIND_GOOGLE in accounts.ALL_ACCOUNT_KINDS:           
                self.__add_link.set_text("Add Google accounts online")
            else:
                self.__add_link.set_text("Add accounts online")
            self.__add_link_box.pack_start(self.__add_link)
            self.__new_account_vbox.pack_start(self.__add_link_box)
   
        self.__model_tree_iter_by_account = {}
        self.__current_account = None

        self.show_all()
        self.__set_no_accounts_state(True)
        
        if len(self.__account_kinds_model) == 0:
            self.__account_kinds_combo.hide()
            self.__username_label.hide()
            self.__username_entry.hide()
            self.__add_button.hide()
 
        self.__notebook.set_current_page(1)

        self.__connections = gutil.DisconnectSet()

        accts = accounts.get_accounts()
        id = accts.connect('account-added', self.__on_account_added)
        self.__connections.add(accts, id)
        id = accts.connect('account-removed', self.__on_account_removed)
        self.__connections.add(accts, id)

        # google_accounts = accts.get_accounts_with_kind(accounts.KIND_GOOGLE)
        all_accounts = accts.get_all_accounts()
        if len(all_accounts) == 0:
            # do something else
            pass
        else:
            for a in all_accounts:
                self.__on_account_added(accts, a)

    ## should be a destroy() that disconnects connections, but we never destroy anyway

    def __open_account_page(self, l):
        libbig.show_url(urlparse.urljoin(globals.get_baseurl(), "/account"))
   
    @log_except(_logger)      
    def __on_account_added(self, accts, a):
        # TODO: check for account kind when we have a per-account dialog
        if a not in self.__model_tree_iter_by_account:
            _logger.debug("account added %s" % a) 
            tree_iter = None
            if a.get_url() and a.get_url() != '':
                tree_iter = self.__model.append([a.get_kind().get_name() + ': ' + a.get_username() + '@' + a.get_url()])
            else:
                tree_iter = self.__model.append([a.get_kind().get_name() + ': ' + a.get_username()])
                
            self.__model_tree_iter_by_account[a] = tree_iter

            if len(self.__model_tree_iter_by_account) == 1:
                self.__set_no_accounts_state(False)
                self.__current_account = a
                self.__accounts_combo.set_active(0) 
                self.__notebook.set_current_page(0)     

    @log_except(_logger)      
    def __on_account_removed(self, accts, a):
        # we don't want to remove accounts that were disabled, but are still present in GConf here
        if accts.has_account(a):
            return  
        _logger.debug("account removed %s" % a) 
        if a in self.__model_tree_iter_by_account:
            _logger.debug("will remove")
            self.__model.remove(self.__model_tree_iter_by_account[a])
            del self.__model_tree_iter_by_account[a]
            # TODO: possibly can get the index of the deleted selection, and move to the next one 
            if self.__current_account == a and len(self.__model_tree_iter_by_account) > 0:                
                self.__current_account = self.__model_tree_iter_by_account.keys()[0]
                self.__accounts_combo.set_active_iter(self.__model_tree_iter_by_account.values()[0]) 
   
        if len(self.__model_tree_iter_by_account) == 0:   
            self.__set_no_accounts_state(True)

    def __set_no_accounts_state(self, no_accounts):
        if no_accounts:
            self.__accounts_combo.hide()
            self.__password_entry.hide()
            self.__check_box.hide()
            self.__undo_button.hide()
            self.__apply_button.hide()
            self.__remove_button.hide()
            self.__remove_link_box.hide()
            self.__existing_accounts_label.set_text("No existing accounts") 
            self.__existing_accounts_label.set_alignment(0.5, 0.0)
        else:
            self.__existing_accounts_vbox.show_all()
            self.__existing_accounts_label.set_text("Password:")
            self.__existing_accounts_label.set_alignment(0.0, 0.5)    

    def __on_edited_account_changed(self, *args):
        new_iter = self.__accounts_combo.get_active_iter()
        _logger.debug("account changed to %s" % self.__model.get_value(new_iter, 0))        
        for (a, tree_iter) in self.__model_tree_iter_by_account.items():
            # this assumes that text for each entry in the drop down box is different, which it should be       
            if self.__model.get_value(tree_iter, 0) == self.__model.get_value(new_iter, 0):
                self.__current_account = a
                # TODO: this will trigger __on_password_entry_changed, make sure that is harmless
                self.__password_entry.set_text(a.get_password())
                self.__check_box.set_active(a.get_enabled())
                self.__remove_button.set_sensitive(not a.get_kind().get_provided_by_server()) 
                self.__remove_link.set_enabled(a.get_kind().get_provided_by_server())
                return
        _logger.error("new edited account was not found in self.__model_tree_iter_by_account")

    def __on_account_settings_changed(self, widget):
        if self.__password_entry.get_text().strip() != self.__current_account.get_password() or \
           self.__check_box.get_active() != self.__current_account.get_enabled():
            self.__undo_button.set_sensitive(True)  
            self.__apply_button.set_sensitive(True)  
        else:
            self.__undo_button.set_sensitive(False)  
            self.__apply_button.set_sensitive(False)  

    def __on_account_settings_applied(self, widget):
        text = self.__password_entry.get_text()
        enabled = self.__check_box.get_active()
        accounts.get_accounts().save_account_changes(self.__current_account,
                                                     { 'password' : text, 'enabled' : enabled })
        self.__undo_button.set_sensitive(False)  
        self.__apply_button.set_sensitive(False)  

    def __on_account_settings_reset(self, widget):
        self.__password_entry.set_text(self.__current_account.get_password())
        self.__check_box.set_active(self.__current_account.get_enabled())
  
    def __on_account_remove_clicked(self, widget):
        accounts.get_accounts().remove_account(self.__current_account)

    def __on_new_username_changed(self, widget):
        # in the future can check if the input is of the desired form here
        username_entered = (len(self.__username_entry.get_text().strip()) > 0)  
        self.__add_button.set_sensitive(username_entered)        

    def __on_new_username_added(self, widget):
        text = self.__username_entry.get_text()
        username_entered = (len(text.strip()) > 0)
        # don't do anything if the user pressed enter in the username textbox while it is empty 
        if not username_entered:
            return
        
        current_iter = self.__account_kinds_combo.get_active_iter()
        for kind in accounts.ALL_ACCOUNT_KINDS:
            if not kind.get_provided_by_server() and self.__account_kinds_model.get_value(current_iter, 0) == kind.get_name():
                account_tuple = accounts.get_accounts().get_or_create_account(kind, text)
                self.__username_entry.set_text("")
                self.__notebook.set_current_page(0)
                # TODO: Display a special message if the account aready existed (account_tuple[1] is False), also
                # the key should be self.__model_tree_iter_by_account only in that case
                if self.__model_tree_iter_by_account.has_key(account_tuple[0]):
                    account_iter = self.__model_tree_iter_by_account[account_tuple[0]]                
                    self.__accounts_combo.set_active_iter(account_iter)
                else:
                    # acounts system will emit a signal that will cause __on_account_added to be called again, 
                    # but it's ok to call this from here so that we can switch to the new account in the combo box right away
                    self.__on_account_added(None, account_tuple[0])
                    account_iter = self.__model_tree_iter_by_account[account_tuple[0]]   
                    self.__accounts_combo.set_active_iter(account_iter)
                return 
                
        _logger.warn("Did not find an account kind which is not provided by the server that matched the account kind in the dropdown %s" % self.__account_kinds_model.get_value(current_iter, 0))

    def __on_delete_event(self, dialog, event):
        self.hide()
        return True
        
class Link(gtk.EventBox):
    __gsignals__ = {
        "clicked" : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, ()),
    }
    
    def __init__(self,**kwargs):
        super(Link, self).__init__(**kwargs)
        self.__text = None
        self.__enabled = True
        self.__label = gtk.Label()
        self.add(self.__label)
        self.connect("button-press-event", self.__on_button_press)
        self.connect("enter_notify_event", self.__on_enter) 
        self.connect("leave_notify_event", self.__on_leave) 
        self.add_events(gtk.gdk.BUTTON_PRESS_MASK
                        & gtk.gdk.ENTER_NOTIFY_MASK
                        & gtk.gdk.LEAVE_NOTIFY_MASK)

    def set_alignment(self, x, y):
        self.__label.set_alignment(x, y)
        
    def set_ellipsize(self, do_ellipsize):
        self.__label.set_ellipsize(do_ellipsize)

    def __on_button_press(self, self2, e):
        if e.button == 1 and self.__enabled:
            self.emit("clicked")
            return True
        return False

    def get_text(self):
        return self.__text
    
    def set_text(self, text):
        self.__text = text
        self.set_markup(gobject.markup_escape_text(text))
    
    def set_enabled(self, enabled):        
        self.__enabled = enabled
        self.set_markup(gobject.markup_escape_text(self.__text))

    def set_markup(self, text):
        if  self.__enabled: 
            self.__label.set_markup('<span foreground="blue">%s</span>' % (text,))
        else:
            self.__label.set_markup('<span foreground="#666666">%s</span>' % (text,))

    def __on_enter(self, w, c):
        self.__talk_to_the_hand(True)

    def __on_leave(self, w, c):
        self.__talk_to_the_hand(False)

    def __talk_to_the_hand(self, hand):
        display = self.get_display()
        cursor = None
        if hand and self.__enabled:
            cursor = gtk.gdk.Cursor(display, gtk.gdk.HAND2)
        self.window.set_cursor(cursor)

__dialog = None

@log_except(_logger)      
def open_dialog():
    global __dialog
    if not __dialog:
        __dialog = Dialog()
    __dialog.present()
    

if __name__ == '__main__':

    import gtk, gobject

    import bigboard.libbig
    try:
        import bigboard.bignative as bignative
    except:
        import bignative

    import dbus.glib

    import bigboard.google as google

    gobject.threads_init()

    libbig.logutil.init('DEBUG', ['AsyncHTTP2LibFetcher', 'bigboard.Keyring', 'bigboard.Google', 'bigboard.Accounts', 'bigboard.AccountsDialog'])

    bignative.set_application_name("BigBoard")
    bignative.set_program_name("bigboard")

    google.init()

    open_dialog()

    gtk.main()
    
