/* This is -*- C -*- */
/* vim: set sw=2: */
/* $Id: calc-extra.c,v 1.6 2001/08/21 02:54:51 trow Exp $ */

/*
 * calc-extra.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 * Copyright (C) 2001 The Free Software Foundation
 *
 * Developed by Jon Trowbridge <trow@gnu.org>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <guppi-convenient.h>
#include <guppi-date-series.h>
#include "calc-extra.h"

typedef struct _PriceSlice PriceSlice;
struct _PriceSlice {
  GuppiPriceSeries *ser;
  guint code;
};

static PriceSlice *
ps_new (GuppiPriceSeries *ser, guint code)
{
  PriceSlice *ps = guppi_new0 (PriceSlice, 1);
  ps->ser = ser;
  guppi_ref (ser);
  ps->code = code;
  return ps;
}

static void
ps_free (gpointer ptr)
{
  PriceSlice *ps = (PriceSlice *)ptr;
  guppi_unref0 (ps->ser);
  guppi_free (ps);
}

static void
ps_bounds (GDate *sd, GDate *ed, gpointer ud)
{
  PriceSlice *ps = (PriceSlice *)ud;

  if (sd)
    *sd = *guppi_date_indexed_start (GUPPI_DATE_INDEXED (ps->ser));
  if (ed)
    *ed = *guppi_date_indexed_end (GUPPI_DATE_INDEXED (ps->ser));
}

static gboolean
ps_valid (const GDate *dt, gpointer ud)
{
  PriceSlice *ps = (PriceSlice *)ud;

  return guppi_price_series_valid (ps->ser, dt) & ps->code;
}

static double
ps_get (const GDate *dt, gpointer ud)
{
  PriceSlice *ps = (PriceSlice *)ud;
  
  return guppi_price_series_get (ps->ser, ps->code, dt);
}

static gint
ps_get_many (const GDate *dt, gint count, double *buf, gpointer ud)
{
  PriceSlice *ps = (PriceSlice *)ud;

  return guppi_price_series_get_many (ps->ser, dt, count,
				      NULL,
				      ps->code & PRICE_OPEN  ? buf : NULL,
				      ps->code & PRICE_HIGH  ? buf : NULL,
				      ps->code & PRICE_LOW   ? buf : NULL,
				      ps->code & PRICE_CLOSE ? buf : NULL);
}

static gint
ps_get_range (const GDate *sd, const GDate *ed,
	      double *tbuf, double *buf, gint bufsize, gpointer ud)
{
  PriceSlice *ps = (PriceSlice *)ud;

  return guppi_price_series_get_range (ps->ser, sd, ed,
				       tbuf,
				       ps->code & PRICE_OPEN  ? buf : NULL,
				       ps->code & PRICE_HIGH  ? buf : NULL,
				       ps->code & PRICE_LOW   ? buf : NULL,
				       ps->code & PRICE_CLOSE ? buf : NULL);
}

static gboolean
ps_get_bounds (const GDate *sd, const GDate *ed, double *min, double *max, gpointer ud)
{
  PriceSlice *ps = (PriceSlice *)ud;

  return guppi_price_series_get_bounds (ps->ser, ps->code, sd, ed, 
					min, max);
}

GuppiData*
guppi_date_series_new_sliced_price_series (GuppiPriceSeries *ser, guint code)
{
  PriceSlice *ps;
  GuppiData *slice;

  g_return_val_if_fail (ser && GUPPI_IS_PRICE_SERIES (ser), NULL);

  ps = ps_new (ser, code);

  slice = guppi_data_new ("GuppiDateSeriesCalc");
  gtk_object_set (GTK_OBJECT (slice),
		  "bounds_fn", ps_bounds,
		  "valid_fn", ps_valid,
		  "get_fn", ps_get,
		  "get_many_fn", ps_get_many,
		  "get_range_fn", ps_get_range,
		  "get_bounds_fn", ps_get_bounds,
		  "user_data", ps,
		  "user_data_destroy_fn", ps_free,
		  NULL);

  return slice;
}


GuppiData *
guppi_date_series_new_price_series_open (GuppiPriceSeries *ser)
{
  g_return_val_if_fail (ser && GUPPI_IS_PRICE_SERIES (ser), NULL);

  return guppi_date_series_new_sliced_price_series (ser, PRICE_OPEN);
}

GuppiData *
guppi_date_series_new_price_series_low (GuppiPriceSeries *ser)
{
  g_return_val_if_fail (ser && GUPPI_IS_PRICE_SERIES (ser), NULL);

  return guppi_date_series_new_sliced_price_series (ser, PRICE_LOW);
}

GuppiData *
guppi_date_series_new_price_series_high (GuppiPriceSeries *ser)
{
  g_return_val_if_fail (ser && GUPPI_IS_PRICE_SERIES (ser), NULL);

  return guppi_date_series_new_sliced_price_series (ser, PRICE_HIGH);
}

GuppiData *
guppi_date_series_new_price_series_close (GuppiPriceSeries *ser)
{
  g_return_val_if_fail (ser && GUPPI_IS_PRICE_SERIES (ser), NULL);

  return guppi_date_series_new_sliced_price_series (ser, PRICE_CLOSE);
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

typedef struct _MovAvg MovAvg;
struct _MovAvg {
  GuppiDateSeries *ser;
  gint days;
};

static MovAvg *
movavg_new (GuppiDateSeries *ser, gint days)
{
  MovAvg *movavg = guppi_new0 (MovAvg, 1);

  movavg->ser = ser;
  movavg->days = days;

  guppi_ref (ser);

  return movavg;
}

static void
movavg_free (gpointer ptr)
{
  MovAvg *movavg = (MovAvg *)ptr;

  guppi_unref0 (movavg->ser);
  guppi_free (movavg);
}

static void
movavg_bounds (GDate *sd, GDate *ed, gpointer ud)
{
  MovAvg *movavg = (MovAvg *)ud;

  if (sd) {
    *sd = *guppi_date_indexed_start (GUPPI_DATE_INDEXED (movavg->ser));
    if (g_date_valid (sd))
      guppi_date_indexed_step (GUPPI_DATE_INDEXED (movavg->ser), 
			       sd, movavg->days-1);
  }

  if (ed)
    *ed = *guppi_date_indexed_end (GUPPI_DATE_INDEXED (movavg->ser));
}

static gboolean
movavg_valid (GDate *dt, gpointer ud)
{
  MovAvg *movavg = (MovAvg *)ud;

  return guppi_date_indexed_valid (GUPPI_DATE_INDEXED (movavg->ser), dt);
}

static double
movavg_get (GDate *dt, gpointer ud)
{
  MovAvg *movavg = (MovAvg *)ud;
  double *buf = guppi_new0 (double, movavg->days);
  gint i, N;
  double sum = 0;

  N = guppi_date_series_get_many (movavg->ser, dt, -movavg->days, buf);

  g_assert (N > 0);

  for (i=0; i<N; ++i)
    sum += buf[i];

  guppi_free (buf);

  return sum/N;
}

#if 0
static gint
movavg_get_many (GDate *dt, gint count, double *buf, gpointer ud)
{
  MovAvg *movavg = (MovAvg *)ud;

  return 0;
}
#endif

static gint
movavg_get_range (GDate *sd, GDate *ed, 
		  double *in_tbuf, double *in_buf,
		  gint bufsize, gpointer ud)
{
  MovAvg *movavg = (MovAvg *)ud;
  GDate sd2 = *sd;
  gint i, N, j;
  double *tbuf;
  double *buf;
  double sum = 0;

  guppi_date_indexed_step (GUPPI_DATE_INDEXED (movavg->ser),
			   &sd2,
			   - movavg->days);
  N = (gint)g_date_julian (ed) - (gint)g_date_julian (&sd2) + 1;

  tbuf = guppi_new0 (double, N);
  buf = guppi_new0 (double, N);

  N = guppi_date_series_get_range_timecoded (movavg->ser, 
					     &sd2, ed, tbuf, buf, N);

  if (N < movavg->days) {
    guppi_free (tbuf);
    guppi_free (buf);
    return 0;
  }

  for (i=0; i<movavg->days-1; ++i)
    sum += buf[i];

  j = 0;
  for (i=movavg->days-1; i<N && j<bufsize; ++i) {
    sum += buf[i];
    if (i >= movavg->days)
      sum -= buf[i - movavg->days];

    in_tbuf[j] = tbuf[i];
    in_buf[j] = sum / movavg->days;
    ++j;
  }

  guppi_free (tbuf);
  guppi_free (buf);

  return j;
}

GuppiData *
guppi_date_series_new_moving_average (GuppiDateSeries *ser,
				      gint days)
{
  GuppiData *d;

  g_return_val_if_fail (ser && GUPPI_IS_DATE_SERIES (ser), NULL);
  g_return_val_if_fail (days >= 1, NULL);

  d = guppi_data_new ("GuppiDateSeriesCalc");
  gtk_object_set (GTK_OBJECT (d),
		  "bounds_fn", movavg_bounds,
		  "valid_fn", movavg_valid,
		  "get_fn", movavg_get,
#if 0
		  "get_many_fn", movavg_get_many,
#endif
		  "get_range_fn", movavg_get_range,
		  "user_data", movavg_new (ser, days),
		  "user_data_destroy_fn", movavg_free,
		  NULL);

  return d;
}


/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

typedef struct _FnSeries FnSeries;
struct _FnSeries {
  GuppiFnWrapper *fn;
  GuppiDateSeries *ser;
};

static FnSeries *
fn_new (GuppiFnWrapper *fn, GuppiDateSeries *ser)
{
  FnSeries *fs = guppi_new0 (FnSeries, 1);
  fs->fn = fn;
  fs->ser = ser;
  guppi_ref (fn);
  guppi_ref (ser);

  return fs;
}

static void
fn_free (gpointer ptr)
{
  FnSeries *fs = (FnSeries *)ptr;
  guppi_unref0 (fs->fn);
  guppi_unref0 (fs->ser);
  guppi_free (fs);
}

static void
fn_bounds (GDate *sd, GDate *ed, gpointer ud)
{
  FnSeries *fs = (FnSeries *)ud;

  if (sd) *sd = *guppi_date_indexed_start (GUPPI_DATE_INDEXED (fs->ser));
  if (ed) *ed = *guppi_date_indexed_end (GUPPI_DATE_INDEXED (fs->ser));
}

static gboolean
fn_valid (GDate *dt, gpointer ud)
{
  FnSeries *fs = (FnSeries *)ud;

  return guppi_date_indexed_valid (GUPPI_DATE_INDEXED (fs->ser), dt);
}

static double
fn_get (GDate *dt, gpointer ud)
{
  FnSeries *fs = (FnSeries *)ud;
  double x;

  x = guppi_date_series_get (fs->ser, dt);
  return guppi_fn_wrapper_eval_d__d (fs->fn, x);
}

static gint
fn_get_many (GDate *dt, gint count, double *buf, gpointer ud)
{
  FnSeries *fs = (FnSeries *)ud;
  gint N;

  N = guppi_date_series_get_many (fs->ser, dt, count, buf);

  guppi_fn_wrapper_eval_d__d_bulk (fs->fn, buf, N);

  return N;
}

static gint
fn_get_range (GDate *sd, GDate *ed,
	      double *tbuf, double *buf,
	      gint bufsize, gpointer ud)
{
  FnSeries *fs = (FnSeries *)ud;
  gint N;

  N = guppi_date_series_get_range_timecoded (fs->ser, sd, ed,
					     tbuf, buf, bufsize);

  guppi_fn_wrapper_eval_d__d_bulk (fs->fn, buf, N);

  return N;
}



GuppiData *
guppi_date_series_new_func (GuppiFnWrapper *fn,
			    GuppiDateSeries *ser)
{
  GuppiData *d;

  g_return_val_if_fail (fn && GUPPI_IS_FN_WRAPPER (fn), NULL);
  g_return_val_if_fail (ser && GUPPI_IS_DATE_SERIES (ser), NULL);

  d = guppi_data_new ("GuppiDateSeriesCalc");

  gtk_object_set (GTK_OBJECT (d),
		  "bounds_fn", fn_bounds,
		  "valid_fn", fn_valid,
		  "get_fn", fn_get,
		  "get_many_fn", fn_get_many,
		  "get_range_fn", fn_get_range,
		  "user_data", fn_new (fn, ser),
		  "user_data_destroy_fn", fn_free,
		  NULL);

  return d;
}

/* $Id: calc-extra.c,v 1.6 2001/08/21 02:54:51 trow Exp $ */
