/* This is -*- C -*- */
/* $Id: guppi-configure.c,v 1.6 2001/10/10 06:48:09 trow Exp $ */

/*
 * guppi-configure.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "guppi-configure.h"

#include "guppi-convenient.h"
#include "guppi-multiview.h"


typedef struct _GuppiConfigurePrivate GuppiConfigurePrivate;
struct _GuppiConfigurePrivate {
  GuppiConfigItem *root_item;

  GtkWidget *tree;
  GtkWidget *multiview;

  GuppiConfigItem *current_item;
};

#define priv(x) ((GuppiConfigurePrivate*)((x)->opaque_internals))

static GtkObjectClass *parent_class = NULL;

static void
guppi_configure_destroy (GtkObject * obj)
{
  if (parent_class->destroy)
    parent_class->destroy (obj);
}

static void
guppi_configure_finalize (GtkObject * obj)
{
  GuppiConfigure *conf = GUPPI_CONFIGURE (obj);
  GuppiConfigurePrivate *p = priv (conf);

  guppi_finalized (obj);

  guppi_config_item_destroy_tree (p->root_item);

  guppi_free (conf->opaque_internals);
  conf->opaque_internals = NULL;

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

static void
guppi_configure_class_init (GuppiConfigureClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;

  parent_class = gtk_type_class (gtk_hpaned_get_type ());

  object_class->destroy = guppi_configure_destroy;
  object_class->finalize = guppi_configure_finalize;
}

static void
guppi_configure_init (GuppiConfigure * obj)
{
  obj->opaque_internals = guppi_new0 (GuppiConfigurePrivate, 1);
}

GtkType
guppi_configure_get_type (void)
{
  static GtkType guppi_configure_type = 0;
  if (!guppi_configure_type) {
    static const GtkTypeInfo guppi_configure_info = {
      "GuppiConfigure",
      sizeof (GuppiConfigure),
      sizeof (GuppiConfigureClass),
      (GtkClassInitFunc) guppi_configure_class_init,
      (GtkObjectInitFunc) guppi_configure_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_configure_type = gtk_type_unique (gtk_hpaned_get_type (),
					    &guppi_configure_info);
  }
  return guppi_configure_type;
}

static void
tree_select_row_cb (GtkWidget * w, GList * lnode, gint col,
		    gpointer user_data)
{
  GtkCTreeNode *ctree_node;
  GuppiConfigurePrivate *p;
  GuppiConfigItem *item;
  GtkWidget *item_widget;

  g_return_if_fail (w != NULL && GTK_IS_CTREE (w));
  g_return_if_fail (user_data != NULL && GUPPI_IS_CONFIGURE (user_data));

  p = priv (GUPPI_CONFIGURE (user_data));

  ctree_node = GTK_CTREE_NODE (lnode);
  item = (GuppiConfigItem *) gtk_ctree_node_get_row_data (GTK_CTREE (w),
							  ctree_node);

  if (item == NULL)
    return;

  item_widget = guppi_config_item_widget (item);

  if (!guppi_multiview_contains (GUPPI_MULTIVIEW (p->multiview), item_widget))
    guppi_multiview_prepend_child (GUPPI_MULTIVIEW (p->multiview),
				   item_widget);

  guppi_multiview_set_current (GUPPI_MULTIVIEW (p->multiview), item_widget);

  p->current_item = item;

  gtk_paned_set_position (GTK_PANED (user_data), -1);
}


void
guppi_configure_construct (GuppiConfigure * conf)
{
  GuppiConfigurePrivate *p;

  g_return_if_fail (conf != NULL && GUPPI_IS_CONFIGURE (conf));

  p = priv (conf);

  p->tree = gtk_ctree_new (1, 0);
  gtk_clist_set_column_auto_resize (GTK_CLIST (p->tree), 0, TRUE);
  gtk_clist_set_selection_mode (GTK_CLIST (p->tree), GTK_SELECTION_BROWSE);
  gtk_ctree_set_expander_style (GTK_CTREE (p->tree),
				GTK_CTREE_EXPANDER_TRIANGLE);
  gtk_ctree_set_line_style (GTK_CTREE (p->tree), GTK_CTREE_LINES_NONE);

  gtk_signal_connect (GTK_OBJECT (p->tree),
		      "tree_select_row",
		      GTK_SIGNAL_FUNC (tree_select_row_cb), conf);

  gtk_paned_pack1 (GTK_PANED (conf), p->tree, FALSE, TRUE);
  gtk_widget_show (p->tree);

  p->multiview = guppi_multiview_new ();
  gtk_paned_pack2 (GTK_PANED (conf), p->multiview, TRUE, FALSE);
  gtk_widget_show (p->multiview);

  gtk_paned_set_position (GTK_PANED (conf), -1);
}

GtkWidget *
guppi_configure_new (void)
{
  GtkWidget *w = GTK_WIDGET (guppi_type_new (guppi_configure_get_type ()));

  guppi_configure_construct (GUPPI_CONFIGURE (w));

  return w;
}

static void
remove_iter_fn (GuppiConfigItem * item, gpointer ptr)
{
  GtkContainer *container = GTK_CONTAINER (ptr);

  if (guppi_config_item_has_widget (item))
    gtk_container_remove (container, guppi_config_item_widget (item));
}

void
guppi_configure_set_item_tree (GuppiConfigure * conf, GuppiConfigItem * root)
{
  GuppiConfigurePrivate *p;

  g_return_if_fail (conf != NULL && GUPPI_IS_CONFIGURE (conf));
  g_return_if_fail (root != NULL && GUPPI_IS_CONFIG_ITEM (root));

  p = priv (conf);

  guppi_config_item_widget_transfer (root, p->root_item);

  /* Remove "excess" widgets */
  if (p->root_item)
    guppi_config_item_foreach (p->root_item, remove_iter_fn, p->multiview);

  gtk_clist_freeze (GTK_CLIST (p->tree));
  gtk_clist_clear (GTK_CLIST (p->tree));

  guppi_config_item_build_ctree (root, GTK_CTREE (p->tree));

  gtk_clist_thaw (GTK_CLIST (p->tree));

  if (p->current_item)
    p->current_item = guppi_config_item_find_similar (root, p->current_item);

  if (p->current_item == NULL)
    p->current_item = root;

  if (p->root_item) {
    guppi_config_item_destroy_tree (p->root_item);
  }
  p->root_item = root;

  guppi_ref (p->root_item);
  guppi_sink (p->root_item);

  guppi_configure_select (conf, p->current_item);
}

void
guppi_configure_select (GuppiConfigure * conf, GuppiConfigItem * item)
{
  GuppiConfigurePrivate *p;
  GtkCTreeNode *node;

  g_return_if_fail (conf != NULL && GUPPI_IS_CONFIGURE (conf));

  g_return_if_fail (item != NULL && GUPPI_IS_CONFIG_ITEM (item));
  p = priv (GUPPI_CONFIGURE (conf));

  g_return_if_fail (p->root_item != NULL);

  node = gtk_ctree_find_by_row_data (GTK_CTREE (p->tree),
				     gtk_ctree_node_nth (GTK_CTREE (p->tree),
							 0), item);

  gtk_ctree_select (GTK_CTREE (p->tree), node);

  /* Made sure the node is visible by walking up the tree and expanding
     all of the node's parents */

  if (!gtk_ctree_is_viewable (GTK_CTREE (p->tree), node)) {
    while (node) {
      node = GTK_CTREE_ROW (node)->parent;
      if (node)
	gtk_ctree_expand (GTK_CTREE (p->tree), node);
    }
  }
}

gboolean
guppi_configure_contains_tag (GuppiConfigure * conf, gconstpointer tag)
{
  GuppiConfigurePrivate *p;

  g_return_val_if_fail (conf != NULL && GUPPI_IS_CONFIGURE (conf), FALSE);
  p = priv (conf);

  if (p->root_item == NULL)
    return FALSE;

  return guppi_config_item_find_by_tag (p->root_item, tag) != NULL;
}


void
guppi_configure_select_by_tag (GuppiConfigure * conf, gconstpointer tag)
{
  GuppiConfigurePrivate *p;
  GuppiConfigItem *item;

  g_return_if_fail (conf != NULL && GUPPI_IS_CONFIGURE (conf));
  p = priv (conf);

  g_return_if_fail (p->root_item != NULL);

  item = guppi_config_item_find_by_tag (p->root_item, tag);
  g_return_if_fail (item != NULL);

  guppi_configure_select (conf, item);
}

/* $Id: guppi-configure.c,v 1.6 2001/10/10 06:48:09 trow Exp $ */
