/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* vim: set sw=8: */

/*
 * guppi-gnumeric-interpreter.c:
 *
 * Copyright (C) 2001 Jody Goldberg (jgoldberg@home.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "guppi-gnumeric-interpreter.h"
#include "guppi-gnumeric-manager.h"
#include "guppi-gnumeric-vector.h"
#include "guppi-data-table-bundle.h"

#include <guppi-root-group-state.h>
#include <guppi-group-view-layout.h>
#include <guppi-memory.h>
#include <guppi-seq-string.h>
#include <guppi-seq-scalar.h>
#include <guppi-metrics.h>
#include <guppi-enums.h>
#include <guppi-marker.h>
#include <guppi-defaults.h>
#include <guppi-rgb.h>

#include <gal/util/e-xml-utils.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/xmlmemory.h>

#define GAP	3.6	/* 1/20 inch */

static GuppiData *
xlplots_stock_labels (int const n)
{
	static GPtrArray* strings = NULL;
	GuppiSeqString *labels;
	int i;

	if (strings == NULL)
		strings = g_ptr_array_new ();

	for (i = strings->len ; i < n ; ) {
		char *str = guppi_strdup_printf ("%d", ++i);
		g_ptr_array_add (strings, str);
	}

	{
		GuppiData* tmp = guppi_seq_string_new ();
		labels = GUPPI_SEQ_STRING (tmp);
	}
	for (i = 0; i < n ; ++i)
		guppi_seq_string_append_nc (labels, g_ptr_array_index (strings, i));

	return GUPPI_DATA (labels);
}

static void
series_dimension_new (xmlNode *series, char const *name, int id)
{
	xmlNode *vec = xmlNewChild (series, series->ns, "Dimension", NULL);
	xmlSetProp (vec, "element", name);
	e_xml_set_integer_prop_by_name (vec, "ID", id);
}

static GupGnmVector *
series_get_vector (GupGnmManager const *manager, xmlNode *series,
		   char const *target)
{
	xmlNode *dim;
	xmlChar *element;
	
	for (dim = series->xmlChildrenNode; dim != NULL ; dim = dim->next) {
		if (strcmp (dim->name, "Dimension"))
			continue;
		element = xmlGetProp (dim, "element");
		if (element != NULL) {
			if (strcmp (element, target) == 0) {
				int id = e_xml_get_integer_prop_by_name_with_default (dim, "ID", -1);
				xmlFree (element);
				if (id < 0)
					return NULL;
				return gup_gnm_manager_get_vector (manager, id);
			}
			xmlFree (element);
		}
	}
	return NULL;
}

static guint32
series_get_color (GupGnmManager const *manager, xmlNode *series)
{
	int i = e_xml_get_integer_prop_by_name_with_default (series,
		"index", -1);

	g_return_val_if_fail (i >= 0, 0);

	return guppi_color_palette_get (manager->series.colours, i);
}

static GuppiMarker
stock_marker_alien (int indx)
{
	static GuppiMarker const markers[] = {
		GUPPI_MARKER_FILLED_DIAMOND,
		GUPPI_MARKER_FILLED_SQUARE,
		GUPPI_MARKER_FILLED_TRIANGLE,
		GUPPI_MARKER_X,
		GUPPI_MARKER_AST,
		GUPPI_MARKER_FILLED_CIRCLE,
		GUPPI_MARKER_CROSS,
		GUPPI_MARKER_HALF_BAR,
		GUPPI_MARKER_BAR,
	};
	static int const marker_count = sizeof(markers)/sizeof(GuppiMarker);
	return markers [indx % marker_count];
}

static GuppiMarker
series_get_marker (GupGnmManager const *manager, xmlNode *series)
{
	int i = e_xml_get_integer_prop_by_name_with_default (series,
		"index", -1);

	g_return_val_if_fail (i >= 0, 0);

	return g_array_index (manager->series.markers, GuppiMarker, i);
}

static GuppiElementState *
unimplemented_plot (void)
{
	return guppi_element_state_new("text",
		"font",	guppi_default_font_large (),
		"text",	"Unimplemented",
		NULL);
}

static GSList *
bar_plot (GupGnmManager *manager, xmlNode *plot)
{
	GuppiElementState *bar_state;
	GuppiColorPalette *colours;
	xmlNodePtr first_series;
	int count = 0, rows = 0;

	xmlNode *desc = e_xml_get_child_by_name (
		e_xml_get_child_by_name (plot, "Type"), "Bar");
	gboolean const is_horizontal	=
		e_xml_get_bool_prop_by_name_with_default (desc, "horizontal", TRUE);
	gboolean const is_stacked	=
		e_xml_get_bool_prop_by_name_with_default (desc, "stacked", FALSE);
	gboolean const as_percentage	=
		e_xml_get_bool_prop_by_name_with_default (desc, "as_percentage", FALSE);

	GuppiData *categories = NULL;
	xmlNode *series = e_xml_get_child_by_name (plot, "Data");
	GuppiDataTable *table = guppi_data_table_bundle_new ();

	g_assert (table != NULL);
	g_return_val_if_fail (series != NULL, NULL);

	count = 0;

	first_series = series->xmlChildrenNode;

	/* First, walk through the tree and count how many series we have. */
	for (series = first_series; series != NULL; series = series->next) {
		GupGnmVector *vec;
		
		if (strcmp (series->name, "Series"))
			continue;
		vec = series_get_vector (manager, series, "values");
		if (vec != NULL) {
			GuppiData *data = gup_gnm_vector_data_get (vec, TRUE);
			gint i1 = guppi_seq_max_index (GUPPI_SEQ (data)) + 1;
			if (i1 > rows)
				rows = i1;
		}

		++count;
	}

	guppi_data_table_set_dimensions (table, rows, count);

	count = 0;
	colours = guppi_color_palette_new ();

	/* Next, walk through the tree again and populate our data table.*/
	for (series = first_series; series != NULL ; series = series->next) {
		GupGnmVector *vec;

		if (strcmp (series->name, "Series"))
			continue;
		vec = series_get_vector (manager, series, "values");
		if (vec != NULL) {
			GuppiData *data = gup_gnm_vector_data_get (vec, TRUE);
			guppi_data_table_bundle_set_col_data (GUPPI_DATA_TABLE_BUNDLE (table),
				count, GUPPI_SEQ_SCALAR (data));

			vec = series_get_vector (manager, series, "categories");
			if (vec != NULL) {
				GuppiData *d = gup_gnm_vector_data_get (vec, FALSE);
				if (categories == NULL) {
					categories = d;
					guppi_data_table_bundle_set_row_labels (GUPPI_DATA_TABLE_BUNDLE (table), 
										GUPPI_SEQ_STRING (categories));
					g_message ("categories:");
					guppi_data_spew_xml (d);
				} else if (categories != d) {
					g_warning ("Multiple categories for bars in the same plot is not supported");
				}
			}
		}

		guppi_color_palette_set (colours, count,
			series_get_color (manager, series));
		count++;
	}

	{
		gint R, C;
		guppi_data_table_get_dimensions (table, &R, &C);
		g_message ("count=%d  R=%d  C=%d", count, R, C);
	}

	/* Build our state objects */
	bar_state = guppi_element_state_new ("barchart",
		"data", 		table,
		"stacked",	  	is_stacked || as_percentage,
		"normalize_stacks",	as_percentage,
		"vertical_bars",	!is_horizontal,
		"use_stock_colors",	FALSE,

		/* Ideally this would be 0 for an
		 * un-anti-aliased un-scaled hair width
		 * line */
		"edge_thickness",	.5,

		"bar_colors",		colours,

		/* What is this measured in ? */
		"cluster_margin",	0.3,
		NULL);

	return g_slist_append (NULL, bar_state);
}

static GSList *
line_plot (GupGnmManager *manager, xmlNode *plot)
{
	GuppiElementState *linegraph_state;
	GuppiData *x_data, *y_data;
	GupGnmVector *vec;
	GSList *res = NULL;
	guint32 color;
	xmlNode *series = e_xml_get_child_by_name (plot, "Data");

	g_return_val_if_fail (series != NULL, NULL);

	for (series = series->xmlChildrenNode ; series ; series = series->next) {
		if (strcmp (series->name, "Series"))
			continue;

		vec = series_get_vector (manager, series, "values");
		if (vec == NULL)
			return NULL;
		y_data = gup_gnm_vector_data_get (vec, TRUE);

		vec = series_get_vector (manager, series, "categories");
		if (vec == NULL)
			return NULL;
		x_data = gup_gnm_vector_data_get (vec, TRUE);

		color = series_get_color (manager, series);
		linegraph_state = guppi_element_state_new ("linegraph",
							   "x_data",  x_data,
							   "y_data",  y_data,
							   "color", color,
							   NULL);

		res = g_slist_append (res, linegraph_state);
	}
	return res;
}

static GSList *
pie_plot (GupGnmManager *manager, xmlNode *plot)
{
	GuppiColorPalette *palette;
	GuppiElementState *pie_state;

	xmlNode *desc = e_xml_get_child_by_name (
		e_xml_get_child_by_name (plot, "Type"), "Pie");
	double const radians_of_first_pie =
		e_xml_get_double_prop_by_name_with_default (desc, "radians_of_first_pie", 0.);
	double const separation_percent_of_diameter =
		e_xml_get_integer_prop_by_name_with_default (desc, "separation_percent_of_diameter", 0);

	/* Use only the first */
	xmlNode *series = e_xml_get_child_by_name (
		e_xml_get_child_by_name (plot, "Data"), "Series");
	GupGnmVector *vec;
	GuppiData *values, *categories;

	vec = series_get_vector (manager, series, "values");
	if (vec == NULL)
		return NULL;
	values = gup_gnm_vector_data_get (vec, TRUE);

	vec = series_get_vector (manager, series, "categories");
	categories = (vec != NULL)
		? gup_gnm_vector_data_get (vec, FALSE)
		: xlplots_stock_labels (guppi_seq_size (GUPPI_SEQ (values)));

	palette = guppi_color_palette_new ();
	guppi_color_palette_set_alien_stock (palette);

	/* Build our state objects */
	pie_state = guppi_element_state_new ("pie",
		"data",			values,
		"label_data",		categories,
		"slice_colors",		palette,
		"show_percentage",	FALSE,
		"radius_maximize",	TRUE,
		"radius_lock",		FALSE,

		/* Ideally this would be 0 for an
		 * un-anti-aliased un-scaled hair width
		 * line */
		"edge_width",		0.6,

		"base_offset",		separation_percent_of_diameter,
		"base_angle",		radians_of_first_pie,
		"use_stock_colors",	0,
		NULL);
	return g_slist_append (NULL, pie_state);
}

static GSList *
scatter_plot (GupGnmManager *manager, xmlNode *plot)
{
	GuppiElementState *scatter_state;
	GuppiData *x_data, *y_data, *bubble_data;
	GupGnmVector *vec;
	guint32 color;
	GSList *res = NULL;
	xmlNode *series = e_xml_get_child_by_name (plot, "Data");

	g_return_val_if_fail (series != NULL, NULL);

	for (series = series->xmlChildrenNode ; series ; series = series->next) {
		if (strcmp (series->name, "Series"))
			continue;

		vec = series_get_vector (manager, series, "values");
		if (vec == NULL)
			return NULL;
		y_data = gup_gnm_vector_data_get (vec, TRUE);
		if (y_data == NULL)
			continue;

		vec = series_get_vector (manager, series, "categories");
		if (vec == NULL)
			return NULL;
		x_data = gup_gnm_vector_data_get (vec, TRUE);
		if (x_data == NULL)
			continue;

		color = series_get_color (manager, series);
		scatter_state = guppi_element_state_new ("scatter",
			"x_data",  x_data,
			"y_data",  y_data,
			"color",   color,
			NULL);

		vec = series_get_vector (manager, series, "bubbles");
		if (vec != NULL) {
			bubble_data = gup_gnm_vector_data_get (vec, TRUE);
			if (bubble_data == NULL)
				continue;

			guppi_element_state_set (scatter_state,
				"data_size1",	 bubble_data,
				"marker",  GUPPI_MARKER_CIRCLE,
				NULL);
		} else {
			GuppiMarker marker = series_get_marker (manager, series);
			guppi_element_state_set (scatter_state,
				"size1", 	.75,
				"marker",	marker,
				NULL);
		}
		res = g_slist_append (res, scatter_state);
	}
	return res;
}

static void
arrange_vectors_data (GupGnmManager const *manager,
		      xmlNode *plot, xmlNode *data,
		      char const **extras, int extra_count)
{
	int j, i = 0, cat = -1, count = 0;
	int max = manager->arrangement_len - extra_count;

	if (manager->arrangement_len > 1) {
		cat = manager->data_ids[0];
		i = 1;
	}

	for (;i < max ; i++) {
		xmlNode *series;
		
		series = xmlNewChild (data, data->ns, "Series", NULL);
		e_xml_set_integer_prop_by_name (series, "index", count++);

		if (manager->header_ids[i] >= 0)
			series_dimension_new (series, "labels",
					      manager->header_ids[i]);
		series_dimension_new (series, "values", manager->data_ids[i]);
		if (cat >= 0)
			series_dimension_new (series, "categories", cat);
		for (j = 0 ; j < extra_count ; j++)
			series_dimension_new (series, extras[j],
				manager->data_ids[++i]);
	}
}

static void
bar_arrange_data (GupGnmManager const *manager,
		  xmlNode *plot, xmlNode *data)
{
	arrange_vectors_data (manager, plot, data, NULL, 0);
}

static void
line_arrange_data (GupGnmManager const *manager,
		   xmlNode *plot, xmlNode *data)
{
	arrange_vectors_data (manager, plot, data, NULL, 0);
}

static void
pie_arrange_data (GupGnmManager const *manager,
		  xmlNode *plot, xmlNode *data)
{
	int i = 0;
	xmlNode *series = xmlNewChild (data, data->ns, "Series", NULL);
	e_xml_set_integer_prop_by_name (series, "index", 0);

	if (manager->arrangement_len > 1) {
		series_dimension_new (series, "categories", manager->data_ids[0]);
		i = 1;
	}

	series_dimension_new (series, "values", manager->data_ids[i]);
	if (manager->header_ids[i] >= 0)
		series_dimension_new (series, "labels", manager->header_ids[i]);
}

static void
scatter_arrange_data (GupGnmManager const *manager,
		      xmlNode *plot, xmlNode *data)
{
	static char const *bubbles [] = { "bubbles", NULL };
	xmlNode *desc = e_xml_get_child_by_name (
		e_xml_get_child_by_name (plot, "Type"), "Scatter");
	gboolean const has_bubbles =
		e_xml_get_bool_prop_by_name_with_default (desc, "has_bubbles", FALSE);
	arrange_vectors_data (manager, plot, data, bubbles, has_bubbles ? 1 : 0);
}

static void
enforce_pref (GuppiElementView *view)
{
	guppi_element_view_force_preferred_view (view, GUPPI_X_AXIS, TRUE);
	guppi_element_view_force_preferred_view (view, GUPPI_Y_AXIS, TRUE);
}

typedef struct {
	char const *name;
	char const *element;
	gboolean    optional;
	gboolean    shared;	/* common to all series */
} SeriesElements;
#define SERIES_ELEMENT_LAST	{ NULL, FALSE, FALSE }

/* Name is always added */
static SeriesElements const bar_spec[] = {
	{ N_("Values"), 	 "values",	FALSE, FALSE },
	{ N_("Category Labels"), "categories",	TRUE, TRUE },
	SERIES_ELEMENT_LAST
};
static SeriesElements const line_spec[] = {
	{ N_("Values"), 	 "values",	FALSE, FALSE },
	{ N_("Category Labels"), "categories",	TRUE, TRUE },
	SERIES_ELEMENT_LAST
};
static SeriesElements const pie_spec[] = {
	{ N_("Values"), 	 "values",	FALSE, FALSE },
	{ N_("Category Labels"), "categories",	TRUE, TRUE },
	SERIES_ELEMENT_LAST
};
static SeriesElements const scatter_spec[] = {
	{ N_("X Values"),	"categories",	FALSE, FALSE },
	{ N_("Y Values"),	"values",	TRUE, FALSE },
	{ N_("Bubble Size"),	"bubbles",	TRUE, FALSE },
	SERIES_ELEMENT_LAST
};

typedef struct {
	char const *name;
	SeriesElements const *spec;
	gboolean	singleton;
	GSList * (*plot)	 (GupGnmManager *manager, xmlNode *plot);
	void     (*arrange_data) (GupGnmManager const *manager, xmlNode *plot, xmlNode *data);
	void     (*process_view) (GuppiElementView *view);
} PlotDescriptor;

static PlotDescriptor const plot_types[] = {
	{ "Bar",     bar_spec,     FALSE, bar_plot,	bar_arrange_data,     enforce_pref },
	{ "Line",    line_spec,    FALSE, line_plot,	line_arrange_data,    NULL },
	{ "Pie",     pie_spec,     TRUE,  pie_plot,	pie_arrange_data,     NULL },
	{ "Scatter", scatter_spec, FALSE, scatter_plot,	scatter_arrange_data, NULL },
	{ NULL,      NULL,	   FALSE, NULL,		NULL },
};
    
static PlotDescriptor const *
plot_get_descriptor (xmlNode *plot)
{
	xmlNode *type_node = e_xml_get_child_by_name (plot, "Type");
	char const *type;
	int i;

	g_return_val_if_fail (type_node != NULL, NULL);

	type = type_node->xmlChildrenNode->name;
	for (i = 0 ; plot_types[i].name != NULL; i++)
		if (!strcmp (plot_types [i].name, type))
			return plot_types + i;
	return NULL;
}

GuppiRootGroupView *
guppi_gnumeric_manager_make_plot (GupGnmManager *manager)
{
	PlotDescriptor const *descriptor = NULL;
	xmlNode *node, *plot;
	guppi_compass_t legend_location;
	gboolean singleton = FALSE;

	/* Models */
	GuppiElementState *root_state;
	GuppiElementState *plot_state, *legend_state;
	GSList *plot_states = NULL;

	/* View Objects */
	GuppiGroupView	 *root_view;
	GuppiElementView *plot_view, *legend_view = NULL;
	GuppiElementView *frame_view, *x_axis_view, *y_axis_view;
	
	GuppiGroupView *group_view;

	g_return_val_if_fail (IS_GUP_GNM_MANAGER (manager), NULL);
	g_return_val_if_fail (manager->doc != NULL,  NULL);

	plot = e_xml_get_child_by_name (manager->doc->xmlRootNode, "Plots");

	g_return_val_if_fail (plot != NULL,  NULL);

	plot_states = NULL;
	for (plot = plot->xmlChildrenNode ; plot ; plot = plot->next) {
		if (strcmp (plot->name, "Plot"))
			continue;

		if (NULL != (descriptor = plot_get_descriptor (plot))) {
			plot_states = g_slist_concat (plot_states,
				descriptor->plot (manager, plot));
			singleton = descriptor->singleton;
		}
	}

	plot_state = NULL;
	if (plot_states != NULL) {
		GSList *tmp = plot_states;
		plot_states = plot_states->next;
		plot_state = tmp->data;
	} else
		plot_state = unimplemented_plot ();

	/* Legend */
	legend_state = NULL;
	legend_location = GUPPI_COMPASS_INVALID;
	node = e_xml_get_child_by_name (manager->doc->xmlRootNode, "Legend");
	if (node != NULL)
		node = e_xml_get_child_by_name (node, "Position");
	if (node != NULL) {
		xmlChar *tmp = xmlNodeGetContent (node);
		if (tmp != NULL) {
			legend_location = guppi_str2compass (tmp);
			xmlFree (tmp);

			legend_state = guppi_element_state_new("legend",
				"swatch_width",  6.,
				"swatch_height", 6.,

				/* Ideally this would be 0 for an
				* un-anti-aliased un scale hair width
				* line */
				"edge_thickness", .5,

				"edge_margin", guppi_in2pt(1/16.0),
				NULL);
			legend_view = guppi_element_state_make_view (legend_state);
			guppi_unref (legend_state);

			guppi_element_state_changed_delayed (plot_state);

			if (singleton) {

				/* This is an evil, pie-specific hack */
				if (!strcmp (descriptor->name, "Pie")) {

					GuppiData *d = NULL;
					GuppiColorPalette *palette = NULL;
					guppi_element_state_get (plot_state,
								 "label_data", &d,
								 "slice_colors", &palette,
								 NULL);
					guppi_element_state_set (legend_state,
								 "labels", d,
								 "swatch_colors", palette, 
								 NULL);
				}
				
			} else if (manager->series.names) {
				guppi_element_state_set (legend_state,
							 "labels",	 manager->series.names,
							 "swatch_colors", manager->series.colours,
							 NULL);

				g_message ("manager->series.names");
				guppi_data_spew_xml (manager->series.names);
			}
		}
	}

	/* Frame (i.e. Grid Lines) */
	if (singleton) {
		frame_view = NULL;
	} else {
		GuppiElementState *frame_state = guppi_element_state_new ("frame",
			/* Ideally this would be 0 for an
			 * un-anti-aliased un scale hair width
			 * line */
			"minor_rule_thickness", .25,
			"major_rule_thickness", .5,
			"minor_rule_color",	RGBA_BLACK,
			"major_rule_color",	RGBA_BLACK,
			"frame_color",		RGBA_BLACK,
			"frame_thickness",	.25,
			NULL);
		frame_view  = guppi_element_view_new (frame_state, NULL);
		guppi_unref (frame_state);
	}

	/* Axes */
	if (singleton) {
		x_axis_view = y_axis_view = NULL;
	} else {
		GuppiElementState *x_axis_state = guppi_element_state_new ("axis",
			"minor_tick_thickness", .25,
			"major_tick_thickness", .5,
			"position", GUPPI_SOUTH,
			NULL);
		GuppiElementState *y_axis_state = guppi_element_state_new ("axis",
			"minor_tick_thickness", .25,
			"major_tick_thickness", .5,
			"position", GUPPI_WEST,
			NULL);
		x_axis_view = guppi_element_view_new (x_axis_state, NULL);
		y_axis_view = guppi_element_view_new (y_axis_state, NULL);
		guppi_unref (x_axis_state);
		guppi_unref (y_axis_state);
	}

	root_state = guppi_root_group_state_new ();
	root_view = GUPPI_GROUP_VIEW (guppi_element_state_make_view (root_state));
	guppi_unref (root_state);

	plot_view = guppi_element_state_make_view (plot_state);
	if (descriptor && descriptor->process_view)
		descriptor->process_view (plot_view);

	/* This view contains our plots and (if they exist) our axes */
	group_view = guppi_group_view_new ();

	if (frame_view) {
		static guint32 const grey = RGBA_TO_UINT(0xb6,0xb6,0xae,0xff);
		GuppiElementState *back_state = guppi_element_state_new ("background",
#ifdef EYE_CANDY
			"color", 0xe0e0ffff,
			"color_final", 0xffffffff,
			"gradient_start", GUPPI_SOUTH,
#else
			"color",	grey,
			"color_final",	grey,
#endif
			NULL);
		GuppiElementView *background_view =
			guppi_element_view_new (back_state, NULL);
		guppi_unref (back_state);

		guppi_group_view_layout_same_place (group_view, background_view, frame_view);
		guppi_group_view_layout_same_place (group_view, frame_view, plot_view);
		guppi_element_view_connect_axis_markers (plot_view, GUPPI_X_AXIS, frame_view, GUPPI_X_AXIS);
		guppi_element_view_connect_axis_markers (plot_view, GUPPI_Y_AXIS, frame_view, GUPPI_Y_AXIS);
	}

	if (y_axis_view) {
		guppi_group_view_layout_flush_top  (group_view, y_axis_view, GAP);
		guppi_group_view_layout_flush_left (group_view, y_axis_view, GAP);
		guppi_group_view_layout_horizontally_aligned (group_view, y_axis_view, plot_view, 0.);
		guppi_group_view_layout_flush_right (group_view, plot_view, GAP);

		guppi_element_view_connect_axis_markers (plot_view, GUPPI_Y_AXIS, y_axis_view, GUPPI_Y_AXIS);

	} else {
		guppi_group_view_layout_fill_horizontally (group_view, plot_view, GAP, GAP);
	}

	if (x_axis_view) {
		guppi_group_view_layout_flush_bottom (group_view, x_axis_view, GAP);
		guppi_group_view_layout_flush_right (group_view, x_axis_view, GAP);
		guppi_group_view_layout_vertically_aligned (group_view, plot_view, x_axis_view, 0.);
		guppi_group_view_layout_flush_top (group_view, plot_view, GAP);

		guppi_element_view_connect_axis_markers (plot_view, GUPPI_X_AXIS, x_axis_view, GUPPI_X_AXIS);
	} else {
		guppi_group_view_layout_fill_vertically (group_view, plot_view, GAP, GAP);
	}

	if (plot_states != NULL) {
		GSList *ptr = plot_states;
		for (; ptr != NULL; ptr = ptr->next) {
			GuppiElementView *p2_view = guppi_element_state_make_view (ptr->data);
			guppi_group_view_layout_same_place (group_view, plot_view, p2_view);
			guppi_element_view_connect_axis_markers (plot_view, GUPPI_X_AXIS, p2_view, GUPPI_X_AXIS);
			guppi_element_view_connect_axis_markers (plot_view, GUPPI_Y_AXIS, p2_view, GUPPI_Y_AXIS);
			guppi_unref (ptr->data);
		}
		g_slist_free (plot_states);
	}

	g_message ("legend_location: %d", legend_location);
	if (legend_location == GUPPI_NORTH) {
		guppi_group_view_layout_flush_top (root_view,
			legend_view, GAP);
		guppi_group_view_layout_vertically_adjacent (root_view,
							     legend_view, (GuppiElementView *) group_view, GAP);
		guppi_group_view_layout_center_horizontally (root_view, legend_view);
	} else
		guppi_group_view_layout_flush_top (root_view, (GuppiElementView *) group_view, GAP);
	if (legend_location == GUPPI_SOUTH) {
		guppi_group_view_layout_flush_bottom (root_view,
			legend_view, GAP);
		guppi_group_view_layout_vertically_adjacent (root_view,
			(GuppiElementView *) group_view, legend_view, GAP);
		guppi_group_view_layout_center_horizontally (root_view, legend_view);
	} else
		guppi_group_view_layout_flush_bottom (root_view, (GuppiElementView *) group_view, GAP);
	if (legend_location == GUPPI_EAST) {
		guppi_group_view_layout_flush_right (root_view,
			legend_view, GAP);
		guppi_group_view_layout_horizontally_adjacent (root_view,
			(GuppiElementView *) group_view, legend_view, GAP);
		guppi_group_view_layout_center_vertically (root_view, legend_view);
	} else
		guppi_group_view_layout_flush_right (root_view, (GuppiElementView *) group_view, GAP);
	if (legend_location == GUPPI_WEST) {
		guppi_group_view_layout_flush_left (root_view,
			legend_view, GAP);
		guppi_group_view_layout_horizontally_adjacent (root_view,
			legend_view, (GuppiElementView *) group_view, GAP);
		guppi_group_view_layout_center_vertically (root_view, legend_view);
	} else
		guppi_group_view_layout_flush_left (root_view, (GuppiElementView *) group_view, GAP);

	guppi_element_view_set_preferred_view_all (plot_view);

	guppi_root_group_view_set_size (GUPPI_ROOT_GROUP_VIEW (root_view),
		6*72, 6*72);
#if 0
	guppi_root_group_item_set_resize_semantics (GUPPI_ROOT_GROUP_ITEM (view->item),
						    ROOT_GROUP_RESIZE_FILL_SPACE);
#endif


	guppi_unref (plot_state);

	return GUPPI_ROOT_GROUP_VIEW (root_view);
}

void
gup_gnm_manager_generate_series (GupGnmManager *manager)
{
	PlotDescriptor const *descriptor;
	xmlNode *node, *plot, *data;

	g_return_if_fail (IS_GUP_GNM_MANAGER (manager));
	g_return_if_fail (manager->doc != NULL);

	/* FIXME We only handle 1 plot per chart until I understand layout
	 * better. Use the first */
	node = e_xml_get_child_by_name (manager->doc->xmlRootNode, "Plots");

	g_return_if_fail (node != NULL);

	plot = e_xml_get_child_by_name (node, "Plot");

	data = e_xml_get_child_by_name (plot, "Data");
	if (data != NULL) {
		xmlUnlinkNode (data);
		xmlFreeNode (data);
	}

	g_return_if_fail (manager->arrangement_len >= 1);
	g_return_if_fail (manager->data_ids != NULL);
	g_return_if_fail (manager->header_ids != NULL);

	gup_gnm_manager_clear_wrappers (manager);

	if (NULL != (descriptor = plot_get_descriptor (plot)))
		descriptor->arrange_data (manager, plot,
			xmlNewChild (plot, plot->ns, "Data", NULL));

	gup_gnm_manager_markup_spec (manager);
}

static int
ensure_index (GPtrArray *elements, xmlNode *element)
{
	int tmp = e_xml_get_integer_prop_by_name_with_default (element,
							       "index", -1);
	if (tmp < 0) {
		tmp = elements->len;
		e_xml_set_integer_prop_by_name (element, "index", tmp);
	}
	if (tmp >= elements->len)
		g_ptr_array_set_size (elements, tmp+1);
	g_ptr_array_index (elements, tmp) = element;

	return tmp;
}

/**
 * gup_gnm_manager_markup_spec :
 */
void
gup_gnm_manager_markup_spec (GupGnmManager *manager)
{
	PlotDescriptor const *descriptor;
	xmlNode *plot, *ser, *layout;
	GPtrArray *plots, *series, *names;
	GuppiSeq *seq;
	int i;

	g_return_if_fail (IS_GUP_GNM_MANAGER (manager));
	g_return_if_fail (manager->doc != NULL);

	plots = g_ptr_array_new ();
	series = g_ptr_array_new ();
	names = g_ptr_array_new ();

	/* 0) Clear the names */
	seq = GUPPI_SEQ (manager->series.names);
	if (guppi_seq_nonempty (seq)) {
		guppi_seq_delete_range (seq,
			guppi_seq_min_index (seq),
			guppi_seq_max_index (seq));
		for (i = manager->vectors->len ; i-- > 0 ; )
			gup_gnm_vector_clear_names (
				g_ptr_array_index (manager->vectors, i));
	}
	/* 0.1) reset the colours to stock */
	guppi_color_palette_set_alien_stock (manager->series.colours);

	/* 0.2) clear out the markers too */
	g_array_set_size (manager->series.markers, 0);

	plot = e_xml_get_child_by_name (manager->doc->xmlRootNode, "Plots");

	g_return_if_fail (plot != NULL);

	for (plot = plot->xmlChildrenNode ; plot ; plot = plot->next) {
		if (strcmp (plot->name, "Plot"))
			continue;
		/* 1) Make sure all the plots have index numbers. */
		ensure_index (plots, plot);

		/* 2) clean out old descriptors */
		layout = e_xml_get_child_by_name (plot, "DataLayout");
		if (layout != NULL) {
			xmlUnlinkNode (layout);
			xmlFreeNode (layout);
		}

		/* 3) Add shiny new descriptors */
		if (NULL != (descriptor = plot_get_descriptor (plot))) {
			int i;
			xmlNode *dim, *layout = xmlNewChild (plot, plot->ns,
				"DataLayout", NULL);

			dim = xmlNewChild (layout, layout->ns, "Dimension", 
					   _("Name"));
			xmlSetProp (dim, "element", "labels");
			for (i = 0; descriptor->spec [i].name != NULL ; i++) {
				dim = xmlNewChild (layout, layout->ns, "Dimension", 
						   _(descriptor->spec[i].name));
				xmlSetProp (dim, "element", 
					descriptor->spec[i].element);
				if (!descriptor->spec[i].optional)
					e_xml_set_bool_prop_by_name (dim, "required", TRUE);
				if (descriptor->spec[i].shared)
					e_xml_set_bool_prop_by_name (dim, "shared", TRUE);
			}
		}

		ser = e_xml_get_child_by_name (plot, "Data");
		if (ser == NULL)
			continue;
		for (ser = ser->xmlChildrenNode ; ser ; ser = ser->next) {
			int indx;
			char *name = NULL;
			GupGnmVector *vector;

			if (strcmp (ser->name, "Series"))
				continue;

			/* 4) Make sure all the series have index numbers. */
			indx = ensure_index (series, ser);

			/* 5) handle series names */
			vector = series_get_vector (manager, ser, "labels");
			if (vector != NULL) {
				GuppiData *d = gup_gnm_vector_data_get (vector, FALSE);
				GuppiSeqString *seqs = GUPPI_SEQ_STRING (d);

				if (seqs != NULL) {
					gup_gnm_vector_mark_as_name (vector, indx);
					name = guppi_strdup (guppi_seq_string_get (seqs, 0));
				}
			} else {
				xmlChar *tmp = xmlGetProp (ser, "name");
				if (tmp != NULL) {
					name = guppi_strdup (tmp);
					xmlFree (tmp);
				}
			}
			if (name == NULL)
				name = guppi_strdup_printf ("Series%d", indx+1);
			xmlSetProp (ser, "name", name);

			if (indx >= names->len) {
				g_ptr_array_set_size (names, indx+1);
				g_array_set_size (manager->series.markers, indx+1);
			}
			g_ptr_array_index (names, indx) = name;
			g_array_index (manager->series.markers, GuppiMarker, indx) = stock_marker_alien (indx);

#if 0
			/* 6) Handle series format */
			/* TODO : Parse Format tags */
			if (format.fill_colour) {
			}
#endif
		}
	}

	for (i = 0; i < names->len ; i++)
		guppi_seq_string_append_nc (GUPPI_SEQ_STRING (manager->series.names),
			g_ptr_array_index (names, i));

	g_ptr_array_free (names, TRUE);
	g_ptr_array_free (series, TRUE);
	g_ptr_array_free (plots, TRUE);
}

/**
 * gup_gnm_manager_store_series_name :
 * @manager :
 * @i :
 * @name :
 *
 * Store a copy of the supplied string in the series_name list,
 * and update the xml.
 */
void
gup_gnm_manager_store_series_name (GupGnmManager *manager, int i,
				   char const *name)
{
	xmlNode *plot, *ser;

	g_return_if_fail (IS_GUP_GNM_MANAGER (manager));
	g_return_if_fail (manager->series.names != NULL);

	guppi_seq_string_set (GUPPI_SEQ_STRING (manager->series.names),
		i, name);

	if (manager->doc == NULL)
		return;

	plot = e_xml_get_child_by_name (manager->doc->xmlRootNode, "Plots");

	g_return_if_fail (plot != NULL);

	for (plot = plot->xmlChildrenNode ; plot ; plot = plot->next) {
		if (strcmp (plot->name, "Plot"))
			continue;
		ser = e_xml_get_child_by_name (plot, "Data");
		if (ser == NULL)
			continue;
		for (ser = ser->xmlChildrenNode ; ser ; ser = ser->next) {
			int indx;
			if (strcmp (ser->name, "Series"))
				continue;
			indx = e_xml_get_integer_prop_by_name_with_default (
				ser, "index", -1);

			if (indx == i) {
				xmlSetProp (ser, "name", name);
				return;
			}
		}
	}
}
