/*
 *    WMAcpiLoad - A dockapp to monitor ACPI status
 *    Copyright (C) 2002  Thomas Nemeth <tnemeth@free.fr>
 *
 *    Patch by Alan Carriou <cariou_alan@yahoo.fr> (C) 2004-2005
 *
 *    Based on work by Seiichi SATO <ssato@sh.rim.or.jp>
 *    Copyright (C) 2001,2002  Seiichi SATO <ssato@sh.rim.or.jp>
 *    and on work by Mark Staggs <me@markstaggs.net>
 *    Copyright (C) 2002  Mark Staggs <me@markstaggs.net>

 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.

 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.

 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include "lib_acpi.h"
#include "wmacpiload.h"
#include "thermal.h"

/************************************************************************
 * Macro definitions
 ************************************************************************/

/* used to parse the temperature file */
#define TEMP_UNIT " C"

/************************************************************************
 * Global constants definitions
 ************************************************************************/

const char *const thermal_zone_dir = "/thermal_zone/";
const char *const thermal_zone_status_file = "/temperature";

/************************************************************************
 * Function definitions
 ************************************************************************/

void *
thermal_zone_init(struct dirent *fd)
{
    Thermal_Zone *tz;

    if (!fd)
        return NULL;

    tz = malloc(sizeof(*tz));
    if (!tz) {
        fprintf(stderr, "%s : could not allocate memory "
                "for thermal zone information concerning "
                "thermal zone '%s'\n", PACKAGE, fd->d_name);
        return NULL;
    }

    tz->name = NULL;
    tz->tempfile = NULL;
    tz->next = NULL;
    tz->temp = 0;

    tz->name = malloc(strlen(fd->d_name) + 1);
    if (!tz->name) {
        fprintf(stderr, "%s : could not allocate memory "
                "for thermal zone name concerning thermal zone '%s'\n",
                PACKAGE, fd->d_name);
        free(tz);
        return NULL;
    }
    strcpy(tz->name, fd->d_name);

    tz->tempfile =
        strcat4("/proc/acpi", thermal_zone_dir, tz->name,
                thermal_zone_status_file);
    if (!tz->tempfile) {
        fprintf(stderr, "%s : could not allocate memory "
                "for thermal zone tempfile concerning "
                "thermal zone '%s'\n", PACKAGE, tz->name);
        free(tz->name);
        free(tz);
        return NULL;
    }

    if (verbose)
        printf("Found thermal zone %s\n", tz->name);

    return tz;
}

void
thermal_zone_next(AcpiInfos *infos)
{
    if (infos && infos->thermal) {
        if (infos->thermal->next == NULL)
            infos->thermal = infos->thermal_first;
        else
            infos->thermal = infos->thermal->next;

        if (verbose)
            printf("Now monitoring thermal zone %s\n",
                   infos->thermal->name);
    }
}

void
thermal_zone_update_status(Thermal_Zone *tz)
{
    char buf[6];                /* allows up to "999 C" */
    temperature temp;

    if (tz != NULL) {
        if (parse_file(tz->tempfile,
                       "temperature:", buf, sizeof buf) != NULL) {
            char *endptr;

            temp = strtoul(buf, &endptr, 10);
            /*
             * The format is "temperature:    xx C" where xx is the
             * temperature in celcius degrees. So we check if we have the
             * " C" left in the buffer to see if all went well. 
             */
            if (strcmp(endptr, TEMP_UNIT) != 0) {
                fprintf(stderr, "%s : Unconsistent temperature  "
                        "about thermal zone %s\n", PACKAGE, tz->name);
                temp = 0;
            }
        } else {
            temp = 0;
        }
        tz->temp = temp;
    }
}

void
thermal_zone_free(Thermal_Zone *tz)
{
    if (!tz) {
        MYFREE(tz->name);
        MYFREE(tz->tempfile);
        free(tz);
    }
}
