/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "main.h"
#include "shared.h"


/* Signal handler */
static void sig_handler( int signal );

/*------------------------------------------------------------------------*/

  int
main (int argc, char *argv[])
{
  /* Command line option returned by getopt() */
  int option;

  /* New and old actions for sigaction() */
  struct sigaction sa_new, sa_old;


  /* Initialize new actions */
  sa_new.sa_handler = sig_handler;
  sigemptyset( &sa_new.sa_mask );
  sa_new.sa_flags = 0;

  /* Register function to handle signals */
  sigaction( SIGINT,  &sa_new, &sa_old );
  sigaction( SIGSEGV, &sa_new, 0 );
  sigaction( SIGFPE,  &sa_new, 0 );
  sigaction( SIGTERM, &sa_new, 0 );
  sigaction( SIGABRT, &sa_new, 0 );
  sigaction( SIGCONT, &sa_new, 0 );

  /* Process command line options. Defaults below. */
  rc_data.rtlsdr_center_freq = 137100000;
  sat_type = NOAA_19;
  while( (option = getopt(argc, argv, "b:hf:v") ) != -1 )
    switch( option )
    {
      case 'b' : /* LPF bandwidth */
        rc_data.rtlsdr_lpf_bw = atoi( optarg );
        break;

      case 'f' : /* Get center frequency */
        rc_data.rtlsdr_center_freq = (uint32_t)(atoi(optarg) );
      break;

      case 'h' : /* Print usage and exit */
        Usage();
        return(0);

      case 'v' : /* Print version */
        puts( PACKAGE_STRING );
        return(0);

      default: /* Print usage and exit */
        Usage();
        exit(-1);

    } /* End of switch( option ) */

  /* Start GTK+ */
  gtk_init( &argc, &argv );

  /* Defaults/initialization */
  rc_data.default_dur = duration = 600;
  image_file[0]   = '\0';
  samples_file[0] = '\0';
  SetFlag( IMAGE_NORMALIZE | IMAGE_COLORIZE );
  SetFlag( CARRIER_SPECTRUM );
  rc_data.ifft_decimate = IFFT_RTLSDR_DECIMATE;
  rc_data.ifft_scale    = IFFT_SDR_SCALE;

  /* Create file path to xwxapt glade path */
  Strlcpy( rc_data.xwxapt_glade,
      getenv("HOME"), sizeof(rc_data.xwxapt_glade) );
  Strlcat( rc_data.xwxapt_glade,
      "/xwxapt/xwxapt.glade", sizeof(rc_data.xwxapt_glade) );

  /* Check for the Glade config file */
  FILE *fp = fopen( rc_data.xwxapt_glade, "r" );
  if( fp == NULL )
  {
    fprintf( stderr, "xwxapt: cannot open xwxapt Glade GUI Description file.\n" );
    perror( rc_data.xwxapt_glade );
    fprintf( stderr, "xwxapt: trying to create xwxapt config directory "
                     "from the installation prefix file tree.\n" );

    /* Find the binary's path (location in file system) */
    char exe_path[256], file_path[288];
    
    /* Read the file path to xwxapt executable */
    size_t len = sizeof( exe_path );
    int bytes = (int)readlink( "/proc/self/exe", exe_path, len );
    if( bytes <= 0 )
    {
      fprintf( stderr, "xwxapt: cannot read xwxapt binary's location.\n" );
      perror( "/proc/self/exe" );
      exit( -1 );
    }

    /* Remove "/bin/xwxapt" from the path with room for termination */
    bytes -= sizeof( "/bin/xwxapt" ) - 1;
    if( bytes < 1 )
    {
      fprintf( stderr, "xwxapt: cannot create file path to examples/xwxapt.\n" );
      exit( -1 );
    }

    /* Create file path to xwxapt examples directory */
    exe_path[bytes] = '\0';
    Strlcpy( file_path, exe_path, sizeof(file_path) );
    Strlcat( file_path, "/share/examples/xwxapt", sizeof(file_path) );
    fprintf( stderr, "xwxapt: creating xwxapt config directory from: %s\n", file_path );

    /* Create system command to copy examples/xwxapt to ~/.xwxapt */
    char syscmnd[512];
    Strlcpy( syscmnd, "cp -r ", sizeof(syscmnd) );
    Strlcat( syscmnd, file_path, sizeof(syscmnd) );
    Strlcat( syscmnd, " ", sizeof(syscmnd) );
    Strlcat( syscmnd, getenv("HOME"), sizeof(syscmnd) );
    Strlcat( syscmnd, "/xwxapt",   sizeof(syscmnd) );
    int ret = system( syscmnd );
    if( ret == -1 )
    {
      fprintf( stderr,"xwxapt: cannot create xwxapt's working directory.\n" );
      perror( file_path );
      exit( -1 );
    }
  } /* if( fp == NULL ) */
  else fclose( fp );

  /* Create xwxapt window */
  main_window = create_main_window( &main_window_builder );
  gtk_window_set_title( GTK_WINDOW(main_window), PACKAGE_STRING );
  gtk_widget_show( main_window );

  /* Create global sync status widgets' table */
  status_table = Builder_Get_Object( main_window_builder, "status_frame" );

  /* Create global sync status indicator icon */
  sync_icon = Builder_Get_Object( main_window_builder, "sync_ind" );

  /* Create the text view scroller */
  text_scroller = Builder_Get_Object( main_window_builder, "text_scrolledwindow" );

  /* Get text buffer */
  text_buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW(
        Builder_Get_Object(main_window_builder, "textview")) );

  /* Get waterfall and scope widgets */
  scope     = Builder_Get_Object( main_window_builder, "demod_drawingarea" );
  waterfall = Builder_Get_Object( main_window_builder, "ifft_drawingarea" );

  /* Create some rendering tags */
  gtk_text_buffer_create_tag( text_buffer, "black",
      "foreground", "black", NULL);
  gtk_text_buffer_create_tag( text_buffer, "red",
      "foreground", "red", NULL);
  gtk_text_buffer_create_tag( text_buffer, "orange",
      "foreground", "orange", NULL);
  gtk_text_buffer_create_tag( text_buffer, "green",
      "foreground", "darkgreen", NULL);
  gtk_text_buffer_create_tag( text_buffer, "bold",
      "weight", PANGO_WEIGHT_BOLD, NULL);

  /* Create the progress bars */
  level_pbar = Builder_Get_Object( main_window_builder, "signal_lev" );
  sync_pbar  = Builder_Get_Object( main_window_builder, "sync_lev" );

  /* Get sizes of displays */
  GtkAllocation alloc;
  gtk_widget_get_allocation( waterfall, &alloc );
  Fft_Drawingarea_Size_Alloc( &alloc );
  gtk_widget_get_allocation( scope, &alloc );
  scope_width  = alloc.width;
  scope_height = alloc.height;

  /* Null pointers in filter data structs */
  filter_data_i.a = NULL;
  filter_data_i.b = NULL;
  filter_data_i.x = NULL;
  filter_data_i.y = NULL;
  filter_data_q.a = NULL;
  filter_data_q.b = NULL;
  filter_data_q.x = NULL;
  filter_data_q.y = NULL;

  char ver[28];
  snprintf( ver, sizeof(ver), _("Welcome to %s"), PACKAGE_STRING );
  Show_Message( ver, "bold" );

  g_idle_add( Load_Config, NULL );

  gtk_main ();

  return( 0 );
}

/*------------------------------------------------------------------------*/

/*  Initialize_Top_Window()
 *
 *  Initializes xwxapt's top window
 */

  void
Initialize_Top_Window( void )
{
  /* The scrolled window image container */
  GtkWidget *image_scroller;
  char text[12];

  /* Show current satellite */
  GtkLabel *label = GTK_LABEL(
      Builder_Get_Object(main_window_builder, "sat_label") );
  snprintf( text, sizeof(text), "%s-APT", sat_names[sat_type] );
  gtk_label_set_text( label, text );

  /* Show Center_Freq to frequency entry */
  Enter_Center_Freq( rc_data.rtlsdr_center_freq );

  /* Show Bandwidth to B/W entry */
  Enter_Filter_BW();

  /* Kill existing pixbuf */
  if( image_pixbuf != NULL )
  {
    g_object_unref( image_pixbuf );
    image_pixbuf = NULL;
  }

  /* Create new pixbuff for sat images */
  switch( sat_type )
  {
    case NOAA_15: case NOAA_18: case NOAA_19: /* Create pixbuff for NOAA images */
      image_pixbuf = gdk_pixbuf_new( GDK_COLORSPACE_RGB, FALSE, 8,
          NOAA_IMAGE_WIDTH, duration );
      break;

    case METEOR: /* Create pixbuff for METEOR images */
      image_pixbuf = gdk_pixbuf_new( GDK_COLORSPACE_RGB, FALSE, 8,
          METEOR_IMAGE_WIDTH/2, duration );
  }

  /* Error, not enough memory */
  if( image_pixbuf == NULL)
  {
    Show_Message( _("Failed to allocated memory to pixbuf - Quit"), "red" );
    Error_Dialog();
    return;
  }

  /* Fill pixbuf with background color */
  gdk_pixbuf_fill( image_pixbuf, 0xaaaaaaff );

  /* Get details of pixbuf */
  pixel_buf  = gdk_pixbuf_get_pixels(     image_pixbuf );
  rowstride  = gdk_pixbuf_get_rowstride(  image_pixbuf );
  n_channels = gdk_pixbuf_get_n_channels( image_pixbuf );

  /* Globalize image to be displayed */
  apt_image = Builder_Get_Object( main_window_builder, "apt_image" );

  /* Set apt image from pixbuff */
  gtk_image_set_from_pixbuf( GTK_IMAGE(apt_image), image_pixbuf );
  gtk_widget_show( apt_image );

  /* Set window size as required */
  gint wdt, hgt;
  image_scroller = Builder_Get_Object(
      main_window_builder, "image_scrolledwindow" );

  if( sat_type == METEOR )
    wdt = METEOR_IMAGE_WIDTH / 2;
  else
    wdt = NOAA_IMAGE_WIDTH;

  if( duration <= rc_data.default_dur )
    wdt += 9;
  else
    wdt += 27;

  if( duration <= rc_data.window_height )
    hgt = duration + 8;
  else
    hgt = rc_data.window_height + 8;

  gtk_widget_set_size_request( image_scroller, wdt, hgt );
  gtk_window_resize( GTK_WINDOW(main_window), 10, 10 );

} /* Initialize_Top_Window() */

/*------------------------------------------------------------------------*/

/*  sig_handler()
 *
 *  Signal Action Handler function
 */

static void sig_handler( int signal )
{
  /* Internal wakeup call */
  if( signal == SIGCONT ) return;

  Cleanup();

  fprintf( stderr, "\n" );
  switch( signal )
  {
    case SIGINT :
      fprintf( stderr, "%s\n", _("xwxapt: Exiting via User Interrupt") );
      exit(-1);

    case SIGSEGV :
      fprintf( stderr, "%s\n", _("xwxapt: Segmentation Fault") );
      exit(-1);

    case SIGFPE :
      fprintf( stderr, "%s\n", _("xwxapt: Floating Point Exception") );
      exit(-1);

    case SIGABRT :
      fprintf( stderr, "%s\n", _("xwxapt: Abort Signal received") );
      exit(-1);

    case SIGTERM :
      fprintf( stderr, "%s\n", _("xwxapt: Termination Request received") );
      exit(-1);
  }

} /* End of sig_handler() */

/*------------------------------------------------------------------------*/

