/* delete.c -  delete a key
 *      Copyright (C) 2001 Werner Koch (dd9jn), g10 Code GmbH
 *      Copyright (C) 2001-2005 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <assert.h>

#include "util.h"
#include "context.h"
#include "ops.h"
#include "key.h"


enum delete_problem {
    No_Such_Key         = 1,
    Delete_Seckey_First = 2,
    Ambigiuous_Specs    = 3
};


static void
delete_status_handler (gpgme_ctx_t ctx, gpg_status_code_t code, char * args)
{
    if (ctx->out_of_core)
        return;

    switch (code) {
    case STATUS_EOF:
        break;

    case STATUS_DELETE_PROBLEM:
        ctx->problem = atol (args);
        break;

    default:
        break; /* ignore all other codes */
    }
} /* delete_status_handler */


static const char *
delete_command_handler (void * opaque, gpg_status_code_t code, const char * key)
{
    gpgme_ctx_t ctx = opaque;

    if( !code )
        return NULL;

    if (code == STATUS_GET_BOOL && !strcmp ("delete_key.secret.okay", key)
        || code == STATUS_GET_BOOL && !strcmp ("delete_key.okay", key))
        return "Y";

    return NULL;
} /* delete_command_handler */


static gpgme_error_t
delete_start (gpgme_ctx_t ctx, const gpgme_key_t key,
              const char *pattern, int allow_secret)
{
    gpgme_error_t rc = 0;
    const char *s;

    if (!key && !pattern) {
        rc = mk_error (Invalid_Value);
        goto leave;
    }

    fail_on_pending_request (ctx);
    ctx->pending = 1;

    _gpgme_gpg_release (&ctx->gpg);
    rc = _gpgme_gpg_new (&ctx->gpg);
    if( rc )
        goto leave;

    _gpgme_gpg_set_status_handler ( ctx->gpg, delete_status_handler, ctx );
    _gpgme_gpg_set_command_handler( ctx->gpg, delete_command_handler, ctx );
    if( ctx->use_logging )
        _gpgme_gpg_set_logging_handler( ctx->gpg, ctx );

    /* build the commandline */
    _gpgme_gpg_add_arg ( ctx->gpg, allow_secret?
                         "--delete-secret-and-public-key" : "--delete-key" );

    _gpgme_gpg_add_arg ( ctx->gpg, "--" );
    if (key != NULL) {
        s = gpgme_key_get_string_attr (key, GPGME_ATTR_KEYID, NULL, 0);
        if(!s) {
            rc = mk_error (Invalid_Key);
            goto leave;
        }
    }
    else
        s = pattern;
    _gpgme_gpg_add_arg (ctx->gpg, s);
    rc = _gpgme_gpg_spawn( ctx->gpg, ctx ); /* do it */

leave:
    if( rc ) {
        ctx->pending = 0;
        _gpgme_gpg_release( &ctx->gpg );
    }
    return rc;
} /* delete_start */


static gpgme_error_t
get_delete_result (gpgme_ctx_t ctx)
{
    gpgme_error_t err = 0;

    if (ctx->problem == No_Such_Key)
        err = mk_error (Invalid_Key);
    else if (ctx->problem == Delete_Seckey_First)
        err = mk_error( Conflict );
    else if (gpgme_get_process_rc (ctx))
        err = mk_error (General_Error);
    return err;
}


gpgme_error_t
gpgme_op_delete (gpgme_ctx_t ctx, const gpgme_key_t key, int allow_secret)
{
    gpgme_error_t err;

    err = delete_start (ctx, key, NULL, allow_secret);
    if (!err) {
        gpgme_wait (ctx, 1);
        ctx->pending = 0;
        err = get_delete_result (ctx);
    }
    return err;
} /* gpgme_op_delete */


/* interface */
gpgme_error_t
gpgme_op_delete_key (const char *keyid, int allow_secret)
{
    gpgme_error_t err = 0;
    gpgme_key_t key = NULL;
    gpgme_ctx_t ctx;

    if (!keyid)
        return mk_error (Invalid_Value);

    err = gpgme_new(&ctx);
    if (!err)
        key = _gpgme_key_new_fromkeyid (keyid);
    if (!err && key)
        err = gpgme_op_delete (ctx, key, allow_secret);

    gpgme_release (ctx);
    gpgme_key_release (key);
    return err;
} /* gpgme_op_key_delete */


gpgme_error_t
gpgme_op_delete_keys (gpgme_recipients_t recp, int allow_secret)
{
    gpgme_error_t err;
    gpgme_ctx_t ctx;
    struct user_id_s * u;
    char *list;
    size_t n=0;

    if (!recp)
        return mk_error (Invalid_Value);

    for  (u=recp->list; u; u=u->next)
        n += strlen (u->name) + 1 + 2;
    list = calloc (1, n+1);
    if (!list)
        return mk_error (Out_Of_Core);

    for (u = recp->list; u; u = u->next) {
        strcat (list, u->name);
        strcat (list, " ");
    }

    /*DEBUG1 ("delete_keys: '%s'\r\n", list);*/
    err = gpgme_new (&ctx);
    if (err)
        return err;
    err = delete_start (ctx, NULL, list, allow_secret);
    if (!err) {
        gpgme_wait (ctx, 1);
        ctx->pending = 0;
        err = get_delete_result (ctx);
    }

    gpgme_release (ctx);
    safe_free (list);
    return err;
} /* gpgme_op_delete_keys */
