/* sym-encrypt.c -  symmetric encrypt functions
 *	Copyright (C) 2002, 2003, 2005 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "util.h"
#include "context.h"
#include "ops.h"


struct symenc_result_s {
    int failed;
};


static gpgme_error_t
create_result_struct (gpgme_ctx_t ctx)
{
    assert (!ctx->result.symenc);
    ctx->result.symenc = calloc (1, sizeof *ctx->result.symenc);
    if (!ctx->result.symenc)
	return mk_error (Out_Of_Core);
    ctx->result_type = RESULT_TYPE_SYMENC;

    return 0;
} /* create_result_struct */


void
_gpgme_release_symenc_result (_symenc_result_t res)
{
    safe_free( res );
} /* _gpgme_relese_symenc_result */


static void
symenc_status_handler (gpgme_ctx_t ctx, gpg_status_code_t code, char *args)
{
    if (ctx->out_of_core)
        return;

    if (ctx->result_type == RESULT_TYPE_NONE) {
	if (create_result_struct (ctx)) {
	    ctx->out_of_core = 1;
	    return;
	}
    }

    assert (ctx->result_type == RESULT_TYPE_SYMENC);

    switch (code) {
    case STATUS_MISSING_PASSPHRASE: 
	ctx->result.symenc->failed = 1;
	break;

    case STATUS_END_DECRYPTION:
	ctx->result.symenc->failed = 0;
	break;

    case STATUS_PROGRESS:
	if (ctx->cb.progress)
	    _gpgme_progress_handler (ctx, args);
	break;

    default:
	break;
    }
} /* symenc_status_handler */


static gpgme_error_t
symenc_start( gpgme_ctx_t ctx, gpgme_data_t plain, gpgme_data_t ciph )
{
    int rc = 0;
    
    fail_on_pending_request( ctx );
    ctx->pending = 1;
    
    /* create a process object */
    _gpgme_gpg_release( &ctx->gpg ); 
    rc = _gpgme_gpg_new( &ctx->gpg );
    if( rc )
        goto leave;

    if( ctx->use_logging )
	_gpgme_gpg_set_logging_handler( ctx->gpg, ctx );
    _gpgme_gpg_set_status_handler ( ctx->gpg, symenc_status_handler, ctx );
    if( ctx->passphrase_value ) {
        rc = _gpgme_add_passphrase( ctx );
        if( rc )
            goto leave;
    }
    
    /* build the commandline */
    _gpgme_gpg_add_arg (ctx->gpg, "--symmetric");
    _gpgme_encrypt_add_cipher (ctx);
    if (ctx->no_compress)
	_gpgme_gpg_add_arg (ctx->gpg, "-z 0");
    if (ctx->force_mdc)
	_gpgme_gpg_add_arg (ctx->gpg, "--force-mdc");
    if (ctx->use_armor)
        _gpgme_gpg_add_arg (ctx->gpg, "--armor");
    
    /* Check the supplied data */
    if( gpgme_data_get_type( plain ) == GPGME_DATA_TYPE_NONE ) {
        rc = mk_error( No_Data );
        goto leave;
    }
    _gpgme_data_set_mode( plain, GPGME_DATA_MODE_OUT );
    if ( !ciph || gpgme_data_get_type( ciph ) != GPGME_DATA_TYPE_NONE ) {
        rc = mk_error (Invalid_Value);
        goto leave;
    }
    _gpgme_data_set_mode( ciph, GPGME_DATA_MODE_IN );
    /* Tell the gpg object about the data */
    if( ctx->use_tmpfiles ) {
        _gpgme_gpg_add_arg ( ctx->gpg, "--output" );
        _gpgme_gpg_add_arg ( ctx->gpg, _gpgme_get_tmpfile( 0 ) );
        _gpgme_data_write_to_tmpfile( plain );
        _gpgme_gpg_add_arg ( ctx->gpg, _gpgme_get_tmpfile( 1 ) );
    }
    else {
        _gpgme_gpg_add_arg ( ctx->gpg, "--output" );
        _gpgme_gpg_add_arg ( ctx->gpg, "-" );
        _gpgme_gpg_add_data ( ctx->gpg, ciph, 1 );
        _gpgme_gpg_add_arg ( ctx->gpg, "--" );
        _gpgme_gpg_add_data ( ctx->gpg, plain, 0 );
    }
    
    /* and kick off the process */
    rc = _gpgme_gpg_spawn ( ctx->gpg, ctx );
    
leave:
    if( rc ) {
        ctx->pending = 0; 
        _gpgme_gpg_release( &ctx->gpg ); 
    }
    return rc;
} /* symenc_start */


gpgme_error_t
gpgme_op_symenc( gpgme_ctx_t ctx, gpgme_data_t in, gpgme_data_t out )
{
    gpgme_error_t err;
    
    err = symenc_start ( ctx, in, out );
    if( !err ) {
        gpgme_wait( ctx, 1 );
        ctx->pending = 0;
        if( ctx->use_tmpfiles ) {
            _gpgme_data_read_from_tmpfile( out );
            _gpgme_del_tmpfiles( ctx->wipe_fnc );
        }
	assert( ctx->result.symenc );
	if( ctx->result.symenc->failed )
            err = mk_error( No_Passphrase );
	else if( gpgme_get_process_rc( ctx ) )
	    err = mk_error( Internal_GPG_Problem );
    }
    return err;
} /* gpgme_op_symenc */


gpgme_error_t
gpgme_op_clip_symenc( gpgme_ctx_t ctx )
{
    gpgme_error_t err;
    gpgme_data_t plain = NULL;
    gpgme_data_t ciph = NULL;
    
    gpgme_control( ctx, GPGME_CTRL_ARMOR, 1 );
    err = gpgme_data_new_from_clipboard (&plain);
    if( !err )        
	err = gpgme_data_new( &ciph );
    if( !err )        
	err = gpgme_op_symenc( ctx, plain, ciph );
    if( !err ) {        
	gpgme_data_change_version( &ciph );
	gpgme_data_release_and_set_clipboard( ciph );
    }
    gpgme_data_release( plain );

    return err;
} /* gpgme_op_clip_symenc */