# mouse.py: mouse configuration data
#
# Copyright 1999-2008 Red Hat, Inc.
#
# This software may be freely redistributed under the terms of the GNU
# library public license.
#
# You should have received a copy of the GNU Library Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# effective API:
# .Mouse()
# .read()
# .mouseToMouse()
# .set()

from rhpl.simpleconfig import SimpleConfigFile
import os
import rhpl.translate as translate
from rhpl.translate import _, N_
import dbus

translate.textdomain('rhpxl')

class Mouse (SimpleConfigFile):
    mouseModels = {
        # (shortname: make, model, gpm protocol, X protocol, input protocol, device, emulate3)
        "ascii": (N_("ASCII"), N_("MieMouse (serial)"),
                  "ms3", "IntelliMouse", "ms3", "ttyS", 0),
        "generic": (N_("Generic"), N_("2 Button Mouse (serial)"),
                    "Microsoft", "Microsoft", "bare", "ttyS", 1),
        "generic3": (N_("Generic"), N_("3 Button Mouse (serial)"),
                     "Microsoft", "Microsoft", "ms", "ttyS", 0),
        "genericps/2": (N_("Generic"), N_("2 Button Mouse (PS/2)"),
                        "imps2", "IMPS/2", None, "input/mice", 1),
        "generic3ps/2": (N_("Generic"), N_("3 Button Mouse (PS/2)"),
                         "imps2", "IMPS/2", None, "input/mice", 0),
        "genericwheelps/2": (N_("Generic"), N_("Wheel Mouse (PS/2)"),
                             "imps2", "IMPS/2", None, "input/mice", 0),
        "genericusb": (N_("Generic"), N_("2 Button Mouse (USB)"),
                       "imps2", "IMPS/2", None, "input/mice", 1),
        "generic3usb": (N_("Generic"), N_("3 Button Mouse (USB)"),
                        "imps2", "IMPS/2", None, "input/mice", 0),
        "genericwheelusb": (N_("Generic"), N_("Wheel Mouse (USB)"),
                            "imps2", "IMPS/2", None, "input/mice", 0),
        "geniusnm": (N_("Genius"), N_("NetMouse (serial)"),
                     "ms3", "IntelliMouse", "ms3", "ttyS", 1),
        "thinking": (N_("Kensington"), N_("Thinking Mouse (serial)"),
                     "Microsoft", "ThinkingMouse", "bare", "ttyS", 1),
        "logitech": (N_("Logitech"), N_("C7 Mouse (serial, old C7 type)"),
                     "Logitech", "Logitech", "mman", "ttyS", 0),
        "logitechcc": (N_("Logitech"), N_("CC Series (serial)"),
                       "logim", "MouseMan", "mman", "ttyS", 0),
        "logimman": (N_("Logitech"), N_("MouseMan/FirstMouse (serial)"),
                     "MouseMan", "MouseMan", "mman", "ttyS", 0),
        "logimman+": (N_("Logitech"), N_("MouseMan+/FirstMouse+ (serial)"),
                      "pnp", "IntelliMouse", "ms3", "ttyS", 0),
        "microsoft": (N_("Microsoft"), N_("Compatible Mouse (serial)"),
                      "Microsoft", "Microsoft", "bare", "ttyS", 1), 
        "msnew": (N_("Microsoft"), N_("Rev 2.1A or higher (serial)"),
                  "pnp", "Auto", "ms", "ttyS", 1), 
        "msintelli": (N_("Microsoft"), N_("IntelliMouse (serial)"),
                      "ms3", "IntelliMouse", "ms3", "ttyS", 0),
        "mousesystems": (N_("Mouse Systems"), N_("Mouse (serial)"),
                         "MouseSystems", "MouseSystems", "msc", "ttyS", 1),
        "mmseries": (N_("MM"), N_("MM Series (serial)"),
                     "MMSeries", "MMSeries", "bare", "ttyS", 1),
        "mmhittab": (N_("MM"), N_("MM HitTablet (serial)"),
                     "MMHitTab", "MMHittab", "bare", "ttyS", 1),
        "none": ("No", "mouse" , "none", "none", None, None, 0),
        "sun": (N_("Sun"), N_("Mouse"), "sun", "sun", "sun", "sunmouse", 0),
        "synaptics": (N_("Synaptics"), N_("Synaptics Touchpad"), 
                      "imps2", "IMPS/2", None, "input/mice", 1),
        "vmmouse": (N_("VMWare"), N_("VMWare Mouse"), 
                      "vmmouse", "vmmouse", None, "input/mice", 1),
	}

    # note that mice is deprecated; things shouldn't be accessing it directly
    # but, to keep things simpler for now, we'll keep it around
    mice = {}
    for (key, val) in mouseModels.items():
        shortname = key
        try:
            (make, model, gpmproto, xproto, iproto, device, emulate3) = val
        except Exception, e:
            continue
	newkey = "%s - %s" % (make, model)
        mice[newkey] = (gpmproto, xproto, iproto, device, emulate3, shortname)
    
    
    def mouseToMouse(self):
        types = {}
        for (type, val) in self.mouseModels.items():
            (make, model, gpmproto, xproto, iproto, device, emulate3) = val
            types[type] = "%s - %s" % (make, model)
        return types
        

    # XXX fixme - externalize
    def __init__ (self, skipProbe = 0):
        self.info = {}
        self.device = None
        self.mousetype = None
        self.emulate = 0
        self.setMouse ("generic3ps/2")
	self.wasProbed = 0
	self.probedList = None

	if not skipProbe:
	    self.probe()

        self.orig_mouse = self.get()

    def get_Orig(self):
        return self.orig_mouse
        
    def probed(self):
	return self.wasProbed

    def getAllProbed(self):
	return self.probedList

    def anyUSBMice(self):
	# first check current configured mouse
	if self.device == "input/mice":
	    return 1
	
	# now check all probed mice
	if not self.probedList:
	    return 0
	
	for l in self.probedList:
	    (device, module, desc) = l
	    if device == "input/mice":
		return 1

	return 0
    
    def probe (self, frob = 0):
	try:
	    bus = dbus.SystemBus()
	    hal = dbus.Interface(bus.get_object('org.freedesktop.Hal',
						'/org/freedesktop/Hal/Manager'),
				 'org.freedesktop.Hal.Manager')
	    list = hal.FindDeviceByCapability('input.mouse')
	except:
	    list = None

        if (list):
	    l = list[0]
	    self.setMouse("generic3ps/2")
            self.device = "input/mice"
	    self.wasProbed = 1
            return 1
        else:
            self.setMouse("none")
	    self.device = None
	    self.wasProbed = 0
            return 0
    
    def available (self):
        return self.mice

    def get (self):
	return (self.info ["FULLNAME"], self.emulate)

    def getMouse (self):
        return (self.mousetype, self.emulate)

    def __str__(self):
	if (self.emulate):
	    self.info["XEMU3"] = "yes"
	else:
	    self.info["XEMU3"] = "no"
	return SimpleConfigFile.__str__(self)

    def makeLink(self, root):
	try:
	    os.unlink(root + "/dev/mouse")
	except:
	    pass
	if (self.device):
	    os.symlink(self.device, root + "/dev/mouse")

    def getDevice(self):
	return self.device

    def shortDescription(self):
        if self.info.has_key("FULLNAME"):
            return self.info["FULLNAME"]
        else:
            return _("Unable to probe")

    def setDevice(self, device):
	self.device = device

    def set (self, mouse = None, emulateThreeButtons = -1, thedev = None):
        (gpm, x11, inp, dev, em, shortname) = self.mice[mouse]
        self.setMouse(shortname, emulateThreeButtons, thedev)

    def setMouse (self, shortname, emul = -1, thedev = None):
        (make, model, gpm, x11, inp, dev, em) = self.mouseModels[shortname]
        mouse = "%s - %s" % (make, model)

        emulateThreeButtons = emul

        self.mousetype = shortname
        self.info["MOUSETYPE"] = gpm
        self.info["XMOUSETYPE"] = x11
	if inp:
	    self.info["IMOUSETYPE"] = inp
        self.info["FULLNAME"] = mouse
        if emulateThreeButtons != -1:
            self.emulate = emulateThreeButtons
        else:
            self.emulate = em
        if thedev:
            self.device = thedev
	if not self.device:
            self.device = dev

    def read(self, instPath = "/"):
        SimpleConfigFile.read(self, instPath + "/etc/sysconfig/mouse")
        self.beenset = 1

    def write(self, instPath):
        if self.info["FULLNAME"] == "None - None":
            return
	f = open(instPath + "/etc/sysconfig/mouse", "w")
	f.write(str (self))
	f.close()
	self.makeLink(instPath)


    def writeKS(self, f):
        f.write("mouse")

        for arg in self.getArgList():
            f.write(" " + arg)
        f.write("\n")


    def getArgList(self):
        args = []

        if self.info["FULLNAME"]:
            mouseName = self.info["FULLNAME"]
            args.append(self.mice[mouseName][4])
        if self.device:
            args.append("--device %s" %(self.device))
        if self.emulate:
            args.append("--emulthree")
        
        return args
