#!/usr/bin/python2

## system-config-display - an Xorg configuration tool
## Copyright (c) 2002-2003 Red Hat, Inc.

## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.

## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.

## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


import string
import gobject
import sys
import time
import os
import shutil
import signal
import xf86config
import math
import getopt
import tempfile
import locale
import rhpxl.monitor
import rhpxl.videocard
import rhpl.keyboard
from rhpl.translate import _, N_,textdomain
from rhpxl.xhwstate import *

TRUE = 1
FALSE = 0

##
## I18N
## 
import gettext
gettext.bindtextdomain ("system-config-display", "/usr/share/locale")
gettext.textdomain ("system-config-display")
_=gettext.gettext

logFile = "/var/log/Xorg.setup.log"
xserverpid = 0
wm_pid = 0

nameTag = _("Display")
commentTag = _("Configure resolution and colors")

def verbose(str):
    if verbose_output:
        print str

# TODO:
# probe hsync/vsync seems to give different results than db, which one to trust?
# Better way to detect when DRI is available.

def finished_dialog(config_file, backup_file, moved_old_file):
    message = _("Display settings changed\n\n")
    if not xserverpid:
        message = message + _("You need to log out and restart the X server for the changes to take effect.\n")
    if backup_file:
        message = message + _("Configuration was written to %s, original configuration saved as %s.\n") % (config_file, backup_file)
    else:
        message = message + _("Configuration was written to %s.\n") % (config_file)
    if moved_old_file:
        message = message + _("\nNote: The file /etc/X11/XF86Config-4 has been deprecated and we now always use /etc/X11/xorg.conf. The XF86Config-4 file has been saved as XF86Config-4.deprecated.")
        
    m = gtk.MessageDialog(None, gtk.DIALOG_MODAL,
                          gtk.MESSAGE_INFO, gtk.BUTTONS_OK, message)
    m.set_position(gtk.WIN_POS_CENTER)
    m.run()
    m.destroy()

def fork_off_server(xconfig):
    tmp_config = tempfile.mktemp("xorg.config")
    verbose(_("Writing temporary config file to %s") % tmp_config)
    xconfig.write(tmp_config)
    os.environ["DISPLAY"]=":17.0"
    verbose(_("Trying to start X server"))
    args = [ "/usr/bin/Xorg", "-config", tmp_config, "-ac", "-v", "-nolisten", "tcp", "-s",
             "1440", "-terminate", "-dpms", "-logfile", "/dev/null", "-br", ":17"]
    
    serverpid = os.fork()
    if serverpid == 0: #child
        try:
            err = os.open(logFile, os.O_RDWR | os.O_CREAT, 0644)
            if err < 0:
                sys.stderr.write(_("error opening %s\n" % logFile))
            else:
                os.dup2(err, 2)
                os.close(err)
        except:
            # oh well
            pass
        
        os.execv(args[0], args)
        sys.exit (1)
            
    return (serverpid, tmp_config)

def start_x_server(xconfig):
    try:
        (server, tmp_config) = fork_off_server(xconfig)
    except:
	import traceback
        server = None
	(type, value, tb) = sys.exc_info()
	list = traceback.format_exception (type, value, tb)
	text = string.joinfields (list, "")
	print text
    
    # give time for the server to fail (if it is going to fail...)
    if not server:
        verbose(_("X SERVER FAILED"));
        raise RuntimeError, "X server failed to start"

    count = 0

    if verbose_output:
        sys.stdout.write(_("Waiting for X server to start...log located in %s\n") % logFile)
        sys.stdout.flush()

    for i in range(5):
        time.sleep(1)
        if verbose_output:
            sys.stdout.write("%s..." % (i+1))
            sys.stdout.flush()
        
    while count < 60:
        if verbose_output:
            sys.stdout.write(".")
            sys.stdout.flush()
        pid = 0
        try:
            pid, status = os.waitpid (server, os.WNOHANG)
        except OSError, (errno, msg):
            print __name__, "waitpid:", msg
	if pid:
            verbose(_("X SERVER FAILED"));
	    raise RuntimeError, "X server failed to start"
	try:
	    os.stat ("/tmp/.X11-unix/X17")
	    break
	except OSError:
	    pass
	time.sleep(1)
	count = count + 1
  
    # We dont need the configuration file anymore.
    os.unlink(tmp_config)

    if count < 60:
        verbose (_("X server started successfully."))
        return server
    else:
        return 0

def start_window_manager():
    wm_pid = os.fork()

    if (not wm_pid):
        path = '/usr/bin/metacity'
        args = ['--display=:1']
        os.execvp(path, args)

    status = 0
    try:
        pid, status = os.waitpid (wm_pid, os.WNOHANG)

    except OSError, (errno, msg):
        print "in except"
        print __name__, "waitpid:", msg

    if status:
        raise RuntimeError, "Window manager failed to start"

    return wm_pid

def handle_set_commands(set_values, hardware_state):
    # Set up values that limit resolution first
    if set_values.has_key("hsync"):
        hardware_state.set_hsync(set_values["hsync"])
    if set_values.has_key("vsync"):
        hardware_state.set_vsync(set_values["vsync"])
    if set_values.has_key("driver"):
        hardware_state.set_videocard_driver(set_values["driver"])
    if set_values.has_key("resolution"):
        hardware_state.set_resolution(set_values["resolution"])
    if set_values.has_key("depth"):
        depth = string.atoi(set_values["depth"])
        hardware_state.set_colordepth(depth)

def kill_x_server():
    if xserverpid:
        try:
            os.kill(xserverpid, 15)
            os.waitpid(xserverpid, 0)
        except:
            pass
   
def kill_window_manager():
    if wm_pid:
        try:
            os.kill(wm_pid, 15)
            os.waitpid(wm_pid, 0)
        except:
            pass

def usage():
    print \
_("""Usage: system-config-display [OPTIONS]
Options:
  -h, --help        display this help and exit
  -v, --verbose     display what the program is doing more verbosely
  -o, --output=     the filename of the config file to be output
      --reconfig    don't base configuration on existing config files
      --noui        don't show the gui, implicit when using --set options
      --forceui     force the ui to be shown, needed when using --set options
                    to change some setting, but you still want to show the ui
      --set-<key>=  change the value of a specific configuration key.

        currently supported keys are:

        resolution  the screen resolution used
        depth       the color depth in bits
                    most drivers support 8, 15, 16 and 24
        driver      the graphics card driver to use
        vsync       monitor vertical sync rates allowed (in Hz)
        hsync       monitor horizontal sync rates allowed (in kHz)
                    
""")

if __name__ == '__main__':
    signal.signal(signal.SIGINT, signal.SIG_DFL)

textdomain("system-config-display")

output_file = "/etc/X11/xorg.conf"
output_file_specified = FALSE

try:
    (opts, rest_args) = getopt.getopt(sys.argv[1:],
                               "vo:h",
                               ["help", "reconfig", "noui", "verbose",
                                "forceui", "output=",
                                "set-hsync=", "set-vsync=",
                                "set-driver=",
                                "set-resolution=", "set-depth="])
except (getopt.GetoptError), e:
    print e
    print
    usage()
    sys.exit(1)

reconfig = FALSE
no_ui = FALSE
force_ui = FALSE
set_values = {}
verbose_output = FALSE

for (opt, value) in opts:
    if opt == "-v" or opt == "--verbose":
        verbose_output = TRUE
    if opt == "--reconfig":
        reconfig = TRUE
    if opt == "-h" or opt == "--help":
        usage()
        sys.exit(1)
    if opt == "--noui":
        no_ui = TRUE
    if opt == "--forceui":
        force_ui = TRUE
    if opt == "-o" or opt == "--output":
        output_file = value
        output_file_specified = TRUE
    if opt[:6] == "--set-":
        no_ui = TRUE
        set_values[opt[6:]] = value

if force_ui:
    no_ui = FALSE

vc = rhpxl.videocard.VideoCardInfo()
card_list = vc.videocards
hardware_state = None

if len(card_list) <= 0:
    print "No video cards were found on this machine, exiting"
    sys.exit(0)

loop_count = 0
while loop_count < len(card_list):
    xconfig = None
    xconfigpath = None
    xserverpid = 0

    if not reconfig:
        (xconfig, xconfigpath) = xf86config.readConfigFile()
        if xconfig:
            verbose(_("Read configuration file %s") % xconfigpath)
        else:
            verbose(_("Could not find existing X configuration"))

    hardware_state = XF86HardwareState(xconfig)

    current_card =  card_list[loop_count]

    verbose(_("Trying with card: %s") % `current_card.getDescription()`)
    if xconfig == None:
        # If not in reconfig don't override driver changes (eg vesa/s3virge)
        current_card_driver = current_card.getDriver()

        hardware_state.set_videocard_driver(current_card_driver)
        hardware_state.set_videocard_name(current_card.getDescription())
        hardware_state.set_videocard_PCIBus(current_card.getPCIBus())
        hardware_state.set_videocard_PCIDev(current_card.getPCIDev())
        hardware_state.set_videocard_PCIFn(current_card.getPCIFn())        

        keyboard = rhpl.keyboard.Keyboard()
	keyboard.read()
        xconfig = hardware_state.generate_xconfig(keyboard = keyboard)

    handle_set_commands(set_values, hardware_state)

    if not no_ui:
        no_x = FALSE
        try:
            import gtk
            reload(gtk)
        except:
            no_x = TRUE

        if no_x:
            # X not already running.  Try to bring up a server.
            try:
                xserverpid = start_x_server(xconfig)
            except:
                xconfig = None
                print _("Couldn't start X server on card %d"%loop_count)
                loop_count = loop_count + 1;

                if loop_count == len(card_list):
                    if reconfig: 
                        print _("Error, failed to start X server.")
                        sys.exit(1)
                    else:
                        print _("Couldn't start X server with old config, trying with a fresh configuration")
                        reconfig = TRUE;
                        loop_count = 0
                        
                continue

        if xserverpid:
            # if we had to start an X server, make it look half-way decent.
            start_window_manager()
            import gtk
            reload(gtk)        
            import gtk.gdk
            root = gtk.gdk.get_default_root_window()
            arrow = gtk.gdk.Cursor(gtk.gdk.LEFT_PTR)
            root.set_cursor (arrow)

        # Work around bug in pygtk where it sets LC_NUMERIC
        locale.setlocale(locale.LC_NUMERIC,"C")
        import gtk.glade
        # TODO: try to import gtk+, on fail, try to start new X
        import xConfigDialog
        dialog = xConfigDialog.XConfigDialog(hardware_state, xconfig, rhpxl.videocard.VideoCardInfo())
        dialog.update_ui()
        if not dialog.run():
            kill_x_server()
            kill_window_manager()
            sys.exit(1)
    break

if hardware_state:
    hardware_state.merge_into(xconfig)

if xconfigpath and not output_file_specified:
    output_file = xconfigpath
    if output_file == "/etc/X11/XF86Config-4":
        output_file = "/etc/X11/xorg.conf"

moved_old_file = FALSE
if os.access("/etc/X11/XF86Config-4", os.F_OK):
    verbose(_("Moving /etc/X11/XF86Config-4 to /etc/X11/XF86Config-4.deprecated"))
    os.rename("/etc/X11/XF86Config-4", "/etc/X11/XF86Config-4.deprecated")
    moved_old_file = TRUE
            
backup_file = None
if os.access(output_file, os.F_OK):
    backup_file = output_file + ".backup"
    verbose(_("Backing up %s to %s") % (output_file, backup_file))
    try:
        shutil.copy(output_file, backup_file)
    except:
        pass

verbose(_("Writing configuration to %s") % output_file)
xconfig.comment = "# Xorg configuration created by system-config-display\n"
xconfig.write(output_file)

try:
    verbose(_("Removing old /etc/X11/X"))
    os.unlink("/etc/X11/X")
except:
    pass

verbose(_("Creating /etc/X11/X symlink"))
try:
    os.symlink("../../usr/bin/Xorg", "/etc/X11/X")
except:
    pass


# Tell gdm to restart the server when the user logs out
# Otherwise the xserver won't re-read the config file
gdmfifofile = "/var/gdm/.gdmfifo"
if os.access(gdmfifofile, os.F_OK):
    verbose(_("Kicking gdm"))
    try:
        fd = os.open(gdmfifofile, os.O_WRONLY | os.O_APPEND | os.O_NONBLOCK)
        if fd >= 0:
            os.write(fd,"\nDIRTY_SERVERS\n");
            os.close(fd)
    except:
        pass

if not no_ui:
    finished_dialog(output_file, backup_file, moved_old_file)

kill_x_server()
kill_window_manager()
