/*
 * Copyright (c) 2015 Goldman Sachs.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Eclipse Distribution License v. 1.0 which accompany this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 */

package org.eclipse.collections.impl.bag.immutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.io.Serializable;

import org.eclipse.collections.api.ShortIterable;
import org.eclipse.collections.api.LazyShortIterable;
import org.eclipse.collections.api.bag.ImmutableBag;
import org.eclipse.collections.api.bag.primitive.ShortBag;
import org.eclipse.collections.api.bag.primitive.ImmutableShortBag;
import org.eclipse.collections.api.bag.primitive.MutableShortBag;
import org.eclipse.collections.api.block.function.primitive.ShortToObjectFunction;
import org.eclipse.collections.api.block.function.primitive.ObjectShortToObjectFunction;
import org.eclipse.collections.api.block.predicate.primitive.ShortPredicate;
import org.eclipse.collections.api.block.procedure.primitive.ShortIntProcedure;
import org.eclipse.collections.api.block.procedure.primitive.ShortProcedure;
import org.eclipse.collections.api.iterator.ShortIterator;
import org.eclipse.collections.api.list.primitive.MutableShortList;
import org.eclipse.collections.api.set.primitive.MutableShortSet;
import org.eclipse.collections.impl.bag.mutable.primitive.ShortHashBag;
import org.eclipse.collections.impl.block.procedure.checked.primitive.CheckedShortIntProcedure;
import org.eclipse.collections.impl.iterator.UnmodifiableShortIterator;

/**
 * ImmutableShortHashBag is the non-modifiable equivalent of {@link ShortHashBag}.
 * This file was automatically generated from template file immutablePrimitiveHashBag.stg.
 *
 * @since 4.0.
 */
final class ImmutableShortHashBag implements ImmutableShortBag, Serializable
{
    private static final long serialVersionUID = 1L;

    private final MutableShortBag delegate;

    private ImmutableShortHashBag(short[] newElements)
    {
        this.checkOptimizedSize(newElements.length);
        this.delegate = ShortHashBag.newBagWith(newElements);
    }

    private void checkOptimizedSize(int length)
    {
        if (length <= 1)
        {
            throw new IllegalArgumentException("Use ShortBags.immutable.with() to instantiate an optimized collection");
        }
    }

    public static ImmutableShortHashBag newBagWith(short... elements)
    {
        return new ImmutableShortHashBag(elements);
    }

    public ImmutableShortBag newWith(short element)
    {
        return ShortHashBag.newBag(this.delegate).with(element).toImmutable();
    }

    public ImmutableShortBag newWithout(short element)
    {
        ShortHashBag hashBag = ShortHashBag.newBag(this.delegate);
        hashBag.remove(element);
        return hashBag.toImmutable();
    }

    public ImmutableShortBag newWithAll(ShortIterable elements)
    {
        ShortHashBag bag = ShortHashBag.newBag(this.delegate);
        bag.addAll(elements);
        return bag.toImmutable();
    }

    public ImmutableShortBag newWithoutAll(ShortIterable elements)
    {
        ShortHashBag bag = ShortHashBag.newBag(this.delegate);
        bag.removeAll(elements);
        return bag.toImmutable();
    }

    public int size()
    {
        return this.delegate.size();
    }

    public boolean isEmpty()
    {
        return this.delegate.isEmpty();
    }

    public boolean notEmpty()
    {
        return this.delegate.notEmpty();
    }

    public boolean contains(short value)
    {
        return this.delegate.contains(value);
    }

    public boolean containsAll(ShortIterable source)
    {
        return this.delegate.containsAll(source);
    }

    public boolean containsAll(short... elements)
    {
        return this.delegate.containsAll(elements);
    }

    public void forEach(ShortProcedure procedure)
    {
        this.each(procedure);
    }

    /**
     * @since 7.0.
     */
    public void each(ShortProcedure procedure)
    {
        this.delegate.forEach(procedure);
    }

    public ImmutableShortBag select(ShortPredicate predicate)
    {
        return this.delegate.select(predicate).toImmutable();
    }

    public ImmutableShortBag reject(ShortPredicate predicate)
    {
        return this.delegate.reject(predicate).toImmutable();
    }

    public <V> ImmutableBag<V> collect(ShortToObjectFunction<? extends V> function)
    {
        return this.delegate.collect(function).toImmutable();
    }

    public MutableShortList toList()
    {
        return this.delegate.toList();
    }

    public int sizeDistinct()
    {
        return this.delegate.sizeDistinct();
    }

    public int occurrencesOf(short item)
    {
        return this.delegate.occurrencesOf(item);
    }

    public void forEachWithOccurrences(ShortIntProcedure shortIntProcedure)
    {
        this.delegate.forEachWithOccurrences(shortIntProcedure);
    }

    public short detectIfNone(ShortPredicate predicate, short ifNone)
    {
        return this.delegate.detectIfNone(predicate, ifNone);
    }

    public int count(ShortPredicate predicate)
    {
        return this.delegate.count(predicate);
    }

    public boolean anySatisfy(ShortPredicate predicate)
    {
        return this.delegate.anySatisfy(predicate);
    }

    public long sum()
    {
        return this.delegate.sum();
    }

    public short min()
    {
        return this.delegate.min();
    }

    public short max()
    {
        return this.delegate.max();
    }

    public short maxIfEmpty(short defaultValue)
    {
        return this.delegate.maxIfEmpty(defaultValue);
    }

    public short minIfEmpty(short defaultValue)
    {
        return this.delegate.minIfEmpty(defaultValue);
    }

    public double average()
    {
        return this.delegate.average();
    }

    public double median()
    {
        return this.delegate.median();
    }

    public short[] toSortedArray()
    {
        return this.delegate.toSortedArray();
    }

    public MutableShortList toSortedList()
    {
        return this.delegate.toSortedList();
    }

    public boolean noneSatisfy(ShortPredicate predicate)
    {
        return this.delegate.noneSatisfy(predicate);
    }

    public boolean allSatisfy(ShortPredicate predicate)
    {
        return this.delegate.allSatisfy(predicate);
    }

    public <T> T injectInto(T injectedValue, ObjectShortToObjectFunction<? super T, ? extends T> function)
    {
        return this.delegate.injectInto(injectedValue, function);
    }

    @Override
    public boolean equals(Object obj)
    {
        return this.delegate.equals(obj);
    }

    @Override
    public int hashCode()
    {
        return this.delegate.hashCode();
    }

    public MutableShortSet toSet()
    {
        return this.delegate.toSet();
    }

    public MutableShortBag toBag()
    {
        return this.delegate.toBag();
    }

    public ImmutableShortBag toImmutable()
    {
        return this;
    }

    public LazyShortIterable asLazy()
    {
        return this.delegate.asLazy();
    }

    public short[] toArray()
    {
        return this.delegate.toArray();
    }

    @Override
    public String toString()
    {
        return this.delegate.toString();
    }

    public String makeString()
    {
        return this.delegate.makeString();
    }

    public String makeString(String separator)
    {
        return this.delegate.makeString(separator);
    }

    public String makeString(String start, String separator, String end)
    {
        return this.delegate.makeString(start, separator, end);
    }

    public void appendString(Appendable appendable)
    {
        this.delegate.appendString(appendable);
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.delegate.appendString(appendable, separator);
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        this.delegate.appendString(appendable, start, separator, end);
    }

    public ShortIterator shortIterator()
    {
        return new UnmodifiableShortIterator(this.delegate.shortIterator());
    }

    private Object writeReplace()
    {
        return new ImmutableShortBagSerializationProxy(this);
    }

    protected static class ImmutableShortBagSerializationProxy implements Externalizable
    {
        private static final long serialVersionUID = 1L;

        private ShortBag bag;

        @SuppressWarnings("UnusedDeclaration")
        public ImmutableShortBagSerializationProxy()
        {
            // Empty constructor for Externalizable class
        }

        protected ImmutableShortBagSerializationProxy(ShortBag bag)
        {
            this.bag = bag;
        }

        public void writeExternal(final ObjectOutput out) throws IOException
        {
            out.writeInt(this.bag.sizeDistinct());
            try
            {
                this.bag.forEachWithOccurrences(new CheckedShortIntProcedure()
                {
                    @Override
                    public void safeValue(short item, int count) throws IOException
                    {
                        out.writeShort(item);
                        out.writeInt(count);
                    }
                });
            }
            catch (RuntimeException e)
            {
                if (e.getCause() instanceof IOException)
                {
                    throw (IOException) e.getCause();
                }
                throw e;
            }
        }

        public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
        {
            int size = in.readInt();
            MutableShortBag deserializedBag = new ShortHashBag();

            for (int i = 0; i < size; i++)
            {
                deserializedBag.addOccurrences(in.readShort(), in.readInt());
            }

            this.bag = deserializedBag;
        }

        protected Object readResolve()
        {
            return this.bag.toImmutable();
        }
    }
}
