/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.logging.aeri.core.filters;

import static org.apache.commons.lang3.StringUtils.defaultIfEmpty;

import java.util.regex.Pattern;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.epp.logging.aeri.core.util.WildcardPatterns;
import org.eclipse.jdt.annotation.Nullable;

public class StatusIgnorePattern {

    public enum ExceptionStacktracePosition {
        TOP, BOTTOM, ANY
    }

    public enum StatusChainPosition {
        FIRST, ANY
    }

    @Nullable
    public static StatusIgnorePattern fromString(String s) {
        int firstSep = s.indexOf(':');
        if (firstSep == -1) {
            return null;
        }
        String pluginPattern = s.substring(0, firstSep);
        StatusIgnorePattern.StatusChainPosition statusChainPosition = StatusChainPosition.ANY;
        if (pluginPattern.startsWith("^")) {
            pluginPattern = pluginPattern.substring(1);
            statusChainPosition = StatusChainPosition.FIRST;
        }
        String tail = s.substring(firstSep + 1);
        int secondSep = tail.indexOf(':');
        if (secondSep == -1) {
            return null;
        }
        String exceptionPattern = tail.substring(0, secondSep);
        StatusIgnorePattern.ExceptionStacktracePosition exceptionStacktracePosition = ExceptionStacktracePosition.ANY;
        if (exceptionPattern.startsWith("^")) {
            exceptionPattern = exceptionPattern.substring(1);
            exceptionStacktracePosition = ExceptionStacktracePosition.TOP;
        } else if (exceptionPattern.startsWith("$")) {
            exceptionPattern = exceptionPattern.substring(1);
            exceptionStacktracePosition = ExceptionStacktracePosition.BOTTOM;
        }
        String messagePattern = tail.substring(secondSep + 1);
        StatusIgnorePattern ip = new StatusIgnorePattern();
        ip.pluginPattern = WildcardPatterns.convert(defaultIfEmpty(pluginPattern, "*"));
        ip.statusChainPosition = statusChainPosition;
        ip.exceptionPattern = WildcardPatterns.convert(defaultIfEmpty(exceptionPattern, "*"));
        ip.exceptionPosition = exceptionStacktracePosition;
        ip.messagePattern = WildcardPatterns.convert(defaultIfEmpty(messagePattern, "*"));
        return ip;
    }

    private Pattern pluginPattern;
    private Pattern exceptionPattern;
    private Pattern messagePattern;
    private StatusIgnorePattern.ExceptionStacktracePosition exceptionPosition;
    private StatusIgnorePattern.StatusChainPosition statusChainPosition;

    public Pattern getPluginPattern() {
        return pluginPattern;
    }

    public Pattern getExceptionPattern() {
        return exceptionPattern;
    }

    public Pattern getMessagePattern() {
        return messagePattern;
    }

    public StatusIgnorePattern.StatusChainPosition getStatusChainPosition() {
        return statusChainPosition;
    }

    public StatusIgnorePattern.ExceptionStacktracePosition getExceptionPosition() {
        return exceptionPosition;
    }

    public boolean matches(IStatus status) {
        if (matches(status.getPlugin(), status.getException(), status.getMessage())) {
            return true;
        } else if (statusChainPosition == StatusChainPosition.ANY) {
            for (IStatus child : status.getChildren()) {
                if (matches(child)) {
                    return true;
                }
            }
        }
        return false;
    }

    public boolean matches(String pluginId, Throwable exception, String message) {
        boolean pluginMatches = pluginPattern.matcher(pluginId).matches();
        boolean messageMatches = messagePattern.matcher(message).matches();
        if (pluginMatches && messageMatches) {
            switch (exceptionPosition) {
            case ANY: {
                Throwable current = exception;
                while (current != null) {
                    if (exceptionPattern.matcher(current.getClass().getName()).matches()) {
                        return true;
                    }
                    current = current.getCause();
                }
                break;
            }
            case BOTTOM: {
                Throwable current = exception;
                while (current.getCause() != null) {
                    current = current.getCause();
                }
                if (exceptionPattern.matcher(current.getClass().getName()).matches()) {
                    return true;
                }
                break;
            }
            case TOP: {
                if (exceptionPattern.matcher(exception.getClass().getName()).matches()) {
                    return true;
                }
                break;
            }
            default:
                break;
            }
        }
        return false;
    }

}
