/**
 * Copyright (c) 2018 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 */
package org.eclipse.papyrus.robotics.ros2.codegen.common.utils;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.papyrus.designer.languages.common.profile.Codegen.NoCodeGen;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.External;
import org.eclipse.papyrus.designer.transformation.base.utils.TransformationException;
import org.eclipse.papyrus.robotics.codegen.common.utils.ApplyProfiles;
import org.eclipse.papyrus.robotics.codegen.common.utils.ComponentUtils;
import org.eclipse.papyrus.robotics.core.utils.InteractionUtils;
import org.eclipse.papyrus.robotics.core.utils.PortUtils;
import org.eclipse.papyrus.robotics.profile.robotics.commpattern.CommunicationPattern;
import org.eclipse.papyrus.robotics.profile.robotics.services.ServiceDefinitionModel;
import org.eclipse.papyrus.robotics.ros2.base.ProcessUtils;
import org.eclipse.papyrus.robotics.ros2.base.Ros2Constants;
import org.eclipse.papyrus.robotics.ros2.base.Ros2ProcessBuilder;
import org.eclipse.papyrus.robotics.ros2.codegen.common.Activator;
import org.eclipse.papyrus.robotics.ros2.codegen.common.message.CreateMsgPackage;
import org.eclipse.papyrus.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Namespace;
import org.eclipse.uml2.uml.ParameterableElement;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.PrimitiveType;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.TemplateBinding;
import org.eclipse.uml2.uml.TemplateParameterSubstitution;
import org.eclipse.uml2.uml.Type;
import org.eclipse.xtext.xbase.lib.Exceptions;

@SuppressWarnings("all")
public class MessageUtils {
  public static final String MESSAGE = "msg";

  public static final String SERVICE = "srv";

  public static final String ACTION = "action";

  /**
   * Return the message package, i.e. the first package (navigating up) that applies the
   * ServiceDefinitionModel stereotype
   */
  public static org.eclipse.uml2.uml.Package getMessagePackage(final NamedElement ne) {
    try {
      if (((ne == null) || ne.eIsProxy())) {
        return null;
      }
      Element pkg = ne.getNearestPackage();
      while ((pkg instanceof org.eclipse.uml2.uml.Package)) {
        {
          boolean _isApplied = StereotypeUtil.isApplied(pkg, ServiceDefinitionModel.class);
          if (_isApplied) {
            return ((org.eclipse.uml2.uml.Package) pkg);
          }
          pkg = ((org.eclipse.uml2.uml.Package)pkg).getOwner();
        }
      }
      String _format = String.format("Cannot find service definition module for element %s.", ne.getName());
      throw new TransformationException(_format);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Get the qualified ROS 2 message name (i.e. packageName/messageKind/messageName)
   * from a data type (communication object).
   */
  public static String getROS2qMsgName(final DataType dt) {
    String name = dt.getName();
    final org.eclipse.uml2.uml.Package pkg = MessageUtils.getMessagePackage(dt);
    return String.format("%s/%s/%s", pkg.getName().toLowerCase(), MessageUtils.MESSAGE, name);
  }

  /**
   * Return the qualified ROS 2 message name (i.e. packageName/messageKind/messageName)
   * from a service definition. It removes the prefix from the service definition
   * @param sd a service definition
   */
  public static String getROS2qMsgName(final Interface sd) {
    final String name = InteractionUtils.getNameWoPrefix(sd);
    final CommunicationPattern pattern = InteractionUtils.getCommunicationPattern(InteractionUtils.getTemplateBinding(sd));
    String kind = MessageUtils.MESSAGE;
    boolean _isQuery = InteractionUtils.isQuery(pattern);
    if (_isQuery) {
      kind = MessageUtils.SERVICE;
    } else {
      boolean _isAction = InteractionUtils.isAction(pattern);
      if (_isAction) {
        kind = MessageUtils.ACTION;
      }
    }
    final org.eclipse.uml2.uml.Package pkg = MessageUtils.getMessagePackage(sd);
    return String.format("%s/%s/%s", pkg.getName().toLowerCase(), kind, name);
  }

  /**
   * Mark a datatype as external (C++ stereotype), since it is used within a message definition and
   * ROS will general code for it.
   * Instead of its fully qualified name, the type will only be qualified with the ROS 2 package
   * name
   */
  public static void makeDTExternal(final DataType dt) {
    External external = StereotypeUtil.<External>applyApp(dt, External.class);
    if ((external == null)) {
      ApplyProfiles.applyCppProfile(dt);
      external = StereotypeUtil.<External>applyApp(dt, External.class);
    }
    final String qName = MessageUtils.getROS2qMsgName(dt);
    external.setName(qName.replace("/", Namespace.SEPARATOR));
    String _escapeCamlCase = RosHelpers.escapeCamlCase(qName);
    String _plus = (_escapeCamlCase + ".hpp");
    external.setIncPath(_plus);
  }

  /**
   * We should not generate code for the service definition
   */
  public static NoCodeGen makeSvcDefExternal(final TemplateBinding tb) {
    NoCodeGen _xblockexpression = null;
    {
      final Element intf = tb.getOwner();
      _xblockexpression = StereotypeUtil.<NoCodeGen>applyApp(intf, NoCodeGen.class);
    }
    return _xblockexpression;
  }

  /**
   * Transform the service definition into an external ROS 2 type.
   * This is useful to queries and actions for which the name of the ROS 2 service
   * or action is used in generated files (not the name of one of the communication
   * objects)
   */
  public static Interface getServiceType(final Port port) {
    final TemplateBinding tb = InteractionUtils.getTemplateBinding(port);
    MessageUtils.makeSvcDefExternal(tb);
    final Interface sd = InteractionUtils.getServiceDefinition(tb);
    External external = StereotypeUtil.<External>applyApp(sd, External.class);
    if ((external == null)) {
      ApplyProfiles.applyCppProfile(sd);
      external = StereotypeUtil.<External>applyApp(sd, External.class);
    }
    final CommunicationPattern pattern = InteractionUtils.getCommunicationPattern(tb);
    final Stereotype patternSt = StereotypeUtil.apply(pattern.getBase_Collaboration(), NoCodeGen.class);
    if ((patternSt == null)) {
      ApplyProfiles.applyCommonProfile(pattern.getBase_Collaboration());
      StereotypeUtil.apply(pattern.getBase_Collaboration(), NoCodeGen.class);
    }
    final String qName = MessageUtils.getROS2qMsgName(sd);
    external.setName(qName.replace("/", Namespace.SEPARATOR));
    String _escapeCamlCase = RosHelpers.escapeCamlCase(qName);
    String _plus = (_escapeCamlCase + ".hpp");
    external.setIncPath(_plus);
    return sd;
  }

  /**
   * Map a UML primitive type to a native ROS type
   */
  public static String primitiveTypeMap(final Type type) {
    String _switchResult = null;
    String _name = type.getName();
    if (_name != null) {
      switch (_name) {
        case "Integer":
          _switchResult = "int32";
          break;
        case "String":
          _switchResult = "string";
          break;
        case "VSL_Expression":
          _switchResult = "string";
          break;
        case "Real":
          _switchResult = "float64";
          break;
        case "UnlimitedNatural":
          _switchResult = "uint64";
          break;
        case "Boolean":
          _switchResult = "bool";
          break;
        default:
          String _xifexpression = null;
          if ((type.getName().endsWith("_t") && type.getNamespace().getName().equals("AnsiCLibrary"))) {
            String _name_1 = type.getName();
            int _length = type.getName().length();
            int _minus = (_length - 2);
            return _name_1.substring(0, _minus);
          } else {
            _xifexpression = type.getName();
          }
          _switchResult = _xifexpression;
          break;
      }
    } else {
      String _xifexpression = null;
      if ((type.getName().endsWith("_t") && type.getNamespace().getName().equals("AnsiCLibrary"))) {
        String _name_1 = type.getName();
        int _length = type.getName().length();
        int _minus = (_length - 2);
        return _name_1.substring(0, _minus);
      } else {
        _xifexpression = type.getName();
      }
      _switchResult = _xifexpression;
    }
    return _switchResult;
  }

  public static UniqueEList<String> getMsgFileNames(final org.eclipse.uml2.uml.Package msgPackage) {
    final UniqueEList<String> msgFileNames = new UniqueEList<String>();
    List<DataType> _messages = InteractionUtils.getMessages(msgPackage);
    for (final DataType msg : _messages) {
      String _escapeUnderscore = RosHelpers.escapeUnderscore(msg.getName());
      String _plus = (_escapeUnderscore + ".msg");
      msgFileNames.add(_plus);
    }
    return msgFileNames;
  }

  public static UniqueEList<String> getSrvFileNames(final org.eclipse.uml2.uml.Package msgPackage) {
    final UniqueEList<String> srvFileNames = new UniqueEList<String>();
    List<Interface> _queries = InteractionUtils.getQueries(msgPackage);
    for (final Interface sd : _queries) {
      String _escapeUnderscore = RosHelpers.escapeUnderscore(InteractionUtils.getNameWoPrefix(sd));
      String _plus = (_escapeUnderscore + ".srv");
      srvFileNames.add(_plus);
    }
    return srvFileNames;
  }

  public static UniqueEList<String> getActFileNames(final org.eclipse.uml2.uml.Package msgPackage) {
    final UniqueEList<String> actFileNames = new UniqueEList<String>();
    List<Interface> _actions = InteractionUtils.getActions(msgPackage);
    for (final Interface sd : _actions) {
      String _escapeUnderscore = RosHelpers.escapeUnderscore(InteractionUtils.getNameWoPrefix(sd));
      String _plus = (_escapeUnderscore + ".action");
      actFileNames.add(_plus);
    }
    return actFileNames;
  }

  public static void createMessagesOrServices(final CreateMsgPackage msgPkgCreator, final org.eclipse.uml2.uml.Class component) {
    EList<Port> _ownedPorts = component.getOwnedPorts();
    for (final Port port : _ownedPorts) {
      {
        final TemplateBinding tb = InteractionUtils.getTemplateBinding(port);
        if ((tb != null)) {
          msgPkgCreator.createMsgPkgs(tb);
        }
      }
    }
  }

  /**
   * Obtain communication objects for query - request
   */
  public static NamedElement getRequest(final TemplateBinding tb) {
    return MessageUtils.getActual("Request", tb);
  }

  /**
   * Obtain communication objects for query - response
   *   or action - reponse
   */
  public static NamedElement getResponse(final TemplateBinding tb) {
    return MessageUtils.getActual("Response", tb);
  }

  /**
   * Obtain communication objects for action - goal
   */
  public static NamedElement getGoal(final TemplateBinding tb) {
    return MessageUtils.getActual("Goal", tb);
  }

  /**
   * Obtain communication objects for action - feedback
   */
  public static NamedElement getFeedback(final TemplateBinding tb) {
    return MessageUtils.getActual("Feedback", tb);
  }

  /**
   * Obtain an actual from a given formal
   */
  public static NamedElement getActual(final String formalName, final TemplateBinding tb) {
    EList<TemplateParameterSubstitution> _parameterSubstitutions = tb.getParameterSubstitutions();
    for (final TemplateParameterSubstitution tps : _parameterSubstitutions) {
      {
        final ParameterableElement actual = tps.getActual();
        if ((formalName.equals(InteractionUtils.getTPName(tps.getFormal())) && (actual instanceof NamedElement))) {
          return ((NamedElement) actual);
        }
      }
    }
    return null;
  }

  /**
   * Obtain communication objects for push and send, return first
   * message object
   */
  public static NamedElement getMessage(final TemplateBinding tb) {
    EList<TemplateParameterSubstitution> _parameterSubstitutions = tb.getParameterSubstitutions();
    for (final TemplateParameterSubstitution tps : _parameterSubstitutions) {
      {
        final ParameterableElement actual = tps.getActual();
        if ((actual instanceof NamedElement)) {
          return ((NamedElement) actual);
        }
      }
    }
    return null;
  }

  public static String getFileName(final TemplateBinding tb) {
    String names = null;
    EList<TemplateParameterSubstitution> _parameterSubstitutions = tb.getParameterSubstitutions();
    for (final TemplateParameterSubstitution tps : _parameterSubstitutions) {
      {
        final ParameterableElement pe = tps.getActual();
        if ((pe instanceof NamedElement)) {
          final String name = ((NamedElement) pe).getName();
          if ((names == null)) {
            names = name;
          } else {
            names = (names + name);
          }
        }
      }
    }
    return names;
  }

  /**
   * Convenience method:
   * Return the list of package names that are required by a component
   * @param the component
   */
  public static List<String> calcDependencies(final org.eclipse.uml2.uml.Class component) {
    return MessageUtils.calcDependencies(Collections.<org.eclipse.uml2.uml.Class>singletonList(component));
  }

  /**
   * Return the list of package names that are required by a list of components
   * @param the component list (a unique list in order to avoid duplicates)
   */
  public static List<String> calcDependencies(final List<org.eclipse.uml2.uml.Class> components) {
    final UniqueEList<String> list = new UniqueEList<String>();
    list.add("rclcpp");
    boolean _isRegistered = ComponentUtils.isRegistered(components);
    if (_isRegistered) {
      list.add("rclcpp_components");
    }
    list.add("rclcpp_lifecycle");
    boolean _haveActions = MessageUtils.haveActions(components);
    if (_haveActions) {
      list.add("rclcpp_action");
      list.add("rcl_action");
    }
    List<org.eclipse.uml2.uml.Package> _calcUsedMessagePackages = MessageUtils.calcUsedMessagePackages(components);
    for (final org.eclipse.uml2.uml.Package pkg : _calcUsedMessagePackages) {
      {
        final String pkgName = pkg.getName().toLowerCase();
        boolean _contains = list.contains(pkgName);
        boolean _not = (!_contains);
        if (_not) {
          list.add(pkgName);
        }
      }
    }
    return list;
  }

  /**
   * Calculate the message-packages referenced by the messages, services
   * and actions of a message package
   */
  public static UniqueEList<org.eclipse.uml2.uml.Package> calcDependencies(final org.eclipse.uml2.uml.Package msgPackage) {
    final UniqueEList<Type> commObjects = new UniqueEList<Type>();
    final UniqueEList<Type> transitiveHull = new UniqueEList<Type>();
    final UniqueEList<org.eclipse.uml2.uml.Package> msgPackages = new UniqueEList<org.eclipse.uml2.uml.Package>();
    commObjects.addAll(InteractionUtils.getMessages(msgPackage));
    List<Interface> _queries = InteractionUtils.getQueries(msgPackage);
    for (final Interface sd : _queries) {
      commObjects.addAll(InteractionUtils.getCommObjects(InteractionUtils.getTemplateBinding(sd)));
    }
    List<Interface> _actions = InteractionUtils.getActions(msgPackage);
    for (final Interface sd_1 : _actions) {
      commObjects.addAll(InteractionUtils.getCommObjects(InteractionUtils.getTemplateBinding(sd_1)));
    }
    for (final Type commObject : commObjects) {
      MessageUtils.allCommObjects(transitiveHull, commObject);
    }
    for (final Type commObject_1 : transitiveHull) {
      if ((!(commObject_1 instanceof PrimitiveType))) {
        msgPackages.add(MessageUtils.getMessagePackage(commObject_1));
      }
    }
    msgPackages.remove(msgPackage);
    return msgPackages;
  }

  /**
   * Calculate the transitive hull for a given communication object
   */
  public static void allCommObjects(final List<Type> transitiveHull, final Type commObject) {
    transitiveHull.add(commObject);
    if ((commObject instanceof Classifier)) {
      EList<Property> _attributes = ((Classifier)commObject).getAttributes();
      for (final Property attribute : _attributes) {
        if (((attribute.getType() != null) && (!transitiveHull.contains(attribute.getType())))) {
          MessageUtils.allCommObjects(transitiveHull, attribute.getType());
        }
      }
    }
  }

  /**
   * Return the list of packages that are required by a list of components
   * @param the component list
   */
  public static List<org.eclipse.uml2.uml.Package> calcUsedMessagePackages(final List<org.eclipse.uml2.uml.Class> components) {
    final ArrayList<org.eclipse.uml2.uml.Package> list = new ArrayList<org.eclipse.uml2.uml.Package>();
    for (final org.eclipse.uml2.uml.Class component : components) {
      EList<Port> _allPorts = PortUtils.getAllPorts(component);
      for (final Port port : _allPorts) {
        {
          final org.eclipse.uml2.uml.Package pkg = MessageUtils.getMessagePackage(InteractionUtils.getServiceDefinition(port));
          if (((pkg != null) && (!list.contains(pkg)))) {
            list.add(pkg);
          }
        }
      }
    }
    return list;
  }

  /**
   * return true, if one of the component port is an action port
   */
  public static boolean hasActions(final org.eclipse.uml2.uml.Class component) {
    EList<Port> _allPorts = PortUtils.getAllPorts(component);
    for (final Port port : _allPorts) {
      boolean _isAction = InteractionUtils.isAction(InteractionUtils.getCommunicationPattern(port));
      if (_isAction) {
        return true;
      }
    }
    return false;
  }

  /**
   * return true, if one of the components has action ports
   */
  public static boolean haveActions(final List<org.eclipse.uml2.uml.Class> components) {
    for (final org.eclipse.uml2.uml.Class component : components) {
      boolean _hasActions = MessageUtils.hasActions(component);
      if (_hasActions) {
        return true;
      }
    }
    return false;
  }

  /**
   * Return a hash map of available ROS 2 package names
   * 
   * @return the map containing package names
   */
  public static Map<String, Boolean> ros2AvailMsgPkgs() {
    final Ros2ProcessBuilder pbMsg = new Ros2ProcessBuilder(Ros2Constants.PKG, Ros2Constants.LIST);
    final HashMap<String, Boolean> availPackages = new HashMap<String, Boolean>();
    try {
      final Process p = pbMsg.start();
      final InputStream es = p.getErrorStream();
      int _read = es.read();
      boolean _equals = (_read == (-1));
      if (_equals) {
        InputStream _inputStream = p.getInputStream();
        InputStreamReader _inputStreamReader = new InputStreamReader(_inputStream);
        final BufferedReader results = new BufferedReader(_inputStreamReader);
        String line = "";
        boolean found = false;
        while ((((line = results.readLine()) != null) && (!found))) {
          availPackages.put(line.trim(), Boolean.valueOf(true));
        }
        results.close();
      } else {
        ProcessUtils.logErrors(p);
      }
    } catch (final Throwable _t) {
      if (_t instanceof IOException) {
        final IOException e = (IOException)_t;
        Activator.log.error(e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return availPackages;
  }
}
