/**
 * Copyright (c) 2017 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 *  Shuai Li (CEA LIST) <shuai.li@cea.fr> - Bug 530651
 *  Shuai Li (CEA LIST) <shuai.li@cea.fr> - Bug 531771
 *  Yoann Farre (CIL4Sys) <yoann.farre@cil4sys.com> - Bug 543072
 */
package org.eclipse.papyrus.designer.components.modellibs.core.transformations;

import com.google.common.base.Objects;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.components.FCM.Assembly;
import org.eclipse.papyrus.designer.components.modellibs.core.transformations.Constants;
import org.eclipse.papyrus.designer.components.transformation.PortInfo;
import org.eclipse.papyrus.designer.components.transformation.PortUtils;
import org.eclipse.papyrus.designer.components.transformation.component.PrefixConstants;
import org.eclipse.papyrus.designer.components.transformation.extensions.IOOTrafo;
import org.eclipse.papyrus.designer.transformation.base.utils.CopyUtils;
import org.eclipse.papyrus.designer.transformation.base.utils.TransformationException;
import org.eclipse.papyrus.designer.transformation.core.transformations.LazyCopier;
import org.eclipse.papyrus.uml.tools.utils.ConnectorUtil;
import org.eclipse.papyrus.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.AggregationKind;
import org.eclipse.uml2.uml.Association;
import org.eclipse.uml2.uml.Behavior;
import org.eclipse.uml2.uml.BehavioredClassifier;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.ConnectableElement;
import org.eclipse.uml2.uml.Connector;
import org.eclipse.uml2.uml.ConnectorEnd;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.InterfaceRealization;
import org.eclipse.uml2.uml.MultiplicityElement;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.StructuralFeature;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.xtend2.lib.StringConcatenation;

/**
 * This class realizes the transformation from component-based to object-oriented
 * models. It includes the replacement of ports and connectors. Ports are
 * replaced with attributes and access operations, connectors within a composite
 * by an operation that creates the initial setup. It is an abstract class that is
 * refined for C++ and Java
 * 
 * 1. add an operation that allows to retrieve the reference to an interface provided
 *    by a port. This operation has a mapping to a specific name, e.g. get_<port_name>
 * 
 * 2. add an operation that allows to connect a specific port.
 *    the connect_q operation (*including a storage attribute*) for a port with a required interface
 * 
 * 3. add an implementation for the getcnx_q operation for a port
 *    with a required interface (the operation itself has been added before)
 */
@SuppressWarnings("all")
public abstract class AbstractCompToOO implements IOOTrafo {
  protected LazyCopier copier;
  
  protected String progLang;
  
  public static final String NL = "\n";
  
  @Override
  public void init(final LazyCopier copier, final org.eclipse.uml2.uml.Class bootloader) {
    PrefixConstants.init(PrefixConstants.CIFvariant.UML);
    this.copier = copier;
  }
  
  /**
   * de-reference an attribute. Use language specific mechanism, e.g. attributeName-> in case of a C++ pointer
   */
  public abstract String nameRef(final Property attribute);
  
  /**
   * apply a stereotypes that transforms an elements into a reference, a pointer in case of C++
   */
  public abstract void applyRef(final Element element);
  
  /**
   * Return the reference of an attribute
   */
  public abstract String getRef(final Property attribute);
  
  @Override
  public void addPortOperations(final org.eclipse.uml2.uml.Class implementation) {
    boolean _isAbstract = implementation.isAbstract();
    boolean _not = (!_isAbstract);
    if (_not) {
      this.addGetPortOperation(implementation);
    }
    this.addConnectPortOperation(implementation);
  }
  
  /**
   * Add the get_p operation for each port with a provided interface. It also
   * adds a suitable implementation that evaluates delegation connectors from
   * the port to a property within the composite. The delegation target could
   * either be a normal class (no port) or an inner component.
   * 
   * @param implementation
   */
  public void addGetPortOperation(final org.eclipse.uml2.uml.Class implementation) {
    EList<PortInfo> _flattenExtendedPorts = PortUtils.flattenExtendedPorts(PortUtils.getAllPorts2(implementation));
    for (final PortInfo portInfo : _flattenExtendedPorts) {
      {
        final List<Interface> providedIntfs = portInfo.getProvideds();
        if (((providedIntfs != null) && (!providedIntfs.isEmpty()))) {
          if (((providedIntfs.size() == 1) && (portInfo.getPort().getType() instanceof Interface))) {
            this.addGetPortOperation(implementation, portInfo, providedIntfs.get(0), portInfo.getName());
          } else {
            for (final Interface providedIntf : providedIntfs) {
              String _name = portInfo.getName();
              String _name_1 = providedIntf.getName();
              String _plus = (_name + _name_1);
              this.addGetPortOperation(implementation, portInfo, providedIntf, _plus);
            }
          }
        }
      }
    }
  }
  
  public Object addGetPortOperation(final org.eclipse.uml2.uml.Class implementation, final PortInfo portInfo, final Interface providedIntf, final String portName) {
    Object _xblockexpression = null;
    {
      final String opName = (PrefixConstants.getP_Prefix + portName);
      Operation op = implementation.getOwnedOperation(opName, null, null);
      Object _xifexpression = null;
      if ((op != null)) {
        Parameter _xifexpression_1 = null;
        Type _type = op.getType();
        boolean _notEquals = (!Objects.equal(_type, providedIntf));
        if (_notEquals) {
          _xifexpression_1 = op.createOwnedParameter(Constants.retParamName, providedIntf);
        }
        _xifexpression = _xifexpression_1;
      } else {
        boolean _xblockexpression_1 = false;
        {
          op = implementation.createOwnedOperation(opName, null, null, providedIntf);
          final Parameter retParam = op.getOwnedParameters().get(0);
          retParam.setName(Constants.retParamName);
          this.applyRef(retParam);
          Behavior _createOwnedBehavior = implementation.createOwnedBehavior(opName, 
            UMLPackage.eINSTANCE.getOpaqueBehavior());
          final OpaqueBehavior behavior = ((OpaqueBehavior) _createOwnedBehavior);
          op.getMethods().add(behavior);
          final List<ConnectorEnd> ces = ConnectorUtil.getDelegations(implementation, portInfo.getModelPort());
          String body = null;
          boolean _isEmpty = ces.isEmpty();
          boolean _not = (!_isEmpty);
          if (_not) {
            body = "return ";
            int i = 0;
            while (((i < ces.size()) && body.equals("return "))) {
              {
                final Property part = ces.get(i).getPartWithPort();
                final ConnectableElement role = ces.get(i).getRole();
                if ((role instanceof Port)) {
                  final Port rolePort = ((Port) role);
                  boolean _contains = rolePort.getProvideds().contains(providedIntf);
                  if (_contains) {
                    int _size = rolePort.getProvideds().size();
                    boolean _greaterThan = (_size > 1);
                    if (_greaterThan) {
                      String _body = body;
                      StringConcatenation _builder = new StringConcatenation();
                      String _nameRef = this.nameRef(part);
                      _builder.append(_nameRef);
                      _builder.append(PrefixConstants.getP_Prefix);
                      String _name = ((Port)role).getName();
                      _builder.append(_name);
                      String _name_1 = providedIntf.getName();
                      _builder.append(_name_1);
                      _builder.append("();");
                      body = (_body + _builder);
                    } else {
                      String _body_1 = body;
                      StringConcatenation _builder_1 = new StringConcatenation();
                      String _nameRef_1 = this.nameRef(part);
                      _builder_1.append(_nameRef_1);
                      _builder_1.append(PrefixConstants.getP_Prefix);
                      String _name_2 = ((Port)role).getName();
                      _builder_1.append(_name_2);
                      _builder_1.append("();");
                      body = (_body_1 + _builder_1);
                    }
                  }
                } else {
                  if ((role instanceof Property)) {
                    final Type roleType = ((Property) role).getType();
                    if (((roleType instanceof BehavioredClassifier) && (((BehavioredClassifier) roleType).getInterfaceRealization(null, providedIntf) != null))) {
                      String _body_2 = body;
                      String _name_3 = ((Property)role).getName();
                      body = (_body_2 + _name_3);
                    }
                  }
                }
                i++;
              }
            }
          } else {
            InterfaceRealization _interfaceRealization = implementation.getInterfaceRealization(null, providedIntf);
            boolean implementsIntf = (_interfaceRealization != null);
            if ((!implementsIntf)) {
              final Interface providedIntfInCopy = this.copier.<Interface>getCopy(providedIntf);
              InterfaceRealization _interfaceRealization_1 = implementation.getInterfaceRealization(null, providedIntfInCopy);
              boolean _tripleNotEquals = (_interfaceRealization_1 != null);
              implementsIntf = _tripleNotEquals;
            }
            if (implementsIntf) {
              body = "return this;";
            } else {
              String _format = String.format(
                "Interface <%s> provided by port <%s> of class <%s> is not implemented by the component itself nor does the port delegate to a part", 
                providedIntf.getName(), portName, implementation.getName());
              throw new RuntimeException(_format);
            }
          }
          behavior.getLanguages().add(this.progLang);
          _xblockexpression_1 = behavior.getBodies().add(body);
        }
        _xifexpression = Boolean.valueOf(_xblockexpression_1);
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * Add a connect_<portName> operation for ports with a required interface.
   * Whereas operation and a behavior is added for each owned port, a behavior
   * (method) is needed for ports inherited from a component type (the
   * behavior is implementation specific, as it needs to take delegation to
   * parts into account)
   * 
   * @param implementation
   */
  public void addConnectPortOperation(final org.eclipse.uml2.uml.Class implementation) {
    EList<PortInfo> _flattenExtendedPorts = PortUtils.flattenExtendedPorts(PortUtils.getAllPorts2(implementation));
    for (final PortInfo portInfo : _flattenExtendedPorts) {
      {
        final List<Interface> requiredIntfs = portInfo.getRequireds();
        if (((requiredIntfs != null) && (!requiredIntfs.isEmpty()))) {
          if (((requiredIntfs.size() == 1) && (portInfo.getPort().getType() instanceof Interface))) {
            this.addConnectPortOperation(implementation, portInfo, requiredIntfs.get(0), portInfo.getName());
          } else {
            for (final Interface requiredIntf : requiredIntfs) {
              String _name = portInfo.getName();
              String _name_1 = requiredIntf.getName();
              String _plus = (_name + _name_1);
              this.addConnectPortOperation(implementation, portInfo, requiredIntf, _plus);
            }
          }
        }
      }
    }
  }
  
  public Boolean addConnectPortOperation(final org.eclipse.uml2.uml.Class implementation, final PortInfo portInfo, final Interface requiredIntf, final String portName) {
    Boolean _xblockexpression = null;
    {
      final String opName = (PrefixConstants.connectQ_Prefix + portName);
      Boolean _xifexpression = null;
      Operation _ownedOperation = implementation.getOwnedOperation(opName, null, null);
      boolean _tripleNotEquals = (_ownedOperation != null);
      if (_tripleNotEquals) {
        _xifexpression = null;
      } else {
        boolean _xblockexpression_1 = false;
        {
          Operation op = implementation.createOwnedOperation(opName, null, null);
          final boolean multiPort = ((portInfo.getUpper() > 1) || (portInfo.getUpper() == (-1)));
          final Parameter refParam = op.createOwnedParameter("ref", requiredIntf);
          this.applyRef(refParam);
          Behavior _createOwnedBehavior = implementation.createOwnedBehavior(opName, 
            UMLPackage.eINSTANCE.getOpaqueBehavior());
          final OpaqueBehavior behavior = ((OpaqueBehavior) _createOwnedBehavior);
          op.getMethods().add(behavior);
          final List<ConnectorEnd> ces = ConnectorUtil.getDelegations(implementation, portInfo.getModelPort());
          String body = "";
          boolean _isEmpty = ces.isEmpty();
          boolean _not = (!_isEmpty);
          if (_not) {
            int i = 0;
            while (((i < ces.size()) && body.isEmpty())) {
              {
                final Property part = ces.get(i).getPartWithPort();
                final ConnectableElement role = ces.get(i).getRole();
                if ((role instanceof Port)) {
                  boolean _contains = ((Port) role).getRequireds().contains(requiredIntf);
                  if (_contains) {
                    body = part.getName();
                    String _name = ((Port)role).getName();
                    String targetOpName = (PrefixConstants.connectQ_Prefix + _name);
                    int _size = ((Port) role).getRequireds().size();
                    boolean _greaterThan = (_size > 1);
                    if (_greaterThan) {
                      String _targetOpName = targetOpName;
                      String _name_1 = requiredIntf.getName();
                      targetOpName = (_targetOpName + _name_1);
                    }
                    StringConcatenation _builder = new StringConcatenation();
                    String _nameRef = this.nameRef(part);
                    _builder.append(_nameRef);
                    _builder.append(targetOpName);
                    _builder.append("(ref);");
                    body = _builder.toString();
                  }
                } else {
                  if (((part.getType() instanceof Classifier) && ((Classifier) part.getType()).getAllUsedInterfaces().contains(requiredIntf))) {
                    String _body = body;
                    StringConcatenation _builder_1 = new StringConcatenation();
                    String _name_2 = part.getName();
                    _builder_1.append(_name_2);
                    _builder_1.append(";");
                    body = (_body + _builder_1);
                  }
                }
                i++;
              }
            }
          } else {
            final String attributeName = (PrefixConstants.attributePrefix + portName);
            Property attr = implementation.getOwnedAttribute(attributeName, null);
            if (((attr == null) || (attr instanceof Port))) {
              attr = implementation.createOwnedAttribute(attributeName, requiredIntf);
              CopyUtils.copyMultElemModifiers(portInfo.getPort(), attr);
              attr.setAggregation(AggregationKind.SHARED_LITERAL);
            }
            body = attributeName;
            if (multiPort) {
              String _body = body;
              body = (_body + "[index]");
            }
            String _body_1 = body;
            body = (_body_1 + " = ref;");
          }
          behavior.getLanguages().add(this.progLang);
          behavior.getBodies().add(body);
          boolean _xifexpression_1 = false;
          if (((PrefixConstants.getConnQ_Prefix.length() > 0) && (!ces.isEmpty()))) {
            boolean _xblockexpression_2 = false;
            {
              final String getConnOpName = (PrefixConstants.getConnQ_Prefix + portName);
              Operation getConnOp = implementation.getOwnedOperation(getConnOpName, null, null);
              if ((getConnOp == null)) {
                getConnOp = implementation.createOwnedOperation(getConnOpName, null, null, requiredIntf);
                final Parameter retParam = op.getOwnedParameters().get(0);
                retParam.setName(Constants.retParamName);
                this.applyRef(retParam);
              }
              Behavior _createOwnedBehavior_1 = implementation.createOwnedBehavior(getConnOpName, 
                UMLPackage.eINSTANCE.getOpaqueBehavior());
              final OpaqueBehavior getConnBehavior = ((OpaqueBehavior) _createOwnedBehavior_1);
              getConnOp.getMethods().add(getConnBehavior);
              final String name = (PrefixConstants.attributePrefix + portName);
              StringConcatenation _builder = new StringConcatenation();
              _builder.append("return ");
              _builder.append(name);
              _builder.append(";");
              body = _builder.toString();
              behavior.getLanguages().add(this.progLang);
              _xblockexpression_2 = behavior.getBodies().add(body);
            }
            _xifexpression_1 = _xblockexpression_2;
          }
          _xblockexpression_1 = _xifexpression_1;
        }
        _xifexpression = Boolean.valueOf(_xblockexpression_1);
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * Add an operation "createConnections" that implements the connections
   * between composite parts. It only takes the assembly connections into
   * account, since delegation connectors are handled by the get_ and connect_
   * port operations above.
   * 
   * @param implementation
   */
  @Override
  public void addConnectionOperation(final org.eclipse.uml2.uml.Class compositeImplementation) throws TransformationException {
    String createConnBody = "";
    final Map<MultiplicityElement, Integer> indexMap = new HashMap<MultiplicityElement, Integer>();
    EList<Connector> _ownedConnectors = compositeImplementation.getOwnedConnectors();
    for (final Connector connector : _ownedConnectors) {
      boolean _isAssembly = ConnectorUtil.isAssembly(connector);
      if (_isAssembly) {
        int _size = connector.getEnds().size();
        boolean _notEquals = (_size != 2);
        if (_notEquals) {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("Connector <");
          String _name = connector.getName();
          _builder.append(_name);
          _builder.append("> does not have two ends. This is currently not supported");
          throw new TransformationException(_builder.toString());
        }
        final ConnectorEnd end1 = connector.getEnds().get(0);
        final ConnectorEnd end2 = connector.getEnds().get(1);
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("// realization of connector <");
        String _name_1 = connector.getName();
        _builder_1.append(_name_1);
        _builder_1.append(">");
        String cmd = (_builder_1.toString() + AbstractCompToOO.NL);
        if (((end1.getRole() instanceof Port) && PortUtils.isExtendedPort(((Port) end1.getRole())))) {
          ConnectableElement _role = end1.getRole();
          final Port port = ((Port) _role);
          final EList<PortInfo> subPorts = PortUtils.flattenExtendedPort(port);
          for (final PortInfo subPort : subPorts) {
            {
              String _cmd = cmd;
              StringConcatenation _builder_2 = new StringConcatenation();
              _builder_2.append("  ");
              _builder_2.append("// realization of connection for sub-port ");
              String _name_2 = subPort.getPort().getName();
              _builder_2.append(_name_2, "  ");
              String _plus = (_builder_2.toString() + AbstractCompToOO.NL);
              cmd = (_cmd + _plus);
              String _cmd_1 = cmd;
              String _connectPorts = this.connectPorts(connector, end1, end2, subPort.getPort());
              cmd = (_cmd_1 + _connectPorts);
              String _cmd_2 = cmd;
              String _connectPorts_1 = this.connectPorts(connector, end2, end1, subPort.getPort());
              cmd = (_cmd_2 + _connectPorts_1);
            }
          }
        } else {
          String _cmd = cmd;
          String _connectPorts = this.connectPorts(connector, end1, end2, null);
          cmd = (_cmd + _connectPorts);
          String _cmd_1 = cmd;
          String _connectPorts_1 = this.connectPorts(connector, end2, end1, null);
          cmd = (_cmd_1 + _connectPorts_1);
        }
        String _createConnBody = createConnBody;
        createConnBody = (_createConnBody + (cmd + AbstractCompToOO.NL));
      }
    }
    int _length = createConnBody.length();
    boolean _greaterThan = (_length > 0);
    if (_greaterThan) {
      final Operation operation = compositeImplementation.createOwnedOperation(Constants.CREATE_CONNECTIONS, null, null);
      Behavior _createOwnedBehavior = compositeImplementation.createOwnedBehavior(operation.getName(), 
        UMLPackage.eINSTANCE.getOpaqueBehavior());
      final OpaqueBehavior behavior = ((OpaqueBehavior) _createOwnedBehavior);
      behavior.getLanguages().add(this.progLang);
      behavior.getBodies().add(createConnBody);
      behavior.setSpecification(operation);
    }
  }
  
  /**
   * Create the body code that creates a connection between the two ends
   * of a connector. This function checks whether the first end really is a receptacle
   * and the second really is a facet.
   * 
   * @param indexMap
   *            a map of indices that are used in case of multiplex
   *            receptacles
   * @param connector
   *            a connector
   * @param receptacleEnd
   *            an end of the connector that may point to a receptacle port
   * @param facetEnd
   *            an end of the connector that may point to a facet port
   * @param subPort
   *            a sub-port in case of extended ports
   * @return
   * @throws TransformationException
   */
  public String connectPorts(final Connector connector, final ConnectorEnd receptacleEnd, final ConnectorEnd facetEnd, final Port subPort) throws TransformationException {
    final Association association = connector.getType();
    if (((receptacleEnd.getRole() instanceof Port) && (facetEnd.getRole() instanceof Port))) {
      ConnectableElement _role = facetEnd.getRole();
      final Port facetPort = ((Port) _role);
      ConnectableElement _role_1 = receptacleEnd.getRole();
      final Port receptaclePort = ((Port) _role_1);
      final PortInfo facetPI = PortInfo.fromSubPort(facetPort, subPort);
      final PortInfo receptaclePI = PortInfo.fromSubPort(receptaclePort, subPort);
      if (((facetPI.getProvided() != null) && (receptaclePI.getRequired() != null))) {
        final Property facetPart = facetEnd.getPartWithPort();
        final Property receptaclePart = receptacleEnd.getPartWithPort();
        String subPortName = "";
        if ((subPort != null)) {
          String _subPortName = subPortName;
          String _name = subPort.getName();
          String _plus = ("_" + _name);
          subPortName = (_subPortName + _plus);
        }
        String result = "";
        int _size = receptaclePI.getRequireds().size();
        boolean _greaterThan = (_size > 1);
        if (_greaterThan) {
          List<Interface> _requireds = receptaclePI.getRequireds();
          for (final Interface requiredInterface : _requireds) {
            {
              String _name_1 = receptaclePI.getName();
              String _name_2 = requiredInterface.getName();
              String receptaclePortName = (_name_1 + _name_2);
              String facetPortName = "";
              boolean _contains = facetPI.getProvideds().contains(requiredInterface);
              if (_contains) {
                String _facetPortName = facetPortName;
                String _name_3 = facetPI.getName();
                facetPortName = (_facetPortName + _name_3);
                int _size_1 = facetPI.getProvideds().size();
                boolean _greaterThan_1 = (_size_1 > 1);
                if (_greaterThan_1) {
                  String _facetPortName_1 = facetPortName;
                  String _name_4 = requiredInterface.getName();
                  facetPortName = (_facetPortName_1 + _name_4);
                }
              }
              boolean _isEmpty = facetPortName.isEmpty();
              boolean _not = (!_isEmpty);
              if (_not) {
                StringConcatenation _builder = new StringConcatenation();
                String _nameRef = this.nameRef(receptaclePart);
                _builder.append(_nameRef);
                _builder.append("connect_");
                _builder.append(receptaclePortName);
                _builder.append(subPortName);
                final String setter = _builder.toString();
                StringConcatenation _builder_1 = new StringConcatenation();
                String _nameRef_1 = this.nameRef(facetPart);
                _builder_1.append(_nameRef_1);
                _builder_1.append("get_");
                _builder_1.append(facetPortName);
                _builder_1.append(subPortName);
                _builder_1.append("()");
                final String getter = _builder_1.toString();
                String _result = result;
                StringConcatenation _builder_2 = new StringConcatenation();
                _builder_2.append(setter);
                _builder_2.append("(");
                _builder_2.append(getter);
                _builder_2.append(");");
                String _plus_1 = (_builder_2.toString() + AbstractCompToOO.NL);
                result = (_result + _plus_1);
              }
            }
          }
        } else {
          String facetPortName = "";
          if (((facetPI.getProvideds().size() > 1) && facetPI.getProvideds().contains(receptaclePI.getRequired()))) {
            String _facetPortName = facetPortName;
            String _name_1 = facetPI.getName();
            String _name_2 = receptaclePI.getRequired().getName();
            String _plus_1 = (_name_1 + _name_2);
            facetPortName = (_facetPortName + _plus_1);
          } else {
            int _size_1 = facetPI.getProvideds().size();
            boolean _equals = (_size_1 == 1);
            if (_equals) {
              facetPortName = facetPI.getName();
            }
          }
          boolean _isEmpty = facetPortName.isEmpty();
          boolean _not = (!_isEmpty);
          if (_not) {
            StringConcatenation _builder = new StringConcatenation();
            String _nameRef = this.nameRef(receptaclePart);
            _builder.append(_nameRef);
            _builder.append("connect_");
            String _name_3 = receptaclePort.getName();
            _builder.append(_name_3);
            _builder.append(subPortName);
            final String setter = _builder.toString();
            StringConcatenation _builder_1 = new StringConcatenation();
            String _nameRef_1 = this.nameRef(facetPart);
            _builder_1.append(_nameRef_1);
            _builder_1.append("get_");
            _builder_1.append(facetPortName);
            _builder_1.append(subPortName);
            _builder_1.append("()");
            final String getter = _builder_1.toString();
            String _result = result;
            StringConcatenation _builder_2 = new StringConcatenation();
            _builder_2.append(setter);
            _builder_2.append("(");
            _builder_2.append(getter);
            _builder_2.append(");");
            String _plus_2 = (_builder_2.toString() + AbstractCompToOO.NL);
            result = (_result + _plus_2);
          }
        }
        return result;
      }
    } else {
      ConnectableElement _role_2 = receptacleEnd.getRole();
      if ((_role_2 instanceof Port)) {
        ConnectableElement _role_3 = receptacleEnd.getRole();
        final Port receptaclePort_1 = ((Port) _role_3);
        Interface _required = PortUtils.getRequired(receptaclePort_1);
        boolean _tripleNotEquals = (_required != null);
        if (_tripleNotEquals) {
          ConnectableElement _role_4 = facetEnd.getRole();
          final Property facetPart_1 = ((Property) _role_4);
          final Property receptaclePart_1 = facetEnd.getPartWithPort();
          String result_1 = "";
          int _size_2 = receptaclePort_1.getRequireds().size();
          boolean _greaterThan_1 = (_size_2 > 1);
          if (_greaterThan_1) {
            EList<Interface> _requireds_1 = receptaclePort_1.getRequireds();
            for (final Interface requiredInterface_1 : _requireds_1) {
              {
                String _name_4 = receptaclePort_1.getName();
                String _name_5 = requiredInterface_1.getName();
                String receptaclePortName = (_name_4 + _name_5);
                StringConcatenation _builder_3 = new StringConcatenation();
                String _nameRef_2 = this.nameRef(receptaclePart_1);
                _builder_3.append(_nameRef_2);
                _builder_3.append("connect_");
                _builder_3.append(receptaclePortName);
                final String setter_1 = _builder_3.toString();
                StringConcatenation _builder_4 = new StringConcatenation();
                String _ref = this.getRef(facetPart_1);
                _builder_4.append(_ref);
                final String getter_1 = _builder_4.toString();
                String _result_1 = result_1;
                StringConcatenation _builder_5 = new StringConcatenation();
                _builder_5.append(setter_1);
                _builder_5.append("(");
                _builder_5.append(getter_1);
                _builder_5.append(");");
                String _plus_3 = (_builder_5.toString() + AbstractCompToOO.NL);
                result_1 = (_result_1 + _plus_3);
              }
            }
          } else {
            StringConcatenation _builder_3 = new StringConcatenation();
            String _nameRef_2 = this.nameRef(receptaclePart_1);
            _builder_3.append(_nameRef_2);
            _builder_3.append("connect_");
            String _name_4 = receptaclePort_1.getName();
            _builder_3.append(_name_4);
            final String setter_1 = _builder_3.toString();
            StringConcatenation _builder_4 = new StringConcatenation();
            String _ref = this.getRef(facetPart_1);
            _builder_4.append(_ref);
            final String getter_1 = _builder_4.toString();
            String _result_1 = result_1;
            StringConcatenation _builder_5 = new StringConcatenation();
            _builder_5.append(setter_1);
            _builder_5.append("(");
            _builder_5.append(getter_1);
            _builder_5.append(");");
            String _plus_3 = (_builder_5.toString() + AbstractCompToOO.NL);
            result_1 = (_result_1 + _plus_3);
          }
          return result_1;
        }
      } else {
        ConnectableElement _role_5 = facetEnd.getRole();
        if ((_role_5 instanceof Port)) {
          ConnectableElement _role_6 = facetEnd.getRole();
          final Port facetPort_1 = ((Port) _role_6);
          Interface _provided = PortUtils.getProvided(facetPort_1);
          boolean _tripleNotEquals_1 = (_provided != null);
          if (_tripleNotEquals_1) {
            final Property facetPart_2 = facetEnd.getPartWithPort();
            ConnectableElement _role_7 = facetEnd.getRole();
            final Property receptaclePart_2 = ((Property) _role_7);
            String result_2 = "";
            int _size_3 = facetPort_1.getProvideds().size();
            boolean _greaterThan_2 = (_size_3 > 1);
            if (_greaterThan_2) {
              EList<Interface> _provideds = facetPort_1.getProvideds();
              for (final Interface providedInterface : _provideds) {
                {
                  String _name_5 = facetPort_1.getName();
                  String _name_6 = providedInterface.getName();
                  String facetPortName_1 = (_name_5 + _name_6);
                  final String setter_2 = receptaclePart_2.getName();
                  StringConcatenation _builder_6 = new StringConcatenation();
                  String _nameRef_3 = this.nameRef(facetPart_2);
                  _builder_6.append(_nameRef_3);
                  _builder_6.append("get_");
                  _builder_6.append(facetPortName_1);
                  _builder_6.append("();");
                  final String getter_2 = _builder_6.toString();
                  String _result_2 = result_2;
                  StringConcatenation _builder_7 = new StringConcatenation();
                  _builder_7.append(setter_2);
                  _builder_7.append(" = ");
                  _builder_7.append(getter_2);
                  _builder_7.append(";");
                  String _plus_4 = (_builder_7.toString() + AbstractCompToOO.NL);
                  result_2 = (_result_2 + _plus_4);
                }
              }
            } else {
              final String setter_2 = receptaclePart_2.getName();
              StringConcatenation _builder_6 = new StringConcatenation();
              String _nameRef_3 = this.nameRef(facetPart_2);
              _builder_6.append(_nameRef_3);
              _builder_6.append("get_");
              String _name_5 = facetPort_1.getName();
              _builder_6.append(_name_5);
              _builder_6.append("();");
              final String getter_2 = _builder_6.toString();
              String _result_2 = result_2;
              StringConcatenation _builder_7 = new StringConcatenation();
              _builder_7.append(setter_2);
              _builder_7.append(" = ");
              _builder_7.append(getter_2);
              _builder_7.append(";");
              String _plus_4 = (_builder_7.toString() + AbstractCompToOO.NL);
              result_2 = (_result_2 + _plus_4);
            }
            return result_2;
          }
        } else {
          if ((association != null)) {
            ConnectableElement _role_8 = facetEnd.getRole();
            final Property facetPart_3 = ((Property) _role_8);
            ConnectableElement _role_9 = receptacleEnd.getRole();
            final Property receptaclePart_3 = ((Property) _role_9);
            final Property assocProp1 = association.getMemberEnd(null, facetPart_3.getType());
            if (((assocProp1 != null) && assocProp1.isNavigable())) {
              StringConcatenation _builder_8 = new StringConcatenation();
              String _name_6 = receptaclePart_3.getName();
              _builder_8.append(_name_6);
              _builder_8.append(".");
              String _name_7 = assocProp1.getName();
              _builder_8.append(_name_7);
              String setter_3 = _builder_8.toString();
              StringConcatenation _builder_9 = new StringConcatenation();
              String _name_8 = facetPart_3.getName();
              _builder_9.append(_name_8);
              final String getter_3 = _builder_9.toString();
              StringConcatenation _builder_10 = new StringConcatenation();
              _builder_10.append(setter_3);
              _builder_10.append(" = ");
              _builder_10.append(getter_3);
              _builder_10.append(";");
              return (_builder_10.toString() + AbstractCompToOO.NL);
            }
          } else {
            String _name_9 = connector.getName();
            String _plus_5 = ("Connector <" + _name_9);
            String _plus_6 = (_plus_5 + 
              "> does not use ports, but it is not typed (only connectors between ports should not be typed)");
            throw new TransformationException(_plus_6);
          }
        }
      }
    }
    return "";
  }
  
  /**
   * Return true, if the bootloader is responsible for the instantiation of a
   * part.
   * 
   * If a part is a component type or an abstract implementation, it cannot be
   * instantiated. Thus, a heir has to be selected in the deployment plan.
   * Since the selection might be different for different instances of the
   * composite, the instantiation is not done by the component itself, but by
   * the bootloader. The bootloader also has to instantiate, if different
   * allocation variants are required. (this is for instance the case for
   * distribution connectors and for the system itself)
   * 
   * If possible, we want to let composites instantiate sub-components, since
   * this eases the transition to systems which support reconfiguration.
   * 
   * [TODO: optimization: analyze whether the deployment plan selects a single
   * implementation. If yes, let the composite instantiate]
   * 
   * @param implementation a composite component
   * @return
   */
  public static boolean instantiateViaBootloader(final org.eclipse.uml2.uml.Class implementation) {
    return (implementation.isAbstract() || StereotypeUtil.isApplied(implementation, Assembly.class));
  }
  
  /**
   * Return whether a part needs to be instantiated by the bootloader instead
   * by the composite in which it is contained. The criteria is based on the
   * question whether the containing composite is flattened, as it is the case
   * for the system component and the interaction components for distribution.
   * 
   * @param part
   * @return
   */
  public static boolean instantiateViaBootloader(final StructuralFeature part) {
    if ((part != null)) {
      Type _type = part.getType();
      if ((_type instanceof org.eclipse.uml2.uml.Class)) {
        Type _type_1 = part.getType();
        final org.eclipse.uml2.uml.Class implementation = ((org.eclipse.uml2.uml.Class) _type_1);
        return AbstractCompToOO.instantiateViaBootloader(implementation);
      } else {
        return false;
      }
    }
    return false;
  }
  
  /**
   * Returns true if the port delegates to multiple parts/ports. Checks in depth.
   * @param port
   * @return port has multiple delegations in depth
   */
  public boolean hasMultipleDelegationsInDepth(final Port port) {
    final Element owner = port.getOwner();
    if ((owner instanceof org.eclipse.uml2.uml.Class)) {
      final List<ConnectorEnd> ces = ConnectorUtil.getDelegations(((org.eclipse.uml2.uml.Class) owner), port);
      int _size = ces.size();
      boolean _greaterThan = (_size > 1);
      if (_greaterThan) {
        return true;
      }
      int _size_1 = ces.size();
      boolean _equals = (_size_1 == 1);
      if (_equals) {
        final ConnectorEnd ce = ces.get(0);
        final ConnectableElement role = ce.getRole();
        if ((role instanceof Port)) {
          return this.hasMultipleDelegationsInDepth(((Port) role));
        }
      }
    }
    return false;
  }
}
