/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.timing.view;

import java.awt.Component;
import java.util.List;

import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;

import org.eclipse.core.runtime.Status;
import org.eclipse.lsat.timing.Activator;
import org.eclipse.lsat.timing.util.ChartCsvExporter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.trace4cps.common.jfreechart.ui.widgets.ChartPanelComposite;
import org.eclipse.ui.IViewReference;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.ViewPart;
import org.eclipse.ui.statushandlers.StatusManager;

public class XYPlotView extends ViewPart {

    private ChartPanelComposite chartComposite;
    private final ChartCsvExporter exporter = new ChartCsvExporter();

    /**
     * The ID of the view as specified by the extension.
     */
    public static final String ID = "org.eclipse.lsat.timing.view.XYPlotView";

    private static final String SUBMENU_SELECT_PLOT = "Select plot";
    private static final String SUBMENU_SAVE_AS = "Save as";
    private static final String CSV_MENU_ITEM_TEXT = "CSV...";

    public static void showPlots(final List<XYPlotData> plotData) {
        XYPlotView[] view = new XYPlotView[1];
        PlatformUI.getWorkbench().getDisplay().syncExec(() -> view[0] = getDefault());
        if (view[0] != null) {
            view[0].setJFreeChart(plotData);
        }
    }

    private static XYPlotView getDefault() {
        IWorkbench wb = PlatformUI.getWorkbench();
        IWorkbenchWindow awbw = wb.getActiveWorkbenchWindow();
        for (IWorkbenchWindow window: wb.getWorkbenchWindows()) {
            for (IWorkbenchPage page: window.getPages()) {
                for (IViewReference viewRef: page.getViewReferences()) {
                    if (ID.equals(viewRef.getId())) {
                        XYPlotView view = (XYPlotView)viewRef.getView(true);
                        awbw.getActivePage().activate(view);
                        return view;
                    }
                }
            }
        }
        try {
            return (XYPlotView)awbw.getActivePage().showView(ID, null, IWorkbenchPage.VIEW_ACTIVATE);
        } catch (PartInitException e) {
            StatusManager.getManager().handle(new Status(Status.ERROR, Activator.PLUGIN_ID, e.getMessage(), e));
            return null;
        }
    }

    /**
     * This is a callback that will allow us to create the viewer and initialize it.
     */
    @Override
    public void createPartControl(Composite parent) {
        chartComposite = new ChartPanelComposite(parent, SWT.H_SCROLL | SWT.V_SCROLL);
    }

    /**
     * Passing the focus request to the viewer's control.
     */
    @Override
    public void setFocus() {
        if (chartComposite != null) {
            chartComposite.setFocus();
        }
    }

    private void setJFreeChart(final List<XYPlotData> plotData) {
        if (null == chartComposite) {
            System.err.println("Not initialized yet!");
            return;
        }

        updatePopupMenu(plotData);

        chartComposite.setChart(plotData.get(0).getChart());
    }

    private void updatePopupMenu(List<XYPlotData> plotData) {
        JPopupMenu popup = chartComposite.getChartPanel().getPopupMenu();

        JMenu saveMenu = findSubmenu(popup, SUBMENU_SAVE_AS);
        if (saveMenu == null) {
            return;
        }

        JMenuItem csvItem = new JMenuItem(CSV_MENU_ITEM_TEXT);
        csvItem.addActionListener(e -> {
            exporter.exportChart(chartComposite.getChartPanel(), chartComposite.getXYPlot().getDatasets(), chartComposite.getChart().getTitle().getText());
        });

        removeSaveAsCSVMenu(saveMenu);
        saveMenu.add(csvItem);

        removeSelectPlotMenu(popup);
        JMenu submenu = buildPlotSelectionMenu(plotData);
        popup.addSeparator();
        popup.add(submenu);
    }

    private void removeSaveAsCSVMenu(JMenu menu) {
        for (int i = menu.getItemCount() - 1; i >= 0; i--) {
            JMenuItem item = menu.getItem(i);
            if (item != null && CSV_MENU_ITEM_TEXT.equals(item.getText())) {
                menu.remove(i);
            }
        }
    }

    private void removeSelectPlotMenu(JPopupMenu popup) {
        int count = popup.getComponentCount();
        if (count == 0) {
            return;
        }

        Component selectPlotComponent = popup.getComponent(count - 1);
        Component separatorComponent = popup.getComponent(count - 2);

        popup.remove(selectPlotComponent);
        popup.remove(separatorComponent);
    }

    private JMenu buildPlotSelectionMenu(List<XYPlotData> plotData) {
        JMenu submenu = new JMenu(SUBMENU_SELECT_PLOT);
        submenu.setName(SUBMENU_SELECT_PLOT);

        for (int i = 0; i < plotData.size(); i++) {
            final int index = i;
            JMenuItem item = new JMenuItem(plotData.get(index).getName());
            item.addActionListener(e ->
                    chartComposite.setChart(plotData.get(index).getChart()));
            submenu.add(item);
        }
        return submenu;
    }

    private JMenu findSubmenu(JPopupMenu popup, String text) {
        for (Component c : popup.getComponents()) {
            if (c instanceof JMenu m && text.equals(m.getText())) {
                return m;
            }
        }
        return null;
    }
}
