/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.timing.view;

import java.awt.BasicStroke;
import java.awt.Paint;
import java.awt.Stroke;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.lsat.activity.teditor.validation.ActivityValidator;
import org.eclipse.lsat.motioncalculator.MotionProfile;
import org.eclipse.lsat.motioncalculator.MotionProfileParameter;
import org.eclipse.lsat.motioncalculator.MotionSegment;
import org.eclipse.lsat.motioncalculator.PositionInfo;
import org.eclipse.lsat.motioncalculator.util.MotionSegmentUtilities;
import org.eclipse.lsat.motioncalculator.util.PositionInfoUtilities;
import org.eclipse.lsat.timing.Activator;
import org.eclipse.lsat.timing.calculator.MotionCalculatorExtension;
import org.eclipse.lsat.timing.util.MotionCalculatorHelper;
import org.eclipse.lsat.timing.util.MoveHelper;
import org.jfree.chart.ChartFactory;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.plot.ValueMarker;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.XYItemRenderer;
import org.jfree.chart.renderer.xy.XYLineAndShapeRenderer;
import org.jfree.chart.title.TextTitle;
import org.jfree.chart.ui.RectangleAnchor;
import org.jfree.data.xy.XYSeries;
import org.jfree.data.xy.XYSeriesCollection;

import activity.Move;
import setting.SettingUtil;
import setting.Settings;

public class MotionViewJob extends Job {
    public static final String SHOW_DEBUG_INFO = "org.eclipse.lsat.timing.view.debug";

    private final Move move;

    public MotionViewJob(Move move) {
        super("Plot " + ActivityValidator.id(move));
        this.move = move;
    }

    @Override
    protected IStatus run(IProgressMonitor monitor) {
        try {
            monitor.beginTask("Calculating plot info", IProgressMonitor.UNKNOWN);

            Settings settings = SettingUtil.getSettings(move.getPeripheral().eResource());
            MotionCalculatorExtension motionCalculator = MotionCalculatorExtension.getSelectedMotionCalculator();
            MotionCalculatorHelper util = new MotionCalculatorHelper(settings, motionCalculator);
            List<Move> concatenatedMove = util.getConcatenatedMove(move);
            List<MotionSegment> motionSegments = util.createMotionSegments(concatenatedMove);
            List<Double> motionTimes = motionCalculator.calculateTimes(motionSegments);
            Collection<PositionInfo> plotInfo = motionCalculator.getPositionInfo(motionSegments);
            Collection<String> setPoints = PositionInfoUtilities.getSetPointIds(plotInfo);
            MotionProfile motionProfile = MotionSegmentUtilities.getMotionProfiles(motionSegments).iterator().next();

            if (null == plotInfo || plotInfo.size() == 0 || monitor.isCanceled()) {
                return Status.CANCEL_STATUS;
            }

            monitor.subTask("Rendering plot");

            List<XYPlotData> motionPlots = new ArrayList<>();
            final int dataSize = Collections.max(plotInfo, (el1, el2) -> el1.getDataSize() - el2.getDataSize()).getDataSize();
            Iterator<MotionProfileParameter> motionProfileParameters = motionProfile.getParameters().iterator();
            for (int paramIndex = 1; motionProfileParameters.hasNext() && paramIndex < dataSize; paramIndex++) {
                MotionProfileParameter param = (paramIndex > 1) ? motionProfileParameters.next() : null;
                final String seriesName = (param != null) ? param.getName() : "Position";
                JFreeChart jfreechart = null;
                Iterator<String> iter = setPoints.iterator();
                for (int i = 0; iter.hasNext(); i++) {
                    final String setPoint = iter.next();
                    XYSeries setPointSeries = new XYSeries(seriesName + " " + setPoint);
                    PositionInfo posInfo = plotInfo.stream().filter(p -> setPoint.equals(p.getSetPointId())).findFirst()
                            .get();
                    posInfo.getData(paramIndex)
                            .forEach(e -> setPointSeries.add(e.getTime(), e.getPosition()));
                    XYSeriesCollection dataSet = new XYSeriesCollection(setPointSeries);

                    if (param != null) {
                        List<Double> motionMaxima = MotionSegmentUtilities.getMotionMaxima(motionSegments, setPoint,
                                param.getKey());
                        XYSeries minimumSeries = new XYSeries("Minimim " + seriesName.toLowerCase() + " " + setPoint);
                        minimumSeries.setDescription(seriesName);
                        XYSeries maximumSeries = new XYSeries("Maximum " + seriesName.toLowerCase() + " " + setPoint);
                        for (int n = 0; n < motionMaxima.size(); n++) {
                            double startTime = (n > 0) ? motionTimes.get(n - 1).doubleValue() : 0;
                            double endTime = motionTimes.get(n).doubleValue();
                            double maxValue = motionMaxima.get(n).doubleValue();
                            minimumSeries.add(startTime, -maxValue);
                            maximumSeries.add(startTime, maxValue);
                            minimumSeries.add(endTime, -maxValue);
                            maximumSeries.add(endTime, maxValue);
                        }
                        dataSet.addSeries(minimumSeries);
                        dataSet.addSeries(maximumSeries);
                    }

                    if (i == 0) {
                        jfreechart = ChartFactory.createXYLineChart(getTitle(concatenatedMove), "Time", setPoint,
                                dataSet, PlotOrientation.VERTICAL, true, true, false);
                        jfreechart.addSubtitle(new TextTitle(getSubTitle(concatenatedMove)));
                        updateRenderer(jfreechart.getXYPlot(), i);
                    } else {
                        XYPlot xyplot = jfreechart.getXYPlot();
                        NumberAxis numberaxis = new NumberAxis(setPoint);
                        numberaxis.setAutoRangeIncludesZero(false);
                        xyplot.setRangeAxis(i, numberaxis);
                        xyplot.setDataset(i, dataSet);
                        xyplot.mapDatasetToRangeAxis(i, i);
                        updateRenderer(xyplot, i);
                    }
                }

                XYPlot xyplot = jfreechart.getXYPlot();
                xyplot.getDataset(0).getGroup().toString();
                xyplot.setDomainPannable(true);
                xyplot.setRangePannable(true);

                for (int i = 0; i < motionTimes.size(); i++) {
                    // Assuming that a move arrives at it destination when all its setpoints arrived at their location.
                    // Thus time of the move is the maximum time it takes for its setpoint moves
                    ValueMarker marker = new ValueMarker(motionTimes.get(i));
                    Move move = concatenatedMove.get(i);
                    String text;
                    if (move.isPositionMove()) {
                        text = String.format("%s %s at %.4f", i < motionTimes.size() - 1 ? "Passing" : "At",
                                move.getTargetPosition().getName(), motionTimes.get(i));
                    } else {
                        text = String.format("Completed %s at %.4f", move.getDistance().getName(), motionTimes.get(i));
                    }
                    marker.setLabel(text);
                    marker.setLabelAnchor(RectangleAnchor.BOTTOM_LEFT);
                    xyplot.addDomainMarker(marker);
                }

                motionPlots.add(new XYPlotData(seriesName, jfreechart));
            }

            XYPlotView.showPlots(motionPlots);

            return Status.OK_STATUS;
        } catch (Exception e) {
            return new Status(Status.ERROR, Activator.PLUGIN_ID, e.getMessage(), e);
        } finally {
            monitor.done();
        }
    }

    private XYItemRenderer updateRenderer(XYPlot xyplot, int index) {
        XYLineAndShapeRenderer renderer = new XYLineAndShapeRenderer(true, false) {
            private static final long serialVersionUID = 6624778799391091328L;

            Stroke regularStroke = new BasicStroke();

            Stroke dashedStroke = new BasicStroke(1.0f, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER, 10.0f,
                    new float[]
                    {10.0f, 6.0f}, 0.0f);

            @Override
            public Stroke getItemStroke(int row, int column) {
                if (row == 0) {
                    return regularStroke;
                } else {
                    return dashedStroke;
                }
            }
        };

        // StandardXYItemRenderer renderer = new StandardXYItemRenderer();
        xyplot.setRenderer(index, renderer);
        final Paint paint = renderer.lookupSeriesPaint(0);
        xyplot.getRangeAxis(index).setLabelPaint(paint);
        xyplot.getRangeAxis(index).setTickLabelPaint(paint);
        return renderer;
    }

    private String getTitle(List<Move> concatenatedMove) {
        if (concatenatedMove.size() == 1) {
            return ActivityValidator.id(concatenatedMove.get(0));
        }
        return "Concatenated move: " + ActivityValidator.names(concatenatedMove);
    }

    private String getSubTitle(List<Move> concatenatedMove) {
        if (concatenatedMove.isEmpty()) {
            return "";
        }
        if (isPositionMove(concatenatedMove)) {
            return getPositionsSubTitle(concatenatedMove);
        } else {
            return getDistancesSubTitle(concatenatedMove);
        }
    }

    private boolean isPositionMove(List<Move> concatenatedMove) {
        return concatenatedMove.stream().anyMatch(e -> e.isPositionMove());
    }

    private String getDistancesSubTitle(List<Move> concatenatedMove) {
        StringBuffer subTitle = new StringBuffer();
        String currentName = MoveHelper.getName(concatenatedMove.get(0));
        int count = 1;
        for (int i = 1; i < concatenatedMove.size(); i++) {
            String name = MoveHelper.getName(concatenatedMove.get(i));
            if (name.equals(currentName)) {
                count++;
            } else {
                appendDistances(subTitle, currentName, count);
                currentName = name;
                count = 1;
            }
        }
        appendDistances(subTitle, currentName, count);
        return subTitle.toString();
    }

    private StringBuffer appendDistances(StringBuffer subTitle, String name, int count) {
        if (subTitle.length() > 0) {
            subTitle.append(" ► ");
        }
        subTitle.append(name);
        if (count > 1) {
            subTitle.append(" (").append(count).append("x)");
        }
        return subTitle;
    }

    private String getPositionsSubTitle(List<Move> concatenatedMove) {
        StringBuffer subTitle = new StringBuffer();
        for (int i = 0; i < concatenatedMove.size(); i++) {
            if (concatenatedMove.get(i).isPositionMove()) {
                Move move = concatenatedMove.get(i);
                if (i == 0) {
                    subTitle.append(move.getSourcePosition().getName());
                }
                subTitle.append(" ► ");
                subTitle.append(move.getTargetPosition().getName());
            }
        }
        return subTitle.toString();
    }
}
