/* --------------------------------------------------------------------------------------------
 * Copyright (c) Remy Suen. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 * ------------------------------------------------------------------------------------------ */
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerCommands = void 0;
var vscode_languageserver_textdocument_1 = require("vscode-languageserver-textdocument");
var vscode_languageserver_types_1 = require("vscode-languageserver-types");
var dockerfile_utils_1 = require("dockerfile-utils");
var main_1 = require("./main");
var DockerCommands = /** @class */ (function () {
    function DockerCommands() {
    }
    DockerCommands.prototype.analyzeDiagnostics = function (diagnostics, textDocumentURI) {
        var commands = [];
        for (var i = 0; i < diagnostics.length; i++) {
            // Diagnostic's code is (number | string), convert it if necessary
            if (typeof diagnostics[i].code === "string") {
                diagnostics[i].code = parseInt(diagnostics[i].code);
            }
            switch (diagnostics[i].code) {
                case dockerfile_utils_1.ValidationCode.CASING_DIRECTIVE:
                    commands.push({
                        title: "Convert directive to lowercase",
                        command: main_1.CommandIds.LOWERCASE,
                        arguments: [textDocumentURI, diagnostics[i].range]
                    });
                    break;
                case dockerfile_utils_1.ValidationCode.CASING_INSTRUCTION:
                    commands.push({
                        title: "Convert instruction to uppercase",
                        command: main_1.CommandIds.UPPERCASE,
                        arguments: [textDocumentURI, diagnostics[i].range]
                    });
                    break;
                case dockerfile_utils_1.ValidationCode.ARGUMENT_EXTRA:
                    commands.push({
                        title: "Remove extra argument",
                        command: main_1.CommandIds.EXTRA_ARGUMENT,
                        arguments: [textDocumentURI, diagnostics[i].range]
                    });
                    break;
                case dockerfile_utils_1.ValidationCode.INVALID_ESCAPE_DIRECTIVE:
                    commands.push({
                        title: "Convert to backslash",
                        command: main_1.CommandIds.DIRECTIVE_TO_BACKSLASH,
                        arguments: [textDocumentURI, diagnostics[i].range]
                    });
                    commands.push({
                        title: "Convert to backtick",
                        command: main_1.CommandIds.DIRECTIVE_TO_BACKTICK,
                        arguments: [textDocumentURI, diagnostics[i].range]
                    });
                    break;
                case dockerfile_utils_1.ValidationCode.INVALID_AS:
                    commands.push({
                        title: "Convert to AS",
                        command: main_1.CommandIds.CONVERT_TO_AS,
                        arguments: [textDocumentURI, diagnostics[i].range]
                    });
                    break;
                case dockerfile_utils_1.ValidationCode.UNKNOWN_HEALTHCHECK_FLAG:
                    commands.push({
                        title: "Convert to --interval",
                        command: main_1.CommandIds.FLAG_TO_HEALTHCHECK_INTERVAL,
                        arguments: [textDocumentURI, diagnostics[i].range]
                    });
                    commands.push({
                        title: "Convert to --retries",
                        command: main_1.CommandIds.FLAG_TO_HEALTHCHECK_RETRIES,
                        arguments: [textDocumentURI, diagnostics[i].range]
                    });
                    commands.push({
                        title: "Convert to --start-period",
                        command: main_1.CommandIds.FLAG_TO_HEALTHCHECK_START_PERIOD,
                        arguments: [textDocumentURI, diagnostics[i].range]
                    });
                    commands.push({
                        title: "Convert to --timeout",
                        command: main_1.CommandIds.FLAG_TO_HEALTHCHECK_TIMEOUT,
                        arguments: [textDocumentURI, diagnostics[i].range]
                    });
                    break;
                case dockerfile_utils_1.ValidationCode.UNKNOWN_ADD_FLAG:
                    commands.push({
                        title: "Convert to --chown",
                        command: main_1.CommandIds.FLAG_TO_CHOWN,
                        arguments: [textDocumentURI, diagnostics[i].range]
                    });
                    break;
                case dockerfile_utils_1.ValidationCode.UNKNOWN_COPY_FLAG:
                    commands.push({
                        title: "Convert to --chown",
                        command: main_1.CommandIds.FLAG_TO_CHOWN,
                        arguments: [textDocumentURI, diagnostics[i].range]
                    });
                    commands.push({
                        title: "Convert to --from",
                        command: main_1.CommandIds.FLAG_TO_COPY_FROM,
                        arguments: [textDocumentURI, diagnostics[i].range]
                    });
                    break;
                case dockerfile_utils_1.ValidationCode.EMPTY_CONTINUATION_LINE:
                    if (diagnostics[i].range.start.line + 1 === diagnostics[i].range.end.line) {
                        commands.push({
                            title: "Remove empty continuation line",
                            command: main_1.CommandIds.REMOVE_EMPTY_CONTINUATION_LINE,
                            arguments: [textDocumentURI, diagnostics[i].range]
                        });
                    }
                    else {
                        commands.push({
                            title: "Remove empty continuation lines",
                            command: main_1.CommandIds.REMOVE_EMPTY_CONTINUATION_LINE,
                            arguments: [textDocumentURI, diagnostics[i].range]
                        });
                    }
                    break;
            }
        }
        return commands;
    };
    DockerCommands.prototype.computeCommandEdits = function (content, command, args) {
        var document = vscode_languageserver_textdocument_1.TextDocument.create("", "", 0, content);
        var range = args[1];
        switch (command) {
            case main_1.CommandIds.LOWERCASE:
                var directive = document.getText().substring(document.offsetAt(range.start), document.offsetAt(range.end));
                return [
                    vscode_languageserver_types_1.TextEdit.replace(range, directive.toLowerCase())
                ];
            case main_1.CommandIds.UPPERCASE:
                var instruction = document.getText().substring(document.offsetAt(range.start), document.offsetAt(range.end));
                return [
                    vscode_languageserver_types_1.TextEdit.replace(range, instruction.toUpperCase())
                ];
            case main_1.CommandIds.EXTRA_ARGUMENT:
                return [
                    vscode_languageserver_types_1.TextEdit.del(range)
                ];
            case main_1.CommandIds.DIRECTIVE_TO_BACKSLASH:
                return [
                    vscode_languageserver_types_1.TextEdit.replace(range, '\\')
                ];
            case main_1.CommandIds.DIRECTIVE_TO_BACKTICK:
                return [
                    vscode_languageserver_types_1.TextEdit.replace(range, '`')
                ];
            case main_1.CommandIds.CONVERT_TO_AS:
                return [
                    vscode_languageserver_types_1.TextEdit.replace(range, "AS")
                ];
            case main_1.CommandIds.FLAG_TO_CHOWN:
                return [
                    vscode_languageserver_types_1.TextEdit.replace(range, "--chown")
                ];
            case main_1.CommandIds.FLAG_TO_HEALTHCHECK_INTERVAL:
                return [
                    vscode_languageserver_types_1.TextEdit.replace(range, "--interval")
                ];
            case main_1.CommandIds.FLAG_TO_HEALTHCHECK_RETRIES:
                return [
                    vscode_languageserver_types_1.TextEdit.replace(range, "--retries")
                ];
            case main_1.CommandIds.FLAG_TO_HEALTHCHECK_START_PERIOD:
                return [
                    vscode_languageserver_types_1.TextEdit.replace(range, "--start-period")
                ];
            case main_1.CommandIds.FLAG_TO_HEALTHCHECK_TIMEOUT:
                return [
                    vscode_languageserver_types_1.TextEdit.replace(range, "--timeout")
                ];
            case main_1.CommandIds.FLAG_TO_COPY_FROM:
                return [
                    vscode_languageserver_types_1.TextEdit.replace(range, "--from")
                ];
            case main_1.CommandIds.REMOVE_EMPTY_CONTINUATION_LINE:
                return [
                    vscode_languageserver_types_1.TextEdit.del(range)
                ];
        }
        return null;
    };
    return DockerCommands;
}());
exports.DockerCommands = DockerCommands;
