/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License (the "License").
 * You may not use this file except in compliance with the License.
 *
 * You can obtain a copy of the license at usr/src/OPENSOLARIS.LICENSE
 * or http://www.opensolaris.org/os/licensing.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at usr/src/OPENSOLARIS.LICENSE.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information: Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 */

/*
 * Copyright 2005 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

#pragma ident	"@(#)mlib_v_SignalFFT_Tbl.c	9.2	07/11/05 SMI"

#include <vis_proto.h>
#include <mlib_signal.h>

/* *********************************************************** */

#include <mlib_SignalFFTUtil.h>

/* *********************************************************** */

#define	_SIGNAL_FFT_TABLE

#include <mlib_v_SignalFFT_Tbl.h>
#include <mlib_v_SignalFFTDisp.h>

/* *********************************************************** */

static const mlib_d64 mlib_v_fftK_SinCosTable[] = {
	0.0,
	1.0,
	0.707106781186547573,
	0.707106781186547462,
	0.923879532511286738,
	0.382683432365089782,
	0.980785280403230431,
	0.195090322016128248,
	0.995184726672196929,
	0.098017140329560604,
	0.998795456205172405,
	0.049067674327418015,
	0.999698818696204250,
	0.024541228522912288,
	0.999924701839144503,
	0.012271538285719925,
	0.999981175282601109,
	0.006135884649154475,
	0.999995293809576191,
	0.003067956762965976,
	0.999998823451701879,
	0.001533980186284766,
	0.999999705862882227,
	0.000766990318742704,
	0.999999926465717892,
	0.000383495187571396,
	0.999999981616429334,
	0.000191747597310703,
	0.999999995404107334,
	0.000095873799095977,
	0.999999998851026861,
	0.000047936899603067,
	0.999999999712756660,
	0.000023968449808418,
	0.999999999928189220,
	0.000011984224905070,
	0.999999999982047250,
	0.000005992112452642,
	0.999999999995511812,
	0.000002996056226335,
	0.999999999998878009,
	0.000001498028113169,
	0.999999999999719447,
	0.000000749014056585,
	0.999999999999929834,
	0.000000374507028292,
	0.999999999999982458,
	0.000000187253514146,
	0.999999999999995670,
	0.000000093626757073,
	0.999999999999998890,
	0.000000046813378537,
	0.999999999999999778,
	0.000000023406689268,
	0.999999999999999889,
	0.000000011703344634,
	1.000000000000000000,
	0.000000005851672317,
	1.000000000000000000,
	0.000000002925836159
};

/* *********************************************************** */

typedef union
{
	mlib_d64 value;
	struct
	{
		mlib_s32 int0, int1;
	} twoint;
} type_mlib_d64;

/* *********************************************************** */

#define	D64_TO_S32(val)	((mlib_s32)(val))

/* *********************************************************** */

void
mlib_v_fftK_CalcTbl(
    void *buff,
    mlib_s32 order,
    mlib_s32 isign)
{
	mlib_s32 layer = order;
	mlib_d64 *tbl = (mlib_d64 *)buff + (1 << (layer - 2));
	mlib_d64 wp_r, wp_i, w_r, w_i, dw0_r, dw1_r, dw2_r;
	mlib_d64 dw3_r, dw0_i, dw1_i, dw2_i, dw3_i;
	mlib_s32 j, p;

	vis_write_gsr(0);

	p = (order - 2) << 1;

	dw1_r = mlib_v_fftK_SinCosTable[p] * 2147418112.0;
	dw1_i = isign * mlib_v_fftK_SinCosTable[p + 1] * 2147418112.0;

	dw2_r = mlib_v_fftK_SinCosTable[p - 2];
	dw2_i = isign * mlib_v_fftK_SinCosTable[p - 1];

	dw3_r = (dw2_r * dw1_r) - (dw2_i * dw1_i);
	dw3_i = (dw2_i * dw1_r) + (dw2_r * dw1_i);

	dw2_r *= 2147418112.0;
	dw2_i *= 2147418112.0;

	dw0_r = 2147418112.0;
	dw0_i = 0;

	wp_r = mlib_v_fftK_SinCosTable[p - 4];
	wp_i = isign * mlib_v_fftK_SinCosTable[p - 3];
	{
		type_mlib_d64 str0, str1;

		str0.twoint.int0 = D64_TO_S32(dw0_r);
		str0.twoint.int1 = D64_TO_S32(dw1_r);
		str1.twoint.int0 = D64_TO_S32(dw2_r);
		str1.twoint.int1 = D64_TO_S32(dw3_r);
		w_r = vis_fpackfix_pair(str0.value, str1.value);
		str0.twoint.int0 = D64_TO_S32(dw0_i);
		str0.twoint.int1 = D64_TO_S32(dw1_i);
		str1.twoint.int0 = D64_TO_S32(dw2_i);
		str1.twoint.int1 = D64_TO_S32(dw3_i);
		w_i = vis_fpackfix_pair(str0.value, str1.value);
	}

	for (j = 0; j < (1 << (order - 3)); j++) {
		mlib_d64 tmp0_r, tmp1_r, tmp2_r, tmp3_r;
		type_mlib_d64 str0, str1;

		tmp0_r = dw0_r;
		dw0_r = (dw0_r * wp_r) - (dw0_i * wp_i);
		tmp1_r = dw1_r;
		dw1_r = (dw1_r * wp_r) - (dw1_i * wp_i);
		tmp2_r = dw2_r;
		dw2_r = (dw2_r * wp_r) - (dw2_i * wp_i);
		tmp3_r = dw3_r;
		dw3_r = (dw3_r * wp_r) - (dw3_i * wp_i);

		tbl[2 * j] = w_r;
		tbl[2 * j + 1] = w_i;

		str0.twoint.int0 = D64_TO_S32(dw0_r);
		str0.twoint.int1 = D64_TO_S32(dw1_r);
		str1.twoint.int0 = D64_TO_S32(dw2_r);
		str1.twoint.int1 = D64_TO_S32(dw3_r);
		w_r = vis_fpackfix_pair(str0.value, str1.value);

		dw0_i = (dw0_i * wp_r) + (tmp0_r * wp_i);
		dw1_i = (dw1_i * wp_r) + (tmp1_r * wp_i);
		dw2_i = (dw2_i * wp_r) + (tmp2_r * wp_i);
		dw3_i = (dw3_i * wp_r) + (tmp3_r * wp_i);

		str0.twoint.int0 = D64_TO_S32(dw0_i);
		str0.twoint.int1 = D64_TO_S32(dw1_i);
		str1.twoint.int0 = D64_TO_S32(dw2_i);
		str1.twoint.int1 = D64_TO_S32(dw3_i);
		w_i = vis_fpackfix_pair(str0.value, str1.value);
	}

	for (layer = order - 1; layer > FFT_VIS_TBL_ORDER; layer--) {
		mlib_d64 *tbl1 = (mlib_d64 *)buff + (1 << (layer - 2));
		mlib_s32 tscale = 4 + (order - layer);
		mlib_s32 t_off = 4 << (order - layer);
		mlib_s32 j, off1, off2, off3;

		if ((order - layer) > 1) {
			off1 = t_off;
			off2 = 2 * t_off;
		} else {
			off1 = 4;
			off2 = 16;
		}

		off3 = off1 + off2;

		vis_write_gsr(6);

#pragma pipeloop(0)
		for (j = 0; j < (1 << (layer - 3)); j++) {
			mlib_u8 *ptbl;

			ptbl = (mlib_u8 *)tbl + (j << tscale);

			w_r = vis_faligndata(vis_ld_u16_i(ptbl, off3), w_r);
			w_i = vis_faligndata(vis_ld_u16_i(ptbl, off3 + 8), w_i);
			w_r = vis_faligndata(vis_ld_u16_i(ptbl, off2), w_r);
			w_i = vis_faligndata(vis_ld_u16_i(ptbl, off3 + 8), w_i);
			w_r = vis_faligndata(vis_ld_u16_i(ptbl, off1), w_r);
			w_i = vis_faligndata(vis_ld_u16_i(ptbl, off3 + 8), w_i);
			w_r = vis_faligndata(vis_ld_u16_i(ptbl, 0), w_r);
			w_i = vis_faligndata(vis_ld_u16_i(ptbl, 8), w_i);

			tbl1[2 * j] = w_r;
			tbl1[2 * j + 1] = w_i;
		}
	}
}

/* *********************************************************** */

const mlib_u32 mlib_v_fftK_WTable_1_M[] = {
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x7fff5a81, 0x0000a57e, 0x0000a57e, 0x8001a57e,
	0x7fff7640, 0x5a8130fb, 0x0000cf04, 0xa57e89bf,
	0x0000cf04, 0xa57e89bf, 0x800189bf, 0xa57ecf04,
	0x7fff7d89, 0x76406a6c, 0x0000e707, 0xcf04b8e3,
	0x5a81471c, 0x30fb18f8, 0xa57e9593, 0x89bf8276,
	0x0000e707, 0xcf04b8e3, 0x80018276, 0x89bf9593,
	0xa57e9593, 0x89bf8276, 0xa57eb8e3, 0xcf04e707,
	0x7fff7f61, 0x7d897a7c, 0x0000f374, 0xe707dad8,
	0x764070e1, 0x6a6c62f1, 0xcf04c3a9, 0xb8e3aecc,
	0x5a815133, 0x471c3c56, 0xa57e9d0e, 0x95938f1e,
	0x30fb2527, 0x18f80c8b, 0x89bf8583, 0x8276809e,
	0x0000f374, 0xe707dad8, 0x8001809e, 0x82768583,
	0xcf04c3a9, 0xb8e3aecc, 0x89bf8f1e, 0x95939d0e,
	0xa57e9d0e, 0x95938f1e, 0xa57eaecc, 0xb8e3c3a9,
	0x89bf8583, 0x8276809e, 0xcf04dad8, 0xe707f374,
	0x7fff7fd7, 0x7f617e9c, 0x0000f9b8, 0xf374ed38,
	0x7d897c29, 0x7a7c7883, 0xe707e0e6, 0xdad8d4e1,
	0x764073b5, 0x70e16dc9, 0xcf04c946, 0xc3a9be32,
	0x6a6c66ce, 0x62f15ed6, 0xb8e3b3c0, 0xaeccaa0b,
	0x5a8155f4, 0x51334c3f, 0xa57ea129, 0x9d0e9931,
	0x471c41cd, 0x3c5636b9, 0x95939236, 0x8f1e8c4a,
	0x30fb2b1e, 0x25271f19, 0x89bf877c, 0x858383d6,
	0x18f812c7, 0x0c8b0647, 0x82768163, 0x809e8028,
	0x0000f9b8, 0xf374ed38, 0x80018028, 0x809e8163,
	0xe707e0e6, 0xdad8d4e1, 0x827683d6, 0x8583877c,
	0xcf04c946, 0xc3a9be32, 0x89bf8c4a, 0x8f1e9236,
	0xb8e3b3c0, 0xaeccaa0b, 0x95939931, 0x9d0ea129,
	0xa57ea129, 0x9d0e9931, 0xa57eaa0b, 0xaeccb3c0,
	0x95939236, 0x8f1e8c4a, 0xb8e3be32, 0xc3a9c946,
	0x89bf877c, 0x858383d6, 0xcf04d4e1, 0xdad8e0e6,
	0x82768163, 0x809e8028, 0xe707ed38, 0xf374f9b8,
	0x7fff7ff5, 0x7fd77fa6, 0x0000fcdb, 0xf9b8f695,
	0x7f617f08, 0x7e9c7e1c, 0xf374f054, 0xed38ea1e,
	0x7d897ce2, 0x7c297b5c, 0xe707e3f4, 0xe0e6dddc,
	0x7a7c7989, 0x7883776b, 0xdad8d7d9, 0xd4e1d1ef,
	0x76407503, 0x73b57254, 0xcf04cc21, 0xc946c673,
	0x70e16f5e, 0x6dc96c23, 0xc3a9c0e9, 0xbe32bb85,
	0x6a6c68a5, 0x66ce64e7, 0xb8e3b64c, 0xb3c0b140,
	0x62f160eb, 0x5ed65cb3, 0xaeccac65, 0xaa0ba7bd,
	0x5a815842, 0x55f4539a, 0xa57ea34c, 0xa1299f14,
	0x51334ebf, 0x4c3f49b3, 0x9d0e9b18, 0x9931975a,
	0x471c447a, 0x41cd3f16, 0x959393dc, 0x923690a1,
	0x3c56398c, 0x36b933de, 0x8f1e8dab, 0x8c4a8afc,
	0x30fb2e10, 0x2b1e2826, 0x89bf8894, 0x877c8676,
	0x25272223, 0x1f191c0b, 0x858384a3, 0x83d6831d,
	0x18f815e1, 0x12c70fab, 0x827681e3, 0x816380f7,
	0x0c8b096a, 0x06470324, 0x809e8059, 0x8028800a,
	0x0000fcdb, 0xf9b8f695, 0x8001800a, 0x80288059,
	0xf374f054, 0xed38ea1e, 0x809e80f7, 0x816381e3,
	0xe707e3f4, 0xe0e6dddc, 0x8276831d, 0x83d684a3,
	0xdad8d7d9, 0xd4e1d1ef, 0x85838676, 0x877c8894,
	0xcf04cc21, 0xc946c673, 0x89bf8afc, 0x8c4a8dab,
	0xc3a9c0e9, 0xbe32bb85, 0x8f1e90a1, 0x923693dc,
	0xb8e3b64c, 0xb3c0b140, 0x9593975a, 0x99319b18,
	0xaeccac65, 0xaa0ba7bd, 0x9d0e9f14, 0xa129a34c,
	0xa57ea34c, 0xa1299f14, 0xa57ea7bd, 0xaa0bac65,
	0x9d0e9b18, 0x9931975a, 0xaeccb140, 0xb3c0b64c,
	0x959393dc, 0x923690a1, 0xb8e3bb85, 0xbe32c0e9,
	0x8f1e8dab, 0x8c4a8afc, 0xc3a9c673, 0xc946cc21,
	0x89bf8894, 0x877c8676, 0xcf04d1ef, 0xd4e1d7d9,
	0x858384a3, 0x83d6831d, 0xdad8dddc, 0xe0e6e3f4,
	0x827681e3, 0x816380f7, 0xe707ea1e, 0xed38f054,
	0x809e8059, 0x8028800a, 0xf374f695, 0xf9b8fcdb,
	0x7fff7ffc, 0x7ff57fe8, 0x0000fe6d, 0xfcdbfb49,
	0x7fd77fc1, 0x7fa67f86, 0xf9b8f826, 0xf695f504,
	0x7f617f37, 0x7f087ed4, 0xf374f1e4, 0xf054eec6,
	0x7e9c7e5e, 0x7e1c7dd5, 0xed38ebaa, 0xea1ee892,
	0x7d897d38, 0x7ce27c88, 0xe707e57d, 0xe3f4e26c,
	0x7c297bc4, 0x7b5c7aee, 0xe0e6df60, 0xdddcdc59,
	0x7a7c7a04, 0x79897908, 0xdad8d958, 0xd7d9d65c,
	0x788377f9, 0x776b76d8, 0xd4e1d367, 0xd1efd079,
	0x764075a4, 0x7503745e, 0xcf04cd92, 0xcc21cab2,
	0x73b57306, 0x7254719d, 0xc946c7db, 0xc673c50d,
	0x70e17022, 0x6f5e6e95, 0xc3a9c248, 0xc0e9bf8c,
	0x6dc96cf8, 0x6c236b4a, 0xbe32bcda, 0xbb85ba33,
	0x6a6c698b, 0x68a567bc, 0xb8e3b796, 0xb64cb505,
	0x66ce65dd, 0x64e763ee, 0xb3c0b27f, 0xb140b005,
	0x62f161f0, 0x60eb5fe2, 0xaeccad97, 0xac65ab36,
	0x5ed65dc6, 0x5cb35b9c, 0xaa0ba8e2, 0xa7bda69c,
	0x5a815963, 0x5842571d, 0xa57ea463, 0xa34ca239,
	0x55f454c9, 0x539a5268, 0xa129a01d, 0x9f149e0f,
	0x51334ffa, 0x4ebf4d80, 0x9d0e9c11, 0x9b189a22,
	0x4c3f4afa, 0x49b34869, 0x99319843, 0x975a9674,
	0x471c45cc, 0x447a4325, 0x959394b5, 0x93dc9307,
	0x41cd4073, 0x3f163db7, 0x9236916a, 0x90a18fdd,
	0x3c563af2, 0x398c3824, 0x8f1e8e62, 0x8dab8cf9,
	0x36b9354d, 0x33de326d, 0x8c4a8ba1, 0x8afc8a5b,
	0x30fb2f86, 0x2e102c98, 0x89bf8927, 0x88948806,
	0x2b1e29a3, 0x282626a7, 0x877c86f7, 0x867685fb,
	0x252723a6, 0x2223209f, 0x85838511, 0x84a3843b,
	0x1f191d93, 0x1c0b1a82, 0x83d68377, 0x831d82c7,
	0x18f8176d, 0x15e11455, 0x8276822a, 0x81e381a1,
	0x12c71139, 0x0fab0e1b, 0x8163812b, 0x80f780c8,
	0x0c8b0afb, 0x096a07d9, 0x809e8079, 0x8059803e,
	0x064704b6, 0x03240192, 0x80288017, 0x800a8003,
	0x0000fe6d, 0xfcdbfb49, 0x80018003, 0x800a8017,
	0xf9b8f826, 0xf695f504, 0x8028803e, 0x80598079,
	0xf374f1e4, 0xf054eec6, 0x809e80c8, 0x80f7812b,
	0xed38ebaa, 0xea1ee892, 0x816381a1, 0x81e3822a,
	0xe707e57d, 0xe3f4e26c, 0x827682c7, 0x831d8377,
	0xe0e6df60, 0xdddcdc59, 0x83d6843b, 0x84a38511,
	0xdad8d958, 0xd7d9d65c, 0x858385fb, 0x867686f7,
	0xd4e1d367, 0xd1efd079, 0x877c8806, 0x88948927,
	0xcf04cd92, 0xcc21cab2, 0x89bf8a5b, 0x8afc8ba1,
	0xc946c7db, 0xc673c50d, 0x8c4a8cf9, 0x8dab8e62,
	0xc3a9c248, 0xc0e9bf8c, 0x8f1e8fdd, 0x90a1916a,
	0xbe32bcda, 0xbb85ba33, 0x92369307, 0x93dc94b5,
	0xb8e3b796, 0xb64cb505, 0x95939674, 0x975a9843,
	0xb3c0b27f, 0xb140b005, 0x99319a22, 0x9b189c11,
	0xaeccad97, 0xac65ab36, 0x9d0e9e0f, 0x9f14a01d,
	0xaa0ba8e2, 0xa7bda69c, 0xa129a239, 0xa34ca463,
	0xa57ea463, 0xa34ca239, 0xa57ea69c, 0xa7bda8e2,
	0xa129a01d, 0x9f149e0f, 0xaa0bab36, 0xac65ad97,
	0x9d0e9c11, 0x9b189a22, 0xaeccb005, 0xb140b27f,
	0x99319843, 0x975a9674, 0xb3c0b505, 0xb64cb796,
	0x959394b5, 0x93dc9307, 0xb8e3ba33, 0xbb85bcda,
	0x9236916a, 0x90a18fdd, 0xbe32bf8c, 0xc0e9c248,
	0x8f1e8e62, 0x8dab8cf9, 0xc3a9c50d, 0xc673c7db,
	0x8c4a8ba1, 0x8afc8a5b, 0xc946cab2, 0xcc21cd92,
	0x89bf8927, 0x88948806, 0xcf04d079, 0xd1efd367,
	0x877c86f7, 0x867685fb, 0xd4e1d65c, 0xd7d9d958,
	0x85838511, 0x84a3843b, 0xdad8dc59, 0xdddcdf60,
	0x83d68377, 0x831d82c7, 0xe0e6e26c, 0xe3f4e57d,
	0x8276822a, 0x81e381a1, 0xe707e892, 0xea1eebaa,
	0x8163812b, 0x80f780c8, 0xed38eec6, 0xf054f1e4,
	0x809e8079, 0x8059803e, 0xf374f504, 0xf695f826,
	0x80288017, 0x800a8003, 0xf9b8fb49, 0xfcdbfe6d,
	0x7fff7ffe, 0x7ffc7ff9, 0x0000ff36, 0xfe6dfda4,
	0x7ff57fef, 0x7fe87fe0, 0xfcdbfc12, 0xfb49fa81,
	0x7fd77fcd, 0x7fc17fb4, 0xf9b8f8ef, 0xf826f75e,
	0x7fa67f96, 0x7f867f74, 0xf695f5cd, 0xf504f43c,
	0x7f617f4c, 0x7f377f20, 0xf374f2ac, 0xf1e4f11c,
	0x7f087eef, 0x7ed47eb9, 0xf054ef8d, 0xeec6edff,
	0x7e9c7e7e, 0x7e5e7e3e, 0xed38ec71, 0xebaaeae4,
	0x7e1c7df9, 0x7dd57db0, 0xea1ee958, 0xe892e7cc,
	0x7d897d61, 0x7d387d0e, 0xe707e642, 0xe57de4b9,
	0x7ce27cb6, 0x7c887c59, 0xe3f4e330, 0xe26ce1a9,
	0x7c297bf7, 0x7bc47b91, 0xe0e6e023, 0xdf60de9e,
	0x7b5c7b25, 0x7aee7ab5, 0xdddcdd1b, 0xdc59db98,
	0x7a7c7a41, 0x7a0479c7, 0xdad8da18, 0xd958d898,
	0x79897949, 0x790878c6, 0xd7d9d71a, 0xd65cd59e,
	0x7883783f, 0x77f977b3, 0xd4e1d424, 0xd367d2ab,
	0x776b7722, 0x76d8768d, 0xd1efd133, 0xd079cfbe,
	0x764075f3, 0x75a47554, 0xcf04ce4b, 0xcd92ccd9,
	0x750374b1, 0x745e740a, 0xcc21cb69, 0xcab2c9fc,
	0x73b5735e, 0x730672ae, 0xc946c890, 0xc7dbc727,
	0x725471f9, 0x719d7140, 0xc673c5c0, 0xc50dc45b,
	0x70e17082, 0x70226fc0, 0xc3a9c2f8, 0xc248c198,
	0x6f5e6efa, 0x6e956e30, 0xc0e9c03a, 0xbf8cbedf,
	0x6dc96d61, 0x6cf86c8e, 0xbe32bd86, 0xbcdabc2f,
	0x6c236bb7, 0x6b4a6adb, 0xbb85badc, 0xba33b98b,
	0x6a6c69fc, 0x698b6919, 0xb8e3b83c, 0xb796b6f1,
	0x68a56831, 0x67bc6745, 0xb64cb5a8, 0xb505b462,
	0x66ce6656, 0x65dd6562, 0xb3c0b31f, 0xb27fb1df,
	0x64e7646b, 0x63ee6370, 0xb140b0a2, 0xb005af68,
	0x62f16271, 0x61f0616e, 0xaeccae31, 0xad97acfe,
	0x60eb6067, 0x5fe25f5d, 0xac65abcd, 0xab36aaa0,
	0x5ed65e4f, 0x5dc65d3d, 0xaa0ba976, 0xa8e2a84f,
	0x5cb35c28, 0x5b9c5b0f, 0xa7bda72c, 0xa69ca60c,
	0x5a8159f3, 0x596358d3, 0xa57ea4f0, 0xa463a3d7,
	0x584257b0, 0x571d5689, 0xa34ca2c2, 0xa239a1b0,
	0x55f4555f, 0x54c95432, 0xa129a0a2, 0xa01d9f98,
	0x539a5301, 0x526851ce, 0x9f149e91, 0x9e0f9d8e,
	0x51335097, 0x4ffa4f5d, 0x9d0e9c8f, 0x9c119b94,
	0x4ebf4e20, 0x4d804ce0, 0x9b189a9d, 0x9a2299a9,
	0x4c3f4b9d, 0x4afa4a57, 0x993198ba, 0x984397ce,
	0x49b3490e, 0x486947c3, 0x975a96e6, 0x96749603,
	0x471c4674, 0x45cc4523, 0x95939524, 0x94b59448,
	0x447a43d0, 0x43254279, 0x93dc9371, 0x9307929e,
	0x41cd4120, 0x40733fc5, 0x923691cf, 0x916a9105,
	0x3f163e67, 0x3db73d07, 0x90a1903f, 0x8fdd8f7d,
	0x3c563ba4, 0x3af23a3f, 0x8f1e8ebf, 0x8e628e06,
	0x398c38d8, 0x3824376f, 0x8dab8d51, 0x8cf98ca1,
	0x36b93603, 0x354d3496, 0x8c4a8bf5, 0x8ba18b4e,
	0x33de3326, 0x326d31b4, 0x8afc8aab, 0x8a5b8a0c,
	0x30fb3041, 0x2f862ecc, 0x89bf8972, 0x892788dd,
	0x2e102d54, 0x2c982bdb, 0x8894884c, 0x880687c0,
	0x2b1e2a61, 0x29a328e5, 0x877c8739, 0x86f786b6,
	0x28262767, 0x26a725e7, 0x86768638, 0x85fb85be,
	0x25272467, 0x23a622e4, 0x8583854a, 0x851184da,
	0x22232161, 0x209f1fdc, 0x84a3846e, 0x843b8408,
	0x1f191e56, 0x1d931ccf, 0x83d683a6, 0x83778349,
	0x1c0b1b46, 0x1a8219bd, 0x831d82f1, 0x82c7829e,
	0x18f81833, 0x176d16a7, 0x8276824f, 0x822a8206,
	0x15e1151b, 0x1455138e, 0x81e381c1, 0x81a18181,
	0x12c71200, 0x11391072, 0x81638146, 0x812b8110,
	0x0fab0ee3, 0x0e1b0d53, 0x80f780df, 0x80c880b3,
	0x0c8b0bc3, 0x0afb0a32, 0x809e808b, 0x80798069,
	0x096a08a1, 0x07d90710, 0x8059804b, 0x803e8032,
	0x0647057e, 0x04b603ed, 0x8028801f, 0x80178010,
	0x0324025b, 0x019200c9, 0x800a8006, 0x80038001,
	0x0000ff36, 0xfe6dfda4, 0x80018001, 0x80038006,
	0xfcdbfc12, 0xfb49fa81, 0x800a8010, 0x8017801f,
	0xf9b8f8ef, 0xf826f75e, 0x80288032, 0x803e804b,
	0xf695f5cd, 0xf504f43c, 0x80598069, 0x8079808b,
	0xf374f2ac, 0xf1e4f11c, 0x809e80b3, 0x80c880df,
	0xf054ef8d, 0xeec6edff, 0x80f78110, 0x812b8146,
	0xed38ec71, 0xebaaeae4, 0x81638181, 0x81a181c1,
	0xea1ee958, 0xe892e7cc, 0x81e38206, 0x822a824f,
	0xe707e642, 0xe57de4b9, 0x8276829e, 0x82c782f1,
	0xe3f4e330, 0xe26ce1a9, 0x831d8349, 0x837783a6,
	0xe0e6e023, 0xdf60de9e, 0x83d68408, 0x843b846e,
	0xdddcdd1b, 0xdc59db98, 0x84a384da, 0x8511854a,
	0xdad8da18, 0xd958d898, 0x858385be, 0x85fb8638,
	0xd7d9d71a, 0xd65cd59e, 0x867686b6, 0x86f78739,
	0xd4e1d424, 0xd367d2ab, 0x877c87c0, 0x8806884c,
	0xd1efd133, 0xd079cfbe, 0x889488dd, 0x89278972,
	0xcf04ce4b, 0xcd92ccd9, 0x89bf8a0c, 0x8a5b8aab,
	0xcc21cb69, 0xcab2c9fc, 0x8afc8b4e, 0x8ba18bf5,
	0xc946c890, 0xc7dbc727, 0x8c4a8ca1, 0x8cf98d51,
	0xc673c5c0, 0xc50dc45b, 0x8dab8e06, 0x8e628ebf,
	0xc3a9c2f8, 0xc248c198, 0x8f1e8f7d, 0x8fdd903f,
	0xc0e9c03a, 0xbf8cbedf, 0x90a19105, 0x916a91cf,
	0xbe32bd86, 0xbcdabc2f, 0x9236929e, 0x93079371,
	0xbb85badc, 0xba33b98b, 0x93dc9448, 0x94b59524,
	0xb8e3b83c, 0xb796b6f1, 0x95939603, 0x967496e6,
	0xb64cb5a8, 0xb505b462, 0x975a97ce, 0x984398ba,
	0xb3c0b31f, 0xb27fb1df, 0x993199a9, 0x9a229a9d,
	0xb140b0a2, 0xb005af68, 0x9b189b94, 0x9c119c8f,
	0xaeccae31, 0xad97acfe, 0x9d0e9d8e, 0x9e0f9e91,
	0xac65abcd, 0xab36aaa0, 0x9f149f98, 0xa01da0a2,
	0xaa0ba976, 0xa8e2a84f, 0xa129a1b0, 0xa239a2c2,
	0xa7bda72c, 0xa69ca60c, 0xa34ca3d7, 0xa463a4f0,
	0xa57ea4f0, 0xa463a3d7, 0xa57ea60c, 0xa69ca72c,
	0xa34ca2c2, 0xa239a1b0, 0xa7bda84f, 0xa8e2a976,
	0xa129a0a2, 0xa01d9f98, 0xaa0baaa0, 0xab36abcd,
	0x9f149e91, 0x9e0f9d8e, 0xac65acfe, 0xad97ae31,
	0x9d0e9c8f, 0x9c119b94, 0xaeccaf68, 0xb005b0a2,
	0x9b189a9d, 0x9a2299a9, 0xb140b1df, 0xb27fb31f,
	0x993198ba, 0x984397ce, 0xb3c0b462, 0xb505b5a8,
	0x975a96e6, 0x96749603, 0xb64cb6f1, 0xb796b83c,
	0x95939524, 0x94b59448, 0xb8e3b98b, 0xba33badc,
	0x93dc9371, 0x9307929e, 0xbb85bc2f, 0xbcdabd86,
	0x923691cf, 0x916a9105, 0xbe32bedf, 0xbf8cc03a,
	0x90a1903f, 0x8fdd8f7d, 0xc0e9c198, 0xc248c2f8,
	0x8f1e8ebf, 0x8e628e06, 0xc3a9c45b, 0xc50dc5c0,
	0x8dab8d51, 0x8cf98ca1, 0xc673c727, 0xc7dbc890,
	0x8c4a8bf5, 0x8ba18b4e, 0xc946c9fc, 0xcab2cb69,
	0x8afc8aab, 0x8a5b8a0c, 0xcc21ccd9, 0xcd92ce4b,
	0x89bf8972, 0x892788dd, 0xcf04cfbe, 0xd079d133,
	0x8894884c, 0x880687c0, 0xd1efd2ab, 0xd367d424,
	0x877c8739, 0x86f786b6, 0xd4e1d59e, 0xd65cd71a,
	0x86768638, 0x85fb85be, 0xd7d9d898, 0xd958da18,
	0x8583854a, 0x851184da, 0xdad8db98, 0xdc59dd1b,
	0x84a3846e, 0x843b8408, 0xdddcde9e, 0xdf60e023,
	0x83d683a6, 0x83778349, 0xe0e6e1a9, 0xe26ce330,
	0x831d82f1, 0x82c7829e, 0xe3f4e4b9, 0xe57de642,
	0x8276824f, 0x822a8206, 0xe707e7cc, 0xe892e958,
	0x81e381c1, 0x81a18181, 0xea1eeae4, 0xebaaec71,
	0x81638146, 0x812b8110, 0xed38edff, 0xeec6ef8d,
	0x80f780df, 0x80c880b3, 0xf054f11c, 0xf1e4f2ac,
	0x809e808b, 0x80798069, 0xf374f43c, 0xf504f5cd,
	0x8059804b, 0x803e8032, 0xf695f75e, 0xf826f8ef,
	0x8028801f, 0x80178010, 0xf9b8fa81, 0xfb49fc12,
	0x800a8006, 0x80038001, 0xfcdbfda4, 0xfe6dff36,
};

/* *********************************************************** */

const mlib_u32 mlib_v_fftK_WTable_1_P[] = {
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x7fff5a81, 0x0000a57e, 0x00005a81, 0x7fff5a81,
	0x7fff7640, 0x5a8130fb, 0x000030fb, 0x5a817640,
	0x0000cf04, 0xa57e89bf, 0x7fff7640, 0x5a8130fb,
	0x7fff7d89, 0x76406a6c, 0x000018f8, 0x30fb471c,
	0x5a81471c, 0x30fb18f8, 0x5a816a6c, 0x76407d89,
	0x0000e707, 0xcf04b8e3, 0x7fff7d89, 0x76406a6c,
	0xa57e9593, 0x89bf8276, 0x5a81471c, 0x30fb18f8,
	0x7fff7f61, 0x7d897a7c, 0x00000c8b, 0x18f82527,
	0x764070e1, 0x6a6c62f1, 0x30fb3c56, 0x471c5133,
	0x5a815133, 0x471c3c56, 0x5a8162f1, 0x6a6c70e1,
	0x30fb2527, 0x18f80c8b, 0x76407a7c, 0x7d897f61,
	0x0000f374, 0xe707dad8, 0x7fff7f61, 0x7d897a7c,
	0xcf04c3a9, 0xb8e3aecc, 0x764070e1, 0x6a6c62f1,
	0xa57e9d0e, 0x95938f1e, 0x5a815133, 0x471c3c56,
	0x89bf8583, 0x8276809e, 0x30fb2527, 0x18f80c8b,
	0x7fff7fd7, 0x7f617e9c, 0x00000647, 0x0c8b12c7,
	0x7d897c29, 0x7a7c7883, 0x18f81f19, 0x25272b1e,
	0x764073b5, 0x70e16dc9, 0x30fb36b9, 0x3c5641cd,
	0x6a6c66ce, 0x62f15ed6, 0x471c4c3f, 0x513355f4,
	0x5a8155f4, 0x51334c3f, 0x5a815ed6, 0x62f166ce,
	0x471c41cd, 0x3c5636b9, 0x6a6c6dc9, 0x70e173b5,
	0x30fb2b1e, 0x25271f19, 0x76407883, 0x7a7c7c29,
	0x18f812c7, 0x0c8b0647, 0x7d897e9c, 0x7f617fd7,
	0x0000f9b8, 0xf374ed38, 0x7ffe7fd7, 0x7f617e9c,
	0xe707e0e6, 0xdad8d4e1, 0x7d897c29, 0x7a7c7883,
	0xcf04c946, 0xc3a9be32, 0x764073b5, 0x70e16dc9,
	0xb8e3b3c0, 0xaeccaa0b, 0x6a6c66ce, 0x62f15ed6,
	0xa57ea129, 0x9d0e9931, 0x5a8155f4, 0x51334c3f,
	0x95939236, 0x8f1e8c4a, 0x471c41cd, 0x3c5636b9,
	0x89bf877c, 0x858383d6, 0x30fb2b1e, 0x25271f19,
	0x82768163, 0x809e8028, 0x18f812c7, 0x0c8b0647,
	0x7fff7ff5, 0x7fd77fa6, 0x00000324, 0x0647096a,
	0x7f617f08, 0x7e9c7e1c, 0x0c8b0fab, 0x12c715e1,
	0x7d897ce2, 0x7c297b5c, 0x18f81c0b, 0x1f192223,
	0x7a7c7989, 0x7883776b, 0x25272826, 0x2b1e2e10,
	0x76407503, 0x73b57254, 0x30fb33de, 0x36b9398c,
	0x70e16f5e, 0x6dc96c23, 0x3c563f16, 0x41cd447a,
	0x6a6c68a5, 0x66ce64e7, 0x471c49b3, 0x4c3f4ebf,
	0x62f160eb, 0x5ed65cb3, 0x5133539a, 0x55f45842,
	0x5a815842, 0x55f4539a, 0x5a815cb3, 0x5ed660eb,
	0x51334ebf, 0x4c3f49b3, 0x62f164e7, 0x66ce68a5,
	0x471c447a, 0x41cd3f16, 0x6a6c6c23, 0x6dc96f5e,
	0x3c56398c, 0x36b933de, 0x70e17254, 0x73b57503,
	0x30fb2e10, 0x2b1e2826, 0x7640776b, 0x78837989,
	0x25272223, 0x1f191c0b, 0x7a7c7b5c, 0x7c297ce2,
	0x18f815e1, 0x12c70fab, 0x7d897e1c, 0x7e9c7f08,
	0x0c8b096a, 0x06470324, 0x7f617fa6, 0x7fd77ff5,
	0x0000fcdb, 0xf9b8f695, 0x7fff7ff5, 0x7fd77fa6,
	0xf374f054, 0xed38ea1e, 0x7f617f08, 0x7e9c7e1c,
	0xe707e3f4, 0xe0e6dddc, 0x7d897ce2, 0x7c297b5c,
	0xdad8d7d9, 0xd4e1d1ef, 0x7a7c7989, 0x7883776b,
	0xcf04cc21, 0xc946c673, 0x76407503, 0x73b57254,
	0xc3a9c0e9, 0xbe32bb85, 0x70e16f5e, 0x6dc96c23,
	0xb8e3b64c, 0xb3c0b140, 0x6a6c68a5, 0x66ce64e7,
	0xaeccac65, 0xaa0ba7bd, 0x62f160eb, 0x5ed65cb3,
	0xa57ea34c, 0xa1299f14, 0x5a815842, 0x55f4539a,
	0x9d0e9b18, 0x9931975a, 0x51334ebf, 0x4c3f49b3,
	0x959393dc, 0x923690a1, 0x471c447a, 0x41cd3f16,
	0x8f1e8dab, 0x8c4a8afc, 0x3c56398c, 0x36b933de,
	0x89bf8894, 0x877c8676, 0x30fb2e10, 0x2b1e2826,
	0x858384a3, 0x83d6831d, 0x25272223, 0x1f191c0b,
	0x827681e3, 0x816380f7, 0x18f815e1, 0x12c70fab,
	0x809e8059, 0x8028800a, 0x0c8b096a, 0x06470324,
	0x7fff7ffc, 0x7ff57fe8, 0x00000192, 0x032404b6,
	0x7fd77fc1, 0x7fa67f86, 0x064707d9, 0x096a0afb,
	0x7f617f37, 0x7f087ed4, 0x0c8b0e1b, 0x0fab1139,
	0x7e9c7e5e, 0x7e1c7dd5, 0x12c71455, 0x15e1176d,
	0x7d897d38, 0x7ce27c88, 0x18f81a82, 0x1c0b1d93,
	0x7c297bc4, 0x7b5c7aee, 0x1f19209f, 0x222323a6,
	0x7a7c7a04, 0x79897908, 0x252726a7, 0x282629a3,
	0x788377f9, 0x776b76d8, 0x2b1e2c98, 0x2e102f86,
	0x764075a4, 0x7503745e, 0x30fb326d, 0x33de354d,
	0x73b57306, 0x7254719d, 0x36b93824, 0x398c3af2,
	0x70e17022, 0x6f5e6e95, 0x3c563db7, 0x3f164073,
	0x6dc96cf8, 0x6c236b4a, 0x41cd4325, 0x447a45cc,
	0x6a6c698b, 0x68a567bc, 0x471c4869, 0x49b34afa,
	0x66ce65dd, 0x64e763ee, 0x4c3f4d80, 0x4ebf4ffa,
	0x62f161f0, 0x60eb5fe2, 0x51335268, 0x539a54c9,
	0x5ed65dc6, 0x5cb35b9c, 0x55f4571d, 0x58425963,
	0x5a815963, 0x5842571d, 0x5a815b9c, 0x5cb35dc6,
	0x55f454c9, 0x539a5268, 0x5ed65fe2, 0x60eb61f0,
	0x51334ffa, 0x4ebf4d80, 0x62f163ee, 0x64e765dd,
	0x4c3f4afa, 0x49b34869, 0x66ce67bc, 0x68a5698b,
	0x471c45cc, 0x447a4325, 0x6a6c6b4a, 0x6c236cf8,
	0x41cd4073, 0x3f163db7, 0x6dc96e95, 0x6f5e7022,
	0x3c563af2, 0x398c3824, 0x70e1719d, 0x72547306,
	0x36b9354d, 0x33de326d, 0x73b5745e, 0x750375a4,
	0x30fb2f86, 0x2e102c98, 0x764076d8, 0x776b77f9,
	0x2b1e29a3, 0x282626a7, 0x78837908, 0x79897a04,
	0x252723a6, 0x2223209f, 0x7a7c7aee, 0x7b5c7bc4,
	0x1f191d93, 0x1c0b1a82, 0x7c297c88, 0x7ce27d38,
	0x18f8176d, 0x15e11455, 0x7d897dd5, 0x7e1c7e5e,
	0x12c71139, 0x0fab0e1b, 0x7e9c7ed4, 0x7f087f37,
	0x0c8b0afb, 0x096a07d9, 0x7f617f86, 0x7fa67fc1,
	0x064704b6, 0x03240192, 0x7fd77fe8, 0x7ff57ffc,
	0x0000fe6d, 0xfcdbfb49, 0x7fff7ffc, 0x7ff57fe8,
	0xf9b8f826, 0xf695f504, 0x7fd77fc1, 0x7fa67f86,
	0xf374f1e4, 0xf054eec6, 0x7f617f37, 0x7f087ed4,
	0xed38ebaa, 0xea1ee892, 0x7e9c7e5e, 0x7e1c7dd5,
	0xe707e57d, 0xe3f4e26c, 0x7d897d38, 0x7ce27c88,
	0xe0e6df60, 0xdddcdc59, 0x7c297bc4, 0x7b5c7aee,
	0xdad8d958, 0xd7d9d65c, 0x7a7c7a04, 0x79897908,
	0xd4e1d367, 0xd1efd079, 0x788377f9, 0x776b76d8,
	0xcf04cd92, 0xcc21cab2, 0x764075a4, 0x7503745e,
	0xc946c7db, 0xc673c50d, 0x73b57306, 0x7254719d,
	0xc3a9c248, 0xc0e9bf8c, 0x70e17022, 0x6f5e6e95,
	0xbe32bcda, 0xbb85ba33, 0x6dc96cf8, 0x6c236b4a,
	0xb8e3b796, 0xb64cb505, 0x6a6c698b, 0x68a567bc,
	0xb3c0b27f, 0xb140b005, 0x66ce65dd, 0x64e763ee,
	0xaeccad97, 0xac65ab36, 0x62f161f0, 0x60eb5fe2,
	0xaa0ba8e2, 0xa7bda69c, 0x5ed65dc6, 0x5cb35b9c,
	0xa57ea463, 0xa34ca239, 0x5a815963, 0x5842571d,
	0xa129a01d, 0x9f149e0f, 0x55f454c9, 0x539a5268,
	0x9d0e9c11, 0x9b189a22, 0x51334ffa, 0x4ebf4d80,
	0x99319843, 0x975a9674, 0x4c3f4afa, 0x49b34869,
	0x959394b5, 0x93dc9307, 0x471c45cc, 0x447a4325,
	0x9236916a, 0x90a18fdd, 0x41cd4073, 0x3f163db7,
	0x8f1e8e62, 0x8dab8cf9, 0x3c563af2, 0x398c3824,
	0x8c4a8ba1, 0x8afc8a5b, 0x36b9354d, 0x33de326d,
	0x89bf8927, 0x88948806, 0x30fb2f86, 0x2e102c98,
	0x877c86f7, 0x867685fb, 0x2b1e29a3, 0x282626a7,
	0x85838511, 0x84a3843b, 0x252723a6, 0x2223209f,
	0x83d68377, 0x831d82c7, 0x1f191d93, 0x1c0b1a82,
	0x8276822a, 0x81e381a1, 0x18f8176d, 0x15e11455,
	0x8163812b, 0x80f780c8, 0x12c71139, 0x0fab0e1b,
	0x809e8079, 0x8059803e, 0x0c8b0afb, 0x096a07d9,
	0x80288017, 0x800a8003, 0x064704b6, 0x03240192,
	0x7fff7ffe, 0x7ffc7ff9, 0x000000c9, 0x0192025b,
	0x7ff57fef, 0x7fe87fe0, 0x032403ed, 0x04b6057e,
	0x7fd77fcd, 0x7fc17fb4, 0x06470710, 0x07d908a1,
	0x7fa67f96, 0x7f867f74, 0x096a0a32, 0x0afb0bc3,
	0x7f617f4c, 0x7f377f20, 0x0c8b0d53, 0x0e1b0ee3,
	0x7f087eef, 0x7ed47eb9, 0x0fab1072, 0x11391200,
	0x7e9c7e7e, 0x7e5e7e3e, 0x12c7138e, 0x1455151b,
	0x7e1c7df9, 0x7dd57db0, 0x15e116a7, 0x176d1833,
	0x7d897d61, 0x7d387d0e, 0x18f819bd, 0x1a821b46,
	0x7ce27cb6, 0x7c887c59, 0x1c0b1ccf, 0x1d931e56,
	0x7c297bf7, 0x7bc47b91, 0x1f191fdc, 0x209f2161,
	0x7b5c7b25, 0x7aee7ab5, 0x222322e4, 0x23a62467,
	0x7a7c7a41, 0x7a0479c7, 0x252725e7, 0x26a72767,
	0x79897949, 0x790878c6, 0x282628e5, 0x29a32a61,
	0x7883783f, 0x77f977b3, 0x2b1e2bdb, 0x2c982d54,
	0x776b7722, 0x76d8768d, 0x2e102ecc, 0x2f863041,
	0x764075f3, 0x75a47554, 0x30fb31b4, 0x326d3326,
	0x750374b1, 0x745e740a, 0x33de3496, 0x354d3603,
	0x73b5735e, 0x730672ae, 0x36b9376f, 0x382438d8,
	0x725471f9, 0x719d7140, 0x398c3a3f, 0x3af23ba4,
	0x70e17082, 0x70226fc0, 0x3c563d07, 0x3db73e67,
	0x6f5e6efa, 0x6e956e30, 0x3f163fc5, 0x40734120,
	0x6dc96d61, 0x6cf86c8e, 0x41cd4279, 0x432543d0,
	0x6c236bb7, 0x6b4a6adb, 0x447a4523, 0x45cc4674,
	0x6a6c69fc, 0x698b6919, 0x471c47c3, 0x4869490e,
	0x68a56831, 0x67bc6745, 0x49b34a57, 0x4afa4b9d,
	0x66ce6656, 0x65dd6562, 0x4c3f4ce0, 0x4d804e20,
	0x64e7646b, 0x63ee6370, 0x4ebf4f5d, 0x4ffa5097,
	0x62f16271, 0x61f0616e, 0x513351ce, 0x52685301,
	0x60eb6067, 0x5fe25f5d, 0x539a5432, 0x54c9555f,
	0x5ed65e4f, 0x5dc65d3d, 0x55f45689, 0x571d57b0,
	0x5cb35c28, 0x5b9c5b0f, 0x584258d3, 0x596359f3,
	0x5a8159f3, 0x596358d3, 0x5a815b0f, 0x5b9c5c28,
	0x584257b0, 0x571d5689, 0x5cb35d3d, 0x5dc65e4f,
	0x55f4555f, 0x54c95432, 0x5ed65f5d, 0x5fe26067,
	0x539a5301, 0x526851ce, 0x60eb616e, 0x61f06271,
	0x51335097, 0x4ffa4f5d, 0x62f16370, 0x63ee646b,
	0x4ebf4e20, 0x4d804ce0, 0x64e76562, 0x65dd6656,
	0x4c3f4b9d, 0x4afa4a57, 0x66ce6745, 0x67bc6831,
	0x49b3490e, 0x486947c3, 0x68a56919, 0x698b69fc,
	0x471c4674, 0x45cc4523, 0x6a6c6adb, 0x6b4a6bb7,
	0x447a43d0, 0x43254279, 0x6c236c8e, 0x6cf86d61,
	0x41cd4120, 0x40733fc5, 0x6dc96e30, 0x6e956efa,
	0x3f163e67, 0x3db73d07, 0x6f5e6fc0, 0x70227082,
	0x3c563ba4, 0x3af23a3f, 0x70e17140, 0x719d71f9,
	0x398c38d8, 0x3824376f, 0x725472ae, 0x7306735e,
	0x36b93603, 0x354d3496, 0x73b5740a, 0x745e74b1,
	0x33de3326, 0x326d31b4, 0x75037554, 0x75a475f3,
	0x30fb3041, 0x2f862ecc, 0x7640768d, 0x76d87722,
	0x2e102d54, 0x2c982bdb, 0x776b77b3, 0x77f9783f,
	0x2b1e2a61, 0x29a328e5, 0x788378c6, 0x79087949,
	0x28262767, 0x26a725e7, 0x798979c7, 0x7a047a41,
	0x25272467, 0x23a622e4, 0x7a7c7ab5, 0x7aee7b25,
	0x22232161, 0x209f1fdc, 0x7b5c7b91, 0x7bc47bf7,
	0x1f191e56, 0x1d931ccf, 0x7c297c59, 0x7c887cb6,
	0x1c0b1b46, 0x1a8219bd, 0x7ce27d0e, 0x7d387d61,
	0x18f81833, 0x176d16a7, 0x7d897db0, 0x7dd57df9,
	0x15e1151b, 0x1455138e, 0x7e1c7e3e, 0x7e5e7e7e,
	0x12c71200, 0x11391072, 0x7e9c7eb9, 0x7ed47eef,
	0x0fab0ee3, 0x0e1b0d53, 0x7f087f20, 0x7f377f4c,
	0x0c8b0bc3, 0x0afb0a32, 0x7f617f74, 0x7f867f96,
	0x096a08a1, 0x07d90710, 0x7fa67fb4, 0x7fc17fcd,
	0x0647057e, 0x04b603ed, 0x7fd77fe0, 0x7fe87fef,
	0x0324025b, 0x019200c9, 0x7ff57ff9, 0x7ffc7ffe,
	0x0000ff36, 0xfe6dfda4, 0x7fff7ffe, 0x7ffc7ff9,
	0xfcdbfc12, 0xfb49fa81, 0x7ff57fef, 0x7fe87fe0,
	0xf9b8f8ef, 0xf826f75e, 0x7fd77fcd, 0x7fc17fb4,
	0xf695f5cd, 0xf504f43c, 0x7fa67f96, 0x7f867f74,
	0xf374f2ac, 0xf1e4f11c, 0x7f617f4c, 0x7f377f20,
	0xf054ef8d, 0xeec6edff, 0x7f087eef, 0x7ed47eb9,
	0xed38ec71, 0xebaaeae4, 0x7e9c7e7e, 0x7e5e7e3e,
	0xea1ee958, 0xe892e7cc, 0x7e1c7df9, 0x7dd57db0,
	0xe707e642, 0xe57de4b9, 0x7d897d61, 0x7d387d0e,
	0xe3f4e330, 0xe26ce1a9, 0x7ce27cb6, 0x7c887c59,
	0xe0e6e023, 0xdf60de9e, 0x7c297bf7, 0x7bc47b91,
	0xdddcdd1b, 0xdc59db98, 0x7b5c7b25, 0x7aee7ab5,
	0xdad8da18, 0xd958d898, 0x7a7c7a41, 0x7a0479c7,
	0xd7d9d71a, 0xd65cd59e, 0x79897949, 0x790878c6,
	0xd4e1d424, 0xd367d2ab, 0x7883783f, 0x77f977b3,
	0xd1efd133, 0xd079cfbe, 0x776b7722, 0x76d8768d,
	0xcf04ce4b, 0xcd92ccd9, 0x764075f3, 0x75a47554,
	0xcc21cb69, 0xcab2c9fc, 0x750374b1, 0x745e740a,
	0xc946c890, 0xc7dbc727, 0x73b5735e, 0x730672ae,
	0xc673c5c0, 0xc50dc45b, 0x725471f9, 0x719d7140,
	0xc3a9c2f8, 0xc248c198, 0x70e17082, 0x70226fc0,
	0xc0e9c03a, 0xbf8cbedf, 0x6f5e6efa, 0x6e956e30,
	0xbe32bd86, 0xbcdabc2f, 0x6dc96d61, 0x6cf86c8e,
	0xbb85badc, 0xba33b98b, 0x6c236bb7, 0x6b4a6adb,
	0xb8e3b83c, 0xb796b6f1, 0x6a6c69fc, 0x698b6919,
	0xb64cb5a8, 0xb505b462, 0x68a56831, 0x67bc6745,
	0xb3c0b31f, 0xb27fb1df, 0x66ce6656, 0x65dd6562,
	0xb140b0a2, 0xb005af68, 0x64e7646b, 0x63ee6370,
	0xaeccae31, 0xad97acfe, 0x62f16271, 0x61f0616e,
	0xac65abcd, 0xab36aaa0, 0x60eb6067, 0x5fe25f5d,
	0xaa0ba976, 0xa8e2a84f, 0x5ed65e4f, 0x5dc65d3d,
	0xa7bda72c, 0xa69ca60c, 0x5cb35c28, 0x5b9c5b0f,
	0xa57ea4f0, 0xa463a3d7, 0x5a8159f3, 0x596358d3,
	0xa34ca2c2, 0xa239a1b0, 0x584257b0, 0x571d5689,
	0xa129a0a2, 0xa01d9f98, 0x55f4555f, 0x54c95432,
	0x9f149e91, 0x9e0f9d8e, 0x539a5301, 0x526851ce,
	0x9d0e9c8f, 0x9c119b94, 0x51335097, 0x4ffa4f5d,
	0x9b189a9d, 0x9a2299a9, 0x4ebf4e20, 0x4d804ce0,
	0x993198ba, 0x984397ce, 0x4c3f4b9d, 0x4afa4a57,
	0x975a96e6, 0x96749603, 0x49b3490e, 0x486947c3,
	0x95939524, 0x94b59448, 0x471c4674, 0x45cc4523,
	0x93dc9371, 0x9307929e, 0x447a43d0, 0x43254279,
	0x923691cf, 0x916a9105, 0x41cd4120, 0x40733fc5,
	0x90a1903f, 0x8fdd8f7d, 0x3f163e67, 0x3db73d07,
	0x8f1e8ebf, 0x8e628e06, 0x3c563ba4, 0x3af23a3f,
	0x8dab8d51, 0x8cf98ca1, 0x398c38d8, 0x3824376f,
	0x8c4a8bf5, 0x8ba18b4e, 0x36b93603, 0x354d3496,
	0x8afc8aab, 0x8a5b8a0c, 0x33de3326, 0x326d31b4,
	0x89bf8972, 0x892788dd, 0x30fb3041, 0x2f862ecc,
	0x8894884c, 0x880687c0, 0x2e102d54, 0x2c982bdb,
	0x877c8739, 0x86f786b6, 0x2b1e2a61, 0x29a328e5,
	0x86768638, 0x85fb85be, 0x28262767, 0x26a725e7,
	0x8583854a, 0x851184da, 0x25272467, 0x23a622e4,
	0x84a3846e, 0x843b8408, 0x22232161, 0x209f1fdc,
	0x83d683a6, 0x83778349, 0x1f191e56, 0x1d931ccf,
	0x831d82f1, 0x82c7829e, 0x1c0b1b46, 0x1a8219bd,
	0x8276824f, 0x822a8206, 0x18f81833, 0x176d16a7,
	0x81e381c1, 0x81a18181, 0x15e1151b, 0x1455138e,
	0x81638146, 0x812b8110, 0x12c71200, 0x11391072,
	0x80f780df, 0x80c880b3, 0x0fab0ee3, 0x0e1b0d53,
	0x809e808b, 0x80798069, 0x0c8b0bc3, 0x0afb0a32,
	0x8059804b, 0x803e8032, 0x096a08a1, 0x07d90710,
	0x8028801f, 0x80178010, 0x0647057e, 0x04b603ed,
	0x800a8006, 0x80038001, 0x0324025b, 0x019200c9,
};

/* *********************************************************** */

const mlib_u32 mlib_v_fftK_WTable_3_M[] = {
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x7fffa57e, 0x00005a81, 0x0000a57e, 0x7fffa57e,
	0x7fff30fb, 0xa57e89bf, 0x000089bf, 0xa57e30fb,
	0x00007640, 0x5a81cf04, 0x7fff30fb, 0xa57e89bf,
	0x7fff6a6c, 0x30fbe707, 0x0000b8e3, 0x89bf8276,
	0xa57e8276, 0x89bfb8e3, 0xa57ee707, 0x30fb6a6c,
	0x0000471c, 0x76407d89, 0x7fff6a6c, 0x30fbe707,
	0x5a8118f8, 0xcf049593, 0xa57e8276, 0x89bfb8e3,
	0x7fff7a7c, 0x6a6c5133, 0x0000dad8, 0xb8e39d0e,
	0x30fb0c8b, 0xe707c3a9, 0x89bf809e, 0x82768f1e,
	0xa57e8f1e, 0x8276809e, 0xa57ec3a9, 0xe7070c8b,
	0x89bf9d0e, 0xb8e3dad8, 0x30fb5133, 0x6a6c7a7c,
	0x00002527, 0x471c62f1, 0x7ffe7a7c, 0x6a6c5133,
	0x76407f61, 0x7d8970e1, 0x30fb0c8b, 0xe707c3a9,
	0x5a813c56, 0x18f8f374, 0xa57e8f1e, 0x8276809e,
	0xcf04aecc, 0x95938583, 0x89bf9d0e, 0xb8e3dad8,
	0x7fff7e9c, 0x7a7c73b5, 0x0000ed38, 0xdad8c946,
	0x6a6c5ed6, 0x513341cd, 0xb8e3aa0b, 0x9d0e9236,
	0x30fb1f19, 0x0c8bf9b8, 0x89bf83d6, 0x809e8028,
	0xe707d4e1, 0xc3a9b3c0, 0x8276877c, 0x8f1e9931,
	0xa57e9931, 0x8f1e877c, 0xa57eb3c0, 0xc3a9d4e1,
	0x82768028, 0x809e83d6, 0xe707f9b8, 0x0c8b1f19,
	0x89bf9236, 0x9d0eaa0b, 0x30fb41cd, 0x51335ed6,
	0xb8e3c946, 0xdad8ed38, 0x6a6c73b5, 0x7a7c7e9c,
	0x000012c7, 0x252736b9, 0x7ffe7e9c, 0x7a7c73b5,
	0x471c55f4, 0x62f16dc9, 0x6a6c5ed6, 0x513341cd,
	0x76407c29, 0x7f617fd7, 0x30fb1f19, 0x0c8bf9b8,
	0x7d897883, 0x70e166ce, 0xe707d4e1, 0xc3a9b3c0,
	0x5a814c3f, 0x3c562b1e, 0xa57e9931, 0x8f1e877c,
	0x18f80647, 0xf374e0e6, 0x82768028, 0x809e83d6,
	0xcf04be32, 0xaecca129, 0x89bf9236, 0x9d0eaa0b,
	0x95938c4a, 0x85838163, 0xb8e3c946, 0xdad8ed38,
	0x7fff7fa6, 0x7e9c7ce2, 0x0000f695, 0xed38e3f4,
	0x7a7c776b, 0x73b56f5e, 0xdad8d1ef, 0xc946c0e9,
	0x6a6c64e7, 0x5ed65842, 0xb8e3b140, 0xaa0ba34c,
	0x513349b3, 0x41cd398c, 0x9d0e975a, 0x92368dab,
	0x30fb2826, 0x1f1915e1, 0x89bf8676, 0x83d681e3,
	0x0c8b0324, 0xf9b8f054, 0x809e800a, 0x802880f7,
	0xe707dddc, 0xd4e1cc21, 0x827684a3, 0x877c8afc,
	0xc3a9bb85, 0xb3c0ac65, 0x8f1e93dc, 0x99319f14,
	0xa57e9f14, 0x993193dc, 0xa57eac65, 0xb3c0bb85,
	0x8f1e8afc, 0x877c84a3, 0xc3a9cc21, 0xd4e1dddc,
	0x827680f7, 0x8028800a, 0xe707f054, 0xf9b80324,
	0x809e81e3, 0x83d68676, 0x0c8b15e1, 0x1f192826,
	0x89bf8dab, 0x9236975a, 0x30fb398c, 0x41cd49b3,
	0x9d0ea34c, 0xaa0bb140, 0x51335842, 0x5ed664e7,
	0xb8e3c0e9, 0xc946d1ef, 0x6a6c6f5e, 0x73b5776b,
	0xdad8e3f4, 0xed38f695, 0x7a7c7ce2, 0x7e9c7fa6,
	0x0000096a, 0x12c71c0b, 0x7fff7fa6, 0x7e9c7ce2,
	0x25272e10, 0x36b93f16, 0x7a7c776b, 0x73b56f5e,
	0x471c4ebf, 0x55f45cb3, 0x6a6c64e7, 0x5ed65842,
	0x62f168a5, 0x6dc97254, 0x513349b3, 0x41cd398c,
	0x76407989, 0x7c297e1c, 0x30fb2826, 0x1f1915e1,
	0x7f617ff5, 0x7fd77f08, 0x0c8b0324, 0xf9b8f054,
	0x7d897b5c, 0x78837503, 0xe707dddc, 0xd4e1cc21,
	0x70e16c23, 0x66ce60eb, 0xc3a9bb85, 0xb3c0ac65,
	0x5a81539a, 0x4c3f447a, 0xa57e9f14, 0x993193dc,
	0x3c5633de, 0x2b1e2223, 0x8f1e8afc, 0x877c84a3,
	0x18f80fab, 0x0647fcdb, 0x827680f7, 0x8028800a,
	0xf374ea1e, 0xe0e6d7d9, 0x809e81e3, 0x83d68676,
	0xcf04c673, 0xbe32b64c, 0x89bf8dab, 0x9236975a,
	0xaecca7bd, 0xa1299b18, 0x9d0ea34c, 0xaa0bb140,
	0x959390a1, 0x8c4a8894, 0xb8e3c0e9, 0xc946d1ef,
	0x8583831d, 0x81638059, 0xdad8e3f4, 0xed38f695,
	0x7fff7fe8, 0x7fa67f37, 0x0000fb49, 0xf695f1e4,
	0x7e9c7dd5, 0x7ce27bc4, 0xed38e892, 0xe3f4df60,
	0x7a7c7908, 0x776b75a4, 0xdad8d65c, 0xd1efcd92,
	0x73b5719d, 0x6f5e6cf8, 0xc946c50d, 0xc0e9bcda,
	0x6a6c67bc, 0x64e761f0, 0xb8e3b505, 0xb140ad97,
	0x5ed65b9c, 0x584254c9, 0xaa0ba69c, 0xa34ca01d,
	0x51334d80, 0x49b345cc, 0x9d0e9a22, 0x975a94b5,
	0x41cd3db7, 0x398c354d, 0x92368fdd, 0x8dab8ba1,
	0x30fb2c98, 0x282623a6, 0x89bf8806, 0x86768511,
	0x1f191a82, 0x15e11139, 0x83d682c7, 0x81e3812b,
	0x0c8b07d9, 0x0324fe6d, 0x809e803e, 0x800a8003,
	0xf9b8f504, 0xf054ebaa, 0x80288079, 0x80f781a1,
	0xe707e26c, 0xdddcd958, 0x82768377, 0x84a385fb,
	0xd4e1d079, 0xcc21c7db, 0x877c8927, 0x8afc8cf9,
	0xc3a9bf8c, 0xbb85b796, 0x8f1e916a, 0x93dc9674,
	0xb3c0b005, 0xac65a8e2, 0x99319c11, 0x9f14a239,
	0xa57ea239, 0x9f149c11, 0xa57ea8e2, 0xac65b005,
	0x99319674, 0x93dc916a, 0xb3c0b796, 0xbb85bf8c,
	0x8f1e8cf9, 0x8afc8927, 0xc3a9c7db, 0xcc21d079,
	0x877c85fb, 0x84a38377, 0xd4e1d958, 0xdddce26c,
	0x827681a1, 0x80f78079, 0xe707ebaa, 0xf054f504,
	0x80288003, 0x800a803e, 0xf9b8fe6d, 0x032407d9,
	0x809e812b, 0x81e382c7, 0x0c8b1139, 0x15e11a82,
	0x83d68511, 0x86768806, 0x1f1923a6, 0x28262c98,
	0x89bf8ba1, 0x8dab8fdd, 0x30fb354d, 0x398c3db7,
	0x923694b5, 0x975a9a22, 0x41cd45cc, 0x49b34d80,
	0x9d0ea01d, 0xa34ca69c, 0x513354c9, 0x58425b9c,
	0xaa0bad97, 0xb140b505, 0x5ed661f0, 0x64e767bc,
	0xb8e3bcda, 0xc0e9c50d, 0x6a6c6cf8, 0x6f5e719d,
	0xc946cd92, 0xd1efd65c, 0x73b575a4, 0x776b7908,
	0xdad8df60, 0xe3f4e892, 0x7a7c7bc4, 0x7ce27dd5,
	0xed38f1e4, 0xf695fb49, 0x7e9c7f37, 0x7fa67fe8,
	0x000004b6, 0x096a0e1b, 0x7fff7fe8, 0x7fa67f37,
	0x12c7176d, 0x1c0b209f, 0x7e9c7dd5, 0x7ce27bc4,
	0x252729a3, 0x2e10326d, 0x7a7c7908, 0x776b75a4,
	0x36b93af2, 0x3f164325, 0x73b5719d, 0x6f5e6cf8,
	0x471c4afa, 0x4ebf5268, 0x6a6c67bc, 0x64e761f0,
	0x55f45963, 0x5cb35fe2, 0x5ed65b9c, 0x584254c9,
	0x62f165dd, 0x68a56b4a, 0x51334d80, 0x49b345cc,
	0x6dc97022, 0x7254745e, 0x41cd3db7, 0x398c354d,
	0x764077f9, 0x79897aee, 0x30fb2c98, 0x282623a6,
	0x7c297d38, 0x7e1c7ed4, 0x1f191a82, 0x15e11139,
	0x7f617fc1, 0x7ff57ffc, 0x0c8b07d9, 0x0324fe6d,
	0x7fd77f86, 0x7f087e5e, 0xf9b8f504, 0xf054ebaa,
	0x7d897c88, 0x7b5c7a04, 0xe707e26c, 0xdddcd958,
	0x788376d8, 0x75037306, 0xd4e1d079, 0xcc21c7db,
	0x70e16e95, 0x6c23698b, 0xc3a9bf8c, 0xbb85b796,
	0x66ce63ee, 0x60eb5dc6, 0xb3c0b005, 0xac65a8e2,
	0x5a81571d, 0x539a4ffa, 0xa57ea239, 0x9f149c11,
	0x4c3f4869, 0x447a4073, 0x99319674, 0x93dc916a,
	0x3c563824, 0x33de2f86, 0x8f1e8cf9, 0x8afc8927,
	0x2b1e26a7, 0x22231d93, 0x877c85fb, 0x84a38377,
	0x18f81455, 0x0fab0afb, 0x827681a1, 0x80f78079,
	0x06470192, 0xfcdbf826, 0x80288003, 0x800a803e,
	0xf374eec6, 0xea1ee57d, 0x809e812b, 0x81e382c7,
	0xe0e6dc59, 0xd7d9d367, 0x83d68511, 0x86768806,
	0xcf04cab2, 0xc673c248, 0x89bf8ba1, 0x8dab8fdd,
	0xbe32ba33, 0xb64cb27f, 0x923694b5, 0x975a9a22,
	0xaeccab36, 0xa7bda463, 0x9d0ea01d, 0xa34ca69c,
	0xa1299e0f, 0x9b189843, 0xaa0bad97, 0xb140b505,
	0x95939307, 0x90a18e62, 0xb8e3bcda, 0xc0e9c50d,
	0x8c4a8a5b, 0x889486f7, 0xc946cd92, 0xd1efd65c,
	0x8583843b, 0x831d822a, 0xdad8df60, 0xe3f4e892,
	0x816380c8, 0x80598017, 0xed38f1e4, 0xf695fb49,
	0x7fff7ff9, 0x7fe87fcd, 0x0000fda4, 0xfb49f8ef,
	0x7fa67f74, 0x7f377eef, 0xf695f43c, 0xf1e4ef8d,
	0x7e9c7e3e, 0x7dd57d61, 0xed38eae4, 0xe892e642,
	0x7ce27c59, 0x7bc47b25, 0xe3f4e1a9, 0xdf60dd1b,
	0x7a7c79c7, 0x7908783f, 0xdad8d898, 0xd65cd424,
	0x776b768d, 0x75a474b1, 0xd1efcfbe, 0xcd92cb69,
	0x73b572ae, 0x719d7082, 0xc946c727, 0xc50dc2f8,
	0x6f5e6e30, 0x6cf86bb7, 0xc0e9bedf, 0xbcdabadc,
	0x6a6c6919, 0x67bc6656, 0xb8e3b6f1, 0xb505b31f,
	0x64e76370, 0x61f06067, 0xb140af68, 0xad97abcd,
	0x5ed65d3d, 0x5b9c59f3, 0xaa0ba84f, 0xa69ca4f0,
	0x58425689, 0x54c95301, 0xa34ca1b0, 0xa01d9e91,
	0x51334f5d, 0x4d804b9d, 0x9d0e9b94, 0x9a2298ba,
	0x49b347c3, 0x45cc43d0, 0x975a9603, 0x94b59371,
	0x41cd3fc5, 0x3db73ba4, 0x92369105, 0x8fdd8ebf,
	0x398c376f, 0x354d3326, 0x8dab8ca1, 0x8ba18aab,
	0x30fb2ecc, 0x2c982a61, 0x89bf88dd, 0x88068739,
	0x282625e7, 0x23a62161, 0x867685be, 0x8511846e,
	0x1f191ccf, 0x1a821833, 0x83d68349, 0x82c7824f,
	0x15e1138e, 0x11390ee3, 0x81e38181, 0x812b80df,
	0x0c8b0a32, 0x07d9057e, 0x809e8069, 0x803e801f,
	0x032400c9, 0xfe6dfc12, 0x800a8001, 0x80038010,
	0xf9b8f75e, 0xf504f2ac, 0x8028804b, 0x807980b3,
	0xf054edff, 0xebaae958, 0x80f78146, 0x81a18206,
	0xe707e4b9, 0xe26ce023, 0x827682f1, 0x83778408,
	0xdddcdb98, 0xd958d71a, 0x84a3854a, 0x85fb86b6,
	0xd4e1d2ab, 0xd079ce4b, 0x877c884c, 0x89278a0c,
	0xcc21c9fc, 0xc7dbc5c0, 0x8afc8bf5, 0x8cf98e06,
	0xc3a9c198, 0xbf8cbd86, 0x8f1e903f, 0x916a929e,
	0xbb85b98b, 0xb796b5a8, 0x93dc9524, 0x967497ce,
	0xb3c0b1df, 0xb005ae31, 0x99319a9d, 0x9c119d8e,
	0xac65aaa0, 0xa8e2a72c, 0x9f14a0a2, 0xa239a3d7,
	0xa57ea3d7, 0xa239a0a2, 0xa57ea72c, 0xa8e2aaa0,
	0x9f149d8e, 0x9c119a9d, 0xac65ae31, 0xb005b1df,
	0x993197ce, 0x96749524, 0xb3c0b5a8, 0xb796b98b,
	0x93dc929e, 0x916a903f, 0xbb85bd86, 0xbf8cc198,
	0x8f1e8e06, 0x8cf98bf5, 0xc3a9c5c0, 0xc7dbc9fc,
	0x8afc8a0c, 0x8927884c, 0xcc21ce4b, 0xd079d2ab,
	0x877c86b6, 0x85fb854a, 0xd4e1d71a, 0xd958db98,
	0x84a38408, 0x837782f1, 0xdddce023, 0xe26ce4b9,
	0x82768206, 0x81a18146, 0xe707e958, 0xebaaedff,
	0x80f780b3, 0x8079804b, 0xf054f2ac, 0xf504f75e,
	0x80288010, 0x80038001, 0xf9b8fc12, 0xfe6d00c9,
	0x800a801f, 0x803e8069, 0x0324057e, 0x07d90a32,
	0x809e80df, 0x812b8181, 0x0c8b0ee3, 0x1139138e,
	0x81e3824f, 0x82c78349, 0x15e11833, 0x1a821ccf,
	0x83d6846e, 0x851185be, 0x1f192161, 0x23a625e7,
	0x86768739, 0x880688dd, 0x28262a61, 0x2c982ecc,
	0x89bf8aab, 0x8ba18ca1, 0x30fb3326, 0x354d376f,
	0x8dab8ebf, 0x8fdd9105, 0x398c3ba4, 0x3db73fc5,
	0x92369371, 0x94b59603, 0x41cd43d0, 0x45cc47c3,
	0x975a98ba, 0x9a229b94, 0x49b34b9d, 0x4d804f5d,
	0x9d0e9e91, 0xa01da1b0, 0x51335301, 0x54c95689,
	0xa34ca4f0, 0xa69ca84f, 0x584259f3, 0x5b9c5d3d,
	0xaa0babcd, 0xad97af68, 0x5ed66067, 0x61f06370,
	0xb140b31f, 0xb505b6f1, 0x64e76656, 0x67bc6919,
	0xb8e3badc, 0xbcdabedf, 0x6a6c6bb7, 0x6cf86e30,
	0xc0e9c2f8, 0xc50dc727, 0x6f5e7082, 0x719d72ae,
	0xc946cb69, 0xcd92cfbe, 0x73b574b1, 0x75a4768d,
	0xd1efd424, 0xd65cd898, 0x776b783f, 0x790879c7,
	0xdad8dd1b, 0xdf60e1a9, 0x7a7c7b25, 0x7bc47c59,
	0xe3f4e642, 0xe892eae4, 0x7ce27d61, 0x7dd57e3e,
	0xed38ef8d, 0xf1e4f43c, 0x7e9c7eef, 0x7f377f74,
	0xf695f8ef, 0xfb49fda4, 0x7fa67fcd, 0x7fe87ff9,
	0x0000025b, 0x04b60710, 0x7fff7ff9, 0x7fe87fcd,
	0x096a0bc3, 0x0e1b1072, 0x7fa67f74, 0x7f377eef,
	0x12c7151b, 0x176d19bd, 0x7e9c7e3e, 0x7dd57d61,
	0x1c0b1e56, 0x209f22e4, 0x7ce27c59, 0x7bc47b25,
	0x25272767, 0x29a32bdb, 0x7a7c79c7, 0x7908783f,
	0x2e103041, 0x326d3496, 0x776b768d, 0x75a474b1,
	0x36b938d8, 0x3af23d07, 0x73b572ae, 0x719d7082,
	0x3f164120, 0x43254523, 0x6f5e6e30, 0x6cf86bb7,
	0x471c490e, 0x4afa4ce0, 0x6a6c6919, 0x67bc6656,
	0x4ebf5097, 0x52685432, 0x64e76370, 0x61f06067,
	0x55f457b0, 0x59635b0f, 0x5ed65d3d, 0x5b9c59f3,
	0x5cb35e4f, 0x5fe2616e, 0x58425689, 0x54c95301,
	0x62f1646b, 0x65dd6745, 0x51334f5d, 0x4d804b9d,
	0x68a569fc, 0x6b4a6c8e, 0x49b347c3, 0x45cc43d0,
	0x6dc96efa, 0x70227140, 0x41cd3fc5, 0x3db73ba4,
	0x7254735e, 0x745e7554, 0x398c376f, 0x354d3326,
	0x76407722, 0x77f978c6, 0x30fb2ecc, 0x2c982a61,
	0x79897a41, 0x7aee7b91, 0x282625e7, 0x23a62161,
	0x7c297cb6, 0x7d387db0, 0x1f191ccf, 0x1a821833,
	0x7e1c7e7e, 0x7ed47f20, 0x15e1138e, 0x11390ee3,
	0x7f617f96, 0x7fc17fe0, 0x0c8b0a32, 0x07d9057e,
	0x7ff57ffe, 0x7ffc7fef, 0x032400c9, 0xfe6dfc12,
	0x7fd77fb4, 0x7f867f4c, 0xf9b8f75e, 0xf504f2ac,
	0x7f087eb9, 0x7e5e7df9, 0xf054edff, 0xebaae958,
	0x7d897d0e, 0x7c887bf7, 0xe707e4b9, 0xe26ce023,
	0x7b5c7ab5, 0x7a047949, 0xdddcdb98, 0xd958d71a,
	0x788377b3, 0x76d875f3, 0xd4e1d2ab, 0xd079ce4b,
	0x7503740a, 0x730671f9, 0xcc21c9fc, 0xc7dbc5c0,
	0x70e16fc0, 0x6e956d61, 0xc3a9c198, 0xbf8cbd86,
	0x6c236adb, 0x698b6831, 0xbb85b98b, 0xb796b5a8,
	0x66ce6562, 0x63ee6271, 0xb3c0b1df, 0xb005ae31,
	0x60eb5f5d, 0x5dc65c28, 0xac65aaa0, 0xa8e2a72c,
	0x5a8158d3, 0x571d555f, 0xa57ea3d7, 0xa239a0a2,
	0x539a51ce, 0x4ffa4e20, 0x9f149d8e, 0x9c119a9d,
	0x4c3f4a57, 0x48694674, 0x993197ce, 0x96749524,
	0x447a4279, 0x40733e67, 0x93dc929e, 0x916a903f,
	0x3c563a3f, 0x38243603, 0x8f1e8e06, 0x8cf98bf5,
	0x33de31b4, 0x2f862d54, 0x8afc8a0c, 0x8927884c,
	0x2b1e28e5, 0x26a72467, 0x877c86b6, 0x85fb854a,
	0x22231fdc, 0x1d931b46, 0x84a38408, 0x837782f1,
	0x18f816a7, 0x14551200, 0x82768206, 0x81a18146,
	0x0fab0d53, 0x0afb08a1, 0x80f780b3, 0x8079804b,
	0x064703ed, 0x0192ff36, 0x80288010, 0x80038001,
	0xfcdbfa81, 0xf826f5cd, 0x800a801f, 0x803e8069,
	0xf374f11c, 0xeec6ec71, 0x809e80df, 0x812b8181,
	0xea1ee7cc, 0xe57de330, 0x81e3824f, 0x82c78349,
	0xe0e6de9e, 0xdc59da18, 0x83d6846e, 0x851185be,
	0xd7d9d59e, 0xd367d133, 0x86768739, 0x880688dd,
	0xcf04ccd9, 0xcab2c890, 0x89bf8aab, 0x8ba18ca1,
	0xc673c45b, 0xc248c03a, 0x8dab8ebf, 0x8fdd9105,
	0xbe32bc2f, 0xba33b83c, 0x92369371, 0x94b59603,
	0xb64cb462, 0xb27fb0a2, 0x975a98ba, 0x9a229b94,
	0xaeccacfe, 0xab36a976, 0x9d0e9e91, 0xa01da1b0,
	0xa7bda60c, 0xa463a2c2, 0xa34ca4f0, 0xa69ca84f,
	0xa1299f98, 0x9e0f9c8f, 0xaa0babcd, 0xad97af68,
	0x9b1899a9, 0x984396e6, 0xb140b31f, 0xb505b6f1,
	0x95939448, 0x930791cf, 0xb8e3badc, 0xbcdabedf,
	0x90a18f7d, 0x8e628d51, 0xc0e9c2f8, 0xc50dc727,
	0x8c4a8b4e, 0x8a5b8972, 0xc946cb69, 0xcd92cfbe,
	0x889487c0, 0x86f78638, 0xd1efd424, 0xd65cd898,
	0x858384da, 0x843b83a6, 0xdad8dd1b, 0xdf60e1a9,
	0x831d829e, 0x822a81c1, 0xe3f4e642, 0xe892eae4,
	0x81638110, 0x80c8808b, 0xed38ef8d, 0xf1e4f43c,
	0x80598032, 0x80178006, 0xf695f8ef, 0xfb49fda4,
};

/* *********************************************************** */

const mlib_u32 mlib_v_fftK_WTable_3_P[] = {
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x7fffa57e, 0x00005a81, 0x00005a81, 0x80015a81,
	0x7fff30fb, 0xa57e89bf, 0x00007640, 0x5a81cf04,
	0x00007640, 0x5a81cf04, 0x8001cf04, 0x5a817640,
	0x7fff6a6c, 0x30fbe707, 0x0000471c, 0x76407d89,
	0xa57e8276, 0x89bfb8e3, 0x5a8118f8, 0xcf049593,
	0x0000471c, 0x76407d89, 0x80019593, 0xcf0418f8,
	0x5a8118f8, 0xcf049593, 0x5a817d89, 0x7640471c,
	0x7fff7a7c, 0x6a6c5133, 0x00002527, 0x471c62f1,
	0x30fb0c8b, 0xe707c3a9, 0x76407f61, 0x7d8970e1,
	0xa57e8f1e, 0x8276809e, 0x5a813c56, 0x18f8f374,
	0x89bf9d0e, 0xb8e3dad8, 0xcf04aecc, 0x95938583,
	0x00002527, 0x471c62f1, 0x80018583, 0x9593aecc,
	0x76407f61, 0x7d8970e1, 0xcf04f374, 0x18f83c56,
	0x5a813c56, 0x18f8f374, 0x5a8170e1, 0x7d897f61,
	0xcf04aecc, 0x95938583, 0x764062f1, 0x471c2527,
	0x7fff7e9c, 0x7a7c73b5, 0x000012c7, 0x252736b9,
	0x6a6c5ed6, 0x513341cd, 0x471c55f4, 0x62f16dc9,
	0x30fb1f19, 0x0c8bf9b8, 0x76407c29, 0x7f617fd7,
	0xe707d4e1, 0xc3a9b3c0, 0x7d897883, 0x70e166ce,
	0xa57e9931, 0x8f1e877c, 0x5a814c3f, 0x3c562b1e,
	0x82768028, 0x809e83d6, 0x18f80647, 0xf374e0e6,
	0x89bf9236, 0x9d0eaa0b, 0xcf04be32, 0xaecca129,
	0xb8e3c946, 0xdad8ed38, 0x95938c4a, 0x85838163,
	0x000012c7, 0x252736b9, 0x80018163, 0x85838c4a,
	0x471c55f4, 0x62f16dc9, 0x9593a129, 0xaeccbe32,
	0x76407c29, 0x7f617fd7, 0xcf04e0e6, 0xf3740647,
	0x7d897883, 0x70e166ce, 0x18f82b1e, 0x3c564c3f,
	0x5a814c3f, 0x3c562b1e, 0x5a8166ce, 0x70e17883,
	0x18f80647, 0xf374e0e6, 0x7d897fd7, 0x7f617c29,
	0xcf04be32, 0xaecca129, 0x76406dc9, 0x62f155f4,
	0x95938c4a, 0x85838163, 0x471c36b9, 0x252712c7,
	0x7fff7fa6, 0x7e9c7ce2, 0x0000096a, 0x12c71c0b,
	0x7a7c776b, 0x73b56f5e, 0x25272e10, 0x36b93f16,
	0x6a6c64e7, 0x5ed65842, 0x471c4ebf, 0x55f45cb3,
	0x513349b3, 0x41cd398c, 0x62f168a5, 0x6dc97254,
	0x30fb2826, 0x1f1915e1, 0x76407989, 0x7c297e1c,
	0x0c8b0324, 0xf9b8f054, 0x7f617ff5, 0x7fd77f08,
	0xe707dddc, 0xd4e1cc21, 0x7d897b5c, 0x78837503,
	0xc3a9bb85, 0xb3c0ac65, 0x70e16c23, 0x66ce60eb,
	0xa57e9f14, 0x993193dc, 0x5a81539a, 0x4c3f447a,
	0x8f1e8afc, 0x877c84a3, 0x3c5633de, 0x2b1e2223,
	0x827680f7, 0x8028800a, 0x18f80fab, 0x0647fcdb,
	0x809e81e3, 0x83d68676, 0xf374ea1e, 0xe0e6d7d9,
	0x89bf8dab, 0x9236975a, 0xcf04c673, 0xbe32b64c,
	0x9d0ea34c, 0xaa0bb140, 0xaecca7bd, 0xa1299b18,
	0xb8e3c0e9, 0xc946d1ef, 0x959390a1, 0x8c4a8894,
	0xdad8e3f4, 0xed38f695, 0x8583831d, 0x81638059,
	0x0000096a, 0x12c71c0b, 0x80018059, 0x8163831d,
	0x25272e10, 0x36b93f16, 0x85838894, 0x8c4a90a1,
	0x471c4ebf, 0x55f45cb3, 0x95939b18, 0xa129a7bd,
	0x62f168a5, 0x6dc97254, 0xaeccb64c, 0xbe32c673,
	0x76407989, 0x7c297e1c, 0xcf04d7d9, 0xe0e6ea1e,
	0x7f617ff5, 0x7fd77f08, 0xf374fcdb, 0x06470fab,
	0x7d897b5c, 0x78837503, 0x18f82223, 0x2b1e33de,
	0x70e16c23, 0x66ce60eb, 0x3c56447a, 0x4c3f539a,
	0x5a81539a, 0x4c3f447a, 0x5a8160eb, 0x66ce6c23,
	0x3c5633de, 0x2b1e2223, 0x70e17503, 0x78837b5c,
	0x18f80fab, 0x0647fcdb, 0x7d897f08, 0x7fd77ff5,
	0xf374ea1e, 0xe0e6d7d9, 0x7f617e1c, 0x7c297989,
	0xcf04c673, 0xbe32b64c, 0x76407254, 0x6dc968a5,
	0xaecca7bd, 0xa1299b18, 0x62f15cb3, 0x55f44ebf,
	0x959390a1, 0x8c4a8894, 0x471c3f16, 0x36b92e10,
	0x8583831d, 0x81638059, 0x25271c0b, 0x12c7096a,
	0x7fff7fe8, 0x7fa67f37, 0x000004b6, 0x096a0e1b,
	0x7e9c7dd5, 0x7ce27bc4, 0x12c7176d, 0x1c0b209f,
	0x7a7c7908, 0x776b75a4, 0x252729a3, 0x2e10326d,
	0x73b5719d, 0x6f5e6cf8, 0x36b93af2, 0x3f164325,
	0x6a6c67bc, 0x64e761f0, 0x471c4afa, 0x4ebf5268,
	0x5ed65b9c, 0x584254c9, 0x55f45963, 0x5cb35fe2,
	0x51334d80, 0x49b345cc, 0x62f165dd, 0x68a56b4a,
	0x41cd3db7, 0x398c354d, 0x6dc97022, 0x7254745e,
	0x30fb2c98, 0x282623a6, 0x764077f9, 0x79897aee,
	0x1f191a82, 0x15e11139, 0x7c297d38, 0x7e1c7ed4,
	0x0c8b07d9, 0x0324fe6d, 0x7f617fc1, 0x7ff57ffc,
	0xf9b8f504, 0xf054ebaa, 0x7fd77f86, 0x7f087e5e,
	0xe707e26c, 0xdddcd958, 0x7d897c88, 0x7b5c7a04,
	0xd4e1d079, 0xcc21c7db, 0x788376d8, 0x75037306,
	0xc3a9bf8c, 0xbb85b796, 0x70e16e95, 0x6c23698b,
	0xb3c0b005, 0xac65a8e2, 0x66ce63ee, 0x60eb5dc6,
	0xa57ea239, 0x9f149c11, 0x5a81571d, 0x539a4ffa,
	0x99319674, 0x93dc916a, 0x4c3f4869, 0x447a4073,
	0x8f1e8cf9, 0x8afc8927, 0x3c563824, 0x33de2f86,
	0x877c85fb, 0x84a38377, 0x2b1e26a7, 0x22231d93,
	0x827681a1, 0x80f78079, 0x18f81455, 0x0fab0afb,
	0x80288003, 0x800a803e, 0x06470192, 0xfcdbf826,
	0x809e812b, 0x81e382c7, 0xf374eec6, 0xea1ee57d,
	0x83d68511, 0x86768806, 0xe0e6dc59, 0xd7d9d367,
	0x89bf8ba1, 0x8dab8fdd, 0xcf04cab2, 0xc673c248,
	0x923694b5, 0x975a9a22, 0xbe32ba33, 0xb64cb27f,
	0x9d0ea01d, 0xa34ca69c, 0xaeccab36, 0xa7bda463,
	0xaa0bad97, 0xb140b505, 0xa1299e0f, 0x9b189843,
	0xb8e3bcda, 0xc0e9c50d, 0x95939307, 0x90a18e62,
	0xc946cd92, 0xd1efd65c, 0x8c4a8a5b, 0x889486f7,
	0xdad8df60, 0xe3f4e892, 0x8583843b, 0x831d822a,
	0xed38f1e4, 0xf695fb49, 0x816380c8, 0x80598017,
	0x000004b6, 0x096a0e1b, 0x80018017, 0x805980c8,
	0x12c7176d, 0x1c0b209f, 0x8163822a, 0x831d843b,
	0x252729a3, 0x2e10326d, 0x858386f7, 0x88948a5b,
	0x36b93af2, 0x3f164325, 0x8c4a8e62, 0x90a19307,
	0x471c4afa, 0x4ebf5268, 0x95939843, 0x9b189e0f,
	0x55f45963, 0x5cb35fe2, 0xa129a463, 0xa7bdab36,
	0x62f165dd, 0x68a56b4a, 0xaeccb27f, 0xb64cba33,
	0x6dc97022, 0x7254745e, 0xbe32c248, 0xc673cab2,
	0x764077f9, 0x79897aee, 0xcf04d367, 0xd7d9dc59,
	0x7c297d38, 0x7e1c7ed4, 0xe0e6e57d, 0xea1eeec6,
	0x7f617fc1, 0x7ff57ffc, 0xf374f826, 0xfcdb0192,
	0x7fd77f86, 0x7f087e5e, 0x06470afb, 0x0fab1455,
	0x7d897c88, 0x7b5c7a04, 0x18f81d93, 0x222326a7,
	0x788376d8, 0x75037306, 0x2b1e2f86, 0x33de3824,
	0x70e16e95, 0x6c23698b, 0x3c564073, 0x447a4869,
	0x66ce63ee, 0x60eb5dc6, 0x4c3f4ffa, 0x539a571d,
	0x5a81571d, 0x539a4ffa, 0x5a815dc6, 0x60eb63ee,
	0x4c3f4869, 0x447a4073, 0x66ce698b, 0x6c236e95,
	0x3c563824, 0x33de2f86, 0x70e17306, 0x750376d8,
	0x2b1e26a7, 0x22231d93, 0x78837a04, 0x7b5c7c88,
	0x18f81455, 0x0fab0afb, 0x7d897e5e, 0x7f087f86,
	0x06470192, 0xfcdbf826, 0x7fd77ffc, 0x7ff57fc1,
	0xf374eec6, 0xea1ee57d, 0x7f617ed4, 0x7e1c7d38,
	0xe0e6dc59, 0xd7d9d367, 0x7c297aee, 0x798977f9,
	0xcf04cab2, 0xc673c248, 0x7640745e, 0x72547022,
	0xbe32ba33, 0xb64cb27f, 0x6dc96b4a, 0x68a565dd,
	0xaeccab36, 0xa7bda463, 0x62f15fe2, 0x5cb35963,
	0xa1299e0f, 0x9b189843, 0x55f45268, 0x4ebf4afa,
	0x95939307, 0x90a18e62, 0x471c4325, 0x3f163af2,
	0x8c4a8a5b, 0x889486f7, 0x36b9326d, 0x2e1029a3,
	0x8583843b, 0x831d822a, 0x2527209f, 0x1c0b176d,
	0x816380c8, 0x80598017, 0x12c70e1b, 0x096a04b6,
	0x7fff7ff9, 0x7fe87fcd, 0x0000025b, 0x04b60710,
	0x7fa67f74, 0x7f377eef, 0x096a0bc3, 0x0e1b1072,
	0x7e9c7e3e, 0x7dd57d61, 0x12c7151b, 0x176d19bd,
	0x7ce27c59, 0x7bc47b25, 0x1c0b1e56, 0x209f22e4,
	0x7a7c79c7, 0x7908783f, 0x25272767, 0x29a32bdb,
	0x776b768d, 0x75a474b1, 0x2e103041, 0x326d3496,
	0x73b572ae, 0x719d7082, 0x36b938d8, 0x3af23d07,
	0x6f5e6e30, 0x6cf86bb7, 0x3f164120, 0x43254523,
	0x6a6c6919, 0x67bc6656, 0x471c490e, 0x4afa4ce0,
	0x64e76370, 0x61f06067, 0x4ebf5097, 0x52685432,
	0x5ed65d3d, 0x5b9c59f3, 0x55f457b0, 0x59635b0f,
	0x58425689, 0x54c95301, 0x5cb35e4f, 0x5fe2616e,
	0x51334f5d, 0x4d804b9d, 0x62f1646b, 0x65dd6745,
	0x49b347c3, 0x45cc43d0, 0x68a569fc, 0x6b4a6c8e,
	0x41cd3fc5, 0x3db73ba4, 0x6dc96efa, 0x70227140,
	0x398c376f, 0x354d3326, 0x7254735e, 0x745e7554,
	0x30fb2ecc, 0x2c982a61, 0x76407722, 0x77f978c6,
	0x282625e7, 0x23a62161, 0x79897a41, 0x7aee7b91,
	0x1f191ccf, 0x1a821833, 0x7c297cb6, 0x7d387db0,
	0x15e1138e, 0x11390ee3, 0x7e1c7e7e, 0x7ed47f20,
	0x0c8b0a32, 0x07d9057e, 0x7f617f96, 0x7fc17fe0,
	0x032400c9, 0xfe6dfc12, 0x7ff57ffe, 0x7ffc7fef,
	0xf9b8f75e, 0xf504f2ac, 0x7fd77fb4, 0x7f867f4c,
	0xf054edff, 0xebaae958, 0x7f087eb9, 0x7e5e7df9,
	0xe707e4b9, 0xe26ce023, 0x7d897d0e, 0x7c887bf7,
	0xdddcdb98, 0xd958d71a, 0x7b5c7ab5, 0x7a047949,
	0xd4e1d2ab, 0xd079ce4b, 0x788377b3, 0x76d875f3,
	0xcc21c9fc, 0xc7dbc5c0, 0x7503740a, 0x730671f9,
	0xc3a9c198, 0xbf8cbd86, 0x70e16fc0, 0x6e956d61,
	0xbb85b98b, 0xb796b5a8, 0x6c236adb, 0x698b6831,
	0xb3c0b1df, 0xb005ae31, 0x66ce6562, 0x63ee6271,
	0xac65aaa0, 0xa8e2a72c, 0x60eb5f5d, 0x5dc65c28,
	0xa57ea3d7, 0xa239a0a2, 0x5a8158d3, 0x571d555f,
	0x9f149d8e, 0x9c119a9d, 0x539a51ce, 0x4ffa4e20,
	0x993197ce, 0x96749524, 0x4c3f4a57, 0x48694674,
	0x93dc929e, 0x916a903f, 0x447a4279, 0x40733e67,
	0x8f1e8e06, 0x8cf98bf5, 0x3c563a3f, 0x38243603,
	0x8afc8a0c, 0x8927884c, 0x33de31b4, 0x2f862d54,
	0x877c86b6, 0x85fb854a, 0x2b1e28e5, 0x26a72467,
	0x84a38408, 0x837782f1, 0x22231fdc, 0x1d931b46,
	0x82768206, 0x81a18146, 0x18f816a7, 0x14551200,
	0x80f780b3, 0x8079804b, 0x0fab0d53, 0x0afb08a1,
	0x80288010, 0x80038001, 0x064703ed, 0x0192ff36,
	0x800a801f, 0x803e8069, 0xfcdbfa81, 0xf826f5cd,
	0x809e80df, 0x812b8181, 0xf374f11c, 0xeec6ec71,
	0x81e3824f, 0x82c78349, 0xea1ee7cc, 0xe57de330,
	0x83d6846e, 0x851185be, 0xe0e6de9e, 0xdc59da18,
	0x86768739, 0x880688dd, 0xd7d9d59e, 0xd367d133,
	0x89bf8aab, 0x8ba18ca1, 0xcf04ccd9, 0xcab2c890,
	0x8dab8ebf, 0x8fdd9105, 0xc673c45b, 0xc248c03a,
	0x92369371, 0x94b59603, 0xbe32bc2f, 0xba33b83c,
	0x975a98ba, 0x9a229b94, 0xb64cb462, 0xb27fb0a2,
	0x9d0e9e91, 0xa01da1b0, 0xaeccacfe, 0xab36a976,
	0xa34ca4f0, 0xa69ca84f, 0xa7bda60c, 0xa463a2c2,
	0xaa0babcd, 0xad97af68, 0xa1299f98, 0x9e0f9c8f,
	0xb140b31f, 0xb505b6f1, 0x9b1899a9, 0x984396e6,
	0xb8e3badc, 0xbcdabedf, 0x95939448, 0x930791cf,
	0xc0e9c2f8, 0xc50dc727, 0x90a18f7d, 0x8e628d51,
	0xc946cb69, 0xcd92cfbe, 0x8c4a8b4e, 0x8a5b8972,
	0xd1efd424, 0xd65cd898, 0x889487c0, 0x86f78638,
	0xdad8dd1b, 0xdf60e1a9, 0x858384da, 0x843b83a6,
	0xe3f4e642, 0xe892eae4, 0x831d829e, 0x822a81c1,
	0xed38ef8d, 0xf1e4f43c, 0x81638110, 0x80c8808b,
	0xf695f8ef, 0xfb49fda4, 0x80598032, 0x80178006,
	0x0000025b, 0x04b60710, 0x80018006, 0x80178032,
	0x096a0bc3, 0x0e1b1072, 0x8059808b, 0x80c88110,
	0x12c7151b, 0x176d19bd, 0x816381c1, 0x822a829e,
	0x1c0b1e56, 0x209f22e4, 0x831d83a6, 0x843b84da,
	0x25272767, 0x29a32bdb, 0x85838638, 0x86f787c0,
	0x2e103041, 0x326d3496, 0x88948972, 0x8a5b8b4e,
	0x36b938d8, 0x3af23d07, 0x8c4a8d51, 0x8e628f7d,
	0x3f164120, 0x43254523, 0x90a191cf, 0x93079448,
	0x471c490e, 0x4afa4ce0, 0x959396e6, 0x984399a9,
	0x4ebf5097, 0x52685432, 0x9b189c8f, 0x9e0f9f98,
	0x55f457b0, 0x59635b0f, 0xa129a2c2, 0xa463a60c,
	0x5cb35e4f, 0x5fe2616e, 0xa7bda976, 0xab36acfe,
	0x62f1646b, 0x65dd6745, 0xaeccb0a2, 0xb27fb462,
	0x68a569fc, 0x6b4a6c8e, 0xb64cb83c, 0xba33bc2f,
	0x6dc96efa, 0x70227140, 0xbe32c03a, 0xc248c45b,
	0x7254735e, 0x745e7554, 0xc673c890, 0xcab2ccd9,
	0x76407722, 0x77f978c6, 0xcf04d133, 0xd367d59e,
	0x79897a41, 0x7aee7b91, 0xd7d9da18, 0xdc59de9e,
	0x7c297cb6, 0x7d387db0, 0xe0e6e330, 0xe57de7cc,
	0x7e1c7e7e, 0x7ed47f20, 0xea1eec71, 0xeec6f11c,
	0x7f617f96, 0x7fc17fe0, 0xf374f5cd, 0xf826fa81,
	0x7ff57ffe, 0x7ffc7fef, 0xfcdbff36, 0x019203ed,
	0x7fd77fb4, 0x7f867f4c, 0x064708a1, 0x0afb0d53,
	0x7f087eb9, 0x7e5e7df9, 0x0fab1200, 0x145516a7,
	0x7d897d0e, 0x7c887bf7, 0x18f81b46, 0x1d931fdc,
	0x7b5c7ab5, 0x7a047949, 0x22232467, 0x26a728e5,
	0x788377b3, 0x76d875f3, 0x2b1e2d54, 0x2f8631b4,
	0x7503740a, 0x730671f9, 0x33de3603, 0x38243a3f,
	0x70e16fc0, 0x6e956d61, 0x3c563e67, 0x40734279,
	0x6c236adb, 0x698b6831, 0x447a4674, 0x48694a57,
	0x66ce6562, 0x63ee6271, 0x4c3f4e20, 0x4ffa51ce,
	0x60eb5f5d, 0x5dc65c28, 0x539a555f, 0x571d58d3,
	0x5a8158d3, 0x571d555f, 0x5a815c28, 0x5dc65f5d,
	0x539a51ce, 0x4ffa4e20, 0x60eb6271, 0x63ee6562,
	0x4c3f4a57, 0x48694674, 0x66ce6831, 0x698b6adb,
	0x447a4279, 0x40733e67, 0x6c236d61, 0x6e956fc0,
	0x3c563a3f, 0x38243603, 0x70e171f9, 0x7306740a,
	0x33de31b4, 0x2f862d54, 0x750375f3, 0x76d877b3,
	0x2b1e28e5, 0x26a72467, 0x78837949, 0x7a047ab5,
	0x22231fdc, 0x1d931b46, 0x7b5c7bf7, 0x7c887d0e,
	0x18f816a7, 0x14551200, 0x7d897df9, 0x7e5e7eb9,
	0x0fab0d53, 0x0afb08a1, 0x7f087f4c, 0x7f867fb4,
	0x064703ed, 0x0192ff36, 0x7fd77fef, 0x7ffc7ffe,
	0xfcdbfa81, 0xf826f5cd, 0x7ff57fe0, 0x7fc17f96,
	0xf374f11c, 0xeec6ec71, 0x7f617f20, 0x7ed47e7e,
	0xea1ee7cc, 0xe57de330, 0x7e1c7db0, 0x7d387cb6,
	0xe0e6de9e, 0xdc59da18, 0x7c297b91, 0x7aee7a41,
	0xd7d9d59e, 0xd367d133, 0x798978c6, 0x77f97722,
	0xcf04ccd9, 0xcab2c890, 0x76407554, 0x745e735e,
	0xc673c45b, 0xc248c03a, 0x72547140, 0x70226efa,
	0xbe32bc2f, 0xba33b83c, 0x6dc96c8e, 0x6b4a69fc,
	0xb64cb462, 0xb27fb0a2, 0x68a56745, 0x65dd646b,
	0xaeccacfe, 0xab36a976, 0x62f1616e, 0x5fe25e4f,
	0xa7bda60c, 0xa463a2c2, 0x5cb35b0f, 0x596357b0,
	0xa1299f98, 0x9e0f9c8f, 0x55f45432, 0x52685097,
	0x9b1899a9, 0x984396e6, 0x4ebf4ce0, 0x4afa490e,
	0x95939448, 0x930791cf, 0x471c4523, 0x43254120,
	0x90a18f7d, 0x8e628d51, 0x3f163d07, 0x3af238d8,
	0x8c4a8b4e, 0x8a5b8972, 0x36b93496, 0x326d3041,
	0x889487c0, 0x86f78638, 0x2e102bdb, 0x29a32767,
	0x858384da, 0x843b83a6, 0x252722e4, 0x209f1e56,
	0x831d829e, 0x822a81c1, 0x1c0b19bd, 0x176d151b,
	0x81638110, 0x80c8808b, 0x12c71072, 0x0e1b0bc3,
	0x80598032, 0x80178006, 0x096a0710, 0x04b6025b,
};

/* *********************************************************** */

const mlib_u32 mlib_v_fftK_WTable_1D_M[] = {
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x3fff2d40, 0x0000d2bf, 0x0000d2bf, 0xc000d2bf,
	0x3fff3b20, 0x2d40187d, 0x0000e782, 0xd2bfc4df,
	0x0000e782, 0xd2bfc4df, 0xc000c4df, 0xd2bfe782,
	0x3fff3ec4, 0x3b203536, 0x0000f383, 0xe782dc71,
	0x2d40238e, 0x187d0c7c, 0xd2bfcac9, 0xc4dfc13b,
	0x0000f383, 0xe782dc71, 0xc000c13b, 0xc4dfcac9,
	0xd2bfcac9, 0xc4dfc13b, 0xd2bfdc71, 0xe782f383,
	0x3fff3fb0, 0x3ec43d3e, 0x0000f9ba, 0xf383ed6c,
	0x3b203870, 0x35363178, 0xe782e1d4, 0xdc71d766,
	0x2d402899, 0x238e1e2b, 0xd2bfce87, 0xcac9c78f,
	0x187d1293, 0x0c7c0645, 0xc4dfc2c1, 0xc13bc04f,
	0x0000f9ba, 0xf383ed6c, 0xc000c04f, 0xc13bc2c1,
	0xe782e1d4, 0xdc71d766, 0xc4dfc78f, 0xcac9ce87,
	0xd2bfce87, 0xcac9c78f, 0xd2bfd766, 0xdc71e1d4,
	0xc4dfc2c1, 0xc13bc04f, 0xe782ed6c, 0xf383f9ba,
	0x3fff3feb, 0x3fb03f4e, 0x0000fcdc, 0xf9baf69c,
	0x3ec43e14, 0x3d3e3c41, 0xf383f073, 0xed6cea70,
	0x3b2039da, 0x387036e4, 0xe782e4a3, 0xe1d4df19,
	0x35363367, 0x31782f6b, 0xdc71d9e0, 0xd766d505,
	0x2d402afa, 0x2899261f, 0xd2bfd094, 0xce87cc98,
	0x238e20e6, 0x1e2b1b5c, 0xcac9c91b, 0xc78fc625,
	0x187d158f, 0x12930f8c, 0xc4dfc3be, 0xc2c1c1eb,
	0x0c7c0963, 0x06450323, 0xc13bc0b1, 0xc04fc014,
	0x0000fcdc, 0xf9baf69c, 0xc000c014, 0xc04fc0b1,
	0xf383f073, 0xed6cea70, 0xc13bc1eb, 0xc2c1c3be,
	0xe782e4a3, 0xe1d4df19, 0xc4dfc625, 0xc78fc91b,
	0xdc71d9e0, 0xd766d505, 0xcac9cc98, 0xce87d094,
	0xd2bfd094, 0xce87cc98, 0xd2bfd505, 0xd766d9e0,
	0xcac9c91b, 0xc78fc625, 0xdc71df19, 0xe1d4e4a3,
	0xc4dfc3be, 0xc2c1c1eb, 0xe782ea70, 0xed6cf073,
	0xc13bc0b1, 0xc04fc014, 0xf383f69c, 0xf9bafcdc,
	0x3fff3ffa, 0x3feb3fd3, 0x0000fe6d, 0xfcdcfb4a,
	0x3fb03f84, 0x3f4e3f0e, 0xf9baf82a, 0xf69cf50f,
	0x3ec43e71, 0x3e143dae, 0xf383f1fa, 0xf073eeee,
	0x3d3e3cc4, 0x3c413bb5, 0xed6cebec, 0xea70e8f7,
	0x3b203a81, 0x39da392a, 0xe782e610, 0xe4a3e339,
	0x387037af, 0x36e43611, 0xe1d4e074, 0xdf19ddc2,
	0x35363452, 0x33673273, 0xdc71db26, 0xd9e0d8a0,
	0x31783075, 0x2f6b2e59, 0xd766d632, 0xd505d3de,
	0x2d402c21, 0x2afa29cd, 0xd2bfd1a6, 0xd094cf8a,
	0x2899275f, 0x261f24d9, 0xce87cd8c, 0xcc98cbad,
	0x238e223d, 0x20e61f8b, 0xcac9c9ee, 0xc91bc850,
	0x1e2b1cc6, 0x1b5c19ef, 0xc78fc6d5, 0xc625c57e,
	0x187d1708, 0x158f1413, 0xc4dfc44a, 0xc3bec33b,
	0x12931111, 0x0f8c0e05, 0xc2c1c251, 0xc1ebc18e,
	0x0c7c0af0, 0x096307d5, 0xc13bc0f1, 0xc0b1c07b,
	0x064504b5, 0x03230192, 0xc04fc02c, 0xc014c005,
	0x0000fe6d, 0xfcdcfb4a, 0xc000c005, 0xc014c02c,
	0xf9baf82a, 0xf69cf50f, 0xc04fc07b, 0xc0b1c0f1,
	0xf383f1fa, 0xf073eeee, 0xc13bc18e, 0xc1ebc251,
	0xed6cebec, 0xea70e8f7, 0xc2c1c33b, 0xc3bec44a,
	0xe782e610, 0xe4a3e339, 0xc4dfc57e, 0xc625c6d5,
	0xe1d4e074, 0xdf19ddc2, 0xc78fc850, 0xc91bc9ee,
	0xdc71db26, 0xd9e0d8a0, 0xcac9cbad, 0xcc98cd8c,
	0xd766d632, 0xd505d3de, 0xce87cf8a, 0xd094d1a6,
	0xd2bfd1a6, 0xd094cf8a, 0xd2bfd3de, 0xd505d632,
	0xce87cd8c, 0xcc98cbad, 0xd766d8a0, 0xd9e0db26,
	0xcac9c9ee, 0xc91bc850, 0xdc71ddc2, 0xdf19e074,
	0xc78fc6d5, 0xc625c57e, 0xe1d4e339, 0xe4a3e610,
	0xc4dfc44a, 0xc3bec33b, 0xe782e8f7, 0xea70ebec,
	0xc2c1c251, 0xc1ebc18e, 0xed6ceeee, 0xf073f1fa,
	0xc13bc0f1, 0xc0b1c07b, 0xf383f50f, 0xf69cf82a,
	0xc04fc02c, 0xc014c005, 0xf9bafb4a, 0xfcdcfe6d,
	0x3fff3ffe, 0x3ffa3ff4, 0x0000ff36, 0xfe6dfda4,
	0x3feb3fe0, 0x3fd33fc3, 0xfcdcfc13, 0xfb4afa82,
	0x3fb03f9b, 0x3f843f6a, 0xf9baf8f2, 0xf82af763,
	0x3f4e3f2f, 0x3f0e3eea, 0xf69cf5d5, 0xf50ff449,
	0x3ec43e9c, 0x3e713e44, 0xf383f2be, 0xf1faf136,
	0x3e143de2, 0x3dae3d77, 0xf073efb0, 0xeeeeee2c,
	0x3d3e3d02, 0x3cc43c84, 0xed6cecac, 0xebeceb2e,
	0x3c413bfc, 0x3bb53b6c, 0xea70e9b3, 0xe8f7e83c,
	0x3b203ad2, 0x3a813a2f, 0xe782e6c9, 0xe610e559,
	0x39da3983, 0x392a38ce, 0xe4a3e3ed, 0xe339e286,
	0x38703811, 0x37af374a, 0xe1d4e124, 0xe074dfc6,
	0x36e4367c, 0x361135a5, 0xdf19de6d, 0xddc2dd19,
	0x353634c5, 0x345233de, 0xdc71dbcb, 0xdb26da82,
	0x336732ee, 0x327331f7, 0xd9e0d93f, 0xd8a0d802,
	0x317830f8, 0x30752ff1, 0xd766d6cb, 0xd632d59b,
	0x2f6b2ee3, 0x2e592dce, 0xd505d471, 0xd3ded34e,
	0x2d402cb1, 0x2c212b8e, 0xd2bfd231, 0xd1a6d11c,
	0x2afa2a64, 0x29cd2934, 0xd094d00e, 0xcf8acf07,
	0x289927fd, 0x275f26c0, 0xce87ce08, 0xcd8ccd11,
	0x261f257d, 0x24d92434, 0xcc98cc21, 0xcbadcb3a,
	0x238e22e6, 0x223d2192, 0xcac9ca5a, 0xc9eec983,
	0x20e62039, 0x1f8b1edb, 0xc91bc8b5, 0xc850c7ee,
	0x1e2b1d79, 0x1cc61c12, 0xc78fc731, 0xc6d5c67c,
	0x1b5c1aa6, 0x19ef1936, 0xc625c5d0, 0xc57ec52d,
	0x187d17c3, 0x1708164c, 0xc4dfc493, 0xc44ac403,
	0x158f14d1, 0x14131353, 0xc3bec37b, 0xc33bc2fd,
	0x129311d3, 0x1111104f, 0xc2c1c288, 0xc251c21d,
	0x0f8c0ec9, 0x0e050d41, 0xc1ebc1bb, 0xc18ec163,
	0x0c7c0bb6, 0x0af00a2a, 0xc13bc115, 0xc0f1c0d0,
	0x0963089c, 0x07d5070d, 0xc0b1c095, 0xc07bc064,
	0x0645057d, 0x04b503ec, 0xc04fc03c, 0xc02cc01f,
	0x0323025b, 0x019200c9, 0xc014c00b, 0xc005c001,
	0x0000ff36, 0xfe6dfda4, 0xc000c001, 0xc005c00b,
	0xfcdcfc13, 0xfb4afa82, 0xc014c01f, 0xc02cc03c,
	0xf9baf8f2, 0xf82af763, 0xc04fc064, 0xc07bc095,
	0xf69cf5d5, 0xf50ff449, 0xc0b1c0d0, 0xc0f1c115,
	0xf383f2be, 0xf1faf136, 0xc13bc163, 0xc18ec1bb,
	0xf073efb0, 0xeeeeee2c, 0xc1ebc21d, 0xc251c288,
	0xed6cecac, 0xebeceb2e, 0xc2c1c2fd, 0xc33bc37b,
	0xea70e9b3, 0xe8f7e83c, 0xc3bec403, 0xc44ac493,
	0xe782e6c9, 0xe610e559, 0xc4dfc52d, 0xc57ec5d0,
	0xe4a3e3ed, 0xe339e286, 0xc625c67c, 0xc6d5c731,
	0xe1d4e124, 0xe074dfc6, 0xc78fc7ee, 0xc850c8b5,
	0xdf19de6d, 0xddc2dd19, 0xc91bc983, 0xc9eeca5a,
	0xdc71dbcb, 0xdb26da82, 0xcac9cb3a, 0xcbadcc21,
	0xd9e0d93f, 0xd8a0d802, 0xcc98cd11, 0xcd8cce08,
	0xd766d6cb, 0xd632d59b, 0xce87cf07, 0xcf8ad00e,
	0xd505d471, 0xd3ded34e, 0xd094d11c, 0xd1a6d231,
	0xd2bfd231, 0xd1a6d11c, 0xd2bfd34e, 0xd3ded471,
	0xd094d00e, 0xcf8acf07, 0xd505d59b, 0xd632d6cb,
	0xce87ce08, 0xcd8ccd11, 0xd766d802, 0xd8a0d93f,
	0xcc98cc21, 0xcbadcb3a, 0xd9e0da82, 0xdb26dbcb,
	0xcac9ca5a, 0xc9eec983, 0xdc71dd19, 0xddc2de6d,
	0xc91bc8b5, 0xc850c7ee, 0xdf19dfc6, 0xe074e124,
	0xc78fc731, 0xc6d5c67c, 0xe1d4e286, 0xe339e3ed,
	0xc625c5d0, 0xc57ec52d, 0xe4a3e559, 0xe610e6c9,
	0xc4dfc493, 0xc44ac403, 0xe782e83c, 0xe8f7e9b3,
	0xc3bec37b, 0xc33bc2fd, 0xea70eb2e, 0xebececac,
	0xc2c1c288, 0xc251c21d, 0xed6cee2c, 0xeeeeefb0,
	0xc1ebc1bb, 0xc18ec163, 0xf073f136, 0xf1faf2be,
	0xc13bc115, 0xc0f1c0d0, 0xf383f449, 0xf50ff5d5,
	0xc0b1c095, 0xc07bc064, 0xf69cf763, 0xf82af8f2,
	0xc04fc03c, 0xc02cc01f, 0xf9bafa82, 0xfb4afc13,
	0xc014c00b, 0xc005c001, 0xfcdcfda4, 0xfe6dff36,
	0x3fff3fff, 0x3ffe3ffc, 0x0000ff9b, 0xff36fed2,
	0x3ffa3ff7, 0x3ff43ff0, 0xfe6dfe09, 0xfda4fd40,
	0x3feb3fe6, 0x3fe03fda, 0xfcdcfc77, 0xfc13fbaf,
	0x3fd33fcb, 0x3fc33fba, 0xfb4afae6, 0xfa82fa1e,
	0x3fb03fa6, 0x3f9b3f90, 0xf9baf956, 0xf8f2f88e,
	0x3f843f77, 0x3f6a3f5c, 0xf82af7c6, 0xf763f6ff,
	0x3f4e3f3f, 0x3f2f3f1f, 0xf69cf638, 0xf5d5f572,
	0x3f0e3efc, 0x3eea3ed8, 0xf50ff4ac, 0xf449f3e6,
	0x3ec43eb0, 0x3e9c3e87, 0xf383f321, 0xf2bef25c,
	0x3e713e5b, 0x3e443e2c, 0xf1faf198, 0xf136f0d4,
	0x3e143dfb, 0x3de23dc8, 0xf073f011, 0xefb0ef4f,
	0x3dae3d92, 0x3d773d5a, 0xeeeeee8d, 0xee2cedcc,
	0x3d3e3d20, 0x3d023ce3, 0xed6ced0c, 0xecacec4c,
	0x3cc43ca4, 0x3c843c63, 0xebeceb8d, 0xeb2eeacf,
	0x3c413c1f, 0x3bfc3bd9, 0xea70ea12, 0xe9b3e955,
	0x3bb53b91, 0x3b6c3b46, 0xe8f7e899, 0xe83ce7df,
	0x3b203af9, 0x3ad23aaa, 0xe782e725, 0xe6c9e66c,
	0x3a813a58, 0x3a2f3a05, 0xe610e5b4, 0xe559e4fe,
	0x39da39af, 0x39833957, 0xe4a3e448, 0xe3ede393,
	0x392a38fc, 0x38ce38a0, 0xe339e2e0, 0xe286e22d,
	0x38703841, 0x381137e0, 0xe1d4e17c, 0xe124e0cc,
	0x37af377d, 0x374a3718, 0xe074e01d, 0xdfc6df6f,
	0x36e436b0, 0x367c3647, 0xdf19dec3, 0xde6dde17,
	0x361135db, 0x35a5356d, 0xddc2dd6e, 0xdd19dcc5,
	0x353634fe, 0x34c5348c, 0xdc71dc1e, 0xdbcbdb78,
	0x34523418, 0x33de33a2, 0xdb26dad4, 0xda82da31,
	0x3367332b, 0x32ee32b1, 0xd9e0d98f, 0xd93fd8ef,
	0x32733235, 0x31f731b8, 0xd8a0d851, 0xd802d7b4,
	0x31783138, 0x30f830b7, 0xd766d718, 0xd6cbd67f,
	0x30753033, 0x2ff12fae, 0xd632d5e6, 0xd59bd550,
	0x2f6b2f27, 0x2ee32e9e, 0xd505d4bb, 0xd471d427,
	0x2e592e14, 0x2dce2d87, 0xd3ded396, 0xd34ed306,
	0x2d402cf9, 0x2cb12c69, 0xd2bfd278, 0xd231d1eb,
	0x2c212bd8, 0x2b8e2b44, 0xd1a6d161, 0xd11cd0d8,
	0x2afa2aaf, 0x2a642a19, 0xd094d051, 0xd00ecfcc,
	0x29cd2980, 0x293428e7, 0xcf8acf48, 0xcf07cec7,
	0x2899284b, 0x27fd27ae, 0xce87ce47, 0xce08cdca,
	0x275f2710, 0x26c02670, 0xcd8ccd4e, 0xcd11ccd4,
	0x261f25ce, 0x257d252b, 0xcc98cc5d, 0xcc21cbe7,
	0x24d92487, 0x243423e1, 0xcbadcb73, 0xcb3acb01,
	0x238e233a, 0x22e62291, 0xcac9ca92, 0xca5aca24,
	0x223d21e8, 0x2192213c, 0xc9eec9b8, 0xc983c94f,
	0x20e62090, 0x20391fe2, 0xc91bc8e7, 0xc8b5c882,
	0x1f8b1f33, 0x1edb1e83, 0xc850c81f, 0xc7eec7be,
	0x1e2b1dd2, 0x1d791d1f, 0xc78fc75f, 0xc731c703,
	0x1cc61c6c, 0x1c121bb7, 0xc6d5c6a8, 0xc67cc650,
	0x1b5c1b01, 0x1aa61a4b, 0xc625c5fa, 0xc5d0c5a7,
	0x19ef1993, 0x193618da, 0xc57ec555, 0xc52dc506,
	0x187d1820, 0x17c31766, 0xc4dfc4b9, 0xc493c46e,
	0x170816aa, 0x164c15ed, 0xc44ac426, 0xc403c3e0,
	0x158f1530, 0x14d11472, 0xc3bec39c, 0xc37bc35b,
	0x141313b3, 0x135312f3, 0xc33bc31c, 0xc2fdc2df,
	0x12931233, 0x11d31172, 0xc2c1c2a5, 0xc288c26d,
	0x111110b0, 0x104f0fee, 0xc251c237, 0xc21dc204,
	0x0f8c0f2b, 0x0ec90e67, 0xc1ebc1d3, 0xc1bbc1a4,
	0x0e050da3, 0x0d410cde, 0xc18ec178, 0xc163c14f,
	0x0c7c0c19, 0x0bb60b53, 0xc13bc127, 0xc115c103,
	0x0af00a8d, 0x0a2a09c7, 0xc0f1c0e0, 0xc0d0c0c0,
	0x09630900, 0x089c0839, 0xc0b1c0a3, 0xc095c088,
	0x07d50771, 0x070d06a9, 0xc07bc06f, 0xc064c059,
	0x064505e1, 0x057d0519, 0xc04fc045, 0xc03cc034,
	0x04b50450, 0x03ec0388, 0xc02cc025, 0xc01fc019,
	0x032302bf, 0x025b01f6, 0xc014c00f, 0xc00bc008,
	0x0192012d, 0x00c90064, 0xc005c003, 0xc001c000,
	0x0000ff9b, 0xff36fed2, 0xc000c000, 0xc001c003,
	0xfe6dfe09, 0xfda4fd40, 0xc005c008, 0xc00bc00f,
	0xfcdcfc77, 0xfc13fbaf, 0xc014c019, 0xc01fc025,
	0xfb4afae6, 0xfa82fa1e, 0xc02cc034, 0xc03cc045,
	0xf9baf956, 0xf8f2f88e, 0xc04fc059, 0xc064c06f,
	0xf82af7c6, 0xf763f6ff, 0xc07bc088, 0xc095c0a3,
	0xf69cf638, 0xf5d5f572, 0xc0b1c0c0, 0xc0d0c0e0,
	0xf50ff4ac, 0xf449f3e6, 0xc0f1c103, 0xc115c127,
	0xf383f321, 0xf2bef25c, 0xc13bc14f, 0xc163c178,
	0xf1faf198, 0xf136f0d4, 0xc18ec1a4, 0xc1bbc1d3,
	0xf073f011, 0xefb0ef4f, 0xc1ebc204, 0xc21dc237,
	0xeeeeee8d, 0xee2cedcc, 0xc251c26d, 0xc288c2a5,
	0xed6ced0c, 0xecacec4c, 0xc2c1c2df, 0xc2fdc31c,
	0xebeceb8d, 0xeb2eeacf, 0xc33bc35b, 0xc37bc39c,
	0xea70ea12, 0xe9b3e955, 0xc3bec3e0, 0xc403c426,
	0xe8f7e899, 0xe83ce7df, 0xc44ac46e, 0xc493c4b9,
	0xe782e725, 0xe6c9e66c, 0xc4dfc506, 0xc52dc555,
	0xe610e5b4, 0xe559e4fe, 0xc57ec5a7, 0xc5d0c5fa,
	0xe4a3e448, 0xe3ede393, 0xc625c650, 0xc67cc6a8,
	0xe339e2e0, 0xe286e22d, 0xc6d5c703, 0xc731c75f,
	0xe1d4e17c, 0xe124e0cc, 0xc78fc7be, 0xc7eec81f,
	0xe074e01d, 0xdfc6df6f, 0xc850c882, 0xc8b5c8e7,
	0xdf19dec3, 0xde6dde17, 0xc91bc94f, 0xc983c9b8,
	0xddc2dd6e, 0xdd19dcc5, 0xc9eeca24, 0xca5aca92,
	0xdc71dc1e, 0xdbcbdb78, 0xcac9cb01, 0xcb3acb73,
	0xdb26dad4, 0xda82da31, 0xcbadcbe7, 0xcc21cc5d,
	0xd9e0d98f, 0xd93fd8ef, 0xcc98ccd4, 0xcd11cd4e,
	0xd8a0d851, 0xd802d7b4, 0xcd8ccdca, 0xce08ce47,
	0xd766d718, 0xd6cbd67f, 0xce87cec7, 0xcf07cf48,
	0xd632d5e6, 0xd59bd550, 0xcf8acfcc, 0xd00ed051,
	0xd505d4bb, 0xd471d427, 0xd094d0d8, 0xd11cd161,
	0xd3ded396, 0xd34ed306, 0xd1a6d1eb, 0xd231d278,
	0xd2bfd278, 0xd231d1eb, 0xd2bfd306, 0xd34ed396,
	0xd1a6d161, 0xd11cd0d8, 0xd3ded427, 0xd471d4bb,
	0xd094d051, 0xd00ecfcc, 0xd505d550, 0xd59bd5e6,
	0xcf8acf48, 0xcf07cec7, 0xd632d67f, 0xd6cbd718,
	0xce87ce47, 0xce08cdca, 0xd766d7b4, 0xd802d851,
	0xcd8ccd4e, 0xcd11ccd4, 0xd8a0d8ef, 0xd93fd98f,
	0xcc98cc5d, 0xcc21cbe7, 0xd9e0da31, 0xda82dad4,
	0xcbadcb73, 0xcb3acb01, 0xdb26db78, 0xdbcbdc1e,
	0xcac9ca92, 0xca5aca24, 0xdc71dcc5, 0xdd19dd6e,
	0xc9eec9b8, 0xc983c94f, 0xddc2de17, 0xde6ddec3,
	0xc91bc8e7, 0xc8b5c882, 0xdf19df6f, 0xdfc6e01d,
	0xc850c81f, 0xc7eec7be, 0xe074e0cc, 0xe124e17c,
	0xc78fc75f, 0xc731c703, 0xe1d4e22d, 0xe286e2e0,
	0xc6d5c6a8, 0xc67cc650, 0xe339e393, 0xe3ede448,
	0xc625c5fa, 0xc5d0c5a7, 0xe4a3e4fe, 0xe559e5b4,
	0xc57ec555, 0xc52dc506, 0xe610e66c, 0xe6c9e725,
	0xc4dfc4b9, 0xc493c46e, 0xe782e7df, 0xe83ce899,
	0xc44ac426, 0xc403c3e0, 0xe8f7e955, 0xe9b3ea12,
	0xc3bec39c, 0xc37bc35b, 0xea70eacf, 0xeb2eeb8d,
	0xc33bc31c, 0xc2fdc2df, 0xebecec4c, 0xecaced0c,
	0xc2c1c2a5, 0xc288c26d, 0xed6cedcc, 0xee2cee8d,
	0xc251c237, 0xc21dc204, 0xeeeeef4f, 0xefb0f011,
	0xc1ebc1d3, 0xc1bbc1a4, 0xf073f0d4, 0xf136f198,
	0xc18ec178, 0xc163c14f, 0xf1faf25c, 0xf2bef321,
	0xc13bc127, 0xc115c103, 0xf383f3e6, 0xf449f4ac,
	0xc0f1c0e0, 0xc0d0c0c0, 0xf50ff572, 0xf5d5f638,
	0xc0b1c0a3, 0xc095c088, 0xf69cf6ff, 0xf763f7c6,
	0xc07bc06f, 0xc064c059, 0xf82af88e, 0xf8f2f956,
	0xc04fc045, 0xc03cc034, 0xf9bafa1e, 0xfa82fae6,
	0xc02cc025, 0xc01fc019, 0xfb4afbaf, 0xfc13fc77,
	0xc014c00f, 0xc00bc008, 0xfcdcfd40, 0xfda4fe09,
	0xc005c003, 0xc001c000, 0xfe6dfed2, 0xff36ff9b,
};

/* *********************************************************** */

const mlib_u32 mlib_v_fftK_WTable_1D_P[] = {
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x3fff2d40, 0x0000d2bf, 0x00002d40, 0x3fff2d40,
	0x3fff3b20, 0x2d40187d, 0x0000187d, 0x2d403b20,
	0x0000e782, 0xd2bfc4df, 0x3fff3b20, 0x2d40187d,
	0x3fff3ec4, 0x3b203536, 0x00000c7c, 0x187d238e,
	0x2d40238e, 0x187d0c7c, 0x2d403536, 0x3b203ec4,
	0x0000f383, 0xe782dc71, 0x3fff3ec4, 0x3b203536,
	0xd2bfcac9, 0xc4dfc13b, 0x2d40238e, 0x187d0c7c,
	0x3fff3fb0, 0x3ec43d3e, 0x00000645, 0x0c7c1293,
	0x3b203870, 0x35363178, 0x187d1e2b, 0x238e2899,
	0x2d402899, 0x238e1e2b, 0x2d403178, 0x35363870,
	0x187d1293, 0x0c7c0645, 0x3b203d3e, 0x3ec43fb0,
	0x0000f9ba, 0xf383ed6c, 0x3fff3fb0, 0x3ec43d3e,
	0xe782e1d4, 0xdc71d766, 0x3b203870, 0x35363178,
	0xd2bfce87, 0xcac9c78f, 0x2d402899, 0x238e1e2b,
	0xc4dfc2c1, 0xc13bc04f, 0x187d1293, 0x0c7c0645,
	0x3fff3feb, 0x3fb03f4e, 0x00000323, 0x06450963,
	0x3ec43e14, 0x3d3e3c41, 0x0c7c0f8c, 0x1293158f,
	0x3b2039da, 0x387036e4, 0x187d1b5c, 0x1e2b20e6,
	0x35363367, 0x31782f6b, 0x238e261f, 0x28992afa,
	0x2d402afa, 0x2899261f, 0x2d402f6b, 0x31783367,
	0x238e20e6, 0x1e2b1b5c, 0x353636e4, 0x387039da,
	0x187d158f, 0x12930f8c, 0x3b203c41, 0x3d3e3e14,
	0x0c7c0963, 0x06450323, 0x3ec43f4e, 0x3fb03feb,
	0x0000fcdc, 0xf9baf69c, 0x3fff3feb, 0x3fb03f4e,
	0xf383f073, 0xed6cea70, 0x3ec43e14, 0x3d3e3c41,
	0xe782e4a3, 0xe1d4df19, 0x3b2039da, 0x387036e4,
	0xdc71d9e0, 0xd766d505, 0x35363367, 0x31782f6b,
	0xd2bfd094, 0xce87cc98, 0x2d402afa, 0x2899261f,
	0xcac9c91b, 0xc78fc625, 0x238e20e6, 0x1e2b1b5c,
	0xc4dfc3be, 0xc2c1c1eb, 0x187d158f, 0x12930f8c,
	0xc13bc0b1, 0xc04fc014, 0x0c7c0963, 0x06450323,
	0x3fff3ffa, 0x3feb3fd3, 0x00000192, 0x032304b5,
	0x3fb03f84, 0x3f4e3f0e, 0x064507d5, 0x09630af0,
	0x3ec43e71, 0x3e143dae, 0x0c7c0e05, 0x0f8c1111,
	0x3d3e3cc4, 0x3c413bb5, 0x12931413, 0x158f1708,
	0x3b203a81, 0x39da392a, 0x187d19ef, 0x1b5c1cc6,
	0x387037af, 0x36e43611, 0x1e2b1f8b, 0x20e6223d,
	0x35363452, 0x33673273, 0x238e24d9, 0x261f275f,
	0x31783075, 0x2f6b2e59, 0x289929cd, 0x2afa2c21,
	0x2d402c21, 0x2afa29cd, 0x2d402e59, 0x2f6b3075,
	0x2899275f, 0x261f24d9, 0x31783273, 0x33673452,
	0x238e223d, 0x20e61f8b, 0x35363611, 0x36e437af,
	0x1e2b1cc6, 0x1b5c19ef, 0x3870392a, 0x39da3a81,
	0x187d1708, 0x158f1413, 0x3b203bb5, 0x3c413cc4,
	0x12931111, 0x0f8c0e05, 0x3d3e3dae, 0x3e143e71,
	0x0c7c0af0, 0x096307d5, 0x3ec43f0e, 0x3f4e3f84,
	0x064504b5, 0x03230192, 0x3fb03fd3, 0x3feb3ffa,
	0x0000fe6d, 0xfcdcfb4a, 0x3fff3ffa, 0x3feb3fd3,
	0xf9baf82a, 0xf69cf50f, 0x3fb03f84, 0x3f4e3f0e,
	0xf383f1fa, 0xf073eeee, 0x3ec43e71, 0x3e143dae,
	0xed6cebec, 0xea70e8f7, 0x3d3e3cc4, 0x3c413bb5,
	0xe782e610, 0xe4a3e339, 0x3b203a81, 0x39da392a,
	0xe1d4e074, 0xdf19ddc2, 0x387037af, 0x36e43611,
	0xdc71db26, 0xd9e0d8a0, 0x35363452, 0x33673273,
	0xd766d632, 0xd505d3de, 0x31783075, 0x2f6b2e59,
	0xd2bfd1a6, 0xd094cf8a, 0x2d402c21, 0x2afa29cd,
	0xce87cd8c, 0xcc98cbad, 0x2899275f, 0x261f24d9,
	0xcac9c9ee, 0xc91bc850, 0x238e223d, 0x20e61f8b,
	0xc78fc6d5, 0xc625c57e, 0x1e2b1cc6, 0x1b5c19ef,
	0xc4dfc44a, 0xc3bec33b, 0x187d1708, 0x158f1413,
	0xc2c1c251, 0xc1ebc18e, 0x12931111, 0x0f8c0e05,
	0xc13bc0f1, 0xc0b1c07b, 0x0c7c0af0, 0x096307d5,
	0xc04fc02c, 0xc014c005, 0x064504b5, 0x03230192,
	0x3fff3ffe, 0x3ffa3ff4, 0x000000c9, 0x0192025b,
	0x3feb3fe0, 0x3fd33fc3, 0x032303ec, 0x04b5057d,
	0x3fb03f9b, 0x3f843f6a, 0x0645070d, 0x07d5089c,
	0x3f4e3f2f, 0x3f0e3eea, 0x09630a2a, 0x0af00bb6,
	0x3ec43e9c, 0x3e713e44, 0x0c7c0d41, 0x0e050ec9,
	0x3e143de2, 0x3dae3d77, 0x0f8c104f, 0x111111d3,
	0x3d3e3d02, 0x3cc43c84, 0x12931353, 0x141314d1,
	0x3c413bfc, 0x3bb53b6c, 0x158f164c, 0x170817c3,
	0x3b203ad2, 0x3a813a2f, 0x187d1936, 0x19ef1aa6,
	0x39da3983, 0x392a38ce, 0x1b5c1c12, 0x1cc61d79,
	0x38703811, 0x37af374a, 0x1e2b1edb, 0x1f8b2039,
	0x36e4367c, 0x361135a5, 0x20e62192, 0x223d22e6,
	0x353634c5, 0x345233de, 0x238e2434, 0x24d9257d,
	0x336732ee, 0x327331f7, 0x261f26c0, 0x275f27fd,
	0x317830f8, 0x30752ff1, 0x28992934, 0x29cd2a64,
	0x2f6b2ee3, 0x2e592dce, 0x2afa2b8e, 0x2c212cb1,
	0x2d402cb1, 0x2c212b8e, 0x2d402dce, 0x2e592ee3,
	0x2afa2a64, 0x29cd2934, 0x2f6b2ff1, 0x307530f8,
	0x289927fd, 0x275f26c0, 0x317831f7, 0x327332ee,
	0x261f257d, 0x24d92434, 0x336733de, 0x345234c5,
	0x238e22e6, 0x223d2192, 0x353635a5, 0x3611367c,
	0x20e62039, 0x1f8b1edb, 0x36e4374a, 0x37af3811,
	0x1e2b1d79, 0x1cc61c12, 0x387038ce, 0x392a3983,
	0x1b5c1aa6, 0x19ef1936, 0x39da3a2f, 0x3a813ad2,
	0x187d17c3, 0x1708164c, 0x3b203b6c, 0x3bb53bfc,
	0x158f14d1, 0x14131353, 0x3c413c84, 0x3cc43d02,
	0x129311d3, 0x1111104f, 0x3d3e3d77, 0x3dae3de2,
	0x0f8c0ec9, 0x0e050d41, 0x3e143e44, 0x3e713e9c,
	0x0c7c0bb6, 0x0af00a2a, 0x3ec43eea, 0x3f0e3f2f,
	0x0963089c, 0x07d5070d, 0x3f4e3f6a, 0x3f843f9b,
	0x0645057d, 0x04b503ec, 0x3fb03fc3, 0x3fd33fe0,
	0x0323025b, 0x019200c9, 0x3feb3ff4, 0x3ffa3ffe,
	0x0000ff36, 0xfe6dfda4, 0x3fff3ffe, 0x3ffa3ff4,
	0xfcdcfc13, 0xfb4afa82, 0x3feb3fe0, 0x3fd33fc3,
	0xf9baf8f2, 0xf82af763, 0x3fb03f9b, 0x3f843f6a,
	0xf69cf5d5, 0xf50ff449, 0x3f4e3f2f, 0x3f0e3eea,
	0xf383f2be, 0xf1faf136, 0x3ec43e9c, 0x3e713e44,
	0xf073efb0, 0xeeeeee2c, 0x3e143de2, 0x3dae3d77,
	0xed6cecac, 0xebeceb2e, 0x3d3e3d02, 0x3cc43c84,
	0xea70e9b3, 0xe8f7e83c, 0x3c413bfc, 0x3bb53b6c,
	0xe782e6c9, 0xe610e559, 0x3b203ad2, 0x3a813a2f,
	0xe4a3e3ed, 0xe339e286, 0x39da3983, 0x392a38ce,
	0xe1d4e124, 0xe074dfc6, 0x38703811, 0x37af374a,
	0xdf19de6d, 0xddc2dd19, 0x36e4367c, 0x361135a5,
	0xdc71dbcb, 0xdb26da82, 0x353634c5, 0x345233de,
	0xd9e0d93f, 0xd8a0d802, 0x336732ee, 0x327331f7,
	0xd766d6cb, 0xd632d59b, 0x317830f8, 0x30752ff1,
	0xd505d471, 0xd3ded34e, 0x2f6b2ee3, 0x2e592dce,
	0xd2bfd231, 0xd1a6d11c, 0x2d402cb1, 0x2c212b8e,
	0xd094d00e, 0xcf8acf07, 0x2afa2a64, 0x29cd2934,
	0xce87ce08, 0xcd8ccd11, 0x289927fd, 0x275f26c0,
	0xcc98cc21, 0xcbadcb3a, 0x261f257d, 0x24d92434,
	0xcac9ca5a, 0xc9eec983, 0x238e22e6, 0x223d2192,
	0xc91bc8b5, 0xc850c7ee, 0x20e62039, 0x1f8b1edb,
	0xc78fc731, 0xc6d5c67c, 0x1e2b1d79, 0x1cc61c12,
	0xc625c5d0, 0xc57ec52d, 0x1b5c1aa6, 0x19ef1936,
	0xc4dfc493, 0xc44ac403, 0x187d17c3, 0x1708164c,
	0xc3bec37b, 0xc33bc2fd, 0x158f14d1, 0x14131353,
	0xc2c1c288, 0xc251c21d, 0x129311d3, 0x1111104f,
	0xc1ebc1bb, 0xc18ec163, 0x0f8c0ec9, 0x0e050d41,
	0xc13bc115, 0xc0f1c0d0, 0x0c7c0bb6, 0x0af00a2a,
	0xc0b1c095, 0xc07bc064, 0x0963089c, 0x07d5070d,
	0xc04fc03c, 0xc02cc01f, 0x0645057d, 0x04b503ec,
	0xc014c00b, 0xc005c001, 0x0323025b, 0x019200c9,
	0x3fff3fff, 0x3ffe3ffc, 0x00000064, 0x00c9012d,
	0x3ffa3ff7, 0x3ff43ff0, 0x019201f6, 0x025b02bf,
	0x3feb3fe6, 0x3fe03fda, 0x03230388, 0x03ec0450,
	0x3fd33fcb, 0x3fc33fba, 0x04b50519, 0x057d05e1,
	0x3fb03fa6, 0x3f9b3f90, 0x064506a9, 0x070d0771,
	0x3f843f77, 0x3f6a3f5c, 0x07d50839, 0x089c0900,
	0x3f4e3f3f, 0x3f2f3f1f, 0x096309c7, 0x0a2a0a8d,
	0x3f0e3efc, 0x3eea3ed8, 0x0af00b53, 0x0bb60c19,
	0x3ec43eb0, 0x3e9c3e87, 0x0c7c0cde, 0x0d410da3,
	0x3e713e5b, 0x3e443e2c, 0x0e050e67, 0x0ec90f2b,
	0x3e143dfb, 0x3de23dc8, 0x0f8c0fee, 0x104f10b0,
	0x3dae3d92, 0x3d773d5a, 0x11111172, 0x11d31233,
	0x3d3e3d20, 0x3d023ce3, 0x129312f3, 0x135313b3,
	0x3cc43ca4, 0x3c843c63, 0x14131472, 0x14d11530,
	0x3c413c1f, 0x3bfc3bd9, 0x158f15ed, 0x164c16aa,
	0x3bb53b91, 0x3b6c3b46, 0x17081766, 0x17c31820,
	0x3b203af9, 0x3ad23aaa, 0x187d18da, 0x19361993,
	0x3a813a58, 0x3a2f3a05, 0x19ef1a4b, 0x1aa61b01,
	0x39da39af, 0x39833957, 0x1b5c1bb7, 0x1c121c6c,
	0x392a38fc, 0x38ce38a0, 0x1cc61d1f, 0x1d791dd2,
	0x38703841, 0x381137e0, 0x1e2b1e83, 0x1edb1f33,
	0x37af377d, 0x374a3718, 0x1f8b1fe2, 0x20392090,
	0x36e436b0, 0x367c3647, 0x20e6213c, 0x219221e8,
	0x361135db, 0x35a5356d, 0x223d2291, 0x22e6233a,
	0x353634fe, 0x34c5348c, 0x238e23e1, 0x24342487,
	0x34523418, 0x33de33a2, 0x24d9252b, 0x257d25ce,
	0x3367332b, 0x32ee32b1, 0x261f2670, 0x26c02710,
	0x32733235, 0x31f731b8, 0x275f27ae, 0x27fd284b,
	0x31783138, 0x30f830b7, 0x289928e7, 0x29342980,
	0x30753033, 0x2ff12fae, 0x29cd2a19, 0x2a642aaf,
	0x2f6b2f27, 0x2ee32e9e, 0x2afa2b44, 0x2b8e2bd8,
	0x2e592e14, 0x2dce2d87, 0x2c212c69, 0x2cb12cf9,
	0x2d402cf9, 0x2cb12c69, 0x2d402d87, 0x2dce2e14,
	0x2c212bd8, 0x2b8e2b44, 0x2e592e9e, 0x2ee32f27,
	0x2afa2aaf, 0x2a642a19, 0x2f6b2fae, 0x2ff13033,
	0x29cd2980, 0x293428e7, 0x307530b7, 0x30f83138,
	0x2899284b, 0x27fd27ae, 0x317831b8, 0x31f73235,
	0x275f2710, 0x26c02670, 0x327332b1, 0x32ee332b,
	0x261f25ce, 0x257d252b, 0x336733a2, 0x33de3418,
	0x24d92487, 0x243423e1, 0x3452348c, 0x34c534fe,
	0x238e233a, 0x22e62291, 0x3536356d, 0x35a535db,
	0x223d21e8, 0x2192213c, 0x36113647, 0x367c36b0,
	0x20e62090, 0x20391fe2, 0x36e43718, 0x374a377d,
	0x1f8b1f33, 0x1edb1e83, 0x37af37e0, 0x38113841,
	0x1e2b1dd2, 0x1d791d1f, 0x387038a0, 0x38ce38fc,
	0x1cc61c6c, 0x1c121bb7, 0x392a3957, 0x398339af,
	0x1b5c1b01, 0x1aa61a4b, 0x39da3a05, 0x3a2f3a58,
	0x19ef1993, 0x193618da, 0x3a813aaa, 0x3ad23af9,
	0x187d1820, 0x17c31766, 0x3b203b46, 0x3b6c3b91,
	0x170816aa, 0x164c15ed, 0x3bb53bd9, 0x3bfc3c1f,
	0x158f1530, 0x14d11472, 0x3c413c63, 0x3c843ca4,
	0x141313b3, 0x135312f3, 0x3cc43ce3, 0x3d023d20,
	0x12931233, 0x11d31172, 0x3d3e3d5a, 0x3d773d92,
	0x111110b0, 0x104f0fee, 0x3dae3dc8, 0x3de23dfb,
	0x0f8c0f2b, 0x0ec90e67, 0x3e143e2c, 0x3e443e5b,
	0x0e050da3, 0x0d410cde, 0x3e713e87, 0x3e9c3eb0,
	0x0c7c0c19, 0x0bb60b53, 0x3ec43ed8, 0x3eea3efc,
	0x0af00a8d, 0x0a2a09c7, 0x3f0e3f1f, 0x3f2f3f3f,
	0x09630900, 0x089c0839, 0x3f4e3f5c, 0x3f6a3f77,
	0x07d50771, 0x070d06a9, 0x3f843f90, 0x3f9b3fa6,
	0x064505e1, 0x057d0519, 0x3fb03fba, 0x3fc33fcb,
	0x04b50450, 0x03ec0388, 0x3fd33fda, 0x3fe03fe6,
	0x032302bf, 0x025b01f6, 0x3feb3ff0, 0x3ff43ff7,
	0x0192012d, 0x00c90064, 0x3ffa3ffc, 0x3ffe3fff,
	0x0000ff9b, 0xff36fed2, 0x3fff3fff, 0x3ffe3ffc,
	0xfe6dfe09, 0xfda4fd40, 0x3ffa3ff7, 0x3ff43ff0,
	0xfcdcfc77, 0xfc13fbaf, 0x3feb3fe6, 0x3fe03fda,
	0xfb4afae6, 0xfa82fa1e, 0x3fd33fcb, 0x3fc33fba,
	0xf9baf956, 0xf8f2f88e, 0x3fb03fa6, 0x3f9b3f90,
	0xf82af7c6, 0xf763f6ff, 0x3f843f77, 0x3f6a3f5c,
	0xf69cf638, 0xf5d5f572, 0x3f4e3f3f, 0x3f2f3f1f,
	0xf50ff4ac, 0xf449f3e6, 0x3f0e3efc, 0x3eea3ed8,
	0xf383f321, 0xf2bef25c, 0x3ec43eb0, 0x3e9c3e87,
	0xf1faf198, 0xf136f0d4, 0x3e713e5b, 0x3e443e2c,
	0xf073f011, 0xefb0ef4f, 0x3e143dfb, 0x3de23dc8,
	0xeeeeee8d, 0xee2cedcc, 0x3dae3d92, 0x3d773d5a,
	0xed6ced0c, 0xecacec4c, 0x3d3e3d20, 0x3d023ce3,
	0xebeceb8d, 0xeb2eeacf, 0x3cc43ca4, 0x3c843c63,
	0xea70ea12, 0xe9b3e955, 0x3c413c1f, 0x3bfc3bd9,
	0xe8f7e899, 0xe83ce7df, 0x3bb53b91, 0x3b6c3b46,
	0xe782e725, 0xe6c9e66c, 0x3b203af9, 0x3ad23aaa,
	0xe610e5b4, 0xe559e4fe, 0x3a813a58, 0x3a2f3a05,
	0xe4a3e448, 0xe3ede393, 0x39da39af, 0x39833957,
	0xe339e2e0, 0xe286e22d, 0x392a38fc, 0x38ce38a0,
	0xe1d4e17c, 0xe124e0cc, 0x38703841, 0x381137e0,
	0xe074e01d, 0xdfc6df6f, 0x37af377d, 0x374a3718,
	0xdf19dec3, 0xde6dde17, 0x36e436b0, 0x367c3647,
	0xddc2dd6e, 0xdd19dcc5, 0x361135db, 0x35a5356d,
	0xdc71dc1e, 0xdbcbdb78, 0x353634fe, 0x34c5348c,
	0xdb26dad4, 0xda82da31, 0x34523418, 0x33de33a2,
	0xd9e0d98f, 0xd93fd8ef, 0x3367332b, 0x32ee32b1,
	0xd8a0d851, 0xd802d7b4, 0x32733235, 0x31f731b8,
	0xd766d718, 0xd6cbd67f, 0x31783138, 0x30f830b7,
	0xd632d5e6, 0xd59bd550, 0x30753033, 0x2ff12fae,
	0xd505d4bb, 0xd471d427, 0x2f6b2f27, 0x2ee32e9e,
	0xd3ded396, 0xd34ed306, 0x2e592e14, 0x2dce2d87,
	0xd2bfd278, 0xd231d1eb, 0x2d402cf9, 0x2cb12c69,
	0xd1a6d161, 0xd11cd0d8, 0x2c212bd8, 0x2b8e2b44,
	0xd094d051, 0xd00ecfcc, 0x2afa2aaf, 0x2a642a19,
	0xcf8acf48, 0xcf07cec7, 0x29cd2980, 0x293428e7,
	0xce87ce47, 0xce08cdca, 0x2899284b, 0x27fd27ae,
	0xcd8ccd4e, 0xcd11ccd4, 0x275f2710, 0x26c02670,
	0xcc98cc5d, 0xcc21cbe7, 0x261f25ce, 0x257d252b,
	0xcbadcb73, 0xcb3acb01, 0x24d92487, 0x243423e1,
	0xcac9ca92, 0xca5aca24, 0x238e233a, 0x22e62291,
	0xc9eec9b8, 0xc983c94f, 0x223d21e8, 0x2192213c,
	0xc91bc8e7, 0xc8b5c882, 0x20e62090, 0x20391fe2,
	0xc850c81f, 0xc7eec7be, 0x1f8b1f33, 0x1edb1e83,
	0xc78fc75f, 0xc731c703, 0x1e2b1dd2, 0x1d791d1f,
	0xc6d5c6a8, 0xc67cc650, 0x1cc61c6c, 0x1c121bb7,
	0xc625c5fa, 0xc5d0c5a7, 0x1b5c1b01, 0x1aa61a4b,
	0xc57ec555, 0xc52dc506, 0x19ef1993, 0x193618da,
	0xc4dfc4b9, 0xc493c46e, 0x187d1820, 0x17c31766,
	0xc44ac426, 0xc403c3e0, 0x170816aa, 0x164c15ed,
	0xc3bec39c, 0xc37bc35b, 0x158f1530, 0x14d11472,
	0xc33bc31c, 0xc2fdc2df, 0x141313b3, 0x135312f3,
	0xc2c1c2a5, 0xc288c26d, 0x12931233, 0x11d31172,
	0xc251c237, 0xc21dc204, 0x111110b0, 0x104f0fee,
	0xc1ebc1d3, 0xc1bbc1a4, 0x0f8c0f2b, 0x0ec90e67,
	0xc18ec178, 0xc163c14f, 0x0e050da3, 0x0d410cde,
	0xc13bc127, 0xc115c103, 0x0c7c0c19, 0x0bb60b53,
	0xc0f1c0e0, 0xc0d0c0c0, 0x0af00a8d, 0x0a2a09c7,
	0xc0b1c0a3, 0xc095c088, 0x09630900, 0x089c0839,
	0xc07bc06f, 0xc064c059, 0x07d50771, 0x070d06a9,
	0xc04fc045, 0xc03cc034, 0x064505e1, 0x057d0519,
	0xc02cc025, 0xc01fc019, 0x04b50450, 0x03ec0388,
	0xc014c00f, 0xc00bc008, 0x032302bf, 0x025b01f6,
	0xc005c003, 0xc001c000, 0x0192012d, 0x00c90064,
};

/* *********************************************************** */

const mlib_u32 mlib_v_fftK_WTable_3D_M[] = {
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x3fffd2bf, 0x00002d40, 0x0000d2bf, 0x3fffd2bf,
	0x3fff187d, 0xd2bfc4df, 0x0000c4df, 0xd2bf187d,
	0x00003b20, 0x2d40e782, 0x3fff187d, 0xd2bfc4df,
	0x3fff3536, 0x187df383, 0x0000dc71, 0xc4dfc13b,
	0xd2bfc13b, 0xc4dfdc71, 0xd2bff383, 0x187d3536,
	0x0000238e, 0x3b203ec4, 0x3fff3536, 0x187df383,
	0x2d400c7c, 0xe782cac9, 0xd2bfc13b, 0xc4dfdc71,
	0x3fff3d3e, 0x35362899, 0x0000ed6c, 0xdc71ce87,
	0x187d0645, 0xf383e1d4, 0xc4dfc04f, 0xc13bc78f,
	0xd2bfc78f, 0xc13bc04f, 0xd2bfe1d4, 0xf3830645,
	0xc4dfce87, 0xdc71ed6c, 0x187d2899, 0x35363d3e,
	0x00001293, 0x238e3178, 0x3fff3d3e, 0x35362899,
	0x3b203fb0, 0x3ec43870, 0x187d0645, 0xf383e1d4,
	0x2d401e2b, 0x0c7cf9ba, 0xd2bfc78f, 0xc13bc04f,
	0xe782d766, 0xcac9c2c1, 0xc4dfce87, 0xdc71ed6c,
	0x3fff3f4e, 0x3d3e39da, 0x0000f69c, 0xed6ce4a3,
	0x35362f6b, 0x289920e6, 0xdc71d505, 0xce87c91b,
	0x187d0f8c, 0x0645fcdc, 0xc4dfc1eb, 0xc04fc014,
	0xf383ea70, 0xe1d4d9e0, 0xc13bc3be, 0xc78fcc98,
	0xd2bfcc98, 0xc78fc3be, 0xd2bfd9e0, 0xe1d4ea70,
	0xc13bc014, 0xc04fc1eb, 0xf383fcdc, 0x06450f8c,
	0xc4dfc91b, 0xce87d505, 0x187d20e6, 0x28992f6b,
	0xdc71e4a3, 0xed6cf69c, 0x353639da, 0x3d3e3f4e,
	0x00000963, 0x12931b5c, 0x3fff3f4e, 0x3d3e39da,
	0x238e2afa, 0x317836e4, 0x35362f6b, 0x289920e6,
	0x3b203e14, 0x3fb03feb, 0x187d0f8c, 0x0645fcdc,
	0x3ec43c41, 0x38703367, 0xf383ea70, 0xe1d4d9e0,
	0x2d40261f, 0x1e2b158f, 0xd2bfcc98, 0xc78fc3be,
	0x0c7c0323, 0xf9baf073, 0xc13bc014, 0xc04fc1eb,
	0xe782df19, 0xd766d094, 0xc4dfc91b, 0xce87d505,
	0xcac9c625, 0xc2c1c0b1, 0xdc71e4a3, 0xed6cf69c,
	0x3fff3fd3, 0x3f4e3e71, 0x0000fb4a, 0xf69cf1fa,
	0x3d3e3bb5, 0x39da37af, 0xed6ce8f7, 0xe4a3e074,
	0x35363273, 0x2f6b2c21, 0xdc71d8a0, 0xd505d1a6,
	0x289924d9, 0x20e61cc6, 0xce87cbad, 0xc91bc6d5,
	0x187d1413, 0x0f8c0af0, 0xc4dfc33b, 0xc1ebc0f1,
	0x06450192, 0xfcdcf82a, 0xc04fc005, 0xc014c07b,
	0xf383eeee, 0xea70e610, 0xc13bc251, 0xc3bec57e,
	0xe1d4ddc2, 0xd9e0d632, 0xc78fc9ee, 0xcc98cf8a,
	0xd2bfcf8a, 0xcc98c9ee, 0xd2bfd632, 0xd9e0ddc2,
	0xc78fc57e, 0xc3bec251, 0xe1d4e610, 0xea70eeee,
	0xc13bc07b, 0xc014c005, 0xf383f82a, 0xfcdc0192,
	0xc04fc0f1, 0xc1ebc33b, 0x06450af0, 0x0f8c1413,
	0xc4dfc6d5, 0xc91bcbad, 0x187d1cc6, 0x20e624d9,
	0xce87d1a6, 0xd505d8a0, 0x28992c21, 0x2f6b3273,
	0xdc71e074, 0xe4a3e8f7, 0x353637af, 0x39da3bb5,
	0xed6cf1fa, 0xf69cfb4a, 0x3d3e3e71, 0x3f4e3fd3,
	0x000004b5, 0x09630e05, 0x3fff3fd3, 0x3f4e3e71,
	0x12931708, 0x1b5c1f8b, 0x3d3e3bb5, 0x39da37af,
	0x238e275f, 0x2afa2e59, 0x35363273, 0x2f6b2c21,
	0x31783452, 0x36e4392a, 0x289924d9, 0x20e61cc6,
	0x3b203cc4, 0x3e143f0e, 0x187d1413, 0x0f8c0af0,
	0x3fb03ffa, 0x3feb3f84, 0x06450192, 0xfcdcf82a,
	0x3ec43dae, 0x3c413a81, 0xf383eeee, 0xea70e610,
	0x38703611, 0x33673075, 0xe1d4ddc2, 0xd9e0d632,
	0x2d4029cd, 0x261f223d, 0xd2bfcf8a, 0xcc98c9ee,
	0x1e2b19ef, 0x158f1111, 0xc78fc57e, 0xc3bec251,
	0x0c7c07d5, 0x0323fe6d, 0xc13bc07b, 0xc014c005,
	0xf9baf50f, 0xf073ebec, 0xc04fc0f1, 0xc1ebc33b,
	0xe782e339, 0xdf19db26, 0xc4dfc6d5, 0xc91bcbad,
	0xd766d3de, 0xd094cd8c, 0xce87d1a6, 0xd505d8a0,
	0xcac9c850, 0xc625c44a, 0xdc71e074, 0xe4a3e8f7,
	0xc2c1c18e, 0xc0b1c02c, 0xed6cf1fa, 0xf69cfb4a,
	0x3fff3ff4, 0x3fd33f9b, 0x0000fda4, 0xfb4af8f2,
	0x3f4e3eea, 0x3e713de2, 0xf69cf449, 0xf1faefb0,
	0x3d3e3c84, 0x3bb53ad2, 0xed6ceb2e, 0xe8f7e6c9,
	0x39da38ce, 0x37af367c, 0xe4a3e286, 0xe074de6d,
	0x353633de, 0x327330f8, 0xdc71da82, 0xd8a0d6cb,
	0x2f6b2dce, 0x2c212a64, 0xd505d34e, 0xd1a6d00e,
	0x289926c0, 0x24d922e6, 0xce87cd11, 0xcbadca5a,
	0x20e61edb, 0x1cc61aa6, 0xc91bc7ee, 0xc6d5c5d0,
	0x187d164c, 0x141311d3, 0xc4dfc403, 0xc33bc288,
	0x0f8c0d41, 0x0af0089c, 0xc1ebc163, 0xc0f1c095,
	0x064503ec, 0x0192ff36, 0xc04fc01f, 0xc005c001,
	0xfcdcfa82, 0xf82af5d5, 0xc014c03c, 0xc07bc0d0,
	0xf383f136, 0xeeeeecac, 0xc13bc1bb, 0xc251c2fd,
	0xea70e83c, 0xe610e3ed, 0xc3bec493, 0xc57ec67c,
	0xe1d4dfc6, 0xddc2dbcb, 0xc78fc8b5, 0xc9eecb3a,
	0xd9e0d802, 0xd632d471, 0xcc98ce08, 0xcf8ad11c,
	0xd2bfd11c, 0xcf8ace08, 0xd2bfd471, 0xd632d802,
	0xcc98cb3a, 0xc9eec8b5, 0xd9e0dbcb, 0xddc2dfc6,
	0xc78fc67c, 0xc57ec493, 0xe1d4e3ed, 0xe610e83c,
	0xc3bec2fd, 0xc251c1bb, 0xea70ecac, 0xeeeef136,
	0xc13bc0d0, 0xc07bc03c, 0xf383f5d5, 0xf82afa82,
	0xc014c001, 0xc005c01f, 0xfcdcff36, 0x019203ec,
	0xc04fc095, 0xc0f1c163, 0x0645089c, 0x0af00d41,
	0xc1ebc288, 0xc33bc403, 0x0f8c11d3, 0x1413164c,
	0xc4dfc5d0, 0xc6d5c7ee, 0x187d1aa6, 0x1cc61edb,
	0xc91bca5a, 0xcbadcd11, 0x20e622e6, 0x24d926c0,
	0xce87d00e, 0xd1a6d34e, 0x28992a64, 0x2c212dce,
	0xd505d6cb, 0xd8a0da82, 0x2f6b30f8, 0x327333de,
	0xdc71de6d, 0xe074e286, 0x3536367c, 0x37af38ce,
	0xe4a3e6c9, 0xe8f7eb2e, 0x39da3ad2, 0x3bb53c84,
	0xed6cefb0, 0xf1faf449, 0x3d3e3de2, 0x3e713eea,
	0xf69cf8f2, 0xfb4afda4, 0x3f4e3f9b, 0x3fd33ff4,
	0x0000025b, 0x04b5070d, 0x3fff3ff4, 0x3fd33f9b,
	0x09630bb6, 0x0e05104f, 0x3f4e3eea, 0x3e713de2,
	0x129314d1, 0x17081936, 0x3d3e3c84, 0x3bb53ad2,
	0x1b5c1d79, 0x1f8b2192, 0x39da38ce, 0x37af367c,
	0x238e257d, 0x275f2934, 0x353633de, 0x327330f8,
	0x2afa2cb1, 0x2e592ff1, 0x2f6b2dce, 0x2c212a64,
	0x317832ee, 0x345235a5, 0x289926c0, 0x24d922e6,
	0x36e43811, 0x392a3a2f, 0x20e61edb, 0x1cc61aa6,
	0x3b203bfc, 0x3cc43d77, 0x187d164c, 0x141311d3,
	0x3e143e9c, 0x3f0e3f6a, 0x0f8c0d41, 0x0af0089c,
	0x3fb03fe0, 0x3ffa3ffe, 0x064503ec, 0x0192ff36,
	0x3feb3fc3, 0x3f843f2f, 0xfcdcfa82, 0xf82af5d5,
	0x3ec43e44, 0x3dae3d02, 0xf383f136, 0xeeeeecac,
	0x3c413b6c, 0x3a813983, 0xea70e83c, 0xe610e3ed,
	0x3870374a, 0x361134c5, 0xe1d4dfc6, 0xddc2dbcb,
	0x336731f7, 0x30752ee3, 0xd9e0d802, 0xd632d471,
	0x2d402b8e, 0x29cd27fd, 0xd2bfd11c, 0xcf8ace08,
	0x261f2434, 0x223d2039, 0xcc98cb3a, 0xc9eec8b5,
	0x1e2b1c12, 0x19ef17c3, 0xc78fc67c, 0xc57ec493,
	0x158f1353, 0x11110ec9, 0xc3bec2fd, 0xc251c1bb,
	0x0c7c0a2a, 0x07d5057d, 0xc13bc0d0, 0xc07bc03c,
	0x032300c9, 0xfe6dfc13, 0xc014c001, 0xc005c01f,
	0xf9baf763, 0xf50ff2be, 0xc04fc095, 0xc0f1c163,
	0xf073ee2c, 0xebece9b3, 0xc1ebc288, 0xc33bc403,
	0xe782e559, 0xe339e124, 0xc4dfc5d0, 0xc6d5c7ee,
	0xdf19dd19, 0xdb26d93f, 0xc91bca5a, 0xcbadcd11,
	0xd766d59b, 0xd3ded231, 0xce87d00e, 0xd1a6d34e,
	0xd094cf07, 0xcd8ccc21, 0xd505d6cb, 0xd8a0da82,
	0xcac9c983, 0xc850c731, 0xdc71de6d, 0xe074e286,
	0xc625c52d, 0xc44ac37b, 0xe4a3e6c9, 0xe8f7eb2e,
	0xc2c1c21d, 0xc18ec115, 0xed6cefb0, 0xf1faf449,
	0xc0b1c064, 0xc02cc00b, 0xf69cf8f2, 0xfb4afda4,
	0x3fff3ffc, 0x3ff43fe6, 0x0000fed2, 0xfda4fc77,
	0x3fd33fba, 0x3f9b3f77, 0xfb4afa1e, 0xf8f2f7c6,
	0x3f4e3f1f, 0x3eea3eb0, 0xf69cf572, 0xf449f321,
	0x3e713e2c, 0x3de23d92, 0xf1faf0d4, 0xefb0ee8d,
	0x3d3e3ce3, 0x3c843c1f, 0xed6cec4c, 0xeb2eea12,
	0x3bb53b46, 0x3ad23a58, 0xe8f7e7df, 0xe6c9e5b4,
	0x39da3957, 0x38ce3841, 0xe4a3e393, 0xe286e17c,
	0x37af3718, 0x367c35db, 0xe074df6f, 0xde6ddd6e,
	0x3536348c, 0x33de332b, 0xdc71db78, 0xda82d98f,
	0x327331b8, 0x30f83033, 0xd8a0d7b4, 0xd6cbd5e6,
	0x2f6b2e9e, 0x2dce2cf9, 0xd505d427, 0xd34ed278,
	0x2c212b44, 0x2a642980, 0xd1a6d0d8, 0xd00ecf48,
	0x289927ae, 0x26c025ce, 0xce87cdca, 0xcd11cc5d,
	0x24d923e1, 0x22e621e8, 0xcbadcb01, 0xca5ac9b8,
	0x20e61fe2, 0x1edb1dd2, 0xc91bc882, 0xc7eec75f,
	0x1cc61bb7, 0x1aa61993, 0xc6d5c650, 0xc5d0c555,
	0x187d1766, 0x164c1530, 0xc4dfc46e, 0xc403c39c,
	0x141312f3, 0x11d310b0, 0xc33bc2df, 0xc288c237,
	0x0f8c0e67, 0x0d410c19, 0xc1ebc1a4, 0xc163c127,
	0x0af009c7, 0x089c0771, 0xc0f1c0c0, 0xc095c06f,
	0x06450519, 0x03ec02bf, 0xc04fc034, 0xc01fc00f,
	0x01920064, 0xff36fe09, 0xc005c000, 0xc001c008,
	0xfcdcfbaf, 0xfa82f956, 0xc014c025, 0xc03cc059,
	0xf82af6ff, 0xf5d5f4ac, 0xc07bc0a3, 0xc0d0c103,
	0xf383f25c, 0xf136f011, 0xc13bc178, 0xc1bbc204,
	0xeeeeedcc, 0xecaceb8d, 0xc251c2a5, 0xc2fdc35b,
	0xea70e955, 0xe83ce725, 0xc3bec426, 0xc493c506,
	0xe610e4fe, 0xe3ede2e0, 0xc57ec5fa, 0xc67cc703,
	0xe1d4e0cc, 0xdfc6dec3, 0xc78fc81f, 0xc8b5c94f,
	0xddc2dcc5, 0xdbcbdad4, 0xc9eeca92, 0xcb3acbe7,
	0xd9e0d8ef, 0xd802d718, 0xcc98cd4e, 0xce08cec7,
	0xd632d550, 0xd471d396, 0xcf8ad051, 0xd11cd1eb,
	0xd2bfd1eb, 0xd11cd051, 0xd2bfd396, 0xd471d550,
	0xcf8acec7, 0xce08cd4e, 0xd632d718, 0xd802d8ef,
	0xcc98cbe7, 0xcb3aca92, 0xd9e0dad4, 0xdbcbdcc5,
	0xc9eec94f, 0xc8b5c81f, 0xddc2dec3, 0xdfc6e0cc,
	0xc78fc703, 0xc67cc5fa, 0xe1d4e2e0, 0xe3ede4fe,
	0xc57ec506, 0xc493c426, 0xe610e725, 0xe83ce955,
	0xc3bec35b, 0xc2fdc2a5, 0xea70eb8d, 0xecacedcc,
	0xc251c204, 0xc1bbc178, 0xeeeef011, 0xf136f25c,
	0xc13bc103, 0xc0d0c0a3, 0xf383f4ac, 0xf5d5f6ff,
	0xc07bc059, 0xc03cc025, 0xf82af956, 0xfa82fbaf,
	0xc014c008, 0xc001c000, 0xfcdcfe09, 0xff360064,
	0xc005c00f, 0xc01fc034, 0x019202bf, 0x03ec0519,
	0xc04fc06f, 0xc095c0c0, 0x06450771, 0x089c09c7,
	0xc0f1c127, 0xc163c1a4, 0x0af00c19, 0x0d410e67,
	0xc1ebc237, 0xc288c2df, 0x0f8c10b0, 0x11d312f3,
	0xc33bc39c, 0xc403c46e, 0x14131530, 0x164c1766,
	0xc4dfc555, 0xc5d0c650, 0x187d1993, 0x1aa61bb7,
	0xc6d5c75f, 0xc7eec882, 0x1cc61dd2, 0x1edb1fe2,
	0xc91bc9b8, 0xca5acb01, 0x20e621e8, 0x22e623e1,
	0xcbadcc5d, 0xcd11cdca, 0x24d925ce, 0x26c027ae,
	0xce87cf48, 0xd00ed0d8, 0x28992980, 0x2a642b44,
	0xd1a6d278, 0xd34ed427, 0x2c212cf9, 0x2dce2e9e,
	0xd505d5e6, 0xd6cbd7b4, 0x2f6b3033, 0x30f831b8,
	0xd8a0d98f, 0xda82db78, 0x3273332b, 0x33de348c,
	0xdc71dd6e, 0xde6ddf6f, 0x353635db, 0x367c3718,
	0xe074e17c, 0xe286e393, 0x37af3841, 0x38ce3957,
	0xe4a3e5b4, 0xe6c9e7df, 0x39da3a58, 0x3ad23b46,
	0xe8f7ea12, 0xeb2eec4c, 0x3bb53c1f, 0x3c843ce3,
	0xed6cee8d, 0xefb0f0d4, 0x3d3e3d92, 0x3de23e2c,
	0xf1faf321, 0xf449f572, 0x3e713eb0, 0x3eea3f1f,
	0xf69cf7c6, 0xf8f2fa1e, 0x3f4e3f77, 0x3f9b3fba,
	0xfb4afc77, 0xfda4fed2, 0x3fd33fe6, 0x3ff43ffc,
	0x0000012d, 0x025b0388, 0x3fff3ffc, 0x3ff43fe6,
	0x04b505e1, 0x070d0839, 0x3fd33fba, 0x3f9b3f77,
	0x09630a8d, 0x0bb60cde, 0x3f4e3f1f, 0x3eea3eb0,
	0x0e050f2b, 0x104f1172, 0x3e713e2c, 0x3de23d92,
	0x129313b3, 0x14d115ed, 0x3d3e3ce3, 0x3c843c1f,
	0x17081820, 0x19361a4b, 0x3bb53b46, 0x3ad23a58,
	0x1b5c1c6c, 0x1d791e83, 0x39da3957, 0x38ce3841,
	0x1f8b2090, 0x21922291, 0x37af3718, 0x367c35db,
	0x238e2487, 0x257d2670, 0x3536348c, 0x33de332b,
	0x275f284b, 0x29342a19, 0x327331b8, 0x30f83033,
	0x2afa2bd8, 0x2cb12d87, 0x2f6b2e9e, 0x2dce2cf9,
	0x2e592f27, 0x2ff130b7, 0x2c212b44, 0x2a642980,
	0x31783235, 0x32ee33a2, 0x289927ae, 0x26c025ce,
	0x345234fe, 0x35a53647, 0x24d923e1, 0x22e621e8,
	0x36e4377d, 0x381138a0, 0x20e61fe2, 0x1edb1dd2,
	0x392a39af, 0x3a2f3aaa, 0x1cc61bb7, 0x1aa61993,
	0x3b203b91, 0x3bfc3c63, 0x187d1766, 0x164c1530,
	0x3cc43d20, 0x3d773dc8, 0x141312f3, 0x11d310b0,
	0x3e143e5b, 0x3e9c3ed8, 0x0f8c0e67, 0x0d410c19,
	0x3f0e3f3f, 0x3f6a3f90, 0x0af009c7, 0x089c0771,
	0x3fb03fcb, 0x3fe03ff0, 0x06450519, 0x03ec02bf,
	0x3ffa3fff, 0x3ffe3ff7, 0x01920064, 0xff36fe09,
	0x3feb3fda, 0x3fc33fa6, 0xfcdcfbaf, 0xfa82f956,
	0x3f843f5c, 0x3f2f3efc, 0xf82af6ff, 0xf5d5f4ac,
	0x3ec43e87, 0x3e443dfb, 0xf383f25c, 0xf136f011,
	0x3dae3d5a, 0x3d023ca4, 0xeeeeedcc, 0xecaceb8d,
	0x3c413bd9, 0x3b6c3af9, 0xea70e955, 0xe83ce725,
	0x3a813a05, 0x398338fc, 0xe610e4fe, 0xe3ede2e0,
	0x387037e0, 0x374a36b0, 0xe1d4e0cc, 0xdfc6dec3,
	0x3611356d, 0x34c53418, 0xddc2dcc5, 0xdbcbdad4,
	0x336732b1, 0x31f73138, 0xd9e0d8ef, 0xd802d718,
	0x30752fae, 0x2ee32e14, 0xd632d550, 0xd471d396,
	0x2d402c69, 0x2b8e2aaf, 0xd2bfd1eb, 0xd11cd051,
	0x29cd28e7, 0x27fd2710, 0xcf8acec7, 0xce08cd4e,
	0x261f252b, 0x2434233a, 0xcc98cbe7, 0xcb3aca92,
	0x223d213c, 0x20391f33, 0xc9eec94f, 0xc8b5c81f,
	0x1e2b1d1f, 0x1c121b01, 0xc78fc703, 0xc67cc5fa,
	0x19ef18da, 0x17c316aa, 0xc57ec506, 0xc493c426,
	0x158f1472, 0x13531233, 0xc3bec35b, 0xc2fdc2a5,
	0x11110fee, 0x0ec90da3, 0xc251c204, 0xc1bbc178,
	0x0c7c0b53, 0x0a2a0900, 0xc13bc103, 0xc0d0c0a3,
	0x07d506a9, 0x057d0450, 0xc07bc059, 0xc03cc025,
	0x032301f6, 0x00c9ff9b, 0xc014c008, 0xc001c000,
	0xfe6dfd40, 0xfc13fae6, 0xc005c00f, 0xc01fc034,
	0xf9baf88e, 0xf763f638, 0xc04fc06f, 0xc095c0c0,
	0xf50ff3e6, 0xf2bef198, 0xc0f1c127, 0xc163c1a4,
	0xf073ef4f, 0xee2ced0c, 0xc1ebc237, 0xc288c2df,
	0xebeceacf, 0xe9b3e899, 0xc33bc39c, 0xc403c46e,
	0xe782e66c, 0xe559e448, 0xc4dfc555, 0xc5d0c650,
	0xe339e22d, 0xe124e01d, 0xc6d5c75f, 0xc7eec882,
	0xdf19de17, 0xdd19dc1e, 0xc91bc9b8, 0xca5acb01,
	0xdb26da31, 0xd93fd851, 0xcbadcc5d, 0xcd11cdca,
	0xd766d67f, 0xd59bd4bb, 0xce87cf48, 0xd00ed0d8,
	0xd3ded306, 0xd231d161, 0xd1a6d278, 0xd34ed427,
	0xd094cfcc, 0xcf07ce47, 0xd505d5e6, 0xd6cbd7b4,
	0xcd8cccd4, 0xcc21cb73, 0xd8a0d98f, 0xda82db78,
	0xcac9ca24, 0xc983c8e7, 0xdc71dd6e, 0xde6ddf6f,
	0xc850c7be, 0xc731c6a8, 0xe074e17c, 0xe286e393,
	0xc625c5a7, 0xc52dc4b9, 0xe4a3e5b4, 0xe6c9e7df,
	0xc44ac3e0, 0xc37bc31c, 0xe8f7ea12, 0xeb2eec4c,
	0xc2c1c26d, 0xc21dc1d3, 0xed6cee8d, 0xefb0f0d4,
	0xc18ec14f, 0xc115c0e0, 0xf1faf321, 0xf449f572,
	0xc0b1c088, 0xc064c045, 0xf69cf7c6, 0xf8f2fa1e,
	0xc02cc019, 0xc00bc003, 0xfb4afc77, 0xfda4fed2,
};

/* *********************************************************** */

const mlib_u32 mlib_v_fftK_WTable_3D_P[] = {
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x3fffd2bf, 0x00002d40, 0x00002d40, 0xc0002d40,
	0x3fff187d, 0xd2bfc4df, 0x00003b20, 0x2d40e782,
	0x00003b20, 0x2d40e782, 0xc000e782, 0x2d403b20,
	0x3fff3536, 0x187df383, 0x0000238e, 0x3b203ec4,
	0xd2bfc13b, 0xc4dfdc71, 0x2d400c7c, 0xe782cac9,
	0x0000238e, 0x3b203ec4, 0xc000cac9, 0xe7820c7c,
	0x2d400c7c, 0xe782cac9, 0x2d403ec4, 0x3b20238e,
	0x3fff3d3e, 0x35362899, 0x00001293, 0x238e3178,
	0x187d0645, 0xf383e1d4, 0x3b203fb0, 0x3ec43870,
	0xd2bfc78f, 0xc13bc04f, 0x2d401e2b, 0x0c7cf9ba,
	0xc4dfce87, 0xdc71ed6c, 0xe782d766, 0xcac9c2c1,
	0x00001293, 0x238e3178, 0xc000c2c1, 0xcac9d766,
	0x3b203fb0, 0x3ec43870, 0xe782f9ba, 0x0c7c1e2b,
	0x2d401e2b, 0x0c7cf9ba, 0x2d403870, 0x3ec43fb0,
	0xe782d766, 0xcac9c2c1, 0x3b203178, 0x238e1293,
	0x3fff3f4e, 0x3d3e39da, 0x00000963, 0x12931b5c,
	0x35362f6b, 0x289920e6, 0x238e2afa, 0x317836e4,
	0x187d0f8c, 0x0645fcdc, 0x3b203e14, 0x3fb03feb,
	0xf383ea70, 0xe1d4d9e0, 0x3ec43c41, 0x38703367,
	0xd2bfcc98, 0xc78fc3be, 0x2d40261f, 0x1e2b158f,
	0xc13bc014, 0xc04fc1eb, 0x0c7c0323, 0xf9baf073,
	0xc4dfc91b, 0xce87d505, 0xe782df19, 0xd766d094,
	0xdc71e4a3, 0xed6cf69c, 0xcac9c625, 0xc2c1c0b1,
	0x00000963, 0x12931b5c, 0xc000c0b1, 0xc2c1c625,
	0x238e2afa, 0x317836e4, 0xcac9d094, 0xd766df19,
	0x3b203e14, 0x3fb03feb, 0xe782f073, 0xf9ba0323,
	0x3ec43c41, 0x38703367, 0x0c7c158f, 0x1e2b261f,
	0x2d40261f, 0x1e2b158f, 0x2d403367, 0x38703c41,
	0x0c7c0323, 0xf9baf073, 0x3ec43feb, 0x3fb03e14,
	0xe782df19, 0xd766d094, 0x3b2036e4, 0x31782afa,
	0xcac9c625, 0xc2c1c0b1, 0x238e1b5c, 0x12930963,
	0x3fff3fd3, 0x3f4e3e71, 0x000004b5, 0x09630e05,
	0x3d3e3bb5, 0x39da37af, 0x12931708, 0x1b5c1f8b,
	0x35363273, 0x2f6b2c21, 0x238e275f, 0x2afa2e59,
	0x289924d9, 0x20e61cc6, 0x31783452, 0x36e4392a,
	0x187d1413, 0x0f8c0af0, 0x3b203cc4, 0x3e143f0e,
	0x06450192, 0xfcdcf82a, 0x3fb03ffa, 0x3feb3f84,
	0xf383eeee, 0xea70e610, 0x3ec43dae, 0x3c413a81,
	0xe1d4ddc2, 0xd9e0d632, 0x38703611, 0x33673075,
	0xd2bfcf8a, 0xcc98c9ee, 0x2d4029cd, 0x261f223d,
	0xc78fc57e, 0xc3bec251, 0x1e2b19ef, 0x158f1111,
	0xc13bc07b, 0xc014c005, 0x0c7c07d5, 0x0323fe6d,
	0xc04fc0f1, 0xc1ebc33b, 0xf9baf50f, 0xf073ebec,
	0xc4dfc6d5, 0xc91bcbad, 0xe782e339, 0xdf19db26,
	0xce87d1a6, 0xd505d8a0, 0xd766d3de, 0xd094cd8c,
	0xdc71e074, 0xe4a3e8f7, 0xcac9c850, 0xc625c44a,
	0xed6cf1fa, 0xf69cfb4a, 0xc2c1c18e, 0xc0b1c02c,
	0x000004b5, 0x09630e05, 0xc000c02c, 0xc0b1c18e,
	0x12931708, 0x1b5c1f8b, 0xc2c1c44a, 0xc625c850,
	0x238e275f, 0x2afa2e59, 0xcac9cd8c, 0xd094d3de,
	0x31783452, 0x36e4392a, 0xd766db26, 0xdf19e339,
	0x3b203cc4, 0x3e143f0e, 0xe782ebec, 0xf073f50f,
	0x3fb03ffa, 0x3feb3f84, 0xf9bafe6d, 0x032307d5,
	0x3ec43dae, 0x3c413a81, 0x0c7c1111, 0x158f19ef,
	0x38703611, 0x33673075, 0x1e2b223d, 0x261f29cd,
	0x2d4029cd, 0x261f223d, 0x2d403075, 0x33673611,
	0x1e2b19ef, 0x158f1111, 0x38703a81, 0x3c413dae,
	0x0c7c07d5, 0x0323fe6d, 0x3ec43f84, 0x3feb3ffa,
	0xf9baf50f, 0xf073ebec, 0x3fb03f0e, 0x3e143cc4,
	0xe782e339, 0xdf19db26, 0x3b20392a, 0x36e43452,
	0xd766d3de, 0xd094cd8c, 0x31782e59, 0x2afa275f,
	0xcac9c850, 0xc625c44a, 0x238e1f8b, 0x1b5c1708,
	0xc2c1c18e, 0xc0b1c02c, 0x12930e05, 0x096304b5,
	0x3fff3ff4, 0x3fd33f9b, 0x0000025b, 0x04b5070d,
	0x3f4e3eea, 0x3e713de2, 0x09630bb6, 0x0e05104f,
	0x3d3e3c84, 0x3bb53ad2, 0x129314d1, 0x17081936,
	0x39da38ce, 0x37af367c, 0x1b5c1d79, 0x1f8b2192,
	0x353633de, 0x327330f8, 0x238e257d, 0x275f2934,
	0x2f6b2dce, 0x2c212a64, 0x2afa2cb1, 0x2e592ff1,
	0x289926c0, 0x24d922e6, 0x317832ee, 0x345235a5,
	0x20e61edb, 0x1cc61aa6, 0x36e43811, 0x392a3a2f,
	0x187d164c, 0x141311d3, 0x3b203bfc, 0x3cc43d77,
	0x0f8c0d41, 0x0af0089c, 0x3e143e9c, 0x3f0e3f6a,
	0x064503ec, 0x0192ff36, 0x3fb03fe0, 0x3ffa3ffe,
	0xfcdcfa82, 0xf82af5d5, 0x3feb3fc3, 0x3f843f2f,
	0xf383f136, 0xeeeeecac, 0x3ec43e44, 0x3dae3d02,
	0xea70e83c, 0xe610e3ed, 0x3c413b6c, 0x3a813983,
	0xe1d4dfc6, 0xddc2dbcb, 0x3870374a, 0x361134c5,
	0xd9e0d802, 0xd632d471, 0x336731f7, 0x30752ee3,
	0xd2bfd11c, 0xcf8ace08, 0x2d402b8e, 0x29cd27fd,
	0xcc98cb3a, 0xc9eec8b5, 0x261f2434, 0x223d2039,
	0xc78fc67c, 0xc57ec493, 0x1e2b1c12, 0x19ef17c3,
	0xc3bec2fd, 0xc251c1bb, 0x158f1353, 0x11110ec9,
	0xc13bc0d0, 0xc07bc03c, 0x0c7c0a2a, 0x07d5057d,
	0xc014c001, 0xc005c01f, 0x032300c9, 0xfe6dfc13,
	0xc04fc095, 0xc0f1c163, 0xf9baf763, 0xf50ff2be,
	0xc1ebc288, 0xc33bc403, 0xf073ee2c, 0xebece9b3,
	0xc4dfc5d0, 0xc6d5c7ee, 0xe782e559, 0xe339e124,
	0xc91bca5a, 0xcbadcd11, 0xdf19dd19, 0xdb26d93f,
	0xce87d00e, 0xd1a6d34e, 0xd766d59b, 0xd3ded231,
	0xd505d6cb, 0xd8a0da82, 0xd094cf07, 0xcd8ccc21,
	0xdc71de6d, 0xe074e286, 0xcac9c983, 0xc850c731,
	0xe4a3e6c9, 0xe8f7eb2e, 0xc625c52d, 0xc44ac37b,
	0xed6cefb0, 0xf1faf449, 0xc2c1c21d, 0xc18ec115,
	0xf69cf8f2, 0xfb4afda4, 0xc0b1c064, 0xc02cc00b,
	0x0000025b, 0x04b5070d, 0xc000c00b, 0xc02cc064,
	0x09630bb6, 0x0e05104f, 0xc0b1c115, 0xc18ec21d,
	0x129314d1, 0x17081936, 0xc2c1c37b, 0xc44ac52d,
	0x1b5c1d79, 0x1f8b2192, 0xc625c731, 0xc850c983,
	0x238e257d, 0x275f2934, 0xcac9cc21, 0xcd8ccf07,
	0x2afa2cb1, 0x2e592ff1, 0xd094d231, 0xd3ded59b,
	0x317832ee, 0x345235a5, 0xd766d93f, 0xdb26dd19,
	0x36e43811, 0x392a3a2f, 0xdf19e124, 0xe339e559,
	0x3b203bfc, 0x3cc43d77, 0xe782e9b3, 0xebecee2c,
	0x3e143e9c, 0x3f0e3f6a, 0xf073f2be, 0xf50ff763,
	0x3fb03fe0, 0x3ffa3ffe, 0xf9bafc13, 0xfe6d00c9,
	0x3feb3fc3, 0x3f843f2f, 0x0323057d, 0x07d50a2a,
	0x3ec43e44, 0x3dae3d02, 0x0c7c0ec9, 0x11111353,
	0x3c413b6c, 0x3a813983, 0x158f17c3, 0x19ef1c12,
	0x3870374a, 0x361134c5, 0x1e2b2039, 0x223d2434,
	0x336731f7, 0x30752ee3, 0x261f27fd, 0x29cd2b8e,
	0x2d402b8e, 0x29cd27fd, 0x2d402ee3, 0x307531f7,
	0x261f2434, 0x223d2039, 0x336734c5, 0x3611374a,
	0x1e2b1c12, 0x19ef17c3, 0x38703983, 0x3a813b6c,
	0x158f1353, 0x11110ec9, 0x3c413d02, 0x3dae3e44,
	0x0c7c0a2a, 0x07d5057d, 0x3ec43f2f, 0x3f843fc3,
	0x032300c9, 0xfe6dfc13, 0x3feb3ffe, 0x3ffa3fe0,
	0xf9baf763, 0xf50ff2be, 0x3fb03f6a, 0x3f0e3e9c,
	0xf073ee2c, 0xebece9b3, 0x3e143d77, 0x3cc43bfc,
	0xe782e559, 0xe339e124, 0x3b203a2f, 0x392a3811,
	0xdf19dd19, 0xdb26d93f, 0x36e435a5, 0x345232ee,
	0xd766d59b, 0xd3ded231, 0x31782ff1, 0x2e592cb1,
	0xd094cf07, 0xcd8ccc21, 0x2afa2934, 0x275f257d,
	0xcac9c983, 0xc850c731, 0x238e2192, 0x1f8b1d79,
	0xc625c52d, 0xc44ac37b, 0x1b5c1936, 0x170814d1,
	0xc2c1c21d, 0xc18ec115, 0x1293104f, 0x0e050bb6,
	0xc0b1c064, 0xc02cc00b, 0x0963070d, 0x04b5025b,
	0x3fff3ffc, 0x3ff43fe6, 0x0000012d, 0x025b0388,
	0x3fd33fba, 0x3f9b3f77, 0x04b505e1, 0x070d0839,
	0x3f4e3f1f, 0x3eea3eb0, 0x09630a8d, 0x0bb60cde,
	0x3e713e2c, 0x3de23d92, 0x0e050f2b, 0x104f1172,
	0x3d3e3ce3, 0x3c843c1f, 0x129313b3, 0x14d115ed,
	0x3bb53b46, 0x3ad23a58, 0x17081820, 0x19361a4b,
	0x39da3957, 0x38ce3841, 0x1b5c1c6c, 0x1d791e83,
	0x37af3718, 0x367c35db, 0x1f8b2090, 0x21922291,
	0x3536348c, 0x33de332b, 0x238e2487, 0x257d2670,
	0x327331b8, 0x30f83033, 0x275f284b, 0x29342a19,
	0x2f6b2e9e, 0x2dce2cf9, 0x2afa2bd8, 0x2cb12d87,
	0x2c212b44, 0x2a642980, 0x2e592f27, 0x2ff130b7,
	0x289927ae, 0x26c025ce, 0x31783235, 0x32ee33a2,
	0x24d923e1, 0x22e621e8, 0x345234fe, 0x35a53647,
	0x20e61fe2, 0x1edb1dd2, 0x36e4377d, 0x381138a0,
	0x1cc61bb7, 0x1aa61993, 0x392a39af, 0x3a2f3aaa,
	0x187d1766, 0x164c1530, 0x3b203b91, 0x3bfc3c63,
	0x141312f3, 0x11d310b0, 0x3cc43d20, 0x3d773dc8,
	0x0f8c0e67, 0x0d410c19, 0x3e143e5b, 0x3e9c3ed8,
	0x0af009c7, 0x089c0771, 0x3f0e3f3f, 0x3f6a3f90,
	0x06450519, 0x03ec02bf, 0x3fb03fcb, 0x3fe03ff0,
	0x01920064, 0xff36fe09, 0x3ffa3fff, 0x3ffe3ff7,
	0xfcdcfbaf, 0xfa82f956, 0x3feb3fda, 0x3fc33fa6,
	0xf82af6ff, 0xf5d5f4ac, 0x3f843f5c, 0x3f2f3efc,
	0xf383f25c, 0xf136f011, 0x3ec43e87, 0x3e443dfb,
	0xeeeeedcc, 0xecaceb8d, 0x3dae3d5a, 0x3d023ca4,
	0xea70e955, 0xe83ce725, 0x3c413bd9, 0x3b6c3af9,
	0xe610e4fe, 0xe3ede2e0, 0x3a813a05, 0x398338fc,
	0xe1d4e0cc, 0xdfc6dec3, 0x387037e0, 0x374a36b0,
	0xddc2dcc5, 0xdbcbdad4, 0x3611356d, 0x34c53418,
	0xd9e0d8ef, 0xd802d718, 0x336732b1, 0x31f73138,
	0xd632d550, 0xd471d396, 0x30752fae, 0x2ee32e14,
	0xd2bfd1eb, 0xd11cd051, 0x2d402c69, 0x2b8e2aaf,
	0xcf8acec7, 0xce08cd4e, 0x29cd28e7, 0x27fd2710,
	0xcc98cbe7, 0xcb3aca92, 0x261f252b, 0x2434233a,
	0xc9eec94f, 0xc8b5c81f, 0x223d213c, 0x20391f33,
	0xc78fc703, 0xc67cc5fa, 0x1e2b1d1f, 0x1c121b01,
	0xc57ec506, 0xc493c426, 0x19ef18da, 0x17c316aa,
	0xc3bec35b, 0xc2fdc2a5, 0x158f1472, 0x13531233,
	0xc251c204, 0xc1bbc178, 0x11110fee, 0x0ec90da3,
	0xc13bc103, 0xc0d0c0a3, 0x0c7c0b53, 0x0a2a0900,
	0xc07bc059, 0xc03cc025, 0x07d506a9, 0x057d0450,
	0xc014c008, 0xc001c000, 0x032301f6, 0x00c9ff9b,
	0xc005c00f, 0xc01fc034, 0xfe6dfd40, 0xfc13fae6,
	0xc04fc06f, 0xc095c0c0, 0xf9baf88e, 0xf763f638,
	0xc0f1c127, 0xc163c1a4, 0xf50ff3e6, 0xf2bef198,
	0xc1ebc237, 0xc288c2df, 0xf073ef4f, 0xee2ced0c,
	0xc33bc39c, 0xc403c46e, 0xebeceacf, 0xe9b3e899,
	0xc4dfc555, 0xc5d0c650, 0xe782e66c, 0xe559e448,
	0xc6d5c75f, 0xc7eec882, 0xe339e22d, 0xe124e01d,
	0xc91bc9b8, 0xca5acb01, 0xdf19de17, 0xdd19dc1e,
	0xcbadcc5d, 0xcd11cdca, 0xdb26da31, 0xd93fd851,
	0xce87cf48, 0xd00ed0d8, 0xd766d67f, 0xd59bd4bb,
	0xd1a6d278, 0xd34ed427, 0xd3ded306, 0xd231d161,
	0xd505d5e6, 0xd6cbd7b4, 0xd094cfcc, 0xcf07ce47,
	0xd8a0d98f, 0xda82db78, 0xcd8cccd4, 0xcc21cb73,
	0xdc71dd6e, 0xde6ddf6f, 0xcac9ca24, 0xc983c8e7,
	0xe074e17c, 0xe286e393, 0xc850c7be, 0xc731c6a8,
	0xe4a3e5b4, 0xe6c9e7df, 0xc625c5a7, 0xc52dc4b9,
	0xe8f7ea12, 0xeb2eec4c, 0xc44ac3e0, 0xc37bc31c,
	0xed6cee8d, 0xefb0f0d4, 0xc2c1c26d, 0xc21dc1d3,
	0xf1faf321, 0xf449f572, 0xc18ec14f, 0xc115c0e0,
	0xf69cf7c6, 0xf8f2fa1e, 0xc0b1c088, 0xc064c045,
	0xfb4afc77, 0xfda4fed2, 0xc02cc019, 0xc00bc003,
	0x0000012d, 0x025b0388, 0xc000c003, 0xc00bc019,
	0x04b505e1, 0x070d0839, 0xc02cc045, 0xc064c088,
	0x09630a8d, 0x0bb60cde, 0xc0b1c0e0, 0xc115c14f,
	0x0e050f2b, 0x104f1172, 0xc18ec1d3, 0xc21dc26d,
	0x129313b3, 0x14d115ed, 0xc2c1c31c, 0xc37bc3e0,
	0x17081820, 0x19361a4b, 0xc44ac4b9, 0xc52dc5a7,
	0x1b5c1c6c, 0x1d791e83, 0xc625c6a8, 0xc731c7be,
	0x1f8b2090, 0x21922291, 0xc850c8e7, 0xc983ca24,
	0x238e2487, 0x257d2670, 0xcac9cb73, 0xcc21ccd4,
	0x275f284b, 0x29342a19, 0xcd8cce47, 0xcf07cfcc,
	0x2afa2bd8, 0x2cb12d87, 0xd094d161, 0xd231d306,
	0x2e592f27, 0x2ff130b7, 0xd3ded4bb, 0xd59bd67f,
	0x31783235, 0x32ee33a2, 0xd766d851, 0xd93fda31,
	0x345234fe, 0x35a53647, 0xdb26dc1e, 0xdd19de17,
	0x36e4377d, 0x381138a0, 0xdf19e01d, 0xe124e22d,
	0x392a39af, 0x3a2f3aaa, 0xe339e448, 0xe559e66c,
	0x3b203b91, 0x3bfc3c63, 0xe782e899, 0xe9b3eacf,
	0x3cc43d20, 0x3d773dc8, 0xebeced0c, 0xee2cef4f,
	0x3e143e5b, 0x3e9c3ed8, 0xf073f198, 0xf2bef3e6,
	0x3f0e3f3f, 0x3f6a3f90, 0xf50ff638, 0xf763f88e,
	0x3fb03fcb, 0x3fe03ff0, 0xf9bafae6, 0xfc13fd40,
	0x3ffa3fff, 0x3ffe3ff7, 0xfe6dff9b, 0x00c901f6,
	0x3feb3fda, 0x3fc33fa6, 0x03230450, 0x057d06a9,
	0x3f843f5c, 0x3f2f3efc, 0x07d50900, 0x0a2a0b53,
	0x3ec43e87, 0x3e443dfb, 0x0c7c0da3, 0x0ec90fee,
	0x3dae3d5a, 0x3d023ca4, 0x11111233, 0x13531472,
	0x3c413bd9, 0x3b6c3af9, 0x158f16aa, 0x17c318da,
	0x3a813a05, 0x398338fc, 0x19ef1b01, 0x1c121d1f,
	0x387037e0, 0x374a36b0, 0x1e2b1f33, 0x2039213c,
	0x3611356d, 0x34c53418, 0x223d233a, 0x2434252b,
	0x336732b1, 0x31f73138, 0x261f2710, 0x27fd28e7,
	0x30752fae, 0x2ee32e14, 0x29cd2aaf, 0x2b8e2c69,
	0x2d402c69, 0x2b8e2aaf, 0x2d402e14, 0x2ee32fae,
	0x29cd28e7, 0x27fd2710, 0x30753138, 0x31f732b1,
	0x261f252b, 0x2434233a, 0x33673418, 0x34c5356d,
	0x223d213c, 0x20391f33, 0x361136b0, 0x374a37e0,
	0x1e2b1d1f, 0x1c121b01, 0x387038fc, 0x39833a05,
	0x19ef18da, 0x17c316aa, 0x3a813af9, 0x3b6c3bd9,
	0x158f1472, 0x13531233, 0x3c413ca4, 0x3d023d5a,
	0x11110fee, 0x0ec90da3, 0x3dae3dfb, 0x3e443e87,
	0x0c7c0b53, 0x0a2a0900, 0x3ec43efc, 0x3f2f3f5c,
	0x07d506a9, 0x057d0450, 0x3f843fa6, 0x3fc33fda,
	0x032301f6, 0x00c9ff9b, 0x3feb3ff7, 0x3ffe3fff,
	0xfe6dfd40, 0xfc13fae6, 0x3ffa3ff0, 0x3fe03fcb,
	0xf9baf88e, 0xf763f638, 0x3fb03f90, 0x3f6a3f3f,
	0xf50ff3e6, 0xf2bef198, 0x3f0e3ed8, 0x3e9c3e5b,
	0xf073ef4f, 0xee2ced0c, 0x3e143dc8, 0x3d773d20,
	0xebeceacf, 0xe9b3e899, 0x3cc43c63, 0x3bfc3b91,
	0xe782e66c, 0xe559e448, 0x3b203aaa, 0x3a2f39af,
	0xe339e22d, 0xe124e01d, 0x392a38a0, 0x3811377d,
	0xdf19de17, 0xdd19dc1e, 0x36e43647, 0x35a534fe,
	0xdb26da31, 0xd93fd851, 0x345233a2, 0x32ee3235,
	0xd766d67f, 0xd59bd4bb, 0x317830b7, 0x2ff12f27,
	0xd3ded306, 0xd231d161, 0x2e592d87, 0x2cb12bd8,
	0xd094cfcc, 0xcf07ce47, 0x2afa2a19, 0x2934284b,
	0xcd8cccd4, 0xcc21cb73, 0x275f2670, 0x257d2487,
	0xcac9ca24, 0xc983c8e7, 0x238e2291, 0x21922090,
	0xc850c7be, 0xc731c6a8, 0x1f8b1e83, 0x1d791c6c,
	0xc625c5a7, 0xc52dc4b9, 0x1b5c1a4b, 0x19361820,
	0xc44ac3e0, 0xc37bc31c, 0x170815ed, 0x14d113b3,
	0xc2c1c26d, 0xc21dc1d3, 0x12931172, 0x104f0f2b,
	0xc18ec14f, 0xc115c0e0, 0x0e050cde, 0x0bb60a8d,
	0xc0b1c088, 0xc064c045, 0x09630839, 0x070d05e1,
	0xc02cc019, 0xc00bc003, 0x04b50388, 0x025b012d,
};

/* *********************************************************** */
