## ----setup, include=FALSE-----------------------------------------------------
knitr::opts_chunk$set(echo = TRUE)

## ----style, echo=FALSE, results='asis'----------------------------------------
BiocStyle::markdown()

## ----install, eval=FALSE------------------------------------------------------
# if (!requireNamespace("BiocManager", quietly = TRUE))
#     install.packages("BiocManager")
# BiocManager::install("SmartPhos")

## ----initialize, warning=FALSE, message=FALSE---------------------------------
library(MultiAssayExperiment)
library(SmartPhos)

## -----------------------------------------------------------------------------
data("dia_example")
dia_example

## -----------------------------------------------------------------------------
se <- dia_example[["Phosphoproteome"]]
colData(se) <- colData(dia_example)
se

## ----message=FALSE------------------------------------------------------------
newSE <- preprocessPhos(seData = se, transform = "log2", normalize = TRUE, 
                        impute = "QRILC")

## -----------------------------------------------------------------------------
plotIntensity(newSE, colorByCol = "replicate")

## -----------------------------------------------------------------------------
plotMissing(newSE)

## -----------------------------------------------------------------------------
# perform PCA
pca <- stats::prcomp(t(assays(newSE)[["imputed"]]), center = TRUE, 
                     scale. = TRUE)
# call the plotting function
p <- plotPCA(pca = pca, se = newSE, color = "replicate")
p

## -----------------------------------------------------------------------------
plotHeatmap(type = "Top variant", newSE, top = 10, 
            annotationCol = c("replicate", "treatment"))

## ----message=FALSE------------------------------------------------------------
dea <- performDifferentialExp(se = newSE, assay = "imputed", method = "limma", 
                              condition = "treatment", reference = "EGF", 
                              target = "1stCrtl")

## -----------------------------------------------------------------------------
dea$seSub
dea$resDE

## -----------------------------------------------------------------------------
plotVolcano(dea$resDE)

## -----------------------------------------------------------------------------
intensityBoxPlot(se = dea$seSub, id = 's447', symbol = "WASL")

## -----------------------------------------------------------------------------
# call addZeroTime function to add zero timepoint to EGF treatment
newSEzero <- addZeroTime(newSE, condition = "treatment", treat = "EGF", 
                         zeroTreat = "1stCrtl", 
                         timeRange = c("10min","100min", "24h"))
# extract the assay
exprMat <- SummarizedExperiment::assay(newSEzero)
# call the clustering function
tsc <- clusterTS(x = exprMat, k = 5)

## -----------------------------------------------------------------------------
tsc$cluster
tsc$plot

## -----------------------------------------------------------------------------
timerange <- unique(newSEzero$timepoint)
plotTimeSeries(newSEzero, type = "expression", geneID = "s40", 
               symbol = "RBM47_T519", condition = "treatment", 
               treatment = "EGF", timerange = timerange)

## ----message=FALSE------------------------------------------------------------
# Load the gene set
genesetPath <- system.file("shiny-app/geneset", package = "SmartPhos")
inGMT <- piano::loadGSC(paste0(genesetPath,"/Cancer_Hallmark.gmt"), type="gmt")
# Call the function
resTab <- enrichDifferential(dea = dea$resDE, type = "Pathway enrichment", 
                             gsaMethod = "PAGE", geneSet = inGMT, 
                             statType = "stat", nPerm = 200, sigLevel = 0.05, 
                             ifFDR = FALSE)
resTab

## ----message=FALSE------------------------------------------------------------
# Load the gene set
genesetPath <- system.file("shiny-app/geneset", package = "SmartPhos")
inGMT <- piano::loadGSC(paste0(genesetPath,
                               "/Chemical_and_Genetic_Perturbations.gmt"), 
                        type="gmt")
# Call the function
clustEnr <- clusterEnrich(clusterTab = tsc$cluster,
                          se = newSE, inputSet = inGMT,
                          filterP = 0.05,
                          ifFDR = FALSE)
clustEnr

## ----message=FALSE------------------------------------------------------------
# Load the ptm set
ptmsetPath <- system.file("shiny-app/ptmset", package = "SmartPhos")
load(paste0(ptmsetPath, "/human_PTM.rda"))
# Call the function
clustEnr <- clusterEnrich(clusterTab = tsc$cluster, se = newSE, 
                          inputSet = ptmSetDb, ptm = TRUE, filterP = 0.05,
                          ifFDR = FALSE)
clustEnr

## ----eval=FALSE---------------------------------------------------------------
# netw <- getDecouplerNetwork("Homo sapiens")
# scoreTab <- calcKinaseScore(dea$resDE, netw, statType = "stat", nPerm = 500)

## -----------------------------------------------------------------------------
sessionInfo()

