## ----setup, include=FALSE-----------------------------------------------------
library(BiocStyle)
# Set standard knitr options
knitr::opts_chunk$set(error=FALSE, message=FALSE, warning=FALSE, tidy = FALSE)

## ----eval = F-----------------------------------------------------------------
# devtools::install_github("BorchLab/Ibex")

## ----eval = F-----------------------------------------------------------------
# if (!require("BiocManager", quietly = TRUE))
#     install.packages("BiocManager")
# 
# BiocManager::install("Ibex")

## ----tidy = FALSE-------------------------------------------------------------
suppressPackageStartupMessages({
  library(bluster)
  library(dplyr)
  library(ggplot2)
  library(Ibex)
  library(kableExtra)
  library(mumosa)
  library(patchwork)
  library(Peptides)
  library(scater)
  library(viridis)
})

## ----echo=FALSE---------------------------------------------------------------
data("ibex_example")
data("ibex_vdj")

## ----tidy = FALSE-------------------------------------------------------------
combined.BCR <- combineExpandedBCR(input.data = list(ibex_vdj),
                                   samples = "Sample1",
                                   filterNonproductive = TRUE)
head(combined.BCR[[1]])[,c(1,11)]

## ----tidy = FALSE-------------------------------------------------------------
model.meta.data <-  read.csv(system.file("extdata", "metadata.csv", 
                                               package = "Ibex"))[,c(1:2,8)]
model.meta.data %>%
  kable("html", escape = FALSE) %>%
  kable_styling(full_width = FALSE) %>%
  scroll_box(width = "100%", height = "400px")


## ----tidy = FALSE-------------------------------------------------------------
Ibex_vectors <- Ibex_matrix(ibex_example, 
                            chain = "Heavy",
                            method = "encoder",
                            encoder.model = "VAE", 
                            encoder.input = "OHE", 
                            species = "Mouse",
                            verbose = FALSE)

ggplot(data = as.data.frame(Ibex_vectors), aes(Ibex_1, Ibex_2)) + 
  geom_point(color = "grey", alpha = 0.7, size = 2) + 
  theme_classic()

Ibex_vectors2 <- Ibex_matrix(ibex_example, 
                             chain = "Heavy",
                             method = "geometric",
                             geometric.theta = pi, 
                             verbose = FALSE)

ggplot(as.data.frame(Ibex_vectors2), aes(x = Ibex_1, y = Ibex_2)) + 
  geom_point(color = "grey", alpha = 0.7, size = 2) + 
  theme_classic()

## ----tidy = FALSE-------------------------------------------------------------
ibex_example <- runIbex(ibex_example, 
                        chain = "Heavy",
                        encoder.input = "kideraFactors", 
                        reduction.name = "Ibex.KF", 
                        species = "Mouse",
                        verbose = FALSE)

## ----tidy = FALSE-------------------------------------------------------------
set.seed(123)
#Generating UMAP from Ibex Neighbors
ibex_example <- runUMAP(ibex_example, 
                        dimred = "Ibex.KF",
                        name = "ibexUMAP")
#Ibex UMAP
plot1 <- plotUMAP(ibex_example, color_by ="Anti-Hen-Egg-Lysozyme", dimred = "ibexUMAP") + 
            theme(legend.position = "bottom")
plot2 <- plotUMAP(ibex_example, color_by = "CTaa", dimred = "ibexUMAP") + 
  scale_color_viridis(discrete = TRUE, option = "B") + 
  guides(color = "none")

plot1 + plot2

## ----tidy = FALSE-------------------------------------------------------------
#Multimodal UMAP
ibex_example <- mumosa::runMultiUMAP(ibex_example, 
                                     dimreds=c("pca", "apca", "Ibex.KF"))
#Multimodal Clustering
output <- rescaleByNeighbors(ibex_example, 
                             dimreds=c("pca", "apca", "Ibex.KF"))
ibex_example$combined.clustering <- clusterRows(output, NNGraphParam())

plot3 <- plotUMAP(ibex_example, 
                  dimred = "MultiUMAP", 
                  color_by = "Anti-Hen-Egg-Lysozyme") + 
            theme(legend.position = "bottom")
plot4 <- plotUMAP(ibex_example, 
                  dimred = "MultiUMAP", 
                  color_by = "combined.clustering") + 
            theme(legend.position = "bottom")
plot5 <- plotUMAP(ibex_example, 
                  dimred = "MultiUMAP", 
                  color_by = "CTaa") + 
  scale_color_manual(values = viridis_pal(option = "B")(length(unique(ibex_example$CTaa)))) +
  guides(color = "none")

plot3 + plot4 + plot5

## ----tidy = FALSE-------------------------------------------------------------
ibex_example <- runUMAP(ibex_example, 
                        dimred = 'pca', 
                        name = "pcaUMAP")

ibex_example <- runUMAP(ibex_example, 
                        dimred = 'apca', 
                        name = "beamUMAP")

plot6 <- plotUMAP(ibex_example, 
                  dimred = "pcaUMAP", 
                  color_by = "combined.clustering") 
plot7 <- plotUMAP(ibex_example, 
                  dimred = "beamUMAP", 
                  color_by = "combined.clustering") 
plot8 <- plotUMAP(ibex_example, 
                  dimred = "ibexUMAP", 
                  color_by = "combined.clustering") 

plot6 + plot7 + plot8 + plot_layout(guides = "collect") &  
  theme(legend.position = "bottom")

## ----tidy = FALSE-------------------------------------------------------------
CoNGA.sce <- CoNGAfy(ibex_example, 
                     method = "mean", 
                     assay = c("RNA", "BEAM"))

CoNGA.sce <- runIbex(CoNGA.sce, 
                     encoder.input = "kideraFactors", 
                     encoder.model = "VAE",
                     reduction.name = "Ibex.KF", 
                     species = "Mouse",
                     verbose = FALSE)

CoNGA.sce <- CoNGA.sce %>%
                  runUMAP(dimred = "Ibex.KF", 
                          name = "ibexUMAP")

plot9 <- plotUMAP(CoNGA.sce, 
                  dimred = "ibexUMAP", 
                  color_by = "Anti-Hen-Egg-Lysozyme", 
                  by.assay.type = "counts") 

plot10 <- plotUMAP(CoNGA.sce, 
                   dimred = "ibexUMAP", 
                   color_by = "H5N1", 
                   by.assay.type = "counts") 

plot9 + plot10 & 
  theme(legend.position = "bottom")

## ----tidy = FALSE-------------------------------------------------------------
sessionInfo()

