## ----setup, echo = FALSE, eval=TRUE, message=FALSE----------------------------
library(BiocStyle)
knitr::opts_knit$set(
  upload.fun = NULL,
  base.url = NULL) # use local files for images
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#"
)
# Override BiocStyle plot hook to avoid css_align issues
knitr::knit_hooks$set(plot = function(x, options) {
  paste0('![', basename(x), '](', x, ')')
})
runchunks = if (Sys.getenv("FORCE_VIGNETTE_REBUILD", "FALSE") == "TRUE") TRUE else FALSE

cache_file <- '../vignette_cache/troubleshooting.RData'
if (!runchunks && file.exists(cache_file)) {
  load(cache_file)
  # If we loaded trimmed objects, reassign them to original names
  if (exists("se.trimmed")) {
    se <- se.trimmed
    rm(se.trimmed)
  }
  if (exists("se4.trimmed")) {
    se4 <- se4.trimmed
    rm(se4.trimmed)
  }
  if (exists("se5.trimmed")) {
    se5 <- se5.trimmed
    rm(se5.trimmed)
  }
} else {
  if (!runchunks) {
    message("Cache file troubleshooting.RData not found - building from scratch")
  }
  runchunks <- TRUE
}
saveout   = runchunks


## ----echo=TRUE, message=FALSE, warning=TRUE, eval=runchunks-------------------
# library(SpiecEasi)
# data(amgut1.filt)
# 
# pargs <- list(seed=10010)
# se3 <- spiec.easi(amgut1.filt, method='mb', lambda.min.ratio=5e-1, nlambda=10, pulsar.params=pargs)

## ----eval=TRUE----------------------------------------------------------------
getOptInd(se3)
sum(getRefit(se3))/2

## ----message=FALSE, warning=FALSE, eval=runchunks-----------------------------
# se4 <- spiec.easi(amgut1.filt, method='mb', lambda.min.ratio=1e-1, nlambda=10, pulsar.params=pargs)

## -----------------------------------------------------------------------------
getStability(se4)
sum(getRefit(se4))/2

## ----message=FALSE, warning=FALSE, eval=runchunks-----------------------------
# se5 <- spiec.easi(amgut1.filt, method='mb', lambda.min.ratio=1e-1, nlambda=100, pulsar.params=pargs)

## ----message=FALSE, warning=FALSE---------------------------------------------
getStability(se5)
sum(getRefit(se5))/2

## ----eval=FALSE---------------------------------------------------------------
# library(parallel)
# cl <- makeCluster(4, type = "SOCK")
# pargs.windows <- list(rep.num=50, seed=10010, cluster=cl)
# se.windows <- spiec.easi(data, method='mb', pulsar.params=pargs.windows)
# stopCluster(cl)

## ----eval=FALSE---------------------------------------------------------------
# # Check stability along lambda path
# getStability(se)
# 
# # Get optimal lambda index
# getOptInd(se)
# 
# # Get summary statistics
# se$select$stars$summary
# 
# # Check network density
# sum(getRefit(se))/2
# 
# # Visualize stability curve
# plot(se$select$stars$summary)
# 
# # Check platform information
# .Platform$OS.type

## -----------------------------------------------------------------------------
sessionInfo()

## ----echo = FALSE, eval=TRUE, message=FALSE-----------------------------------
# Save objects if they exist
if (exists("se") && exists("pargs")) {
  cache_file <- '../vignette_cache/troubleshooting.RData'
  tryCatch({
    # Load trimming function and trim objects to reduce size
    source('../vignette_cache/trim_spiec_easi.R')
    se.trimmed <- trim_spiec_easi(se)
    se4.trimmed <- trim_spiec_easi(se4)
    se5.trimmed <- trim_spiec_easi(se5)
    
    # Save trimmed objects
    save(se.trimmed, se4.trimmed, se5.trimmed, pargs, file=cache_file)
    message("Saved trimmed cache file: ", cache_file, " in directory: ", getwd())
  }, error = function(e) {
    message("Failed to save cache file: ", e$message)
  })
}

