#lang racket/base

;; This module provides abbreviations that are used to construct types
;; and data that show up in types. These are intended for internal use
;; within Typed Racket implementation code.

(require "../utils/utils.rkt"
         "../utils/prefab.rkt"
         "../utils/identifier.rkt"
         racket/list
         syntax/id-set
         racket/match
         (prefix-in c: (contract-req))
         "../rep/rep-utils.rkt"
         "../rep/type-rep.rkt"
         "../rep/type-mask.rkt"
         "../rep/prop-rep.rkt"
         "../rep/object-rep.rkt"
         "../rep/values-rep.rkt"
         "../rep/type-constr.rkt"
         "../rep/free-variance.rkt"
         "../private/user-defined-type-constr.rkt"
         "numeric-tower.rkt"
         (only-in "utils.rkt" self-var imp-var)
         ;; Using this form so all-from-out works
         "base-abbrev.rkt" "match-expanders.rkt"

         ;; signature env req here is so it is statically required by
         ;; the code loaded during typechecking, otherwise we get
         ;; a `reference to a module that is not available` error
         ;; from references generated by init-envs
         "../env/signature-env.rkt"

         (for-syntax racket/base syntax/parse))

(provide (all-defined-out)
         (all-from-out "base-abbrev.rkt" "match-expanders.rkt"
                       "../private/user-defined-type-constr.rkt"))

;; Convenient constructors
(define -App make-App)
(define -has-struct-property make-Has-Struct-Property)

(define -mpair make-MPair)
(define (-Param t1 [t2 t1]) (make-Param t1 t2))
(define -box make-Box)
(define -channel make-Channel)
(define -async-channel make-Async-Channel)
(define -thread-cell make-ThreadCell)
(define -Promise make-Promise)
(define -set make-Set)
(define -treelist make-TreeList)
(define -mvec make-Mutable-Vector)
(define -ivec make-Immutable-Vector)

(define/type-constr (-vec [a : variance:inv])
  (Un (-mvec a) (-ivec a)))

(define make-Vector -vec)
(define -ivec* (make-type-constr make-Immutable-HeterogeneousVector 0 #:kind*? #t #:variances (list variance:co)))
(define -mvec* (make-type-constr make-Mutable-HeterogeneousVector 0 #:kind*? #t #:variances (list variance:inv)))
(define -vec* (make-type-constr make-HeterogeneousVector 0 #:kind*? #t #:variances (list variance:inv)))
(define -Inter (make-type-constr -Inter-fun 0 #f #:kind*? #t #:variances (list variance:co)))
(define -future make-Future)
(define -struct-property make-Struct-Property)
(define -evt make-Evt)
(define -weak-box make-Weak-Box)
(define -CustodianBox make-CustodianBox)
(define -Ephemeron make-Ephemeron)

(define -inst make-Instance)
(define (-prefab key . types)
  (make-Prefab (normalize-prefab-key key (length types)) types))
(define (-prefab-top key field-count)
  (make-PrefabTop (normalize-prefab-key key field-count)))
(define -unit make-Unit)
(define -signature make-Signature)

(define (-seq . args) (make-Sequence args))
(define/cond-contract (-seq-dots args dty dbound)
  (c:-> (c:listof Type?) Type? (c:or/c symbol? c:natural-number/c)
        SequenceDots?)
  (make-SequenceDots args dty dbound))

(define (one-of/c . args)
  (apply Un (map -val args)))

(define/type-constr (-opt [t : variance:co])
  #:productive? #f
  (Un (-val #f) t))

(define (-ne-lst t) (-pair t (-lst t)))

;; Convenient constructor for ValuesDots
;; (wraps arg types with Result)
(define/cond-contract (-values-dots args dty dbound)
  (c:-> (c:listof Type?) Type? (c:or/c symbol? c:natural-number/c)
        ValuesDots?)
  (make-ValuesDots (for/list ([i (in-list args)]) (-result i))
                   dty dbound))

;; Basic types
(define -Self (make-F self-var))
(define -Imp (make-F imp-var))
(define -Listof -lst)
(define -MListof -mlst)
(define/decl -Regexp (Un -PRegexp -Base-Regexp))
(define/decl -Byte-Regexp (Un -Byte-PRegexp -Byte-Base-Regexp))
(define/decl -Pattern (Un -String -Bytes -Regexp -Byte-Regexp))
(define/decl -Module-Path
  (-mu X
       (Un -Symbol -String -Path
           (-lst* (-val 'quote) -Symbol)
           (-lst* (-val 'lib) -String)
           (-lst* (-val 'file) -String)
           (-pair (-val 'planet)
                  (Un (-lst* -Symbol)
                      (-lst* -String)
                      (-lst* -String
                             (-lst*
                              -String -String
                              #:tail (make-Listof
                                      (Un -Nat
                                          (-lst* (Un -Nat (one-of/c '= '+ '-))
                                                 -Nat)))))))
           (-lst* (-val 'submod) X
                  #:tail (-lst (Un -Symbol (-val "..")))))))
(define/decl -Compiled-Expression (Un -Compiled-Module-Expression -Compiled-Non-Module-Expression))
;; in the type (-Syntax t), t is the type of the result of syntax-e, not syntax->datum
(define -Syntax make-Syntax)
(define/decl In-Syntax
  (-mu e
       (Un -Null -Boolean -Symbol -String -Bytes -Keyword -Char -Number
           (make-Vector (-Syntax e))
           (make-Box (-Syntax e))
           (make-Listof (-Syntax e))
           (-pair (-Syntax e) (-Syntax e)))))
(define/decl Any-Syntax (-Syntax In-Syntax))
(define/decl -Stxish (-mu S (Un -Null (-Syntax Univ) (-pair (-Syntax Univ) S))))


(define/type-constr (-Sexpof [t : variance:co])
  (-mu sexp
       (Un -Null
           -Number -Boolean -Symbol -String -Bytes -Keyword -Char
           (-pair sexp sexp)
           (make-Vector sexp)
           (make-Box sexp)
           t)))


(define/decl -Sexp (-Sexpof (Un)))
(define Syntax-Sexp (-Sexpof Any-Syntax))
(define Ident (-Syntax -Symbol))
(define -Mutable-HT make-Mutable-HashTable)
(define -Immutable-HT make-Immutable-HashTable)
(define -Weak-HT make-Weak-HashTable)
(define/type-constr (-HT [a : variance:inv]
                         [b : variance:inv])
  (Un (-Mutable-HT a b) (-Immutable-HT a b) (-Weak-HT a b)))
(define -Prompt-Tagof make-Prompt-Tagof)
(define -Continuation-Mark-Keyof make-Continuation-Mark-Keyof)

(define/decl -Flat
  (-mu flat
       (Un -Null -Number -Boolean -Symbol -String -Bytes -Keyword -Char
           (-pair flat flat)
           (-ivec flat)
           (-Immutable-HT flat flat))))

(define make-HashTable -HT)
(define/decl -Port (Un -Output-Port -Input-Port))
(define/decl -SomeSystemPath (Un -Path -OtherSystemPath))
(define/decl -Pathlike (Un -String -Path))
(define/decl -SomeSystemPathlike (Un -String -SomeSystemPath))
(define/decl -Pathlike* (Un -String -Path (-val 'up) (-val 'same)))
(define/decl -SomeSystemPathlike*
  (Un -String -SomeSystemPath(-val 'up) (-val 'same)))
(define/decl -PathConventionType (Un (-val 'unix) (-val 'windows)))
(define/decl -Log-Level (one-of/c 'fatal 'error 'warning 'info 'debug))
(define/decl -Place-Channel (Un -Place -Base-Place-Channel))

;; note, these are number? #f
(define/decl -ExtFlonumZero (Un -ExtFlonumPosZero -ExtFlonumNegZero -ExtFlonumNan))
(define/decl -PosExtFlonum (Un -PosExtFlonumNoNan -ExtFlonumNan))
(define/decl -NonNegExtFlonum (Un -PosExtFlonum -ExtFlonumZero))
(define/decl -NonNegSignExtFlonum (Un -PosExtFlonum -ExtFlonumPosZero))
(define/decl -NegExtFlonum (Un -NegExtFlonumNoNan -ExtFlonumNan))
(define/decl -NonPosExtFlonum (Un -NegExtFlonum -ExtFlonumZero))
(define/decl -ExtFlonum (Un -NegExtFlonumNoNan -ExtFlonumNegZero -ExtFlonumPosZero -PosExtFlonumNoNan -ExtFlonumNan))

(define/decl -Struct-Type-Property (-struct-property Univ #f))

;; Type alias names
(define (-struct-name name)
  (make-Name name 0 #t))


;; Structs
(define (-struct name parent flds [proc #f] [poly #f] [pred #'dummy] [props (immutable-free-id-set)])
  (make-Struct name parent flds proc poly pred props))

;; Function type constructors
(define/decl top-func (make-Fun (list)))

(define (asym-pred dom rng prop [T+ #f])
  (make-Fun (list (-Arrow (list dom) rng #:props prop #:T+ T+))))

(define (unsafe-shallow:asym-pred dom rng prop)
  (asym-pred dom rng prop #true))

(define/cond-contract make-pred-ty
  (c:case-> (c:-> Type? Type?)
            (c:-> (c:listof Type?) Type? Type? Type?)
            (c:-> (c:listof Type?) Type? Type? Object? Type?)
            (c:-> (c:listof Type?) Type? Type? Object? boolean? Type?))
  (case-lambda
    [(in out t o T+)
     (->* in out : (-PS (-is-type o t) (-not-type o t)) :T+ T+)]
    [(in out t o)
     (make-pred-ty in out t o #f)]
    [(in out t)
     (make-pred-ty in out t (make-Path null (cons 0 0)) #f)]
    [(t)
     (make-pred-ty (list Univ) -Boolean t (make-Path null (cons 0 0)) #f)]))

(define unsafe-shallow:make-pred-ty
  (case-lambda
    [(in out t o)
     (make-pred-ty in out t o #true)]
    [(in out t)
     (make-pred-ty in out t (make-Path null (cons 0 0)) #true)]
    [(t)
     (make-pred-ty (list Univ) -Boolean t (make-Path null (cons 0 0)) #true)]))

(define/decl -true-propset (-PS -tt -ff))
(define/decl -false-propset (-PS -ff -tt))

(define (opt-fn args opt-args result #:rest [rest #f] #:kws [kws null] #:T+ [T+ #f])
  (apply cl->* (for/list ([i (in-range (add1 (length opt-args)))])
                 (make-Fun (list (-Arrow (append args (take opt-args i))
                                         result ;; only the LAST arrow gets the rest arg
                                         #:rest (and (= i (length opt-args)) rest)
                                         #:kws kws
                                         #:T+ T+))))))

(define-syntax (->opt stx)
  (syntax-parse stx
    ((->opt args ... [opt ...] res T+:rng-T+)
     (syntax/loc stx (opt-fn (list args ...) (list opt ...) res #:T+ T+.val)))
    ((->opt args ... [opt ...] res)
     #:with T+ #`#,(default-T+)
     (syntax/loc stx (opt-fn (list args ...) (list opt ...) res #:T+ 'T+)))))

;; from define-new-subtype
(define (-Distinction name sym ty)
  (make-Distinction name sym ty))

;; class utilities

(begin-for-syntax
 (define-syntax-class names+types
   #:attributes (data)
   (pattern [(name:id type) ...]
            #:with data #'(list (list (quote name) type) ...)))

 (define-syntax-class names+types+opt
   #:attributes (data no-opts)
   (pattern [(name:id type opt?) ...]
            #:with data #'(list (list (quote name) type opt?) ...)
            #:with no-opts #'(list (list (quote name) type) ...)))

 (define-splicing-syntax-class -class-clause
   #:attributes (inits fields methods augments)
   (pattern (~seq #:init sub-clauses:names+types+opt)
            #:with inits #'sub-clauses.data
            #:with fields #'null
            #:with methods #'null
            #:with augments #'null)
   (pattern (~seq #:init-field sub-clauses:names+types+opt)
            #:with inits #'sub-clauses.data
            #:with fields #'sub-clauses.no-opts
            #:with methods #'null
            #:with augments #'null)
   (pattern (~seq #:method sub-clauses:names+types)
            #:with inits #'null
            #:with fields #'null
            #:with methods #'sub-clauses.data
            #:with augments #'null)
   (pattern (~seq #:field sub-clauses:names+types)
            #:with inits #'null
            #:with fields #'sub-clauses.data
            #:with methods #'null
            #:with augments #'null)
   (pattern (~seq #:augment sub-clauses:names+types)
            #:with inits #'null
            #:with fields #'null
            #:with methods #'null
            #:with augments #'sub-clauses.data)))

(define-syntax (-class stx)
  (syntax-parse stx
    [(_ (~or (~optional (~seq #:row var:expr)
                        #:defaults ([var #'#f]))
             ?clause:-class-clause) ...)
     #'(make-Class
        var
        (append ?clause.inits ...)
        (append ?clause.fields ...)
        (append ?clause.methods ...)
        (append ?clause.augments ...)
        #f)]))

(define-syntax-rule (-object . ?clauses)
  (make-Instance (-class . ?clauses)))


(define-syntax (-refine/fresh stx)
  (syntax-parse stx
    [(_ x:id t p)
     (syntax/loc stx
       (let ([x (genid (syntax->datum #'x))])
         (-refine t (abstract-obj p (list x)))))]))

;; extract-props : Object Type -> (values Type (listof Prop?))
;; given the fact that 'obj' is of type 'type',
;; look inside of type trying to learn
;; more info about obj
(define (extract-props obj type)
  (cond
    [(Empty? obj) (values type '())]
    [else
     (define props '())
     (define new-type
       (let extract ([rep type]
                     [obj obj])
         (match rep
           [(app int-type->known-bounds
                 (cons maybe-lower-bound maybe-upper-bound))
            #:when (with-refinements?)
            (when maybe-lower-bound
              (set! props (cons (-leq (-lexp maybe-lower-bound) (-lexp obj))
                                props)))
            (when maybe-upper-bound
              (set! props (cons (-leq (-lexp obj) (-lexp maybe-upper-bound))
                                props)))
            rep]
           [(Pair: t1 t2) (make-Pair (extract t1 (-car-of obj))
                                     (extract t2 (-cdr-of obj)))]
           [(Refine-obj: obj t prop)
            (set! props (cons prop props))
            (extract t obj)]
           [(HeterogeneousVector: ts)
            #:when (with-refinements?)
            (set! props (cons (-eq (-vec-len-of obj) (-lexp (length ts)))
                              props))
            rep]
           [_ #:when (and (with-refinements?)
                          (or (eqv? mask:immutable-vector (mask rep))
                              (eqv? mask:mutable-vector (mask rep))
                              (eqv? mask:vector (mask rep))))
              (set! props (cons (-leq (-lexp 0) (-vec-len-of obj))
                                props))
              rep]
           [(Intersection: ts _)
            (apply -unsafe-intersect
                   (for/list ([t (in-list ts)])
                     (extract t obj)))]
           [_ rep])))
     (values new-type props)]))



(define (-list-or-set s) (Un (-lst s) (-set s)))

;; Since generics are not yet supported, we currently overload
;; the set operations to work both on list sets and hash sets.
;; This helper makes those types in the base-env less verbose.
(define-syntax-rule (set-abs s e)
  (cl->*
   (let ([s -set]) e)
   (let ([s -lst]) e)))
