/*
 * Decompiled with CFR 0.152.
 */
package org.tribuo.transform.transformations;

import com.oracle.labs.mlrg.olcut.config.Config;
import com.oracle.labs.mlrg.olcut.provenance.ObjectProvenance;
import com.oracle.labs.mlrg.olcut.provenance.Provenance;
import com.oracle.labs.mlrg.olcut.provenance.primitives.EnumProvenance;
import com.oracle.labs.mlrg.olcut.provenance.primitives.IntProvenance;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.logging.Logger;
import org.tribuo.transform.TransformStatistics;
import org.tribuo.transform.Transformation;
import org.tribuo.transform.TransformationProvenance;
import org.tribuo.transform.Transformer;

public final class BinningTransformation
implements Transformation {
    private static final String NUM_BINS = "numBins";
    private static final String TYPE = "type";
    @Config(description="Number of bins.")
    private int numBins;
    @Config(description="Binning algorithm to use.")
    private BinningType type;

    private BinningTransformation() {
    }

    private BinningTransformation(BinningType type, int numBins) {
        this.type = type;
        this.numBins = numBins;
        this.postConfig();
    }

    public void postConfig() {
        if (this.numBins < 2) {
            throw new IllegalArgumentException("Number of bins must be 2 or greater, found " + this.numBins);
        }
        if (this.type == BinningType.STD_DEVS && (this.numBins & 1) == 1) {
            throw new IllegalArgumentException("Std dev must have an even number of bins, found " + this.numBins);
        }
    }

    @Override
    public TransformStatistics createStats() {
        switch (this.type) {
            case EQUAL_WIDTH: {
                return new EqualWidthStats(this.numBins);
            }
            case EQUAL_FREQUENCY: {
                return new EqualFreqStats(this.numBins);
            }
            case STD_DEVS: {
                return new StdDevStats(this.numBins);
            }
        }
        throw new IllegalStateException("Unknown binning type " + (Object)((Object)this.type));
    }

    public TransformationProvenance getProvenance() {
        return new BinningTransformationProvenance(this);
    }

    public String toString() {
        return "BinningTransformation(type=" + (Object)((Object)this.type) + ",numBins=" + this.numBins + ")";
    }

    public static BinningTransformation equalWidth(int numBins) {
        return new BinningTransformation(BinningType.EQUAL_WIDTH, numBins);
    }

    public static BinningTransformation equalFrequency(int numBins) {
        return new BinningTransformation(BinningType.EQUAL_FREQUENCY, numBins);
    }

    public static BinningTransformation stdDevs(int numDeviations) {
        return new BinningTransformation(BinningType.STD_DEVS, numDeviations * 2);
    }

    private static class BinningTransformer
    implements Transformer {
        private static final long serialVersionUID = 1L;
        private final BinningType type;
        private final double[] bins;
        private final double[] values;

        public BinningTransformer(BinningType type, double[] bins, double[] values) {
            this.type = type;
            this.bins = bins;
            this.values = values;
        }

        @Override
        public double transform(double input) {
            if (input > this.bins[this.bins.length - 1]) {
                return this.values[this.bins.length - 1];
            }
            int index = Arrays.binarySearch(this.bins, input);
            if (index < 0) {
                return this.values[-1 - index];
            }
            return this.values[index];
        }

        public String toString() {
            return "BinningTransformer(type=" + (Object)((Object)this.type) + ",bins=" + Arrays.toString(this.bins) + ",values=" + Arrays.toString(this.values) + ")";
        }
    }

    private static class StdDevStats
    implements TransformStatistics {
        private static final Logger logger = Logger.getLogger(StdDevStats.class.getName());
        private final int numBins;
        private double mean = 0.0;
        private double sumSquares = 0.0;
        private long count = 0L;

        public StdDevStats(int numBins) {
            this.numBins = numBins;
        }

        @Override
        public void observeValue(double value) {
            ++this.count;
            double delta = value - this.mean;
            this.mean += delta / (double)this.count;
            double delta2 = value - this.mean;
            this.sumSquares += delta * delta2;
        }

        @Override
        @Deprecated
        public void observeSparse() {
            this.observeValue(0.0);
        }

        @Override
        public void observeSparse(int sparseCount) {
            this.count += (long)sparseCount;
            double delta = -this.mean;
            this.mean += delta;
            double delta2 = -this.mean;
            this.sumSquares += (double)sparseCount * (delta * delta2);
        }

        @Override
        public Transformer generateTransformer() {
            if (this.sumSquares == 0.0) {
                logger.info("Only observed a single value (" + this.mean + ") when building a BinningTransformer using standard deviation bins.");
            }
            double[] bins = new double[this.numBins];
            double[] values = new double[this.numBins];
            double stdDev = Math.sqrt(this.sumSquares / (double)(this.count - 1L));
            int binCount = -(this.numBins / 2);
            for (int i = 0; i < bins.length; ++i) {
                values[i] = i + 1;
                bins[i] = this.mean + (double)(++binCount) * stdDev;
            }
            return new BinningTransformer(BinningType.STD_DEVS, bins, values);
        }

        public String toString() {
            return "StdDevStatistics(mean=" + this.mean + ",sumSquares=" + this.sumSquares + ",count=" + this.count + ",numBins=" + this.numBins + ")";
        }
    }

    private static class EqualFreqStats
    implements TransformStatistics {
        private static final int DEFAULT_SIZE = 50;
        private final int numBins;
        private double[] observedValues;
        private int count;

        public EqualFreqStats(int numBins) {
            this.numBins = numBins;
            this.observedValues = new double[50];
            this.count = 0;
        }

        @Override
        public void observeValue(double value) {
            if (this.observedValues.length == this.count + 1) {
                this.growArray();
            }
            this.observedValues[this.count] = value;
            ++this.count;
        }

        protected void growArray(int minCapacity) {
            int newCapacity = this.newCapacity(minCapacity);
            this.observedValues = Arrays.copyOf(this.observedValues, newCapacity);
        }

        private int newCapacity(int minCapacity) {
            int oldCapacity = this.observedValues.length;
            int newCapacity = oldCapacity + (oldCapacity >> 1);
            if (newCapacity - minCapacity <= 0) {
                if (minCapacity < 0) {
                    throw new OutOfMemoryError();
                }
                return minCapacity;
            }
            return newCapacity;
        }

        protected void growArray() {
            this.growArray(this.count + 1);
        }

        @Override
        @Deprecated
        public void observeSparse() {
            this.observeValue(0.0);
        }

        @Override
        public void observeSparse(int sparseCount) {
            if (this.observedValues.length < this.count + sparseCount) {
                this.growArray(this.count + sparseCount);
            }
            this.count += sparseCount;
        }

        @Override
        public Transformer generateTransformer() {
            if (this.numBins > this.observedValues.length) {
                throw new IllegalStateException("Needs more values than bins, requested " + this.numBins + " bins, but only found " + this.observedValues.length + " values.");
            }
            Arrays.sort(this.observedValues, 0, this.count);
            double[] bins = new double[this.numBins];
            double[] values = new double[this.numBins];
            int increment = this.count / this.numBins;
            for (int i = 0; i < this.numBins - 1; ++i) {
                bins[i] = this.observedValues[(i + 1) * increment];
                values[i] = i + 1;
            }
            bins[this.numBins - 1] = this.observedValues[this.count - 1];
            values[this.numBins - 1] = this.numBins;
            return new BinningTransformer(BinningType.EQUAL_FREQUENCY, bins, values);
        }

        public String toString() {
            return "EqualFreqStatistics(count=" + this.count + ",numBins=" + this.numBins + ")";
        }
    }

    private static class EqualWidthStats
    implements TransformStatistics {
        private final int numBins;
        private double min = Double.POSITIVE_INFINITY;
        private double max = Double.NEGATIVE_INFINITY;

        public EqualWidthStats(int numBins) {
            this.numBins = numBins;
        }

        @Override
        public void observeValue(double value) {
            if (value < this.min) {
                this.min = value;
            }
            if (value > this.max) {
                this.max = value;
            }
        }

        @Override
        @Deprecated
        public void observeSparse() {
            this.observeValue(0.0);
        }

        @Override
        public void observeSparse(int count) {
            this.observeValue(0.0);
        }

        @Override
        public Transformer generateTransformer() {
            double range = Math.abs(this.max - this.min);
            double increment = range / (double)this.numBins;
            double[] bins = new double[this.numBins];
            double[] values = new double[this.numBins];
            for (int i = 0; i < bins.length; ++i) {
                bins[i] = this.min + (double)(i + 1) * increment;
                values[i] = i + 1;
            }
            return new BinningTransformer(BinningType.EQUAL_WIDTH, bins, values);
        }

        public String toString() {
            return "EqualWidthStatistics(min=" + this.min + ",max=" + this.max + ",numBins=" + this.numBins + ")";
        }
    }

    public static final class BinningTransformationProvenance
    implements TransformationProvenance {
        private static final long serialVersionUID = 1L;
        private final IntProvenance numBins;
        private final EnumProvenance<BinningType> type;

        BinningTransformationProvenance(BinningTransformation host) {
            this.numBins = new IntProvenance(BinningTransformation.NUM_BINS, host.numBins);
            this.type = new EnumProvenance(BinningTransformation.TYPE, (Enum)host.type);
        }

        public BinningTransformationProvenance(Map<String, Provenance> map) {
            this.numBins = (IntProvenance)ObjectProvenance.checkAndExtractProvenance(map, (String)BinningTransformation.NUM_BINS, IntProvenance.class, (String)BinningTransformationProvenance.class.getSimpleName());
            this.type = (EnumProvenance)ObjectProvenance.checkAndExtractProvenance(map, (String)BinningTransformation.TYPE, EnumProvenance.class, (String)BinningTransformationProvenance.class.getSimpleName());
        }

        public String getClassName() {
            return BinningTransformation.class.getName();
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof BinningTransformationProvenance)) {
                return false;
            }
            BinningTransformationProvenance pairs = (BinningTransformationProvenance)o;
            return this.numBins.equals((Object)pairs.numBins) && this.type.equals(pairs.type);
        }

        public int hashCode() {
            return Objects.hash(this.numBins, this.type);
        }

        public Map<String, Provenance> getConfiguredParameters() {
            HashMap<String, Object> map = new HashMap<String, Object>();
            map.put(BinningTransformation.NUM_BINS, this.numBins);
            map.put(BinningTransformation.TYPE, this.type);
            return Collections.unmodifiableMap(map);
        }
    }

    public static enum BinningType {
        EQUAL_WIDTH,
        EQUAL_FREQUENCY,
        STD_DEVS;

    }
}

