﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/fsx/FSxRequest.h>
#include <aws/fsx/FSx_EXPORTS.h>
#include <aws/fsx/model/DeleteVolumeOntapConfiguration.h>
#include <aws/fsx/model/DeleteVolumeOpenZFSConfiguration.h>

#include <utility>

namespace Aws {
namespace FSx {
namespace Model {

/**
 */
class DeleteVolumeRequest : public FSxRequest {
 public:
  AWS_FSX_API DeleteVolumeRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteVolume"; }

  AWS_FSX_API Aws::String SerializePayload() const override;

  AWS_FSX_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{

  inline const Aws::String& GetClientRequestToken() const { return m_clientRequestToken; }
  inline bool ClientRequestTokenHasBeenSet() const { return m_clientRequestTokenHasBeenSet; }
  template <typename ClientRequestTokenT = Aws::String>
  void SetClientRequestToken(ClientRequestTokenT&& value) {
    m_clientRequestTokenHasBeenSet = true;
    m_clientRequestToken = std::forward<ClientRequestTokenT>(value);
  }
  template <typename ClientRequestTokenT = Aws::String>
  DeleteVolumeRequest& WithClientRequestToken(ClientRequestTokenT&& value) {
    SetClientRequestToken(std::forward<ClientRequestTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the volume that you are deleting.</p>
   */
  inline const Aws::String& GetVolumeId() const { return m_volumeId; }
  inline bool VolumeIdHasBeenSet() const { return m_volumeIdHasBeenSet; }
  template <typename VolumeIdT = Aws::String>
  void SetVolumeId(VolumeIdT&& value) {
    m_volumeIdHasBeenSet = true;
    m_volumeId = std::forward<VolumeIdT>(value);
  }
  template <typename VolumeIdT = Aws::String>
  DeleteVolumeRequest& WithVolumeId(VolumeIdT&& value) {
    SetVolumeId(std::forward<VolumeIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>For Amazon FSx for ONTAP volumes, specify whether to take a final backup of
   * the volume and apply tags to the backup. To apply tags to the backup, you must
   * have the <code>fsx:TagResource</code> permission.</p>
   */
  inline const DeleteVolumeOntapConfiguration& GetOntapConfiguration() const { return m_ontapConfiguration; }
  inline bool OntapConfigurationHasBeenSet() const { return m_ontapConfigurationHasBeenSet; }
  template <typename OntapConfigurationT = DeleteVolumeOntapConfiguration>
  void SetOntapConfiguration(OntapConfigurationT&& value) {
    m_ontapConfigurationHasBeenSet = true;
    m_ontapConfiguration = std::forward<OntapConfigurationT>(value);
  }
  template <typename OntapConfigurationT = DeleteVolumeOntapConfiguration>
  DeleteVolumeRequest& WithOntapConfiguration(OntapConfigurationT&& value) {
    SetOntapConfiguration(std::forward<OntapConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>For Amazon FSx for OpenZFS volumes, specify whether to delete all child
   * volumes and snapshots.</p>
   */
  inline const DeleteVolumeOpenZFSConfiguration& GetOpenZFSConfiguration() const { return m_openZFSConfiguration; }
  inline bool OpenZFSConfigurationHasBeenSet() const { return m_openZFSConfigurationHasBeenSet; }
  template <typename OpenZFSConfigurationT = DeleteVolumeOpenZFSConfiguration>
  void SetOpenZFSConfiguration(OpenZFSConfigurationT&& value) {
    m_openZFSConfigurationHasBeenSet = true;
    m_openZFSConfiguration = std::forward<OpenZFSConfigurationT>(value);
  }
  template <typename OpenZFSConfigurationT = DeleteVolumeOpenZFSConfiguration>
  DeleteVolumeRequest& WithOpenZFSConfiguration(OpenZFSConfigurationT&& value) {
    SetOpenZFSConfiguration(std::forward<OpenZFSConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clientRequestToken{Aws::Utils::UUID::PseudoRandomUUID()};

  Aws::String m_volumeId;

  DeleteVolumeOntapConfiguration m_ontapConfiguration;

  DeleteVolumeOpenZFSConfiguration m_openZFSConfiguration;
  bool m_clientRequestTokenHasBeenSet = true;
  bool m_volumeIdHasBeenSet = false;
  bool m_ontapConfigurationHasBeenSet = false;
  bool m_openZFSConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace FSx
}  // namespace Aws
